/*
 *  ncplib.h
 *
 *  Copyright (C) 1995, 1996 by Volker Lendecke
 *
 */

#ifndef _NCPLIB_H
#define _NCPLIB_H

#include <linux/types.h>
#include <linux/ncp.h>
#include <linux/ncp_fs.h>
#include <linux/ipx.h>
#include <sys/param.h>
#include <stdio.h>
#include <time.h>

#include "ipxlib.h"

#ifndef memzero
#include <string.h>
#define memzero(object) memset(&(object), 0, sizeof(object))
#endif

void
str_upper(char *name);

enum connect_state {
	NOT_CONNECTED = 0,
	CONN_PERMANENT,
	CONN_TEMPORARY
};

struct ncp_conn {

	enum connect_state is_connected;

	char server[NCP_BINDERY_NAME_LEN];
	char user  [NCP_BINDERY_NAME_LEN];

	struct ncp_fs_info i;

	/* Fields for use with permanent connections */
	int mount_fid;
	char mount_point[MAXPATHLEN];

	/* Fields for use with temporary connections */
	int ncp_sock;
	int wdog_sock;
	int wdog_pid;
	__u8 sequence;
	int completion;
	int conn_status;
	int reply_size;

	/* Fields used to setup ncp requests */
	int current_size;
	int has_subfunction;
	int verbose;
	int ncp_reply_size;

	int lock;

	char packet[NCP_PACKET_SIZE];
};

struct ncp_conn_spec {
	char server[NCP_BINDERY_NAME_LEN];
	char user[NCP_BINDERY_NAME_LEN];
	uid_t uid;
	int login_type;		/* NCP_BINDERY_USER / NCP_BINDERY_PSERVER */
	char password[NCP_BINDERY_NAME_LEN];
};

struct ncp_search_seq {
	struct nw_search_sequence s;
	int namespace;
};

/* ncp_initialize is the main entry point for user programs which want
   to connect to a NetWare Server. It looks for -S, -U, -P and -n in
   the argument list, opens the connection and removes the arguments
   from the list. It was designed after the X Windows init
   functions. */
int
ncp_initialize(struct ncp_conn *conn,
	       int *argc, char **argv,
	       int login_necessary);

/* You can login as another object by this procedure. As a first use
   pserver comes to mind. */
int
ncp_initialize_as(struct ncp_conn *conn,
		  int *argc, char **argv,
		  int login_necessary, int login_type);


/* Open an existing permanent connection */
int
ncp_open(struct ncp_conn *conn,
	 const struct ncp_conn_spec *spec);

/* Open a connection on an existing mount point */
int
ncp_open_mount(struct ncp_conn *conn,
	       const char *mount_point);

/* Detach from and destroy a permanent connection */
int
ncp_destroy_permanent(struct ncp_conn *conn);

/* Create a temporary connection */
int
ncp_open_temporary(struct ncp_conn *conn,
		   const struct ncp_conn_spec *spec);

/* Detach from a permanent connection or destroy a temporary
   connection */
int
ncp_close(struct ncp_conn *conn);

/* like getmntent, get_ncp_conn_ent scans /etc/mtab for usable
   connections */

struct ncp_conn_ent {
	char server[NCP_BINDERY_NAME_LEN];
	char user[NCP_BINDERY_NAME_LEN];
	uid_t uid;
	char mount_point[MAXPATHLEN];
};

struct ncp_conn_ent *
ncp_get_conn_ent(FILE *filep);

#define NWCLIENT (".nwclient")
#define NWC_NOPASSWORD ("-")

/* find an appropriate connection */

struct ncp_conn_spec *
ncp_find_conn_spec(const char *server, const char *user, const char *password,
		   uid_t uid);

int
ncp_get_file_server_description_strings(struct ncp_conn *conn,
					char target[512]);

int
ncp_get_file_server_time(struct ncp_conn *conn, time_t *target);

int
ncp_get_connlist(struct ncp_conn *conn, __u32 last_id,
		 __u16 object_type, const char *object_name,
		 int *returned_no, __u16 conn_numbers[256]);

int
ncp_get_encryption_key(struct ncp_conn *conn,
		       char *target);
int
ncp_get_bindery_object_id(struct ncp_conn *conn,
			  __u16 object_type,
			  const char *object_name,
			  struct ncp_bindery_object *target);
int
ncp_scan_bindery_object(struct ncp_conn *conn,
			__u32 last_id, __u16 object_type, char *search_string,
			struct ncp_bindery_object *target);
int
ncp_read_property_value(struct ncp_conn *conn,
			int object_type, const char *object_name,
			int segment, const char *prop_name,
			struct nw_property *target);
int
ncp_login_encrypted(struct ncp_conn *conn,
		    const struct ncp_bindery_object *object,
		    const unsigned char *key,
		    const unsigned char *passwd);
int
ncp_login_user(struct ncp_conn *conn,
	       const unsigned char *username,
	       const unsigned char *password);
int
ncp_get_volume_info_with_number(struct ncp_conn *conn, int n,
				struct ncp_volume_info *target);

int
ncp_get_volume_number(struct ncp_conn *conn, const char *name,
		      int *target);

int
ncp_file_search_init(struct ncp_conn *conn,
		     int dir_handle, const char *path,
		     struct ncp_filesearch_info *target);

int
ncp_file_search_continue(struct ncp_conn *conn,
			 struct ncp_filesearch_info *fsinfo,
			 int attributes, const char *path,
			 struct ncp_file_info *target);

int
ncp_get_finfo(struct ncp_conn *conn,
	      int dir_handle, const char *path, const char *name,
	      struct ncp_file_info *target);

int
ncp_open_file(struct ncp_conn *conn,
	      int dir_handle, const char *path,
	      int attr, int access,
	      struct ncp_file_info *target);
int
ncp_close_file(struct ncp_conn *conn, const char *file_id);

int
ncp_create_newfile(struct ncp_conn *conn,
		   int dir_handle, const char *path,
		   int attr,
		   struct ncp_file_info *target);

int
ncp_create_file(struct ncp_conn *conn,
		int dir_handle, const char *path,
		int attr,
		struct ncp_file_info *target);

int
ncp_erase_file(struct ncp_conn *conn,
	       int dir_handle, const char *path,
	       int attr);

int
ncp_rename_file(struct ncp_conn *conn,
		int old_handle, const char *old_path,
		int attr,
		int new_handle, const char *new_path);

int
ncp_create_directory(struct ncp_conn *conn,
		     int dir_handle, const char *path,
		     int inherit_mask);

int
ncp_delete_directory(struct ncp_conn *conn,
		     int dir_handle, const char *path);

int
ncp_rename_directory(struct ncp_conn *conn,
		     int dir_handle,
		     const char *old_path, const char *new_path);

int
ncp_read(struct ncp_conn *conn, const char *file_id,
	 off_t offset, size_t count, char *target);

int
ncp_write(struct ncp_conn *conn, const char *file_id,
	  off_t offset, size_t count, const char *source);

int
ncp_do_lookup(struct ncp_conn *conn,
	      struct nw_info_struct *dir,
	      char *path,	/* may only be one component */
	      struct nw_info_struct *target);

int
ncp_modify_file_or_subdir_dos_info(struct ncp_conn *conn,
				   struct nw_info_struct *file,
				   __u32 info_mask,
				   struct nw_modify_dos_info *info);

int
ncp_del_file_or_subdir(struct ncp_conn *conn,
		       struct nw_info_struct *dir, char *name);


int
ncp_open_create_file_or_subdir(struct ncp_conn *conn,
			       struct nw_info_struct *dir, char *name,
			       int open_create_mode,
			       __u32 create_attributes,
			       int desired_acc_rights,
			       struct nw_file_info *target);

int
ncp_initialize_search(struct ncp_conn *conn,
		      const struct nw_info_struct *dir,
		      int    namespace,
		      struct ncp_search_seq *target);

int
ncp_search_for_file_or_subdir(struct ncp_conn *conn,
			      struct ncp_search_seq *seq,
			      struct nw_info_struct *target);

int
ncp_ren_or_mov_file_or_subdir(struct ncp_conn *conn,
			      struct nw_info_struct *old_dir, char *old_name,
			      struct nw_info_struct *new_dir, char *new_name);

int
ncp_create_queue_job_and_file(struct ncp_conn *conn,
			      __u32 queue_id,
			      struct queue_job *job);

int
ncp_close_file_and_start_job(struct ncp_conn *conn,
			     __u32 queue_id,
			     struct queue_job *job);

int
ncp_attach_to_queue(struct ncp_conn *conn,
		    __u32 queue_id);

int
ncp_detach_from_queue(struct ncp_conn *conn,
		      __u32 queue_id);

int
ncp_service_queue_job(struct ncp_conn *conn, __u32 queue_id, __u16 job_type,
		      struct queue_job *job);

int
ncp_finish_servicing_job(struct ncp_conn *conn, __u32 queue_id,
			 __u32 job_number, __u32 charge_info);

int
ncp_abort_servicing_job(struct ncp_conn *conn, __u32 queue_id,
			__u32 job_number);

#endif /* _NCPLIB_H */
