#ifndef _REFLIST_H
#define _REFLIST_H

/* File: dlist.h
 * --------------
 * Defines the interface for the Dynamiclist ADT.
 * The list_t allows the client to store any number of entrys of any desired
 * base type and is appropriate for a wide variety of storage problems. It 
 * supports efficient entry access, and appending/inserting/deleting entrys
 * as well as optional sorting and searching. In all cases, the list_t imposes 
 * no upper bound on the number of entrys and deals with all its own memory 
 * management. The client specifies the size (in bytes) of the entrys that 
 * will be stored in the list when it is created. Therefter the client and 
 * the list_t can refer to entrys via (void*) ptrs. 
 *
 * Note that if you can use a list_t as a secondary index (that is, a list of
 * pointers to objects stored elsewhere) by just storing pointers in the list_t
 * rather than the structs themselves.
 */


/* Type: list_t
 * ----------------
 * Defines the list_t type itself, which is a _pointer_ type, like char *. Only
 * a call to listNew actually creates a usable list_t.
 */
typedef struct listImplementation *list_t;


/* listCompareFn
 * --------------
 * listCompareFn is a pointer to a client supplied function which the
 * list_t uses to sort or search the entrys. The comparator takes two 
 * (void*) pointers (these will point to entrys) and returns an int.
 * The comparator should indicate the ordering of the two entrys
 * using the same convention as the strcmp library function:
 * If entry1 is "less than" entry2, return a negative number (typically -1).
 * If entry1 is "greater than"  entry2, return a positive number (typically 1).
 * If the two entrys are "equal", return 0.
 */
typedef int (*listCompareFn)(const void *entry1, const void *entry2);


/* listMapFn
 * ----------
 * listMapFn defines the type of functions that can be used to map over
 * the entrys in a list_t.  Each function is called with a pointer to
 * the entry and a client data pointer passed in from the original
 * caller.
 */
typedef void (*listMapFn)(const void *entry, void *clientData);


/* listNew
 * --------
 * Creates a new list_t and returns it. The actual number of entrys in
 * the new list is 0.  The entrySize parameter specifies the number of 
 * bytes that a single entry of this list should take up.  Must be 
 * greater than 0. For example, if you want to store entrys of type 
 * Binky, you would pass sizeof(Binky) as this parameter.
 *
 * The allocNum parameter specifies the initial ALLOCATED length of the 
 * list.  If the client passes a negative or 0 value, room for 
 * DEFAULTALLOC entrys will be allocated. 
 * If more space is required the list doubles it's current size so
 * as to respond appropriately to lists that grow drastically.  
 */
#define DEFAULTALLOC 10
list_t listNew(int allocNum, listCompareFn comparator);


void listReplaceComparator(list_t list, listCompareFn comparator);


/* listFree
 * ----------
 * Frees up all the memory for the list and its entrys. It DOES NOT free 
 * memory owned by pointers embedded in the entrys. This would require 
 * knowledge of the structure of the entrys which the list_t does not have. 
 * After calling this, the value of what "list" is pointing to is undefined.
 */
void listFree(list_t list);


/** 
 * Returns the number of entrys in the list.  Runs in constant time.
 */
int listSize(list_t list);


/* listNth
 * --------
 * Returns a pointer to the entry numbered n in the specified list.  
 * Numbering begins with 0.  It is an error if n is less than 0 or greater 
 * than the number of entrys in the list minus 1.
 *
 * A pointer returned by listNth becomes invalid after any calls which 
 * involve insertion, deletion or sorting the list, as all of 
 * these may rearrange the entry storage.  Runs in constant time.
 */ 
void *listElementAt(list_t list, int n);


/* listAppend
 * -----------
 * Adds a new entry to the end of the specified list.  
 */
void listAppend(list_t list, void *newElem);


/* listInsertAt
 * -------------
 * Copies a new entry into the list, placing it at the specified position n.
 * It is an error if n is less than 0 or greater than the logical length. The 
 * list entrys after position n will be shifted over to make room. The new 
 * entry's contents are copied from the memory pointed to by newElem. Runs 
 * in linear time, but will reallocate the list's space if out of room.
 */
void listInsertAt(list_t list, void *newElem, int n);


/* listDeleteAt
 * -------------
 * Deletes the entry numbered n from the list. It is an error if n is less 
 * than 0 or greater than the logical length minus one. All the entrys
 * after position n will be shifted over to fill the gap. Runs in linear time.
 * If the number of entrys the list can store is twice as large as the number
 * of entrys currently in the list, and more than 8 entrys can currently
 * be stored, the list will be reallocated at a smaller size.
 */
#define NO_REALLOC_FLOOR 8
void listDeleteAt(list_t list, int n);

void listDelete(list_t arrar, void *deleteElem);
 
/* listSort
 * ---------
 * Sorts the specified list into ascending order according to the supplied
 * comparator.  The numbering of the entrys will change to reflect the 
 * new ordering.  Should run in nlogn time.
 */
void listSort(list_t list);


/* listSearch
 * -----------
 * Searches the specified list for an entry whose contents match
 * the entry passed as the key.  Uses the comparator argument as test
 * for equality. The "isSorted" parameter allows the client to specify that 
 * the list is already in sorted order, and thus it should use a faster 
 * binary search.  If isSorted is false, a simple linear search is used.
 * If the key is found, a pointer to the matching entry is returned.
 * Otherwise the function returns NULL.
 */
void *listSearch(list_t list, void *key, int isSorted);


/* listSearchPosition
 * -----------
 * Does a linear search over the given list to find the entry passed,
 * and returns that entry's position in the list (numbered starting
 * with zero).  Returns -1 if not found.
 */
int listSearchPosition(list_t list, void *entry);

int listContains(list_t list, void *entry);

/* listMap
 * -----------
 * Iterates through each entry in the list in order (from entry 0 to
 * entry n-1) and calls the function fn for that entry.  The listMapFn
 * function is called with the address of the list entry and the clientData 
 * pointer passed to listMap.
 * The clientData value allows the client to pass extra state information to 
 * the client-supplied function, if necessary.  If no client data is required, 
 * this argument should be NULL.
 */
void listMap(list_t list, listMapFn fn, void *clientData);

#endif _REFLIST_H
