#include "patch.h"

/************************************************************
 TEXTURE.C
			 /-------------------------\
			||  Textures and PNames    ||
			 \-------------------------/
									
************************************************************/

void InitTextures(WadTextures *wt)
{
	wt->pnames.npnames = 0;
	wt->pnames.patchname = NULL;
	wt->t1.N = 0;
	wt->t1.offs = NULL;
	wt->t1.textures = NULL;
	wt->t2.N = 0;
	wt->t2.offs = NULL;
	wt->t2.textures = NULL;
	wt->t3.N = 0;
	wt->t3.offs = NULL;
	wt->t3.textures = NULL;
}

/* Return a string that represents a patch name, the p-th in PNames */

char *GetPName(PNamesLump *pl,int p)
{
	static char pnbuff[9];

	strcpy(pnbuff,"");
	if (pl->npnames>p)
		strncpy(pnbuff,pl->patchname[p],8);
	pnbuff[8]='\0';

	return (pnbuff);
}

/* Return the patch index in PNames of a patch name */

int GetPIdx(PNamesLump *pl,char *pnamb)
{
	int i;
	char pnam[9];

	strncpy(pnam,pnamb,8);
	pnam[8]='\0';

	for (i=0;i<pl->npnames;i++)
		if (stricmp(pnam,GetPName(pl,i))==0)
			return i;
	return -1;
}


/* Print a patchname */

void PrintPName(WadTextures *wt,int p)
{
	if (wt->pnames.npnames>p)
		printf("%-8s\n",Str8(wt->pnames.patchname[p]));
}

/* Print a list of PNames */

void PrintPNames(WadTextures *wt)
{
	int i;

	for (i=0;i<wt->pnames.npnames;i++)
	    PrintPName(wt,i);
	printf("\n");
}

/* Print a texture description */

void PrintTexture(WadTextures *wt,TextureDesc *td)
{
	int i;

	if (td->npatches>0)
	{
		printf("%-8s %d %d\n",Str8(td->textname),td->twid,td->thgt);
		for (i=0;i<td->npatches;i++)
			printf("*    %-8s %d %d\n",GetPName(&wt->pnames,td->patches[i].pidx),td->patches[i].xoff,td->patches[i].yoff);
	}
}

/* Print a list of Textures and Patches they use ala DeuTex */

void PrintTextureLump(WadTextures *wt, TextureLump *t)
{
	int i;

	for (i=0;i<t->N;i++)
		PrintTexture(wt,t->textures+i);
	
}

/* Print all three existing texture lumps */

void PrintTextures(WadTextures *wt)
{
	if (wt->pnames.npnames>0)
	{
		printf("PNames:\n");
		PrintPNames(wt);
	}
	if (wt->t1.N>0)
	{
		printf("Texture Lump TEXTURE1:\n");
		PrintTextureLump(wt,&wt->t1);
	}
	if (wt->t2.N>0)
	{
		printf("Texture Lump TEXTURE2:\n");
		PrintTextureLump(wt,&wt->t2);
	}
	if (wt->t3.N>0)
	{
		printf("Texture Lump TEXTURE3:\n");
		PrintTextureLump(wt,&wt->t3);
	}
}

// Add the patch name pnam to the PNames lump pl

void AddPName(PNamesLump *pl,char *pnam)
{
	int i,n;

	n = pl->npnames;
	for (i=0;i<n;i++)
		if (strnicmp(pl->patchname[i],pnam,8)==0) return;

	pl->patchname = (char **)realloc(pl->patchname,(n+1)*sizeof(char *));
	assert(pl->patchname);
	if (n==0) pl->patchname[0]=NULL;

	pl->patchname[0] = (char *)realloc(pl->patchname[0],(n+1)*8);
	assert(pl->patchname[0]);
	for (i=1;i<=n;i++)
		pl->patchname[i] = pl->patchname[i-1]+8;

	strncpy(pl->patchname[n],pnam,8);
	pl->npnames++;
}

// Add the texture *td to the texture lump *ol, translating indices if !isbase

void AddTexture(TextureLump *ol,TextureDesc *td, int isbase)
{
	int j;

	// reallocate for one new offset
	ol->offs = (long *)realloc(ol->offs,(1+ol->N) * sizeof(long));
	assert(ol->offs);

	// all offsets increase by the size of the new offset
	for (j=0;j<ol->N;j++)
		ol->offs[j] += sizeof(long);

	// set new offset += TextureDesc - Patch pointer + patches
	if (ol->N)
		ol->offs[ol->N] = ol->offs[ol->N-1] + sizeof(TextureDesc) - sizeof(Patch *)
			+ ol->textures[ol->N-1].npatches * sizeof(Patch);
	else
		ol->offs[ol->N] = sizeof(long)+4;

	// reallocate one more texture description
	ol->textures = (TextureDesc *)realloc(ol->textures,(1+ol->N) * sizeof(TextureDesc));
	assert(ol->textures);

	// copy the new description
	ol->textures[ol->N] = *td;

	// allocate the patch array
	ol->textures[ol->N].patches = (Patch *)malloc(td->npatches * sizeof(Patch));
	assert(ol->textures[ol->N].patches);

	// copy the patch array, translate the patchname indices to output indices
	for (j=0;j<td->npatches;j++)
	{
		ol->textures[ol->N].patches[j] = td->patches[j];
		if (!isbase)
			ol->textures[ol->N].patches[j].pidx = XIdx[td->patches[j].pidx];
	}

	// note one more texture in the lump
	ol->N++;
}


// Read the PNAMES from a wad file

void ReadPNames(WadDir *dir,FILE *wst,WadTextures *wt)
{
	int o,n,i,m,p;

	if ((p=IsIn("PNAMES",dir))>=0)
	{
		o = dir->dirp[p].resptr;
		n = dir->dirp[p].reslen;
		fseek(wst,o,SEEK_SET);

		fread(&m,sizeof(long),1,wst);
		wt->pnames.npnames = m;
		wt->pnames.patchname = (char **)malloc(m*sizeof(char *));
		assert(wt->pnames.patchname);
		wt->pnames.patchname[0] = (char *)malloc(8*m);
		assert(wt->pnames.patchname[0]);
		for (i=1;i<m;i++)
			wt->pnames.patchname[i] = wt->pnames.patchname[i-1]+8;
		fread(wt->pnames.patchname[0],8*m,1,wst);
	}
	else
	{
		wt->pnames.npnames = 0;
		wt->pnames.patchname = NULL;
	}
}

// Read the TEXTUREn from a wad file

void ReadTextures(WadDir *dir,FILE *wst,WadTextures *wt)
{
	int o,n,i,x,p;
	TextureHeader th;

	if ((p=IsIn("TEXTURE1",dir))!=-1)
	{
		o = dir->dirp[p].resptr;
		n = dir->dirp[p].reslen;
		fseek(wst,o,SEEK_SET);

		fread(&(wt->t1.N),sizeof(long),1,wst);
		if (wt->t1.N>0)
		{
			wt->t1.offs = (long *)malloc(wt->t1.N * sizeof(long));
			assert(wt->t1.offs);
			fread(wt->t1.offs,sizeof(long),wt->t1.N,wst);
			wt->t1.textures = (TextureDesc *)malloc(wt->t1.N * sizeof(TextureDesc));
			assert(wt->t1.textures);
			for (i=0;i<wt->t1.N;i++)
			{
				fread(&th,sizeof(TextureHeader),1,wst);
				strncpy(wt->t1.textures[i].textname,th.textname,8);
				wt->t1.textures[i].twid = th.twid;
				wt->t1.textures[i].thgt = th.thgt;
				wt->t1.textures[i].npatches = th.npatches;
				wt->t1.textures[i].res1 = wt->t1.textures[i].res2=0;
				wt->t1.textures[i].res3 = wt->t1.textures[i].res4=0;
				wt->t1.textures[i].patches =
					(Patch *)malloc(wt->t1.textures[i].npatches*sizeof(Patch));
				assert(wt->t1.textures[i].patches!=NULL);
				fread(wt->t1.textures[i].patches,sizeof(Patch),wt->t1.textures[i].npatches,wst);
			}
		}
		else
		{
			wt->t1.textures=NULL;
		}
	}
	else
	{
		wt->t1.N = 0;
		wt->t1.textures=NULL;
	}

	if ((p=IsIn("TEXTURE2",dir))!=-1)
	{
		o = dir->dirp[p].resptr;
		n = dir->dirp[p].reslen;
		fseek(wst,o,SEEK_SET);

		fread(&(wt->t2.N),sizeof(long),1,wst);
		if (wt->t2.N>0)
		{
			wt->t2.offs = (long *)malloc(wt->t2.N * sizeof(long));
			assert(wt->t2.offs);
			fread(wt->t2.offs,sizeof(long),wt->t2.N,wst);
			wt->t2.textures = (TextureDesc *)malloc(wt->t2.N * sizeof(TextureDesc));
			assert(wt->t2.textures);
			for (i=0;i<wt->t2.N;i++)
			{
				fread(wt->t2.textures[i].textname,8,1,wst);
				fseek(wst,2*sizeof(short),SEEK_CUR);
				fread(&(wt->t2.textures[i].twid),sizeof(short),1,wst);
				fread(&(wt->t2.textures[i].thgt),sizeof(short),1,wst);
				fseek(wst,2*sizeof(short),SEEK_CUR);
				fread(&(wt->t2.textures[i].npatches),sizeof(short),1,wst);
				wt->t2.textures[i].patches = (Patch *)malloc(wt->t2.textures[i].npatches*sizeof(Patch));
				assert(wt->t2.textures[i].patches);
				fread(wt->t2.textures[i].patches,sizeof(Patch),wt->t2.textures[i].npatches,wst);
			}
		}
		else
		{
			wt->t2.textures=NULL;
		}
	}
	else
	{
		wt->t2.N = 0;
		wt->t2.textures=NULL;
	}

	if ((p=IsIn("TEXTURE3",dir))!=-1)
	{
		o = dir->dirp[p].resptr;
		n = dir->dirp[p].reslen;
		fseek(wst,o,SEEK_SET);

		fread(&(wt->t3.N),sizeof(long),1,wst);
		if (wt->t3.N>0)
		{
			wt->t3.offs = (long *)malloc(wt->t3.N * sizeof(long));
			assert(wt->t3.offs);
			fread(wt->t3.offs,sizeof(long),wt->t3.N,wst);
			wt->t3.textures = (TextureDesc *)malloc(wt->t3.N * sizeof(TextureDesc));
			assert(wt->t3.textures);
			for (i=0;i<wt->t3.N;i++)
			{
				fread(wt->t3.textures[i].textname,8,1,wst);
				fseek(wst,2*sizeof(short),SEEK_CUR);
				fread(&(wt->t3.textures[i].twid),sizeof(short),1,wst);
				fread(&(wt->t3.textures[i].thgt),sizeof(short),1,wst);
				fseek(wst,2*sizeof(short),SEEK_CUR);
				fread(&(wt->t3.textures[i].npatches),sizeof(short),1,wst);
				wt->t3.textures[i].patches = (Patch *)malloc(wt->t3.textures[i].npatches*sizeof(Patch));
				assert(wt->t3.textures[i].patches);
				fread(wt->t3.textures[i].patches,sizeof(Patch),wt->t3.textures[i].npatches,wst);
			}
		}
		else
		{
			wt->t3.textures=NULL;
		}
	}
	else
	{
		wt->t3.N = 0;
		wt->t3.textures=NULL;
	}
}

// Merging textures starts by merging the pname resources. We then build new
// TEXTURE1, TEXTURE2, and TEXTURE3 resources in memory, converting each patch
// index in the original descriptions to the new merged pnames indices, 
// The new TEXTURE1 contains all textures named in the base TEXTURE1,
// replaced as needed by any textures in the patch wad's TEXTURE1, TEXTURE2,
// or TEXTURE3 resources that have the same name as a base wad TEXTURE1. It
// also contains, at the end, all textures in the patch wad's TEXTURE1 that
// do not occur in any base wad TEXTUREn section.

void MergePNames()
{
	int i,m,n,p;

	// Merge the pnames
	// 1) start the merged list with the base wad's PNAMES
	// 2) for every patchname in patch wad but not base wad add to end

	m = bwt.pnames.npnames+pwt.pnames.npnames;  // output len overestimate
	if (m==0) return;

	owt.pnames.patchname = (char **)malloc(m*sizeof(char *)); // overalloc
	assert(owt.pnames.patchname);
	owt.pnames.patchname[0] = (char *)malloc(m*8);
	assert(owt.pnames.patchname[0]);
	for (i=1;i<m;i++)							// singly allocated array so
		owt.pnames.patchname[i] = owt.pnames.patchname[i-1]+8; // set ptrs
	for (i=0;i<bwt.pnames.npnames;i++)			// copy base's pnames
		strncpy(owt.pnames.patchname[i],bwt.pnames.patchname[i],8);
	owt.pnames.npnames = bwt.pnames.npnames;	// set length so far
  
	n=0;
	while (DoTextures && n<pwt.pnames.npnames)	// while not at end of list
	{
		while						// search for entry NOT in base wad
		(
			n<pwt.pnames.npnames
			&&
			GetPIdx(&bwt.pnames,Str8(pwt.pnames.patchname[n]))>=0
		)
			n++;

		// then add it to end of out wad's pnames
		if (n<pwt.pnames.npnames)
			strncpy(owt.pnames.patchname[owt.pnames.npnames++],
				    pwt.pnames.patchname[n++], 8);
	}

	// shrink to required allocation
	owt.pnames.patchname = (char **)realloc(owt.pnames.patchname,sizeof(char *)*owt.pnames.npnames);
	assert(owt.pnames.patchname);
	owt.pnames.patchname[0] = (char *)realloc(owt.pnames.patchname[0],8*owt.pnames.npnames);
	assert(owt.pnames.patchname[0]);
	for (i=1;i<owt.pnames.npnames;i++)			// singly allocated array so
		owt.pnames.patchname[i] = owt.pnames.patchname[i-1]+8; // set ptrs

	// create index translation table from patch pnames to output pnames
	XIdx = (int *)malloc(owt.pnames.npnames*sizeof(int));
	assert(XIdx);
	for (i=0;i<pwt.pnames.npnames;i++)
	{
		XIdx[i] = GetPIdx(&owt.pnames,pwt.pnames.patchname[i]);
		if (!IsIn(Str8(pwt.pnames.patchname[i]),&pdir))
			XIdx[i] = GetPIdx(&bwt.pnames,pwt.pnames.patchname[i]);
		if (XIdx[i]<0)
		{
			printf("Fatal - %s not defined in base or patch wad\n",Str8(pwt.pnames.patchname[i]));
			exit(1);
		}
	}
}

void MergeTextures()
{
	int i,j,m,n;
	char textnam[9];
	TextureDesc td;

	// First we create the output wads TEXTURE1 resource
	// This contains every texture named in the base wad, but uses
	// the definitions in the patch wad if they exist instead. It
	// also contains all textures in the patch, but not the base,
	// that are in the TEXTURE1 section of the patch.

	// add to output wad all texture definitions superseded by the
	// patch wad, and all definitions from the base wad that are not.

	for (i=0;i<bwt.t1.N;i++)	// for each base texture
	{
		strcpy(textnam,Str8(bwt.t1.textures[i].textname));
		if (i==0)
		{
			strncpy(td.textname,"AASHITTY",8);
			td.res1 = td.res2 = td.res3 = td.res4 = 0;
			td.twid = td.thgt = 64;
			td.npatches = 1;
			td.patches = malloc(sizeof(Patch));
			assert(td.patches);
			td.patches[0].xoff = 0;
			td.patches[0].yoff = 0;
			td.patches[0].pidx = 0;
			td.patches[0].cmap = 0;
			td.patches[0].stepdir=1;
			AddTexture(&owt.t1,&td,1);
			free(td.patches);
		}
		if (stricmp(textnam,"AASHITTY")==0) goto nextone;

		if (DoTextures && !NewOnly)
		{
			for (j=0;j<pwt.t1.N;j++)
			{
				if (stricmp(textnam,Str8(pwt.t1.textures[j].textname))==0)
				{
					// use patch wads TEXTURE1 definition
					AddTexture(&owt.t1,pwt.t1.textures+j,0);
					goto nextone;
				}
			}
			for (j=0;j<pwt.t2.N;j++)
			{
				if (stricmp(textnam,Str8(pwt.t2.textures[j].textname))==0)
				{
					// use patch wads TEXTURE2 definition
					AddTexture(&owt.t1,pwt.t2.textures+j,0);
					goto nextone;
				}
			}
			for (j=0;j<pwt.t3.N;j++)
			{
				if (stricmp(textnam,Str8(pwt.t3.textures[j].textname))==0)
				{
					// use patch wads TEXTURE3 definition
					AddTexture(&owt.t1,pwt.t3.textures+j,0);
					goto nextone;
				}
			}
		}

		// use base wad's TEXTURE1 definition
		AddTexture(&owt.t1,bwt.t1.textures+i,1);

nextone:;
	}

	// Add to output wad all textures from the patches TEXTURE1 that are
	// not in any base wad texture section.

	if (DoTextures)
	{
		for (i=0;i<pwt.t1.N;i++)
		{
			strcpy(textnam,Str8(pwt.t1.textures[i].textname));
			for (j=0;j<bwt.t1.N;j++)
			{
				if (stricmp(textnam,Str8(bwt.t1.textures[j].textname))==0)
					goto nxtone;
			}
			for (j=0;j<bwt.t2.N;j++)
			{
				if (stricmp(textnam,Str8(bwt.t2.textures[j].textname))==0)
					goto nxtone;
			}
			for (j=0;j<bwt.t3.N;j++)
			{
				if (stricmp(textnam,Str8(bwt.t3.textures[j].textname))==0)
					goto nxtone;
			}

			// new texture - append to out wads TEXTURE1 section
			AddTexture(&owt.t1,pwt.t1.textures+i,0);
	nxtone:;
		}
	}
}

