#include "patch.h"

/************************************************************
 PICTURE.C
			 /-------------------------\
			||  DOOM Picture support   ||
			 \-------------------------/
									
************************************************************/

// Return the length of a picture structure as written in a wad
// no allocation

int GetPictureLen(PicData *p)
{
	int offs=0,i=0;
	unsigned char npix,rown;

	if (p->pwid>0)
	{
		offs = p->coloffs[p->pwid-1];
		i = offs - 4*sizeof(short) - p->pwid*sizeof(int);
		rown = p->posts[i++];
		while (rown!=255)
		{
			npix = p->posts[i++];
			i += npix+2;
			rown=p->posts[i++];
		}
	}
	return 4*sizeof(short)+p->pwid*sizeof(int)+i;										 
}

// Convert a rectangular array of numbers to a DOOM format picture
// bytes 0-255 represent palette colors as usual, XPARENT is transparency.
//
// memory is allocated on p->coloffs and p->posts
// max: 256k+rwid actual: postlen+4*(1+rwid)

void ConvertToPicture(BYTE *bytes,int logw,int rwid,int logh,int x,int y,PicData *p)
{
	int i,j,n;
	int pixcnt,trancnt;
	int postlen;
	unsigned char *posts;

	p->pwid = rwid;
	p->phgt = logh;
	p->pxoff = PatchFormat? rwid/2 - 1 : x;
	p->pyoff = PatchFormat? logh - 5 : y;

	p->coloffs = (long *)malloc(rwid*sizeof(int));
	assert(p->coloffs);
	p->posts = (unsigned char *)malloc(1024*256*sizeof(unsigned char));
	assert(p->posts);

	n=postlen=0;
	for (j=0;j<rwid;j++)
	{
		p->coloffs[j] = 4*sizeof(short) + p->pwid*sizeof(int) + postlen;
		pixcnt = 0;
		for (i=0;i<logh;i++)
		{
			if (bytes[logw*i+j]==XPARENT)
			{
				if (pixcnt>0)
				{
					p->posts[postlen++]=0;		// add fill byte at end
					pixcnt=0;
				}
			}
			else
			{
				if (pixcnt==0) 					// start a post
				{
					p->posts[postlen++] = i;
					n = postlen++;
					p->posts[n] = 1;
					p->posts[postlen++] = 0;  	// add fill byte at start
				}
				else p->posts[n]++;

				// add pixel to current post

				p->posts[postlen++] = bytes[logw*i+j];
				pixcnt++;
			}
		}
		if (pixcnt>0)							// terminate last post
			p->posts[postlen++]=0;
		p->posts[postlen++]=255;				// terminate column
	}
	p->posts = (unsigned char *)realloc(p->posts,postlen);
	if (postlen) assert(p->posts);
}

// Load a Doom picture into a rectangular byte array using its internal
// offsets and an externally specified offset
// no allocation

void OverlayPicture(BYTE *bytes,int logw,int logh,PicData *p,int xo,int yo)
{
	int i,j;
	unsigned char rown,npix;
	int postlen;

	postlen=0;
	for (i=0;i<p->pwid;i++)
	{
		postlen = p->coloffs[i] - 4*sizeof(short) - p->pwid*sizeof(int);
		rown=p->posts[postlen++];
		while (rown!=255)
		{
			npix=p->posts[postlen++];
			postlen++;
			for (j=0;j<npix;j++)
			{
				if (!Xlation)
					bytes[logw*(rown+j+yo)+i+xo] = p->posts[postlen++];
				else
					bytes[logw*(rown+j+yo)+i+xo] = XlateTextColor!=TEXTRED? XlateColor(p->posts[postlen++],XlateTextColor,Brightness) : p->posts[postlen++];
			}
			postlen++;
			rown=p->posts[postlen++];
		}
	}
}

// Write a picture structure as a lump in an opened file
// memory balanced

int WritePictureImage(FILE *st,int offs,PicData *p)
{
	int i,j,n,postlen;
	unsigned char rown,npix,z=0;
	short *pchd;
	int *cola;
	unsigned char *post;
	char *img = NULL;

	if (!n) return ftell(st);

	img = (char *) malloc (n=GetPictureLen(p));
	assert(img);

	pchd = (short *)(img);
	cola = (int *)(((char *)pchd) + 4*sizeof(short));
	post = (unsigned char *)(((char *)cola) + p->pwid*sizeof(int));

	memcpy(pchd,&(p->pwid),4*sizeof(short));
	memcpy(cola,p->coloffs,(p->pwid)*sizeof(int));
	memcpy(post,p->posts,n-4*sizeof(short)-p->pwid*sizeof(int));

	fseek(st,offs,SEEK_SET);
	fwrite(img,n,1,st);
	free(img);

	return ftell(st);
}

// Read the 62 font character graphics resources from an IWAD
// no allocation

void InitDoomFont()
{
	int j,n,k;

	for (j=33;j<94;j++)
	{
		k = j-33;
		n = DoomFont[k].pwid = ((short *)FontData[k])[0];
		DoomFont[k].phgt = ((short *)FontData[k])[1];
		DoomFont[k].pxoff = ((short *)FontData[k])[2];
		DoomFont[k].pyoff = ((short *)FontData[k])[3];
		DoomFont[k].coloffs = (long *)(((short *)FontData[k])+4);
		DoomFont[k].posts = (unsigned char *)(DoomFont[k].coloffs+n);

		n = BigFont[k].pwid = ((short *)BigData[k])[0];
		BigFont[k].phgt = ((short *)BigData[k])[1];
		BigFont[k].pxoff = ((short *)BigData[k])[2];
		BigFont[k].pyoff = ((short *)BigData[k])[3];
		BigFont[k].coloffs = (long *)(((short *)BigData[k])+4);
		BigFont[k].posts = (unsigned char *)(BigFont[k].coloffs+n);
	}
}

// Write a short array with -1 transparency convention to .BMP file
// memory balanced

void WriteBMP(BYTE *out,int logw,int logh,char *rbmpname)
{
	unsigned char *cbuff;
	int i,j,c,wid;
	BITMAPFILEHEADER bmfh;
	BITMAPINFOHEADER bmih;
	RGBQUAD	aColors[256];
	FILE *st;
	char bmpname[256],tbmpname[256];
	char *p;

	strcpy(tbmpname,rbmpname);
	p = strchr(tbmpname,'\\');
	if (p!=NULL)
	{
		*p='\0';
		DOSTranslate(p+1,bmpname);
		strcat(tbmpname,"\\");
		strcat(tbmpname,bmpname);
	}

	wid = 4*((logw+3)/4);
	bmfh.bfType = 19778;
	bmfh.bfSize = sizeof(bmfh)+sizeof(bmih)+256L*sizeof(RGBQUAD)+wid*logh;
	bmfh.bfReserved1 = 0;
	bmfh.bfReserved2 = 0;
	bmfh.bfOffBits = sizeof(bmfh)+sizeof(bmih)+256L*sizeof(RGBQUAD);

	bmih.biSize = sizeof(bmih);
	bmih.biWidth = logw;
	bmih.biHeight = logh;
	bmih.biPlanes = 1;
	bmih.biBitCount = 8;
	bmih.biCompression = BI_RGB;
	bmih.biSizeImage = wid*logh;
	bmih.biXPelsPerMeter = 0;
	bmih.biYPelsPerMeter = 0;
	bmih.biClrUsed = 256;
	bmih.biClrImportant = 256;

   	st = fopen(tbmpname,"wb");
	if (st!=NULL)
	{
//		for (i=logw;i<wid;i++) cbuff[i]=0;

		fwrite(&bmfh,sizeof(bmfh),1,st);
		fwrite(&bmih,sizeof(bmih),1,st);
		fwrite(DoomPalette,sizeof(RGBQUAD),256,st);

		for (i=0;i<logh;i++)
			fwrite(out+(logh-1-i)*logw,sizeof(BYTE),wid,st);
		fclose(st);
	}
	else printf("Cannot open %s for write\n",tbmpname);
}

// Reads BMP file with name translated from rbmpname
// Header is read for real width, and logical height
// Logical width is next multiple of 4 greater than or equal to real width
// Bitmap is written to short array *out, allocated to fit
// *out is allocated and read as logical width by logical height
//
// memory is allocated on *out
// max: logw*logh*sizeof(short)

void ReadBMP(BYTE **out,int *logw,int *realw,int *logh,char *rbmpname)
{
	int i,j,c,n;
	BITMAPFILEHEADER bmfh;
	BITMAPINFOHEADER bmih;
	RGBQUAD	aColors[256];
	FILE *st;
	char *buffer=NULL;
	char bmpname[256],tbmpname[256];
	char *p;

	strcpy(tbmpname,rbmpname);
	p = strchr(tbmpname,'\\');
	if (p!=NULL) *p='\0';
	DOSTranslate(p+1,bmpname);
	strcat(tbmpname,"\\");
	strcat(tbmpname,bmpname);

	*logw = *logh = *realw = 0;
   	st = fopen(tbmpname,"rb");
	if (st!=NULL)
	{
		fread(&bmfh,sizeof(bmfh),1,st);
		fread(&bmih,sizeof(bmih),1,st);
		fread(aColors,sizeof(RGBQUAD),256,st);

		*realw = bmih.biWidth;
		*logw = 4*((bmih.biWidth+3)/4);
		*logh = bmih.biHeight;

		n = (*logw)*(*logh);

		buffer = malloc(n);
		assert(buffer);
		fseek(st,bmfh.bfOffBits,SEEK_SET);
		fread(buffer,n,1,st);

		*out = (BYTE *)malloc(n);
		assert(*out);

		for (i=(*logh)-1;i>=0;i--)
			memmove((*out)+((*logh)-1-i)*(*logw),buffer+(*logw)*i,*logw);
		fclose(st);
		free(buffer);
	}
	else printf("Warning: %s not supplied as .BMP, must be base resource\n",tbmpname);
}

static int mxcir[16];
static int mxhir[16];

// Break a text creation control string into segments delimited by changes in
// color, line number, or brightness
// no allocation

void SegmentText(char *text)
{

 	int i;
 	char *p,buff[256];
	int e;
	char colstr[8]=" rbgwsy";
	char valnstr[8]="^-v";
	char halnstr[8]="<|>";
	char bristr[12]=" 01234567";
	PicData *fp;

 	for (i=0;i<16;i++)
 	{
 		mxcir[i]=0;
		mxhir[i]=0;
 	}

	nsegs=0;
 	segs[nsegs].color=TEXTRED;
 	segs[nsegs].bright=0;
 	segs[nsegs].fsel=0;
	segs[nsegs].valn=V_TOP;
	segs[nsegs].haln=H_CENTER;
 	segs[nsegs].row=0;
 	segs[nsegs].col=0;
 	segs[nsegs].ncols=0;
 	segs[nsegs].len=0;
 	segs[nsegs].str[0]='\0';

 	for (i=0;i<strlen(text);i++)
 	{
 		switch(text[i])
 		{
 			case '\\':
				switch(tolower(text[i+1]))
				{
					// color change escape code
					case 'r':
					case 'b':
					case 'g':
					case 's':
					case 'w':
					case 'y':
						e = strchr(colstr,text[i+1]) - colstr;
						if (segs[nsegs].col+segs[nsegs].ncols>mxcir[segs[nsegs].row])
							mxcir[segs[nsegs].row] = segs[nsegs].col+segs[nsegs].ncols;
						nsegs++; 
						segs[nsegs].color = e-1;
						segs[nsegs].bright = segs[nsegs-1].bright;
						segs[nsegs].valn = segs[nsegs-1].valn;
						segs[nsegs].haln = segs[nsegs-1].haln;
						segs[nsegs].fsel = segs[nsegs-1].fsel;
						segs[nsegs].row = segs[nsegs-1].row;
						segs[nsegs].col = segs[nsegs-1].col+segs[nsegs-1].ncols;
						segs[nsegs].ncols = 0;
						segs[nsegs].len = 0;
						segs[nsegs].str[0]='\0';
						i++;
						break;

//					// left align escape code
//					case 'l':
//						i++;
//						CenterText=0;
//						break;

					// vertical alignment escapes
					case '^':
					case '-':
					case 'v':
						e = strchr(valnstr,text[i+1]) - valnstr;
						if (segs[nsegs].col+segs[nsegs].ncols>mxcir[segs[nsegs].row])
							mxcir[segs[nsegs].row] = segs[nsegs].col+segs[nsegs].ncols;
						nsegs++; 
						segs[nsegs].color = segs[nsegs-1].color;
						segs[nsegs].bright = segs[nsegs-1].bright;
						segs[nsegs].valn = e;
						segs[nsegs].haln = segs[nsegs-1].haln;
						segs[nsegs].fsel = segs[nsegs-1].fsel;
						segs[nsegs].row = segs[nsegs-1].row;
						segs[nsegs].col = segs[nsegs-1].col+segs[nsegs-1].ncols;
						segs[nsegs].ncols = 0;
						segs[nsegs].len = 0;
						segs[nsegs].str[0]='\0';
						i++;
						break;

					// horizontal alignment escapes
					case '<':
					case '|':
					case '>':
						e = strchr(halnstr,text[i+1]) - halnstr;
						if (segs[nsegs].col+segs[nsegs].ncols>mxcir[segs[nsegs].row])
							mxcir[segs[nsegs].row] = segs[nsegs].col+segs[nsegs].ncols;
						nsegs++; 
						segs[nsegs].color = segs[nsegs-1].color;
						segs[nsegs].bright = segs[nsegs-1].bright;
						segs[nsegs].haln = e;
						segs[nsegs].valn = segs[nsegs-1].valn;
						segs[nsegs].fsel = segs[nsegs-1].fsel;
						segs[nsegs].row = segs[nsegs-1].row;
						segs[nsegs].col = segs[nsegs-1].col+segs[nsegs-1].ncols;
						segs[nsegs].ncols = 0;
						segs[nsegs].len = 0;
						segs[nsegs].str[0]='\0';
						i++;
						break;

					// font select text escape code
					case 'f':
						e = isdigit(text[i+2])? text[i+++2]-'0' : 0;
						if (segs[nsegs].col+segs[nsegs].ncols>mxcir[segs[nsegs].row])
							mxcir[segs[nsegs].row] = segs[nsegs].col+segs[nsegs].ncols;
						nsegs++; 
						segs[nsegs].color = segs[nsegs-1].color;
						segs[nsegs].bright = segs[nsegs-1].bright;
						segs[nsegs].valn = segs[nsegs-1].valn;
						segs[nsegs].haln = segs[nsegs-1].haln;
						segs[nsegs].fsel = e;
						segs[nsegs].row = segs[nsegs-1].row;
						segs[nsegs].col = segs[nsegs-1].col+segs[nsegs-1].ncols;
						segs[nsegs].ncols = 0;
						segs[nsegs].len = 0;
						segs[nsegs].str[0]='\0';
						i++;
						break;

					// brightness adjust escape code
					case '0':
					case '1':
					case '2':
					case '3':
					case '4':
					case '5':
					case '6':
					case '7':
						e = strchr(bristr,text[i+1]) - bristr;
						if (segs[nsegs].col+segs[nsegs].ncols>mxcir[segs[nsegs].row])
							mxcir[segs[nsegs].row] = segs[nsegs].col+segs[nsegs].ncols;
						nsegs++; 
						segs[nsegs].color = segs[nsegs-1].color;
						segs[nsegs].bright = e-1;
						segs[nsegs].valn = segs[nsegs-1].valn;
						segs[nsegs].haln = segs[nsegs-1].haln;
						segs[nsegs].fsel = segs[nsegs-1].fsel;
						segs[nsegs].row = segs[nsegs-1].row;
						segs[nsegs].col = segs[nsegs-1].col+segs[nsegs-1].ncols;
						segs[nsegs].ncols = 0;
						segs[nsegs].len = 0;
						segs[nsegs].str[0]='\0';
						i++;
						break;

					// new line escape code
					case 'n':
						if (segs[nsegs].col+segs[nsegs].ncols>mxcir[segs[nsegs].row])
							mxcir[segs[nsegs].row] = segs[nsegs].col+segs[nsegs].ncols;
						nsegs++; 
						segs[nsegs].color = segs[nsegs-1].color;
						segs[nsegs].bright = segs[nsegs-1].bright;
						segs[nsegs].valn = segs[nsegs-1].valn;
						segs[nsegs].haln = segs[nsegs-1].haln;
						segs[nsegs].fsel = segs[nsegs-1].fsel;
						segs[nsegs].row = segs[nsegs-1].row+1;
						segs[nsegs].col = 0;
						segs[nsegs].ncols = 0;
						segs[nsegs].len = 0;
						segs[nsegs].str[0]='\0';
						i++;
						break;

					// unknown escape code
					default:
						printf("%d Bad text sign escape code: \\%c\n",i,tolower(text[i+1]));
						break;
				}
				break;
			default:
				// regular text character
				fp = segs[nsegs].fsel? BigFont : DoomFont;
				if (33<=text[i] && text[i]<=95)
				{
					segs[nsegs].ncols += fp[text[i]-33].pwid;
					if (fp[text[i]-33].phgt > mxhir[segs[nsegs].row])
						mxhir[segs[nsegs].row] = fp[text[i]-33].phgt;
					segs[nsegs].str[segs[nsegs].len++] = text[i];
					segs[nsegs].str[segs[nsegs].len] = '\0';
				}
				else if (text[i]!=13 && text[i]!=10)
				{
					segs[nsegs].ncols += 6;
					segs[nsegs].str[segs[nsegs].len++] = ' ';
					segs[nsegs].str[segs[nsegs].len] = '\0';
				}
				break;
			
 		}
 	}
	if (segs[nsegs].col+segs[nsegs].ncols>mxcir[segs[nsegs].row])
		mxcir[segs[nsegs].row] = segs[nsegs].col+segs[nsegs].ncols;
	nsegs++; 
}

// create a PicData structure from a text creation control string
// memory is allocated on outpp->coloffs and outpp->posts by the
// call to ConvertPicture at the end

void RenderString(int targwid,char *strs,BYTE *out,PicData *outpp)
{
	int i,j,k;
	int o,h,w,x,y;
	int coff,colpos;
	int rowhgt[64];
	int logw,logh,maxh,maxc,toth;
	char *tok;
	PicData *fp;
	int nrows,vsty;

	CenterText=1;
	DoubleSize=0;
	SegmentText(strs);
	if (nsegs<=0) return;

	if (Verbose)
	{
		printf("%s text segments:\n",strs);
		for (j=0;j<nsegs;j++)
			printf("%d %d %d %d %d %d %s\n",j,segs[j].row,segs[j].col,segs[j].ncols,segs[j].color,segs[j].len,segs[j].str);
	}

	nrows = segs[nsegs-1].row+1;

	maxh=maxc=0;
	for (i=0,maxc=0;i<nrows;i++)
		if (mxcir[i]>maxc) maxc=mxcir[i];
	for (i=0,maxh=toth=0;i<nrows;i++)
	{
		if (mxhir[i]>maxh) maxh=mxhir[i];
		toth += mxhir[i]+4;
		rowhgt[i] = i>0? rowhgt[i-1]+mxhir[i-1]+4 : 0;
	}

	logw = NextPowerTwo(max(maxc,targwid));
	logh = NextPowerTwo(toth);
	
	vsty=H_CENTER;
	for (j=0;j<nsegs;j++)
	{
		for (k=j+1;k<nsegs && segs[k].row==segs[j].row;k++)
			vsty = segs[k].haln;
		switch(vsty)
		{
			case H_LEFT:
				coff = 0;
				break;
			case H_CENTER:
				coff = (logw-mxcir[segs[j].row])/2;
				break;
			case H_RIGHT:
				coff = (logw-mxcir[segs[j].row]);
				break;
		}
//		printf("coff: %d\n",coff);
		colpos = segs[j].col+coff;
		XlateTextColor=segs[j].color;
		Brightness=segs[j].bright;
		fp = segs[j].fsel? BigFont : DoomFont;
		for (i=0;i<segs[j].len;i++)
		{
			if (33<=segs[j].str[i] && segs[j].str[i]<95)
        	{

		    	w = fp[segs[j].str[i]-33].pwid;
		    	h = fp[segs[j].str[i]-33].phgt;
                o = fp[segs[j].str[i]-33].pyoff;
				x = colpos;
				y = rowhgt[segs[j].row]-o;

				if (h<mxhir[segs[j].row])
				{
					if (segs[j].valn==V_BOTTOM)
						y += mxhir[segs[j].row]-h+o;
					else if (segs[j].valn==V_MIDDLE)
						y += (mxhir[segs[j].row]-h+o)/2;
				}

				Xlation=1;
		    	OverlayPicture(out,logw,logh,&fp[segs[j].str[i]-33],x,y);
				Xlation=0;
		    	colpos += w;
        	}
        	else colpos += 6;
		}
	}
	ConvertToPicture(out,logw,logw,logh,0,0,outpp);
}

// Change the color and brightness of a text color
// no allocation

int XlateColor(int color,int xlate,int bright)
{
	int c;

	c=color;
	if (c<0) return c;

	switch(xlate)
	{
		case TEXTRED:
			return ColorMap[4*bright][c];
		case TEXTBLUE:
			if (176<=c && c<=191) c = blueidx[c-176];
			else if (c>=0) c = blueidx[15];
			return ColorMap[4*bright][c];
		case TEXTGREEN:
			if (176<=c && c<=191) c = greenidx[c-176];
			else if (c>=0) c = greenidx[15];
			return ColorMap[4*bright][c];
		case TEXTWHITE:
			if (176<=c && c<=191) c = whiteidx[c-176];
			else if (c>=0) c = whiteidx[15];
			return ColorMap[4*bright][c];
		case TEXTGRAY:
			if (176<=c && c<=191) c = grayidx[c-176];
			else if (c>=0) c = grayidx[15];
			return ColorMap[4*bright][c];
		case TEXTYELLOW:
			if (176<=c && c<=191) c = yellowidx[c-176];
			else if (c>=0) c = yellowidx[15];
			return ColorMap[4*bright][c];
	}
	return ColorMap[4*bright][c];
}

void DoubleBitMap(BYTE **out,int *logw,int *logh)
{
	int i,j;
	int w,h;
	BYTE *dp;

	w = (*logw)<<1;
	h = (*logh)<<1;
	dp = *out = (BYTE *) realloc(*out, w*h);
	for (i=h-1;i>=0;i--)
		for (j=w-1;j>=0;j--)
			dp[i*w+j] = dp[(i/2)*(w/2)+j/2];
	*logw = w;
	*logh = h;
}

