// *************************
// Expert Quake Overlay Code
// *************************
// Conventions: Pure Block Emulation, Pure Hungarian
// "ed" is an edict. "sz" is a zero terminated string.
#include "g_local.h"
#include "e_overlay.h"

#define NOEXEC return

void ResizeLevelMemory(void** ppMem, size_t sizeNew, size_t sizeOld)
{
	byte* pOld = (byte *)*ppMem;
	byte* pNew = gi.TagMalloc(sizeNew, TAG_LEVEL);
	
	//assert(ppMem != NULL && 
	//assert(sizeNew > 0 && sizeOld > 0);

	gi.dprintf("new:%d", sizeNew);

	memcpy(pNew, pOld, sizeOld);

	#ifdef _DEBUG
	{
		if (sizeNew > sizeOld)
			memset(pNew+sizeNew, 0xCC, sizeNew - sizeOld);
			
		else if (sizeOld > sizeNew)
			memset((pOld)+sizeNew, 0xCC, sizeOld - sizeNew);
	}
	#endif

	gi.TagFree(pOld);
	*ppMem = (void *)pNew;
}

// ==========================
//  Matrix Drawing Functions
// ==========================

void DrawMatrix (void)
{
	short iClient;
	edict_t* pedPlayer;
	NOEXEC;
	for (iClient = 1; iClient <= game.maxclients; iClient++) {
		pedPlayer = g_edicts + iClient;
		if (pedPlayer->inuse) {
			gi.cprintf(mpedCur, PRINT_HIGH, "somebody's kills against you: %i\n",
			pedPlayer->client->paKillsVersus[mpaTranslateRank[mpedCur->client->iRank]]);
		}
	}
}
// =============================
//  Matrix Management Functions
// =============================

//InitMatrix: Allocates and initializes all data for the first time
void InitMatrix (edict_t* pedFirst)
{
	//allocate the translation table.
	//mpaTranslateRank = gi.TagMalloc(MINPLAYERS, TAG_LEVEL);

	//assumption: KillsVersus and TranslateRank elements are 1 byte long
	//assert(ElementSize(mpaTranslateRank) == 1);

	//record how much room is available
	mcPlayersAllocated = MINPLAYERS;

}

//RankPlayer: accepts a pointer to a client and adjusts their position in the rankings
//if necessary. All clients between the player's current ranking and proper ranking will have
//their maTranslateRank position and iRank index adjusted.
void RankPlayer (edict_t* pedPlayer)
{
	//while this player is better than the next player on the list
		//swap their rank indexes
		//swap their translation table positions
		//move to the new next player on the list

	//otherwise while this player is worse than the previous player on the list
		//swap their rank indexes
		//swap their translation table positions
		//move to the new previous player on the list
}

//updates the scores for the target and killer when the killer kills the target
void UpdateMatrixScores (edict_t* pedTarget, edict_t *pedKiller)
{
	NOEXEC;
	//increment target's death counter
	pedTarget->client->cDeaths++;

	//if the killer is a player, increment killer's KillsVersus against target
	if (pedKiller->client)
		pedKiller->client->paKillsVersus[mpaTranslateRank[pedTarget->client->iRank]]++;
	else
		pedTarget->client->paKillsVersus[mpaTranslateRank[pedTarget->client->iRank]]++;
	
	//update killer's ranking
	RankPlayer(pedKiller);
}

//prepares all arrays for a new player, expanding if necessary
void ExpandMatrix (edict_t* pedJoining)
{
	byte iNewPosition;
	byte iClient; //for looping through all the clients
	edict_t *pedPlayer; //ditto
	byte cOldPlayers;

	NOEXEC;

	//if this is the first client of game, let InitMatrix handle setup
//	if (mcPlayersAllocated == 0) {
//		InitMatrix(pedJoining);
//	}

	//if insufficient room for new client
	if (level.players > mcPlayersAllocated) {
		cOldPlayers = mcPlayersAllocated;
		//keep players allocated a power of two
		mcPlayersAllocated *= 2;
		//go through every client in the game
		for (iClient = 1; iClient <= game.maxclients; iClient++) {
			pedPlayer = g_edicts + iClient;
			//if it is a client, grow their KillsVersus array
			if (pedPlayer->inuse)
				ResizeLevelMemory(&(pedPlayer->client->paKillsVersus), mcPlayersAllocated, cOldPlayers);
		}

		//grow the translation table
		ResizeLevelMemory(&mpaTranslateRank, mcPlayersAllocated, cOldPlayers);
	}

	//zero data for client
	pedJoining->client->cDeaths = 0;

	//allocate and zero the client's new KillsVersus array
	//assumption: size of the array is one byte per score
	pedJoining->client->paKillsVersus = gi.TagMalloc(mcPlayersAllocated, TAG_LEVEL);
	memset(pedJoining->client->paKillsVersus, 0, mcPlayersAllocated);

	//place the client at the end of the Clients array
	iNewPosition = (byte)level.players - 1;
//	mpapedClients[iNewPosition] = pedJoining;

	//place the client at the end of the rankings
	pedJoining->client->iRank = iNewPosition;
	mpaTranslateRank[iNewPosition] = iNewPosition;

	//call RankPlayer to correctly rank the client
	RankPlayer(pedJoining);
}

void ContractMatrix (edict_t* pedLeaving)
{
	//move the player to the end of the rankings
	//rank the player, so they'll be at the end
	//remove them from the Clients array and translation table
	//if empty spaces exceed filled spaces
		//if total spaces equal the minimum, don't change anything
		//record the new allocation size
		//go through every other client in the game
			//shrink their KillsVersus array
		//shrink the array of Clients
		//shrink the translation table
}


// =============================
//  Overlay Statusbar Functions
// =============================

// GetPlayerID: Finds the player closest to the viewer's viewpoint
edict_t *GetPlayerID(edict_t *viewer)
{
	edict_t	*target = NULL,	*bestTarget = NULL;
	vec3_t targetDir, viewDir, diffDir;
	// note that setting bestAngleOffset to 0.5 initially means we will
	// return null and not display a name if no players are visible 
	// within about 60 degrees of the viewpoint
	float angleOffset, bestAngleOffset = 0.5;
	trace_t tr;

	while ((target = G_Find (target, FOFS(classname), "player")) != NULL) {
		// get a vector from viewer to target
		VectorSubtract(target->s.origin, viewer->s.origin, targetDir);

		// normalize the vector to the target and change to angular coordinates
		VectorNormalize(targetDir);
	
		// get a vector for the player's view angle
		AngleVectors(viewer->client->v_angle, viewDir, NULL, NULL);

		// angle difference from viewpoint to target
		_VectorSubtract(targetDir, viewDir, diffDir);

		// since both angles were normalized, cLayoutLengthgth of difference vector represents difference in angle.
		angleOffset = VectorLength(diffDir);

		if (angleOffset < bestAngleOffset) {
			// angle to target is smallest so far. check LOS between target and viewer
			tr = gi.trace (viewer->s.origin, NULL, NULL, target->s.origin, 
					viewer, MASK_SHOT|CONTENTS_SLIME|CONTENTS_LAVA);
			if (tr.ent->client != NULL)	{
				// best so far
			 	bestAngleOffset = angleOffset;
				bestTarget = target;
			}
		}
	}
	return bestTarget;
}

// Version of GetPlayerID that works with non-players
edict_t *GetEdictID(edict_t *viewer)
{
	vec3_t forward, endvec;
	trace_t tr;

	AngleVectors(viewer->client->v_angle, forward, NULL, NULL);
	VectorMA(viewer->s.origin, 8192, forward, endvec);

	tr = gi.trace (viewer->s.origin, NULL, NULL, endvec, viewer,
		MASK_SHOT|CONTENTS_SLIME|CONTENTS_LAVA);

	if (tr.ent != g_edicts)	return tr.ent;
	return NULL;
}

//Appends the statusbar to a person's overlay
void AppendBar (char *pszLayout)
{
	char	szEntry[STRING_CHARS];
	edict_t	*pedFacing;
	int		horizPosition;

	pedFacing = GetPlayerID(mpedCur);
	if (curMode & DEBUGON) pedFacing = GetEdictID(mpedCur);
	if (pedFacing == NULL) return; 

	if (curMode & DEBUGON) {
		horizPosition = 160 - 4 * strlen(pedFacing->classname);
		Com_sprintf(szEntry, sizeof(szEntry), "xv %d yb -60 string \"%s\" ", 
			horizPosition, pedFacing->classname);

	} else {
		horizPosition = 160 - 4 * strlen(pedFacing->client->pers.netname);
		Com_sprintf(szEntry, sizeof(szEntry), "xv %d yb -60 string2 \"%s\" ", 
			horizPosition, pedFacing->client->pers.netname);
	}

	if (strlen(szEntry) + strlen(pszLayout) > LAYOUT_SAFE) return;
	strcat (pszLayout, szEntry);
}

void AppendScoresBar (edict_t *pedPlayer, char *pszLayout)
{
	if (ExpertFlags(EXPERT_PLAYERID)) {
		DecodeConfig(pedPlayer);
		if (curMode & BARON) AppendBar(pszLayout);
	}
}

// =========================
//  Overlay Radar Functions
// =========================

//Places a position and picture pair in the array for drawing
void Plot (edict_t *pedBlip, char pic[PIC_CHARS], int flags)
{
	short x, y, z, angle;
	short col, row, old;

	angle = (short)mpedCur->s.angles[YAW];
	if (curMode & NOROTATE) angle = 0; //always north
	if (angle < 0) angle += 360;

	x = mpedCur->s.origin[0] - pedBlip->s.origin[0];
	y = mpedCur->s.origin[1] - pedBlip->s.origin[1];
	z = mpedCur->s.origin[2] - pedBlip->s.origin[2];

	if (angle > 315 || angle < 45) {//N
		col = y;
		row = x;
	} else if (angle < 135) {//E
		col = -x;
		row = y;
	} else if (angle < 225) {//S
		col = -y;
		row = -x;
	} else if (angle < 315) {//W
		col = x;
		row = -y;
	}

	if (curMode & ZVERT) {
		old = row;
		row = z;
		z = old;
	}

	col = (col / hscale) + (scrWidth[curScreen]/2);
	row = (row / vscale) + (scrWidth[curScreen]/2);

	if (flags & PLOT_Z) {
		if (z < BELOW_FEET) pic = PLOT_BELOW;
		else if (z > ABOVE_FEET) pic = PLOT_ABOVE;
	}

	//plot on edge, even if out of range
	if (flags & PLOT_EDGE) {
		if (col > scrWidth[curScreen]) 
			col = scrWidth[curScreen];
		else if (col < 0)
			col = 0;
		if (row > scrHeight[curScreen])
			row = scrHeight[curScreen];
		else if (row < 0)
			row = 0;

	} else if (col > scrWidth[curScreen] || col < 0 ||
			  row > scrHeight[curScreen] || row < 0) return;
		
	if (flags & PLOT_FORCE || fStrMatch(blip[col][row], NOPIC)) {
		strcpy (blip[col][row], pic);
		IsColumn[col] = true;
	} 
}

//Runs Plot for weapons and powerups
void PlotItems(void)
{
	char wnames[9][32] = {"weapon_shotgun","weapon_supershotgun",
		"weapon_machinegun","weapon_chaingun","weapon_grenadelauncher",
		"weapon_rocketlauncher","weapon_hyperblaster","weapon_railgun",
		"weapon_bfg"};
	//all powerups but adrenaline, enviro, breather, ancient head, jacket and combat armor
	char pnames[9][32] = {"item_health_mega","item_pack",
		"item_bandolier","item_power_screen","item_armor_combat",
		"item_invulnerability","item_sicLayoutLengthcer","item_power_shield",
		"item_quad"}; 
	edict_t *pedBlip = NULL;
	short i;

	if (gametype == GAME_SINGLE) {
		while ((pedBlip = G_Find (pedBlip, FOFS(classname), "target_secret")) != NULL) {
			Plot(pedBlip, SECRET, PLOT_Z);
		}
	}

	if (curMode & SHOWWEAP) {
		for (i=0;i<9;i++) {
			while ((pedBlip = G_Find (pedBlip, FOFS(classname), wnames[i])) != NULL) {
				Plot(pedBlip, WEAP, 0);
			}
		}
	}
	if (curMode & SHOWPOWER) {
		for (i=0;i<9;i++) {
			while ((pedBlip = G_Find (pedBlip, FOFS(classname), pnames[i])) != NULL) {
				Plot(pedBlip, POWER, PLOT_EDGE);
			}
		}
	}
}

//Appends all plots, backdrop and stuff to string
void AppendRadar (char *pszLayout)
{
	char	szEntry[STRING_CHARS];
	short	col, row;

	//empty it out
	for (col = 0; col <= scrWidth[curScreen];col++) {
		IsColumn[col] = false;
		for (row=0;row <= scrHeight[curScreen];row++) {
			strcpy(blip[col][row], NOPIC);
		}
	}

	//backdrop
	if (curMode & SHOWBACK) {
		Com_sprintf (szEntry, sizeof(szEntry), "xl %d yt %d picn %s ",
			hpos, vpos, scrPic[curScreen]);
		if (strlen(szEntry) + strlen(pszLayout) > LAYOUT_SAFE) 
			return; //too big
		strcat(pszLayout, szEntry);
	}

	switch (gametype) { 
		case GAME_SINGLE:
		case GAME_DM:
			PlotItems();
			break;
		default: //turn off expert radar
			gi.dprintf("Expert Radar Warning: No plots for current game mode.\n");
			if (ExpertFlags(EXPERT_RADAR)) sv_expflags->value -= EXPERT_RADAR;
			return;
	}

	if (curMode & SHOWSELF) 
		Plot(mpedCur, SELF, 0);
	
	//blips
	for (col = 0; col <= scrWidth[curScreen]; col++) 
	{
		if (IsColumn[col] == false) 
			continue; 
		Com_sprintf (szEntry, sizeof(szEntry), "xl %d ", hpos+col); 
		if (strlen(szEntry) + strlen(pszLayout) > LAYOUT_SAFE) 
			return; 
		strcat(pszLayout, szEntry);
		for (row = 0; row <= scrHeight[curScreen]; row++) 
		{
			if (fStrMatch(blip[col][row], NOPIC)) 
				continue; 
			Com_sprintf (szEntry, sizeof(szEntry), "yt %d picn %s ", vpos+row, blip[col][row]);
			if (strlen(szEntry) + strlen(pszLayout) > LAYOUT_SAFE) 
				return; 
			strcat(pszLayout, szEntry);
		}
	}
}

// ===========================
//  Overlay Control Functions
// ===========================
void EncodeConfig(void)
{
	if (ExpertFlags(EXPERT_RADAR)) {
		SetIntKey("hpos", hpos);
		SetIntKey("vpos", vpos);
		SetIntKey("hscale", hscale);
		SetIntKey("vscale", vscale);
		SetIntKey("screen", curScreen);
	}
	SetIntKey("mode", curMode);
}

void ResetConfig(void)
{
	hpos = INIT_HPOS;
	vpos = INIT_VPOS;
	hscale = INIT_HSCALE;
	vscale = INIT_VSCALE;
	curScreen = INIT_SCREEN;
	curMode = INIT_MODE;
	curMode |= RADARON | BARON | ISVALID;
}

void DecodeConfig(edict_t* pedNew)
{
	if (mpedCur == pedNew) return;
	
	mpedCur = pedNew;
	hpos = KeyToShort("hpos");
	vpos = KeyToShort("vpos");
	hscale = KeyToShort("hscale");
	vscale = KeyToShort("vscale");
	curScreen = KeyToShort("screen");
	curMode = KeyToShort("mode");

	if (ExpertFlags(EXPERT_RADAR)) {
		if (hscale < 2 || vscale < 2 || ~curMode & ISVALID ||
			curScreen < 1 || curScreen > (MAX_SCREENS-1) ) {
			ResetConfig();
			EncodeConfig();
		}
	} else { // just check mode
		if (~curMode & ISVALID) {
			ResetConfig();
			EncodeConfig();
		}
	}
}

void PrintConfig(void)
{
	gi.cprintf(mpedCur, PRINT_HIGH, "Current Overlay Settings:\n");
	gi.cprintf(mpedCur, PRINT_HIGH, "Screen position (horiz/vert): %d / %d.\n", hpos, vpos);
	gi.cprintf(mpedCur, PRINT_HIGH, "Scale (horiz/vert): %d / %d.\n", hscale, vscale);
	gi.cprintf(mpedCur, PRINT_HIGH, "Screen size: %d. Allowed sizes: 1 to %d.\n", curScreen, MAX_SCREENS-1);
	gi.cprintf(mpedCur, PRINT_HIGH, "Mode bitvector: %d.\n", curMode);
}

qboolean OverlayCommand (edict_t *pedPlayer)
{
	char szLayout[LAYOUT_CHARS] = "";
	char* sCommand = gi.argv(0);
	int parameter = atoi(gi.argv(1));

	DecodeConfig(pedPlayer);

	if (fStrMatch(sCommand, "reset")) {
		ResetConfig();

	} else if (fStrMatch(sCommand, "matrix")) {
		if (parameter > 0) curMode |= MATRIXON;
		else curMode ^= MATRIXON;

	} else if (fStrMatch(sCommand, "config")) {
		PrintConfig();

	} else if (fStrMatch(sCommand, "legend")) {
		switch(gametype) {
			case GAME_SINGLE:
				gi.cprintf(mpedCur, PRINT_HIGH, LEGEND_SINGLE);
				break;
			case GAME_DM:
				gi.cprintf(mpedCur, PRINT_HIGH, LEGEND_DM);
				break;
			default:
				gi.cprintf(mpedCur, PRINT_HIGH, "No legend defined for current game mode.\n");
				break;
		}
	} else if (fStrMatch(sCommand, "xyzzy")) {
		AppendRadar(szLayout);
		mpedCur->client->showscores = true;
		strcpy (mpedCur->client->oldlayout, szLayout);
		gi.WriteByte(svc_layout);
		gi.WriteString (szLayout);
		gi.unicast(mpedCur, false);
	
	//mode toggles Note: fStrMatch is case insensitive
	} else if (fStrMatch(sCommand, "STATUSBAR")) {
		if (parameter > 0) curMode |= BARON;
		else curMode ^= BARON;
	} else if (fStrMatch(sCommand, "RADAR")) {
		if (parameter > 0) curMode |= RADARON;
		else curMode ^= RADARON;
	} else if (fStrMatch(sCommand, "SHOWBACK")) {
		if (parameter > 0) curMode |= SHOWBACK;
		else curMode ^= SHOWBACK;
	} else if (fStrMatch(sCommand, "SHOWSELF")) {
		if (parameter > 0) curMode |= SHOWSELF;
		else curMode ^= SHOWSELF;
	} else if (fStrMatch(sCommand, "NOROTATE")) {
		if (parameter > 0) curMode |= NOROTATE;
		else curMode ^= NOROTATE;
	} else if (fStrMatch(sCommand, "ZVERT")) {
		if (parameter > 0) curMode |= ZVERT;
		else curMode ^= ZVERT;
	} else if (fStrMatch(sCommand, "DEBUG")) {
		if (parameter > 0) curMode |= DEBUGON;
		else curMode ^= DEBUGON;
	} else if (fStrMatch(sCommand, "SHOWWEAP")) {
		if (parameter > 0) curMode |= SHOWWEAP;
		else curMode ^= SHOWWEAP;
	} else if (fStrMatch(sCommand, "SHOWPOWER")) {
		if (parameter > 0) curMode |= SHOWPOWER;
		else curMode ^= SHOWPOWER;

	//setting commands
	} else if (fStrMatch(sCommand, "hpos")) {
		if (parameter > 0 && parameter < 1600) hpos = parameter;
		else gi.cprintf(mpedCur, PRINT_HIGH, "Unusable horizontal position setting.\n");
	} else if (fStrMatch(sCommand, "vpos")) {
		if (parameter > 0 && parameter < 1280) vpos = parameter;
		else gi.cprintf(mpedCur, PRINT_HIGH, "Unusable vertical position setting.\n");
	} else if (fStrMatch(sCommand, "scale")) { //both
		if (parameter > 2 && parameter < 8000) hscale = vscale = parameter;
		else gi.cprintf(mpedCur, PRINT_HIGH, "Unusable scale setting.\n");
	} else if (fStrMatch(sCommand, "hscale")) {
		if (parameter > 2 && parameter < 8000) hscale = parameter;
		else gi.cprintf(mpedCur, PRINT_HIGH, "Unusable horizontal scale setting.\n");
	} else if (fStrMatch(sCommand, "vscale")) {
		if (parameter > 2 && parameter < 8000) vscale = parameter;
		else gi.cprintf(mpedCur, PRINT_HIGH, "Unusable vertical scale setting.\n");
	} else if (fStrMatch(sCommand, "size")) {
		if (parameter > 0 && parameter < MAX_SCREENS) curScreen = parameter;
		else gi.cprintf(mpedCur, PRINT_HIGH, "Unusable size setting.\n");
	} else if (fStrMatch(sCommand, "mode")) { //direct changes
		if (parameter > 1 && parameter < 99999) curMode = parameter;
		else gi.cprintf(mpedCur, PRINT_HIGH, "Unusable mode setting.\n");
	} else return false; //command doesn't match

	EncodeConfig();
	OverlayThink(mpedCur, true);
	return true; //command matched
}

void LoadImages (void)
{
	int i;

	if ((int)sv_expflags->value & EXPERT_RADAR) {
		
		for (i=1;i<MAX_SCREENS;i++) //skip first, it's a null entry
			gi.imageindex(scrPic[i]);

		gi.imageindex (PLOT_ABOVE);
		gi.imageindex (PLOT_BELOW);

		gi.imageindex (WEAP);
		gi.imageindex (POWER);
	}
}

void OverlayThink(edict_t *pedViewer, qboolean force)
{
	char szLayout[LAYOUT_CHARS] = "";
	int layoutLength = 0;
	gclient_t *pclCur = pedViewer->client;
	
	if (force)
		pclCur->IsHelp = pclCur->showinventory = false;
	else {
		if (pclCur->IsHelp || pclCur->showinventory) 
			return; //Scoreboard / inventory is up
		if (level.framenum < pclCur->updateframe) 
			return; //Not time yet
	}

	pclCur->showscores = false;
	pclCur->updateframe = level.framenum + EXPERT_UPDATE_FRAMES;
	DecodeConfig(pedViewer);
	
	//main actions
	if (ExpertFlags(EXPERT_PLAYERID) && curMode & BARON) AppendBar(szLayout);
	if (ExpertFlags(EXPERT_RADAR) && curMode & RADARON) AppendRadar(szLayout);

	// If there's nothing to print, quit and leave showscores off
	layoutLength = strlen(szLayout);
	if (layoutLength == 0) {
		strncpy (pclCur->oldlayout, szLayout, LAYOUT_SAFE);
		if (curMode & DEBUGON) 
			gi.cprintf(mpedCur, PRINT_MEDIUM, "No overlay to send\n");
		return;
	} else if (layoutLength > LAYOUT_SAFE) { 
		strncpy (pclCur->oldlayout, szLayout, LAYOUT_SAFE);
		if (curMode & DEBUGON) 
			gi.cprintf(mpedCur, PRINT_MEDIUM, "Overflow too large\n");
		return;
	}

	pclCur->showscores = true;

	// If it's the same as last time, quit but leave showscores ON
	if (!force && fStrMatch(pclCur->oldlayout, szLayout)) {
		if (curMode & DEBUGON) 
			gi.cprintf(mpedCur, PRINT_MEDIUM, "Old and new overlays same\n");
		return;
	}
	
	if (curMode & DEBUGON) 
		gi.cprintf(mpedCur, PRINT_MEDIUM, "Sent %i byte layout\n", layoutLength);

	strcpy (pclCur->oldlayout, szLayout);
	gi.WriteByte(svc_layout);
	gi.WriteString (szLayout);
	gi.unicast(mpedCur, false);
}
