/* **************************************************************************
 *
 *	xjscal.c	XJoystick Calibratior
 *	-------------------------------------
 *	Functions:
 *
 *	void xjscal_help()
 *	void xjscal_version()
 *	void xjscal_exit_CB()
 *	int xjscal_redraw_buttons()
 *	int xjscal_redraw_switches()
 *	int xjscal_redraw_readout()
 *	int xjscal_redraw_coords()
 *	int xjscal_redraw_all()
 *	int xjscal_do_caliberation()
 *	int main()
 *
 *	---
 *
 *
 *
 *
 *
 *
 *
 *
 */

#include "xjscal.h"

#include "images/crosshair.xpm"
#include "images/crosshairm.xbm"
#include "images/exit_button0.xpm"
#include "images/exit_button1.xpm"
#include "images/jscoords.xpm"
#include "images/mode_switch0.xpm"
#include "images/mode_switch1.xpm"
#include "images/num_switch0.xpm"
#include "images/num_switch1.xpm"
#include "images/num_switch2.xpm"
#include "images/num_switch3.xpm"
#include "images/readout.xpm"
#include "images/save_button0.xpm"
#include "images/save_button1.xpm"
#include "images/toplevel_bkg.xpm"




/* ****************************************************************************
 *
 *   void xjscal_help()
 *
 *	Prints help screen.
 *
 *	No return value.
 */
void xjscal_help()
{
 printf("Usage: xjscal [options] [toolkitoptions]\n\n");
 printf("   [options] can be any of the following:\n\n");
 printf("      -d /dev/js0               Start up reading on this joystick node.\n");
 printf("      -f /home/me/.joystick     Joystick caliberation file to write to.\n");
 printf("      --help | -h               Print (this) help screen and exit.\n");
 printf("      --version                 Print version info and exit.\n");
 printf("\n");

 return;
}



/* **************************************************************************
 *
 *   void xjscal_version()
 *
 *	Prints version information.
 *
 *	No return value.
 */
void xjscal_version()
{
 printf("%s Version %s\n\n", XJSCAL_NAME, XJSCAL_VERSION);

 printf(XJSCAL_COPYRIGHT);

 return;
}



/* *********************************************************************
 *
 *   void xjscal_exit_CB()
 *
 *	Exit sequence.
 *
 *	No return value.
 */
void xjscal_exit_CB()
{
	/* Free the pixmaps. */
	XFreePixmap(display, toplevel_bkg_pixmap);
        XFreePixmap(display, exit_button0_pixmap);
        XFreePixmap(display, exit_button1_pixmap);
        XFreePixmap(display, jscoords_pixmap);
        XFreePixmap(display, mode_switch0_pixmap);
        XFreePixmap(display, mode_switch1_pixmap);
        XFreePixmap(display, num_switch0_pixmap);
        XFreePixmap(display, num_switch1_pixmap);
        XFreePixmap(display, num_switch2_pixmap);
        XFreePixmap(display, num_switch3_pixmap);
        XFreePixmap(display, readout_pixmap);
        XFreePixmap(display, save_button0_pixmap);
        XFreePixmap(display, save_button1_pixmap);


        /* Free the colormap. */
        XFreeColormap(display,colormap);

        /* Close the connection to display. */
        XCloseDisplay(display);

        /* Close the joystick device */
        close(cur_js_fd);


        return;
}



/* ********************************************************************************
 *
 *   int xjscal_redraw_buttons()
 *
 *	Redraws all buttons.
 *
 *	Returns non-zero on error.
 */
int xjscal_redraw_buttons()
{
	XClearWindow(display, js_save_button_win);
	if(js_save_button_state)
	{
		XSetWindowBackgroundPixmap(display,
			js_save_button_win, save_button1_pixmap);
	}
	else
	{
                XSetWindowBackgroundPixmap(display,
                        js_save_button_win, save_button0_pixmap);
	}

        XClearWindow(display, js_exit_button_win);
        if(js_exit_button_state)
        {
                XSetWindowBackgroundPixmap(display,
                        js_exit_button_win, exit_button1_pixmap);
        }
        else
        {   
                XSetWindowBackgroundPixmap(display,
                        js_exit_button_win, exit_button0_pixmap);
        }
   

	return(0);
}



/* ********************************************************************************
 *   
 *   int xjscal_redraw_switches()
 *
 *	Redraws all switch windows.
 *
 *	Returns non-zero on error.
 */
int xjscal_redraw_switches()
{
	/* Redraw js_mode_switch_win */
        XClearWindow(display, js_mode_switch_win);
	switch(js_mode_switch_state)
	{
	    case 0:
		XSetWindowBackgroundPixmap(display,
			js_mode_switch_win, mode_switch0_pixmap);
		break;
            case 1:
                XSetWindowBackgroundPixmap(display,
                        js_mode_switch_win, mode_switch1_pixmap);
                break;
	    default:
                XSetWindowBackgroundPixmap(display,
                        js_mode_switch_win, mode_switch0_pixmap);
                break;
	}

        /* Redraw js_num_switch_win */
	XClearWindow(display, js_num_switch_win);
	switch(js_number_switch_state)
	{
            case 0:
                XSetWindowBackgroundPixmap(display,
                        js_num_switch_win, num_switch0_pixmap);
                break;
            case 1:
                XSetWindowBackgroundPixmap(display,
                        js_num_switch_win, num_switch1_pixmap);
                break;
            case 2:
                XSetWindowBackgroundPixmap(display,
                        js_num_switch_win, num_switch2_pixmap);
                break;
            case 3:
                XSetWindowBackgroundPixmap(display,
                        js_num_switch_win, num_switch3_pixmap);
                break;
            default:
                XSetWindowBackgroundPixmap(display,
                        js_num_switch_win, num_switch0_pixmap);
                break;
	}


	return(0);
}


/* *****************************************************************************
 *
 *   int xjscal_redraw_readout()
 *
 *      Redraws readout window.
 *      
 *      Returns non-zero on error.
 */
int xjscal_redraw_readout()
{
	/* Local variables. */
	int js_num;
	char stringa[256];


	XClearWindow(display, js_readout_win);
        XSetWindowBackgroundPixmap(display,
           js_readout_win, readout_pixmap);



	js_num = js_number_switch_state;
	if(js_num >= MAX_JOYSTICKS)
		js_num = 0;


	/* When in caliberation mode, min and max values */
	/* are displayed.                                */
	if(js_mode_switch_state == 1)
	{
		sprintf(stringa, "X MIN: %i",
			js_caliberation_record[js_num].x_min);
                XDrawString(display, js_readout_win, the_GC,
                        32, 31, stringa, strlen(stringa));

                sprintf(stringa, "X MAX: %i",
                        js_caliberation_record[js_num].x_max);
                XDrawString(display, js_readout_win, the_GC,
                        32, 62, stringa, strlen(stringa));

                sprintf(stringa, "Y MIN: %i",
                        js_caliberation_record[js_num].y_min);
                XDrawString(display, js_readout_win, the_GC,
                        32, 91, stringa, strlen(stringa));
                
                sprintf(stringa, "Y MAX: %i",
                        js_caliberation_record[js_num].y_max);
                XDrawString(display, js_readout_win, the_GC,
                        32, 120, stringa, strlen(stringa));

                sprintf(stringa, "NZ: %i",
                        js_caliberation_record[js_num].nullzone);
                XDrawString(display, js_readout_win, the_GC,
                        32, 149, stringa, strlen(stringa));

	}
	else
	{
	    /* Button 1. */
	    if(js.buttons & 1)
                sprintf(stringa, "B1: ON");
	    else
	        sprintf(stringa, "B1: OFF");

            XDrawString(display, js_readout_win, the_GC,
                32, 31, stringa, strlen(stringa));

            /* Button 2. */   
            if(js.buttons & 2)
                sprintf(stringa, "B2: ON");
            else
                sprintf(stringa, "B2: OFF");
        
            XDrawString(display, js_readout_win, the_GC,
                32, 62, stringa, strlen(stringa));


            /* Button 3. */
            if(js.buttons & 3)
                sprintf(stringa, "B3: ON");
            else
                sprintf(stringa, "B3: OFF");
        
            XDrawString(display, js_readout_win, the_GC,
                32, 91, stringa, strlen(stringa));


            /* Button 4. */
            if(js.buttons & 4)
                sprintf(stringa, "B4: ON");
            else
                sprintf(stringa, "B4: OFF");
        
            XDrawString(display, js_readout_win, the_GC,
                32, 120, stringa, strlen(stringa));

            /* Button 5. */
            if(js.buttons & 5)
                sprintf(stringa, "B5: ON");
            else
                sprintf(stringa, "B5: OFF");
        
            XDrawString(display, js_readout_win, the_GC,
                32, 149, stringa, strlen(stringa));

	}


	return(0);
}



/* ******************************************************************************
 *
 *   int xjscal_redraw_coords()
 *
 *	Redraws js_coords_win window.
 *
 *      Returns non-zero on error.
 */
int xjscal_redraw_coords()
{
	/* Local variables. */
	char stringa[128];
	int x;
	double nz;
	double js_x, js_y;
	double js_x_min, js_x_max;
        double js_y_min, js_y_max;

	/* Clear window. */
	XClearWindow(display, js_coords_win);
	XSetWindowBackgroundPixmap(display,
	   js_coords_win, jscoords_pixmap);

	/* Get joystick number. */
	x = js_number_switch_state;
	if(x >= MAX_JOYSTICKS)
		x = MAX_JOYSTICKS - 1;


	/* Set the foreground color. */
	XSetForeground(display, the_GC, readout_text_pixel);


	/* Are we in caliberation mode? */
	if(js_mode_switch_state == 1)
	{
            XUnmapWindow(display, crosshair_win);
            
            sprintf(stringa, "Caliberating:");  
            XDrawString(display, js_coords_win, the_GC,
                27, 58, stringa, strlen(stringa)
            );
            sprintf(stringa, "Move joystick to all");
            XDrawString(display, js_coords_win, the_GC,
                30, 76, stringa, strlen(stringa)
            );
            sprintf(stringa, "extreme corners.");
            XDrawString(display, js_coords_win, the_GC,
                30, 94, stringa, strlen(stringa)
            );

            sprintf(stringa, "Switch back to");
            XDrawString(display, js_coords_win, the_GC,
                30, 122, stringa, strlen(stringa)
            );

            sprintf(stringa, "\"Test\" when done.");
            XDrawString(display, js_coords_win, the_GC,
                30, 140, stringa, strlen(stringa)
            );


	    return(0);
	}


	/* See if joystick is valid. */
	if(js_caliberation_record[x].exists < 0)
	{
	    XUnmapWindow(display, crosshair_win);

	    sprintf(stringa, "No Joystick On:");
            XDrawString(display, js_coords_win, the_GC,
		38, 78, stringa, strlen(stringa)
	    );
	    sprintf(stringa, cur_js_device_node);
            XDrawString(display, js_coords_win, the_GC,
                45, 95, stringa, strlen(stringa)
            );
	}
	else
	{

	    /* Fetch the joystick structure data from joystick. */
	    read(cur_js_fd, &js, JS_RETURN);

	    /* Draw circle indicating nullzone. */
            js_x_min = js_caliberation_record[x].x_min;
            js_x_max = js_caliberation_record[x].x_max;
            js_y_min = js_caliberation_record[x].y_min;
            js_y_max = js_caliberation_record[x].y_max;

	    nz = js_caliberation_record[x].nullzone;
	    /* Nullzone dipicted relative to X axis min and max. */
	    nz = nz / (js_x_max - js_x_min);
	    nz = 152 * nz;
	    js_x = 93 - (nz / 2);
	    js_y = 95 - (nz / 2);
	    XDrawArc(display, js_coords_win, the_GC,
		js_x, js_y,
		nz, nz,
		0, 23040
 	    );


            /* Move the crosshairs. */
            js_x = js.x;
            js_y = js.y;
            
            js_x_min = js_caliberation_record[x].x_min;
            js_x_max = js_caliberation_record[x].x_max;
            js_y_min = js_caliberation_record[x].y_min;
            js_y_max = js_caliberation_record[x].y_max;
            
            js_x = js_x / (js_x_max - js_x_min);
            js_x = (js_x * 150) - 8 + 18;
            js_y = js_y / (js_y_max - js_y_min);
            js_y = (js_y * 154) - 8 + 15;

	    if(js_x > 159)
		js_x = 159;
            if(js_x < 18)
                js_x = 18;
            if(js_y > 159)
                js_y = 159;
            if(js_y < 15) 
                js_y = 15;  

            XMapWindow(display, crosshair_win);
	    XMoveWindow(display, crosshair_win, js_x, js_y);
	}



	return(0);
}



/* ******************************************************************************
 *
 *   int xjscal_redraw_all()
 *       
 *      Redraws all windows.
 *
 *      Returns non-zero on error.
 */
int xjscal_redraw_all()
{
	/* Redraw toplevel. */
	XClearWindow(display, toplevel);
        XSetWindowBackgroundPixmap(display,
           toplevel, toplevel_bkg_pixmap);
	
	/* Redraw buttons. */
	xjscal_redraw_buttons();

	/* Redraw switches. */
	xjscal_redraw_switches();

	/* Redraw readout. */
	xjscal_redraw_readout();


	return(0);
}



/* **************************************************************************
 *
 *   int xjscal_do_caliberation()
 *
 *	Does the caliberation by updating the min and max of the currently
 *	selected joystick record.   This function should be called on
 *	every loop when in caliberation mode.
 *
 */
int xjscal_do_caliberation()
{
	/* Local variables. */
	int x;


	/* Get number of joystick we are caliberating. */
	x = js_number_switch_state;

	/* See if that joystick is valid. */
	if(js_caliberation_record[x].exists < 0)
		return(-1);

	/* Fetch joystick coordinates. */
        read(cur_js_fd, &js, JS_RETURN);


	/* If stick has gone farther than last min or max, */
	/* then get new min or max.                        */
	if(js.x < js_caliberation_record[x].x_min)
		js_caliberation_record[x].x_min = js.x;

	if(js.x > js_caliberation_record[x].x_max)
		js_caliberation_record[x].x_max = js.x;


        if(js.y < js_caliberation_record[x].y_min)
                js_caliberation_record[x].y_min = js.y;
        
        if(js.y > js_caliberation_record[x].y_max)
                js_caliberation_record[x].y_max = js.y;


	return(0);
}



int main(int argc, char *argv[])
{
        /* Local variables */
        int x, y, z;
	char stringa[256];

	Bool is_geometry_set=False;


	/* xjscal Window Geometry. */
        int xjscal_x = 0;
	int xjscal_y = 0;
        unsigned int xjscal_width=320;
        unsigned int xjscal_height=240;

	long readout_update_last = MilliTime();


        /* ******************************************************* */
        /* ***      Set Global Variables to Default Values     *** */
        rootpid = getpid();
        childpid = rootpid;
        child_procs = 0;

        normal_fontname="7x14";         /* Default fontnames. */
        bold_fontname="7x14";

        button_draw_count = 0;

	cur_js_device_node = DEV_JOYSTICK_0;

	js_number_switch_state = 0;     /* 0 to 3, default 0. */
	js_mode_switch_state = 0;	/* 0 or 1, default 0. */

	/* Set joystick structure to defaults. */
	for(x = 0; x < MAX_JOYSTICKS; x++)
	{
		js_caliberation_record[x].exists = -1;
		js_caliberation_record[x].x_min = 15;
                js_caliberation_record[x].x_max = 1200;
                js_caliberation_record[x].y_min = 15;
                js_caliberation_record[x].y_max = 1200;
		js_caliberation_record[x].nullzone = 20;
	}

	js_save_button_state = False;
	js_exit_button_state = False;

	sprintf(js_calibration_file_fullname, "%s/%s",
		getenv("HOME"), JS_CALIBRATION_FILE);


        /* ********************************************* */
        /* ***     Parse Command Line Parameters     *** */
        for(x = 1; x < argc; x++)
	{
          if( 0 == strcmp(argv[x],"--help") )
          {
                xjscal_help();
                exit(0);
          }
          else if( 0 == strcmp(argv[x],"-help") )
          {
                xjscal_help();
                exit(0);
          }  
          else if( 0 == strcmp(argv[x],"-h") )
          {
                xjscal_help();
                exit(0);
          }
          else if( 0 == strcmp(argv[x],"--version") )
          {
                xjscal_version();
                exit(0);
          }
          else if( 0 == strcmp(argv[x],"-d") )
	  {
		x++;
		cur_js_device_node = argv[x];
	  }
          else if( 0 == strcmp(argv[x],"-f") )
          {
                x++;
                sprintf(js_calibration_file_fullname, argv[x]);
          }
          else if( 0 == strcmp(argv[x],"-geometry") )
          {
              x++;
              XParseGeometry(argv[x],
                  &xjscal_x, &xjscal_y,    /* x, y coordinates. */
                  &z, &z                   /* width, height (not used). */
              );
              is_geometry_set=True;
          }
        }



	/* See which device node we need to switch to at startup. */
	if( 0 == strcmp(cur_js_device_node, DEV_JOYSTICK_0) )
	{
		js_number_switch_state = 0;
	}
        else if( 0 == strcmp(cur_js_device_node, DEV_JOYSTICK_1) )
        {
                js_number_switch_state = 1;
        }
        else if( 0 == strcmp(cur_js_device_node, DEV_JOYSTICK_2) )
        {
                js_number_switch_state = 2;
        }
        else if( 0 == strcmp(cur_js_device_node, DEV_JOYSTICK_3) )
        {
                js_number_switch_state = 3;
        }


        /* ****************************************************** */
        /* If the caliberation file does not exist, start up in   */
	/* caliberation mode.                                     */
        if(access(js_calibration_file_fullname, F_OK) != 0)
        {
                js_mode_switch_state = 1;
		x = js_number_switch_state;
                /* Reset the joystick records before caliberating. */
                js_caliberation_record[x].x_min = MAX_JS_X_AXIS_VALUE;
                js_caliberation_record[x].x_max = MIN_JS_X_AXIS_VALUE;
                js_caliberation_record[x].y_min = MAX_JS_Y_AXIS_VALUE;
                js_caliberation_record[x].y_max = MIN_JS_Y_AXIS_VALUE;
        }


        /* ************************************************* */
        /* ***          Initialize JoyStick              *** */

	/* Read caliberation from file. */
	x = JSReadCaliberationFile(js_calibration_file_fullname);

	/* Get which joystick device nodes exist. */
	if(access(DEV_JOYSTICK_0, R_OK) == 0)
		js_caliberation_record[0].exists = 0;
        if(access(DEV_JOYSTICK_1, R_OK) == 0)
                js_caliberation_record[1].exists = 0; 
        if(access(DEV_JOYSTICK_2, R_OK) == 0)
                js_caliberation_record[2].exists = 0; 
        if(access(DEV_JOYSTICK_3, R_OK) == 0)
                js_caliberation_record[3].exists = 0; 

	/* Does the one we want to start up with exist? */
	if(access(cur_js_device_node, R_OK) != 0)
	{
	   fprintf(stderr, "Error: Cannot access device node: %s\n",
		cur_js_device_node);
	}
        if(access(cur_js_device_node, F_OK) != 0)
        {       
           fprintf(stderr, "Error: Non-existant device node: %s\n",
                cur_js_device_node);
        }

	/* Open joystick device. */
        if(JSOpenDevice(cur_js_device_node) != 0)
	{
           fprintf(stderr,
	        "Error: Unable to open joystick device: %s\n",
		cur_js_device_node
	   );
	   fprintf(stderr,
		"Is the joystick device already in use by another program?\n"
	   );
	   fprintf(stderr,
		"Is the joystick module loaded? Type 'lsmod' to check.\n"
	   );
           fprintf(stderr, 
                "Is your joystick plugged in?\n"
	   );
	   fprintf(stderr,
    "If your sound/game card is Plug and Play, is it properly initialized?\n"
 	   );
	}



        /* *********************************************************** */
        /*     Initiate X display connection and set X resources.      */

        /* Open a connection to the Display. */
        if((display=XOpenDisplay(NULL)) == NULL)
        {
           fprintf(stderr,"%s: cannot connect to X server",argv[0]);
           if(getenv("DISPLAY") == NULL)
              fprintf(stderr,", 'DISPLAY' environment variable not set.\n");
           else
              fprintf(stderr," %s\n", XDisplayName(NULL));
           return(-1);
        }


        /* Get some default values of the server. */
        scr_num = DefaultScreen(display);
        scr_ptr = DefaultScreenOfDisplay(display);
        visual = DefaultVisualOfScreen(scr_ptr);
        depth = DefaultDepthOfScreen(scr_ptr);
        black_pix = BlackPixelOfScreen(scr_ptr);
        white_pix = WhitePixelOfScreen(scr_ptr);
        the_GC = DefaultGC(display,scr_num);
        colormap = DefaultColormap(display, scr_num);


        /* Get xfontinfos. */
        XuGetFontInfo(&normal_xfont, normal_fontname);
        XuGetFontInfo(&bold_xfont, bold_fontname);
        XSetFont(display, the_GC, normal_xfont->fid);  /* Use normal_xfont. */


        /* Get color pixels. */
        normal_text_pixel =
           XuGetColor(NORMAL_TEXT_COLOR_SPEC, normal_text_xcolor);
        bold_text_pixel =
           XuGetColor(BOLD_TEXT_COLOR_SPEC, bold_text_xcolor);
        readout_text_pixel =
           XuGetColor(READOUT_TEXT_COLOR_SPEC, readout_text_xcolor);


        /* Set foreground color. */
        XSetForeground(display, the_GC, white_pix);


        /* Setting the window attributes. */
        attributes.background_pixel = white_pix;
        attributes.background_pixmap = ParentRelative;
        attributes.border_pixel = white_pix;
        attributes.border_pixel = None;
        attributes.backing_store = NotUseful;
        attributes.bit_gravity = CenterGravity; 
        attributes.win_gravity = StaticGravity;
        attributes.colormap = DefaultColormapOfScreen(scr_ptr);
        attributes.cursor = XCreateFontCursor(display,XC_left_ptr);


        /* ***      XLib Resources Initialization Finished       *** */
        /* ********************************************************* */


        /* ********************************************************* */
        /* ***             Create Primary Windows                *** */

        toplevel = XCreateWindow(
           display,
           DefaultRootWindow(display),
           xjscal_x,
           xjscal_y,
           xjscal_width,      /* width. */
           xjscal_height,     /* height. */
           0,                 /* border_width. */
           depth,             /* depth from parent. */
           InputOutput,       /* IO option. */
           visual,
           WIN_ATTR_MASK,
           &attributes
        );

        js_coords_win = XCreateSimpleWindow(display,
           toplevel, 130, 0, 190, 177, 0, white_pix, black_pix);

	crosshair_win = XCreateSimpleWindow(display,
	   js_coords_win, 0, 0, 16, 16, 0, white_pix, black_pix);

	js_readout_win = XCreateSimpleWindow(display,
	   toplevel, 0, 0, 114, 163, 0, white_pix, black_pix);

	js_mode_switch_win = XCreateSimpleWindow(display,
	   toplevel, 130, 190, 83, 50, 0, white_pix, black_pix);

	js_num_switch_win = XCreateSimpleWindow(display,
	   toplevel, 0, 160, 78, 80, 0, white_pix, black_pix);

	js_save_button_win = XCreateSimpleWindow(display,
	   toplevel, 213, 195, 46, 26, 0, white_pix, black_pix);

	js_exit_button_win = XCreateSimpleWindow(display,
	   toplevel, 260, 195, 42, 27, 0, white_pix, black_pix);


	/* Load all pixmaps. */
	crosshair_pixmap = readpixmapdata(crosshair_xpm);
	crosshairm_bitmap = XCreateBitmapFromData(display,
		toplevel,
		crosshairm_bits,
		crosshairm_width,
		crosshairm_height
	);
        toplevel_bkg_pixmap = readpixmapdata(toplevel_bkg_xpm);
        exit_button0_pixmap = readpixmapdata(exit_button0_xpm);
        exit_button1_pixmap = readpixmapdata(exit_button1_xpm);
        jscoords_pixmap = readpixmapdata(jscoords_xpm);
        mode_switch0_pixmap = readpixmapdata(mode_switch0_xpm);
        mode_switch1_pixmap = readpixmapdata(mode_switch1_xpm);
        num_switch0_pixmap = readpixmapdata(num_switch0_xpm);
        num_switch1_pixmap = readpixmapdata(num_switch1_xpm);
        num_switch2_pixmap = readpixmapdata(num_switch2_xpm);
        num_switch3_pixmap = readpixmapdata(num_switch3_xpm);
        readout_pixmap = readpixmapdata(readout_xpm);
        save_button0_pixmap = readpixmapdata(save_button0_xpm);
        save_button1_pixmap = readpixmapdata(save_button1_xpm);
 

        /* Set size hints for toplevel. */
        if(is_geometry_set)
        {
           /* Set definate position if is_geometry_set is True. */
           sizehints.flags=USPosition|PSize|PMinSize|PMaxSize;
           sizehints.x=xjscal_x;
           sizehints.y=xjscal_y;
        }
        else
        {
           sizehints.flags=PPosition|PSize|PMinSize|PMaxSize;
        }
        sizehints.min_width=xjscal_width;      /* width and height, we  */
        sizehints.min_height=xjscal_height;    /* set them the same for */
        sizehints.max_width=xjscal_width;      /* min and max so no     */
        sizehints.max_height=xjscal_height;    /* resizing.             */
        sprintf(stringa, "%s %s", XJSCAL_NAME, XJSCAL_VERSION);
        XSetStandardProperties(display,
           toplevel,
           stringa,             /* Window title name. */
           XJSCAL_NAME,         /* Iconified name. */
           jscoords_pixmap,     /* Icon pixmap. */
           argv, argc,          /* Command line arguments. */
           &sizehints           /* Sizehints. */
        );


	/* Select event mask input for each Window. */
	XSelectInput(display, toplevel, TOPLEVEL_EVENTMASK);
        XSelectInput(display, js_save_button_win, BUTTON_EVENTMASK);
        XSelectInput(display, js_exit_button_win, BUTTON_EVENTMASK);
        XSelectInput(display, js_mode_switch_win, BUTTON_EVENTMASK); 
        XSelectInput(display, js_num_switch_win, BUTTON_EVENTMASK); 



	/* Define the cursors for each window. */
        XDefineCursor(display, toplevel,
                XCreateFontCursor(display, XC_left_ptr)
        );


	/* Map Windows. */
        XMapWindow(display, toplevel);
        XMapSubwindows(display, toplevel);

	/* Set up crosshairs window. */
        XSetWindowBackgroundPixmap(display, crosshair_win, crosshair_pixmap);
        XShapeCombineMask(display, crosshair_win,
               ShapeBounding,
               0, 0,
               crosshairm_bitmap,
               ShapeSet
        );    



        /* ******************************************************* */
        /* ***                   Main Loop                     *** */
        while(1)
        {
		cur_millitime = MilliTime();


                /* ************************************************ */
                /* ***             ReFetch XEvents              *** */
                XCheckTypedEvent(display, ButtonPress, &buttonpress_xevent);
                XCheckTypedEvent(display, ButtonRelease, &buttonrelease_xevent);
                XCheckTypedEvent(display, KeyPress, &keypress_xevent);
                XCheckTypedEvent(display, KeyRelease, &keyrelease_xevent);
                XCheckTypedEvent(display, Expose, &expose_xevent);
                XCheckTypedEvent(display, DestroyNotify, &destroy_xevent);


                /* *********************************************** */
                /* ***         Destroy XEvent Check            *** */
                if(destroy_xevent.type == DestroyNotify)
                {   
                        fprintf(stderr,
                                "%s: DestroyNotify recieved.\n",
                                argv[0]
                        );

                        xjscal_exit_CB();
                                
                        /* Clear the DestroyNotify XEvent. */
                        destroy_xevent.type = False;
			destroy_xevent.xany.window = False;
 
                        /* Break out of the main loop. */
                        break;
                }


                /* ************************************************** */
                /* ***             KeyPress XEvent                *** */
                if(keypress_xevent.type == KeyPress)
                {
                    /* "x" key (code 53). */
                    if(keypress_xevent.xkey.keycode == 53)
                    {
                        xjscal_exit_CB();
			break;
                    }
		    /* "s" key (code 39) */
		    else if(keypress_xevent.xkey.keycode == 39)
		    {
			x = JSWriteCaliberationFile(js_calibration_file_fullname);
			if(x < 0)
			{
			   fprintf(stderr,
                             "Error: Joystick caliberation not saved properly.\n"
		           );
			}
		    }
		    /* "1" key (code 10) */
                    else if(keypress_xevent.xkey.keycode == 10)
                    {
                        /* Change switch only in testing mode. */
                        if(js_mode_switch_state == 0)
                        {  

			   close(cur_js_fd);
			   x = JSOpenDevice(DEV_JOYSTICK_0);
			   if(x < 0)
			   {
				js_caliberation_record[0].exists = -1;
			   }
			   cur_js_device_node = DEV_JOYSTICK_0;
			   js_number_switch_state = 0;
			}
                    }
                    /* "2" key (code 11) */
                    else if(keypress_xevent.xkey.keycode == 11)
                    {
                        /* Change switch only in testing mode. */
                        if(js_mode_switch_state == 0)
                        {
                           close(cur_js_fd);
                           x = JSOpenDevice(DEV_JOYSTICK_1);
                           if(x < 0)
                           {
                                js_caliberation_record[1].exists = -1;
                           }
                           cur_js_device_node = DEV_JOYSTICK_1;
                           js_number_switch_state = 1;
			}
                    }
                    /* "3" key (code 12) */
                    else if(keypress_xevent.xkey.keycode == 12)
                    {
                        /* Change switch only in testing mode. */
                        if(js_mode_switch_state == 0)
                        {
                           close(cur_js_fd);
                           x = JSOpenDevice(DEV_JOYSTICK_2);
                           if(x < 0)
                           {
                                js_caliberation_record[2].exists = -1;
                           }    
                           cur_js_device_node = DEV_JOYSTICK_2;
                           js_number_switch_state = 2;
			}
                    }
                    /* "4" key (code 13) */
                    else if(keypress_xevent.xkey.keycode == 13)
                    {
                        /* Change switch only in testing mode. */
                        if(js_mode_switch_state == 0)
			{
                           close(cur_js_fd);
                           x = JSOpenDevice(DEV_JOYSTICK_3);
                           if(x < 0)
                           {
                                js_caliberation_record[3].exists = -1;
                           }
                           cur_js_device_node = DEV_JOYSTICK_3;
                           js_number_switch_state = 3;
			}
                    }
                    /* "t" key (code 28) */
                    else if(keypress_xevent.xkey.keycode == 28)
                    {
                        js_mode_switch_state = 0;
                    }
                    /* "c" key (code 54) */
                    else if(keypress_xevent.xkey.keycode == 54)
                    {
                        x = js_number_switch_state;
                        /* Reset the joystick records before caliberating. */
                        js_caliberation_record[x].x_min = MAX_JS_X_AXIS_VALUE;
                        js_caliberation_record[x].x_max = MIN_JS_X_AXIS_VALUE;
                        js_caliberation_record[x].y_min = MAX_JS_Y_AXIS_VALUE;
                        js_caliberation_record[x].y_max = MIN_JS_Y_AXIS_VALUE;

                        js_mode_switch_state = 1;
                    }
                    /* "," key (code 59) */
                    else if(keypress_xevent.xkey.keycode == 59)
                    {
			x = js_number_switch_state;
			js_caliberation_record[x].nullzone -= 1;

			if(js_caliberation_record[x].nullzone < 0)
			   js_caliberation_record[x].nullzone = 0;
                    }
                    /* "." key (code 60) */
                    else if(keypress_xevent.xkey.keycode == 60)
                    {
                        x = js_number_switch_state;
                        js_caliberation_record[x].nullzone += 1;
		    }


                    /* Set button_draw_count to 0 so buttons will get   */
                    /* drawn, see Expose XEvent checking farther below. */
                    button_draw_count = 0;

                    /* Clear the KeyPress XEvents. */
                    keypress_xevent.type = False;
                    keypress_xevent.xkey.keycode = False;
		}


                /* ************************************************ */
                /* ***           ButtonPress XEvent             *** */
                if(buttonpress_xevent.type == ButtonPress)
                {
                    /* and is it the left pointer button? (1) */
                    if(buttonpress_xevent.xbutton.button == Button1)
                    {
                        if(buttonpress_xevent.xany.window == js_save_button_win)
                            js_save_button_state = True;

                        else if(buttonpress_xevent.xany.window == js_exit_button_win)
			    js_exit_button_state = True;

			else if(buttonpress_xevent.xany.window == js_num_switch_win)
			{
			    /* Change switch only in testing mode. */
			    if(js_mode_switch_state == 0)
			    	js_number_switch_state++;

			    if(js_number_switch_state >= MAX_JOYSTICKS)
				js_number_switch_state = 0;

                            close(cur_js_fd);

			    switch(js_number_switch_state)
			    {
                              case 0:
                                x = JSOpenDevice(DEV_JOYSTICK_0);
				cur_js_device_node = DEV_JOYSTICK_0;
				break;
			      case 1:
                                x = JSOpenDevice(DEV_JOYSTICK_1);
                                cur_js_device_node = DEV_JOYSTICK_1;
                                break;
                              case 2:
                                x = JSOpenDevice(DEV_JOYSTICK_2);
                                cur_js_device_node = DEV_JOYSTICK_2;
                                break;
                              case 3:
                                x = JSOpenDevice(DEV_JOYSTICK_3);
                                cur_js_device_node = DEV_JOYSTICK_3;
                                break;
                              default:
                                x = JSOpenDevice(DEV_JOYSTICK_0);
                                cur_js_device_node = DEV_JOYSTICK_0;
                                break;
			    }
                            if(x < 0)
                            {
                                js_caliberation_record[js_number_switch_state].exists = -1;
                            }
			}
                        else if(buttonpress_xevent.xany.window == js_mode_switch_win)
			{
			    if(js_mode_switch_state == 0)
			    {
                               x = js_number_switch_state;  
                               /* Reset the joystick records before caliberating. */
                               js_caliberation_record[x].x_min = MAX_JS_X_AXIS_VALUE;
                               js_caliberation_record[x].x_max = MIN_JS_X_AXIS_VALUE;
                               js_caliberation_record[x].y_min = MAX_JS_Y_AXIS_VALUE;
                               js_caliberation_record[x].y_max = MIN_JS_Y_AXIS_VALUE;
                        
                               js_mode_switch_state = 1;
			    }
			    else
                            {
                               js_mode_switch_state = 0;
                            }
			}


                        /* Set button_draw_count to 0 so buttons will get   */
                        /* drawn, see Expose XEvent checking farther below. */
                        button_draw_count = 0;
                    
                        /* Clear the ButtonPress XEvents. */
                        buttonpress_xevent.type = False;
                        buttonpress_xevent.xany.window = False;
                    }
		}


                /* ************************************************ */
                /* ***           ButtonRelease XEvent           *** */
                if(buttonrelease_xevent.type == ButtonRelease)
                {
                    /* and is it the left pointer button? (1) */
                    if(buttonrelease_xevent.xbutton.button == Button1)
                    {

                        if(buttonrelease_xevent.xany.window == js_save_button_win)
			{
                           x = JSWriteCaliberationFile(js_calibration_file_fullname);
                           if(x < 0)
                           {
                              fprintf(stderr,
                                "Error: Joystick caliberation not saved properly.\n"
                              );
                            }
                            js_save_button_state = False;
                    	}
                        else if(buttonrelease_xevent.xany.window == js_exit_button_win)
			{
                            js_exit_button_state = False;
			    xjscal_exit_CB();
			    break;
			}

                        /* Set button_draw_count to 0 so buttons will get   */
                        /* drawn, see Expose XEvent checking farther below. */
                        button_draw_count = 0;
		    
                        /* Clear the ButtonRelease XEvents. */
                        buttonrelease_xevent.type = False;
                        buttonrelease_xevent.xany.window = False;
		    }
		}


                /* *********************************************** */
                /* ***         Exposure Event Check            *** */
                if(expose_xevent.type == Expose)
                {
		    xjscal_redraw_all();

                    /* Clear the Expose XEvents. */
                    expose_xevent.type = False;
                    expose_xevent.xany.window = False;
		}
		else
		{
                    /* Redraw buttons 2 times whenever */
                    /* button_draw_count is set to 0.  */
                    if(button_draw_count < 3)
                    {
			xjscal_redraw_buttons();
			xjscal_redraw_switches();
                        button_draw_count++;
                    }

		    /* Coordinates are always redrawn per loop. */
		    xjscal_redraw_coords();

		    /* Update readout ever 250 milliseconds. */
		    if(readout_update_last + 250 < cur_millitime)
		    {
	                xjscal_redraw_readout();
			readout_update_last = cur_millitime;
		    }
		}





		/* ************************************************* */
		/* ***              Do Caliberation              *** */
		if(js_mode_switch_state == 1)
		{
		   xjscal_do_caliberation();
		}



		/* Sleep */
		usleep(XJSCAL_LOOP_INTERVAL);
	}

	return(0);
}
