/* **********************************************************************
 *
 *	swserv.c	ShipWars Server
 *	-------------------------------
 *
 *
 * Copyright (C) 1997, 1998  Taura (learfox@hotmail.com).
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * TAURA BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
 * AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *
 */

#include "../include/swserv.h"



void swserv_help()
{
 printf("Developer's edition, please do not redistribute!\n\n");
 printf("Usage: swserv [rcfile] [options]\n\n");
 printf("   [rcfile] must exist, if it is not specified then it will be searched\n");
 printf("   for in the current working directory.\n\n");
 printf("   [options] can be any of the following:\n\n");
 printf("      -dbin <dbinfile>       Database file to read from.\n");
 printf("      -dbout <dboutfile>     Database file to write to.\n");
 printf("      -p 1701                Port number.\n");
 printf("      --help | -h            Print (this) help screen and exit.\n");
 printf("      --version              Print version information.\n");
 printf("\n");

 return;
}



void swserv_version()
{
 printf("%s Version %s\n\n", SWSERV_NAME, SWSERV_VERSION);

 printf(SWSERV_COPYRIGHT);

 return;
}



void swserv_signalhandle(int signal_recieved)
{
	switch(signal_recieved)
	{
	   /* Interrupt. */
	   case SIGINT:
                /* Log signal incident. */
                LogAppendLineFormatted(primary_log_fullname,
                    "*** Server: Interrupt.");
                fprintf(stderr,
                    "*** Server: Interrupt.\n");

		/* Close all connections. */
/* Need to notify all connections. */
		LogAppendLineFormatted(primary_log_fullname,
                    "*** Server: Closing all connections.");
                fprintf(stderr,
                    "*** Server: Closing all connections.\n");
		NetClearAllConnections();

		/* Emergancy DB Save. */
		DBEmergancySave();

                /* Close listening_socket. */
                close(listening_socket);
                listening_socket = -1;  

                /* Change runlevel to 0. */
                LogAppendLineFormatted(primary_log_fullname,
                    "*** Server: Entering runlevel 0.");
                fprintf(stderr,
                    "*** Server: Entering runlevel 0.\n");
                runlevel = 0;

		break;


	   /* Broken pipe (or socket). */
	   case SIGPIPE:
		/* Watch for SIGPIPE again. */
		signal(SIGPIPE,swserv_signalhandle);

/* Write code to check for disconnected sockets. */

		break;


	   /* Segmentation fault. */
	   case SIGSEGV:
		/* Log signal incident. */
                LogAppendLineFormatted(primary_log_fullname,
                    "*** Server: Segmentation fault.");
                fprintf(stderr,
                    "*** Server: Segmentation fault.\n");

		/* Emergancy DB save. */
		DBEmergancySave();

		/* Close listening_socket. */
                close(listening_socket);
                listening_socket = -1;

                /* Change runlevel to 0. */
                LogAppendLineFormatted(primary_log_fullname,
                    "*** Server: Entering runlevel 0.");
                fprintf(stderr,
                    "*** Server: Entering runlevel 0.\n");
                runlevel = 0;

                break;


	   default:
                DBEmergancySave();
                /* Need to notify all connections. */
                break;
	}

	return;
}



void swserv_shutdown()
{
	/* Local variables. */
	long x;
        char sndbuf[MAX_NET_CMD_SIZE];


	/* Warn all connections of shutdown. */
        sprintf(sndbuf,
            "%i Server shutting down normally.\n%c",
            NET_CMD_GLOBALMESG,
            '\0'
        );

        for(x = 0; x < total_connections; x++)
        {
            if(connection[x].socket < 0)
                continue;

            NetSendDataToConnection(x, sndbuf, 1);
        }


	/* Close all connections. */
	NetClearAllConnections();


	/* Log shutdown. */
        LogAppendLineFormatted(primary_log_fullname,
            "*** Server shutting down normally.");


	/* Close and set listening_socket to -1. */
        close(listening_socket);
	listening_socket = -1;


	/* Set global runlevel to 0, main() loop will exit. */
	runlevel = 0;


	return;
}



void swserv_reset_timming()
{
	/* Local variables. */
	char stringa[256];
	long lx, ly, lz;


fprintf(stderr, "Refreshing timming...\n");
                    

        /* Reset all global timming variables. */
        memory_clean_last = 0;
        socket_poll_last = 0;
        objval_update_last = 0;
        weapons_update_last = 0;


	/* Reset connection last updated. */
        for(ly = 0; ly < total_connections; ly++)
        {
            if(connection[ly].socket < 0)
                continue;
            
            connection[ly].obj_ud_last = 0;
        }

	/* Reset weapons last used on each object. */
        for(ly = 0; ly < total_objects; ly++)
        {
            if(xsw_object[ly].type <= XSW_OBJ_TYPE_GARBAGE)
                continue;
             
            xsw_object[ly].last_updated = 0;
	    xsw_object[ly].animation.last_interval = 0;

            for(lz = 0; lz < MAX_WEAPONS; lz++)
            {
                xsw_object[ly].weapons[lz].last_used = 0;
            }
        }


        /* Log timming reset. */
        strncpy(stringa, "Server: Timming Refreshed.", 256);
        LogAppendLineFormatted(primary_log_fullname, stringa);
    

	return;
}



int main(int argc, char *argv[])
{
	/* Local variables. */
	int x, y, z;
	long lx, ly, lz;
	char stringa[1024];
	char *strptr;

	long lapsed_utime = UTime();
        long cur_utime = UTime();


	/* Set global variables to default values. */
	rootpid = getpid();
        childpid = rootpid;
        child_procs = 0;

	runlevel = 1;

	uptime = time(NULL);	/* Time the server was started up. */

	cur_millitime = MilliTime();
	objval_update_last = MilliTime();
	weapons_update_last = MilliTime();
	memory_clean_last = MilliTime();
	socket_poll_last = MilliTime();

        objval_update_interval = DEFAULT_OBJVAL_UPDATE_INTERVAL;
        weapons_update_interval = DEFAULT_WEAPONS_UPDATE_INTERVAL;

	total_connections = 0;
	highest_connection = 0;

	listening_port = DEFAULT_PORT;
	uset_listening_port = 0;

	strcpy(swserv_home_dir, SWSERV_HOME_DIR);
	strcpy(swserv_etc_dir, SWSERV_ETC_DIR);
        strcpy(swserv_logs_dir, SWSERV_LOGS_DIR);
        strcpy(swserv_bin_dir, SWSERV_BIN_DIR);
        strcpy(swserv_db_dir, SWSERV_DB_DIR);

	strcpy(db_in_fullname, DEFAULT_DB_IN_FILE);
	uset_db_in_fullname = 0;
	strcpy(db_out_fullname, DEFAULT_DB_OUT_FILE);
	uset_db_out_fullname = 0;

	sprintf(primary_log_fullname, "%s/%s",
		SWSERV_LOGS_DIR, PRIMARY_LOG_FILENAME);

	getcwd(stringa, sizeof(stringa));
	sprintf(swserv_rc_fullname, "%s/%s", stringa, SWSERV_RC_FILE);

	strcpy(dbtitle, "Untitled");

	listening_socket = -1;

	for(lx = 0; lx < MAX_CONNECTIONS; lx++)
	{
	    connection[lx].socket = -1;
            connection[lx].conhost[0] = '\0';
            connection[lx].object_num = -1;
            connection[lx].contime = 0;
            connection[lx].bytes_recieved = 0;
            connection[lx].bytes_sent = 0;
            connection[lx].errors_recieved = 0;
            connection[lx].errors_sent = 0;
            connection[lx].badlogins = 0;
            connection[lx].obj_ud_interval = DEFAULT_UPDATE_INTERVAL;
            connection[lx].obj_ud_last = 0;
            connection[lx].conbuf_items = 0;
	}


	/* ********************************************************** */
	/* ***            Parse Command Line Arguments            *** */
        for(x = 1; x < argc; x++)
        {
	    if( 0 == (strcmp(argv[x],"--help")) )
	    {
                swserv_help();
		return(0);
            }
	    else if( 0 == (strcmp(argv[x],"-help")) )
            {
                swserv_help();
                return(0);
            }
            else if( 0 == (strcmp(argv[x],"-h")) )
            {
                swserv_help();
                return(0);   
            }
            else if( 0 == (strcmp(argv[x],"--version")) )
            {
                swserv_version();
                return(0);
            }
            else if( 0 == (strcmp(argv[x],"-dbin")) )
            {
                x++;
                sprintf(db_in_fullname,argv[x]);
		uset_db_in_fullname = 1;
            }
            else if( 0 == (strcmp(argv[x],"-dbout")) )
            {
                x++;
                sprintf(db_out_fullname,argv[x]);
		uset_db_out_fullname = 1;
            }
            else if( 0 == (strcmp(argv[x],"-p")) )
            {
                x++;
		listening_port = atoi(argv[x]);
		uset_listening_port = 1;
            }
            /* Check if argument is prefixed with '-' '+' or '/'. */
            else if(
                    (argv[x][0] == '-') ||
                    (argv[x][0] == '+')
            )
            {
                fprintf(stderr,
                   "%s: Argument %i: Warning: Unsupported argument: `%s'\n", 
                   argv[0], x, argv[x]
                );
            }
	    /* RC file to use. */
	    else
	    {
		strcpy(swserv_rc_fullname, argv[x]);
		fprintf(stderr, "%s: Using restart configuration file: `%s'\n",
			argv[0], swserv_rc_fullname);
	    }
	}


	/* Check for any errors in arguments. */




	/* Print starting message. */
	fprintf(stderr, "Starting %s %s...\n",
		SWSERV_NAME, SWSERV_VERSION);


	/* *********************************************************** */
	/* ***          Load Global Variables from RC File         *** */
	x = -1;

	/* Current or specified directory? */
	if(access(swserv_rc_fullname, R_OK) != -1)
	{
	    x = RCLoadConfigurationFromFile(swserv_rc_fullname);
	}
	else
	{
	    /* Is it in the swserv_etc_dir? */
	    sprintf(swserv_rc_fullname, "%s/%s", swserv_etc_dir, SWSERV_RC_FILE);
	    if(access(swserv_rc_fullname, R_OK) != -1)
	    {
		x = RCLoadConfigurationFromFile(swserv_rc_fullname);
	    }
	    else
	    {
		/* Is it in the global etc directory? */
		sprintf(swserv_rc_fullname, "%s/%s", ETC_DIR, SWSERV_RC_FILE);
                if(access(swserv_rc_fullname, R_OK) != -1)
                {
                    x = RCLoadConfigurationFromFile(swserv_rc_fullname);
                }
	    }
	}
	/* Check for errors. */
	if(x < 0)
	{
	    fprintf(stderr,
 "%s: Error: Unable to obtain parameter values from file: %s\n",
		argv[0],
		SWSERV_RC_FILE
	    );
            fprintf(stderr,
 "This file contains important settings that %s needs to know in order to run.\n",
		 SWSERV_NAME
	    );
            fprintf(stderr,
 "That file is looked for in the following directories:\n");
            fprintf(stderr, 
 "   * Current or Specified Directory\n");
            fprintf(stderr,
 "   * %s\n", swserv_etc_dir);
            fprintf(stderr,
 "   * %s\n", ETC_DIR);

	    return(-1);
	}


	/* ********************************************************* */
	/* ***             Check Parameter Values                *** */
	/* Server toplevel/home directory. */
	if(access(swserv_home_dir, R_OK) == -1)
	{
	    fprintf(stderr,
 "Error: Unable to find directory: `%s'\n",
		swserv_home_dir
	    );
            fprintf(stderr,
 "Error: Please verify that this directory exists or is specified correctly.\n"
            );
            fprintf(stderr,
 "Error: Exiting due to fatal error.\n"
            );
	    return(-1);
	}
	/* Server etc directory. */
        if(access(swserv_etc_dir, R_OK) == -1)
        {
            fprintf(stderr,
 "Error: Unable to find directory: `%s'\n",
                swserv_etc_dir
            );
            fprintf(stderr,
 "Error: Please verify that this directory exists or is specified correctly.\n"
            );
            fprintf(stderr,
 "Error: Exiting due to fatal error.\n"
            );
            return(-1);
        }
	/* Server logs directory. */
        if(access(swserv_logs_dir, R_OK) == -1)
        {   
            fprintf(stderr,
 "Error: Unable to find directory: `%s'\n",
                swserv_logs_dir
            );
            fprintf(stderr,
 "Error: Please verify that this directory exists or is specified correctly.\n"
            );
            fprintf(stderr,
 "Error: Exiting due to fatal error.\n"
            );
            return(-1);
        }
        /* Server bin directory. */
        if(access(swserv_bin_dir, R_OK) == -1)
        {
            fprintf(stderr,
 "Error: Unable to find directory: `%s'\n",
                swserv_bin_dir
            );
            fprintf(stderr,
 "Error: Please verify that this directory exists or is specified correctly.\n"
            );
            fprintf(stderr,   
 "Error: Exiting due to fatal error.\n"
            );
            return(-1);
        }
        /* Server db directory. */
        if(access(swserv_db_dir, R_OK) == -1)
        {
            fprintf(stderr,
 "Error: Unable to find directory: `%s'\n",
                swserv_db_dir
            );
            fprintf(stderr,
 "Error: Please verify that this directory exists or is specified correctly.\n"
            );
            fprintf(stderr,
 "Error: Exiting due to fatal error.\n"
            );
            return(-1);
        }
       	/* Make sure objval_update_interval is not too low. */
	if(objval_update_interval < DEFAULT_OBJVAL_UPDATE_INTERVAL)
	{
	     fprintf(stderr,
 "Warning: ObjectValuesUpdateInterval is set below %i milliseconds.\n",
		DEFAULT_OBJVAL_UPDATE_INTERVAL
	     );
	     fprintf(stderr,
 "Warning: This value may be too low, causing a loss in network bandwidth.\n"
	     );
	}
	/* Make sure weapons_update_interval is not too low. */
	if(weapons_update_interval < DEFAULT_WEAPONS_UPDATE_INTERVAL)
	{
             fprintf(stderr,
 "Warning: WeaponValuesUpdateInterval is set below %i milliseconds.\n",
                DEFAULT_WEAPONS_UPDATE_INTERVAL
             );
             fprintf(stderr,
 "Warning: This value may be too low, causing a loss in network bandwidth.\n"
             );
        }   



	/* ********************************************************* */
	/* ***                   Load Database                   *** */
	total_objects = DBReadFromFile(db_in_fullname);
	if(total_objects < 0)
	{
	    fprintf(stderr,
		"Error: Unable to load database file: `%s'\n",
                db_in_fullname
	    );
	    sprintf(stringa,
                "Error: Unable to load database file: `%s'",
                db_in_fullname
	    );
            LogAppendLineFormatted(primary_log_fullname, stringa);
	}
	total_objects += 1;



        /* ******************************************************** */
        /* ***                  Signal Watching                 *** */
        signal(SIGKILL,swserv_signalhandle);
        signal(SIGABRT,swserv_signalhandle);
        signal(SIGINT,swserv_signalhandle);
	signal(SIGPIPE,swserv_signalhandle);


	/* ********************************************************* */
	/* ***              Setup Listening Socket               *** */
	/* Reset all connections. */
	NetClearAllConnections();

        /* Get listening_socket. */
	listening_socket = socket(AF_INET, SOCK_STREAM, 0);
	if(listening_socket == -1)
	{
            perror("socket");
            return(1);
        }

        server_addr.sin_family = AF_INET;
        server_addr.sin_port = htons(listening_port);
        server_addr.sin_addr.s_addr = INADDR_ANY;
        bzero(&(server_addr.sin_zero), 8);

        /* Bind the socket. */
        x = bind(listening_socket, (struct sockaddr *)&server_addr,
		sizeof(struct sockaddr));
	if(x == -1)
	{
            perror("bind");
            return(1);
        }

        /* Set up listening and backlogged connection limit. */
        if(listen(listening_socket, LISTEN_SOCKET_BACKLOG) == -1) {
            perror("listen");
            return(1);
        }

        /* Set listening_socket to nonblocking. */
        fcntl(listening_socket, F_SETFL, O_NONBLOCK);


	/* ********************************************************** */
	/* ***           Successful Startup Message               *** */
        fprintf(stderr,
            "********************************************************\n"
        );
	fprintf(stderr,
            "*** Server started and listening on port %i.\n",
	    listening_port
	);
	fprintf(stderr,
	    "*** Loaded database file: \"%s\"\n",
	    db_in_fullname
	);

        fprintf(stderr,
            "*** Database title: \"%s\"  Objects: %i\n",
            dbtitle,
            total_objects
        );


	/* Log startup. */
	strcpy(stringa,
            "********************************************************"
	);
	LogAppendLineFormatted(primary_log_fullname, stringa);

        sprintf(stringa,
            "*** Server started and listening on port %i.",
	    listening_port
        );
        LogAppendLineFormatted(primary_log_fullname, stringa);

        sprintf(stringa,
	    "*** Loaded database file: \"%s\"",
            db_in_fullname
        );
        LogAppendLineFormatted(primary_log_fullname, stringa);

        sprintf(stringa,
            "*** Database title: \"%s\"  Objects: %i",
            dbtitle,
            total_objects
        );
        LogAppendLineFormatted(primary_log_fullname, stringa);



	/* ********************************************************** */
	/* ***                   Main Event Loop                  *** */
	runlevel = 2;
	while(runlevel > 0)
	{

            /* ****************************************************** */
            /* ***                  Update Timming                *** */

	    /* Lapsed time per loop in microseconds. */
            lapsed_utime = UTime() - cur_utime;
            while(lapsed_utime < 0)
                    lapsed_utime += 1000000;

            cur_utime = UTime();

	    /* Current time since midnight in milliseconds. */
	    lx = MilliTime();
	    if(lx < cur_millitime)
	    {
		swserv_reset_timming();
	    }
            cur_millitime = lx;


            /* ***************************************************** */
            /* ***                      Sleep                    *** */
            if(lapsed_utime > MAIN_LOOP_INTERVAL)
            {
		usleep(MAIN_LOOP_CATCHUP);
            }
            else
            {
		usleep(MAIN_LOOP_INTERVAL);
            }
 

	    /* ****************************************************** */
	    /* ***         Poll Sockets for New Connections       *** */
	    if(socket_poll_last + SOCKET_POLL_INTERVAL < cur_millitime)
	    {
		/* Poll socket. */
		switch(NetPollSocket(listening_socket))
		{
		     /* No error. */
		     case 0:
			break;

		     /* General error. */
		     case -1:
			break;

		     /* MAX_CONNECTIONS reached, cannot allocate */
		     /* more connection descriptors.             */
		     case -2:
			break;

		     /* Unknown error. */
		     default:
			break;
		}

		/* Resend login request to connections that are connected */
		/* but haven't givin us a name and password (not logged   */
                /* in).                                                   */
                for(lx = 0; lx < total_connections; lx++)
                {
                    /* Skip if connection has no socket. */
                    if(connection[lx].socket < 0)
                        continue;
            
                    /* Skip if connection's object *is* valid. */
                    if( (connection[lx].object_num >= 0) &&
	                (connection[lx].object_num < MAX_OBJECTS)
		    )
                        continue;

		    /* Close socket if timmed out. */
		    if( (connection[lx].contime + LOGIN_TIMEOUT) <
			time(NULL)
		    )
		    {
fprintf(stderr, "Connection %i: Timmed out.\n", lx);
			NetCloseConnection(lx);
                        sprintf(stringa, "Connection %i: Timmed out.",
				lx);
                        LogAppendLineFormatted(primary_log_fullname, stringa);
			continue;
		    }


		    /* Resend login request. */
fprintf(stderr, "Connection %i: Resending login request...\n", lx);
                    sprintf(stringa, "%i\n%c", NET_CMD_LOGIN, '\0');
                    NetSendDataToConnection(lx, stringa, 1);
		}


                socket_poll_last = cur_millitime;
	    }


	    /* ****************************************************** */
	    /* ***     Refresh Database and Connection Total      *** */
	    if(memory_clean_last + MEMORY_CLEAN_INTERVAL < cur_millitime)
	    {
		total_objects = DBCleanMemory();
		total_objects += 1;

		/* total_connections is refreshed with each call to
		 * NetCloseConnection().
                 */

		memory_clean_last = cur_millitime;
	    }


            /* ****************************************************** */
            /* ***                Update Objects                  *** */
            for(lx = 0; lx < total_objects; lx++)
            {
                UpdateObject(lx);
            }


            /* ****************************************************** */
            /* ***           Send Outgoing Network Data           *** */
	    for(lx = 0; lx < total_connections; lx++)
	    {
                /* Check if connection has a socket. */
                if(connection[lx].socket < 0)
                    continue;
 
                /* Check if connection has an object. */
                if(connection[lx].object_num < 0)
                    continue;

		/* Due for connection to recieve update? */
		if((connection[lx].obj_ud_last + connection[lx].obj_ud_interval) <
			cur_millitime)
		{

		    /* Have lz be the connection's object_num. */
                    lz = connection[lx].object_num;

		    /* Send object updates to connection. */
		    for(ly = 0; ly < total_objects; ly++)
		    {
			/* Skip garbage objects. */
			if(xsw_object[ly].type <= XSW_OBJ_TYPE_GARBAGE)
				continue;

			/* Check if connection object is in range with */
			/* object.                                     */
			if( Mu3DInRange(lz, ly, xsw_object[lz].scanner_range) <
			   1)
				continue;

			/* Send object position. */
			NetSendObjectPositionS(lx, ly);
		    }

		    /* Mark last update for this connection. */
		    connection[lx].obj_ud_last = cur_millitime;
		}
	    }


	    /* Send object values update. */
	    if((objval_update_last + objval_update_interval) < cur_millitime)
	    {
                for(lx = 0; lx < total_connections; lx++)
                {
                    /* Check if connection has a socket. */
                    if(connection[lx].socket < 0)
                        continue;
            
                    /* Check if connection has an object. */
                    if(connection[lx].object_num < 0)
                        continue;
                
                    /* Have lz be the connection's object_num. */
                    lz = connection[lx].object_num;
                         
                    /* Send object updates. */  
                    for(ly = 0; ly < total_objects; ly++)
                    {
                        /* Skip garbage objects. */
                        if(xsw_object[ly].type <= XSW_OBJ_TYPE_GARBAGE)
                                continue;

                        /* Check if connection object is in range with */
                        /* object.                                     */
                        if( Mu3DInRange(lz, ly, xsw_object[lz].scanner_range) <
                            1)
                                continue;

			/* Send values. */
                	NetSendObjectValues(lx, ly);

			/* Send maximums only for the following. */
			if( (xsw_object[ly].type == XSW_OBJ_TYPE_STATIC) ||
			    (xsw_object[ly].type == XSW_OBJ_TYPE_DYNAMIC) ||
                            (xsw_object[ly].type == XSW_OBJ_TYPE_CONTROLLED) ||
                            (xsw_object[ly].type == XSW_OBJ_TYPE_PLAYER) ||
                            (xsw_object[ly].type == XSW_OBJ_TYPE_HOME) ||
                            (xsw_object[ly].type == XSW_OBJ_TYPE_AREA)
			)
			{
			    NetSendObjectMaximums(lx, ly);
			}
                    }
                }


                /* Mark last object values update. */
                objval_update_last = cur_millitime;
	    }


	    /* Send weapon values for each ship. */
	    if((weapons_update_last + weapons_update_interval) < cur_millitime)
	    {
		for(lx = 0; lx < total_connections; lx++)
		{
                    /* Check if connection has a socket. */
                    if(connection[lx].socket < 0)
                        continue;
            
                    /* Check if connection has an object. */  
                    if(connection[lx].object_num < 0)
                        continue;

                    /* Have lz be the connection's object_num. */
                    lz = connection[lx].object_num;

                    /* Send object updates. */
                    for(ly = 0; ly < total_objects; ly++)
                    {
                        /* Skip garbage objects. */
                        if(xsw_object[ly].type <= XSW_OBJ_TYPE_GARBAGE)
                                continue;

			/* Update only for player and controlled objects. */
			if(!( (xsw_object[ly].type == XSW_OBJ_TYPE_CONTROLLED) ||
			      (xsw_object[ly].type == XSW_OBJ_TYPE_PLAYER)
			))
				continue;

                        /* Check if connection object is in range with */
                        /* object.                                     */
                        if( Mu3DInRange(lz, ly, xsw_object[lz].scanner_range) <
                            1)
                                continue;

                        /* Send all weapons values of object to connection. */
                        for(y = 0; y < MAX_WEAPONS; y++)
		        {
			    NetSendWeaponValues(
			        lx,
			        ly,
			        y
			    );
		        }
		    }
		}

		/* Mark last weapons update. */
		weapons_update_last = cur_millitime;
	    }



            /* Poll sockets for recieving and handling data. */
            for(lx = 0; lx < total_connections; lx++)
            {
                 NetPollSocketRecvData(lx);
            }


	}

	return(0);
}
