/* *********************************************************************
 *
 *	match.c		Matching Functions
 *	----------------------------------
 *	Functions:
 *
 *	long MatchWeaponsLock(long ref_object, long start_object, double range)
 *	long MatchIntercept(long object_num, char *arg)
 *      long MatchInterceptByNumber(long ref_object, long target_object)
 *	long MatchObjectByName(char *nameofobject, int typeofobject)
 *
 *	---
 *
 *	MatchWeaponsLock() returns the next XSW Object number that is within
 *	range range of object ref_object.   ref_object should be the object
 *	that is doing the weapons lock, and may never be -1.
 *	start_object maybe -1 or any object number less than MAX_OBJECTS.
 *	range should be the the maximum scanner range of ref_object.
 *	Returns the next XSW Object number of a valid (nongarbage or nonerror)
 *	object withing range range of ref_object starting from start_object,
 *	or returns -1 if no match can be made.
 *
 *	MatchIntercept() tries to match object specified in arg.  If arg
 *	is "" or "#off" then -1 is returned.   If the object specified in arg
 *	is found and is of type XSW_OBJ_TYPE_STATIC then that object's number
 *	is returned.  If the object is found but not of type static, then it
 *	is checked to see if it is within scanner range of object_num and if
 *	it is, then it's number is returned or of not in range then -1 is
 *	returned.
 *
 *	MatchObjectByName() takes whole or partial name nameofobject to
 *	try and match all objects except non-garbage or non-error objects
 *	of type typeofobject.   If typeofobject is -1, then type is ignored.
 *	If more than one nameofobject was matched, then -2 is returned.
 *	If one object was matched then the number of that object is returned.
 *
 */

#include "../include/swserv.h"



long MatchWeaponsLock(long ref_object, long start_object, double range) 
{
	/* Local variables. */
	long object_count;
        double x, y, z;
	long found_object = -1;
	double cur_range = 0;


	/* If start_object is -1, set it to 0. */
	if(start_object < 0)
		start_object = 0;


	/* ref_object must be a valid object. */
	if(ref_object < 0)
		return(-1);

	/* range must be positive. */
	if(range < 0)
		range = 1;


	/* Begin search. */
	for( object_count = start_object + 1; object_count < total_objects;
	   object_count++ )
	{
	   /* Skip garbage and error objects. */
	   if( (xsw_object[object_count].type <= XSW_OBJ_TYPE_GARBAGE) ||
               (xsw_object[object_count].type == XSW_OBJ_TYPE_WEAPON) ||
               (xsw_object[object_count].type == XSW_OBJ_TYPE_STREAMWEAPON) ||
               (xsw_object[object_count].type == XSW_OBJ_TYPE_AREA)
	   )
		continue;

	   /* Don't match ref_object. */
	   if(object_count == ref_object)
		continue;

	   /* See if object is in range. */
	   x = xsw_object[object_count].x - xsw_object[ref_object].x;
	   y = xsw_object[object_count].y - xsw_object[ref_object].y;
	   cur_range = range * DBGetObjectVisibility(object_count);
	   if(Mu3DDistance(x, y, 0) > cur_range)
		continue;

	   /* Checks passed, this is the object we want. */
	   found_object = object_count;
	   break;
	}


	return(found_object);
}




long MatchIntercept(long object_num, char *arg)
{
        /* Local variables. */
        double x, y;
        char larg[1024];
	long target_obj;


        /* object_num must be valid. */
        if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
            return(-1);

	/* Sanitize arg and copy it to larg. */
	strncpy(larg, StringStripSpaces(arg), 1024);
	larg[1024 - 1] = '\0';


	/* Turn intercept off? */
	if( (0 == strcmp(larg, "#off")) ||
	    (strlen(larg) < 1)
	)
	{
	    return(-1);
	}


	/* Match target_obj. */
	target_obj = MatchObjectByName(larg, -1);


	/* Did we find a match? */
	if( (target_obj < 0) || (target_obj >= MAX_OBJECTS) )
	    return(-1);

	/* Cannot intercept itself. */
	if(target_obj == object_num)
	    return(-1);


	/* Following types of objects may not be intercepted. */
	if( (xsw_object[target_obj].type <= XSW_OBJ_TYPE_GARBAGE) ||
            (xsw_object[target_obj].type == XSW_OBJ_TYPE_WEAPON) ||
	    (xsw_object[target_obj].type == XSW_OBJ_TYPE_STREAMWEAPON)
	)
	    return(-1);


        /* Static objects don't need range check. */
        if(xsw_object[target_obj].type == XSW_OBJ_TYPE_STATIC)
            return(target_obj);


        /* Is target_obj within scanner range of object_num? */
        x = xsw_object[target_obj].x - xsw_object[object_num].x;
        y = xsw_object[target_obj].y - xsw_object[object_num].y;
/* Need to include size in check. */

        if(Mu3DDistance(x, y, 0) < xsw_object[object_num].scanner_range)
            return(target_obj);

	return(-1);
}



long MatchInterceptByNumber(long ref_object, long target_object)
{
        /* Local variables. */
        double x, y;


        /* target_object value must be valid. */
        if( (target_object < 0) || (target_object >= MAX_OBJECTS) )
                return(-1);


        /* ref_object and target_object cannot be the same. */
        if(ref_object == target_object)
                return(-1);


        /* Skip garbage and error. */
        if( (xsw_object[target_object].type <= XSW_OBJ_TYPE_GARBAGE) ||
            (xsw_object[target_object].type == XSW_OBJ_TYPE_WEAPON) ||
            (xsw_object[target_object].type == XSW_OBJ_TYPE_STREAMWEAPON)
        )
                return(-1);


        /* Static objects always return itself (true). */
        if(xsw_object[target_object].type == XSW_OBJ_TYPE_STATIC)
                return(target_object);


        /* If none of the above, then do scanner range check. */
        x = xsw_object[target_object].x - xsw_object[ref_object].x;
        y = xsw_object[target_object].y - xsw_object[ref_object].y;
        if(Mu3DDistance(x, y, 0) < xsw_object[ref_object].scanner_range)
                return(target_object);


        /* All else return false. */
        return(-1);
}



long MatchObjectByName(char *nameofobject, int typeofobject)
{
	/* Local variables. */
	long x;
	int objects_found = 0;
	long found_object = -1;
	char temp_name[MAX_NAME_LENGTH];
	char *strptr;


	/* nameofobject must be atleast 1 characters long. */
	if(strlen(nameofobject) < 2)
		return(-1);


	/* Number matching? */
	strncpy(temp_name, nameofobject, MAX_NAME_LENGTH);
	temp_name[MAX_NAME_LENGTH - 1] = '\0';
	if(temp_name[0] == '#')
	{
	    strptr = strchr(temp_name, '#');
	    strptr += 1;
	    x = atol(strptr);

	    /* Is x a valid object? */
	    if( (x < 0) || (x >= MAX_OBJECTS) )
		return(-1);

	    /* Is x an object of type garbage or error? */
	    if(xsw_object[x].type <= XSW_OBJ_TYPE_GARBAGE)
		return(-1);

            /* Match all types of objects? */
	    if(typeofobject < 0)
		return(x);

	    /* Match specific type. */
	    if(xsw_object[x].type == typeofobject)
		return(x);

	    /* No match. */
	    return(-1);
	}


	/* Begin searching by name. */
	for(x = 0; x < total_objects; x++)
	{
	    /* Skip objects of type garbage or error. */
	    if(xsw_object[x].type <= XSW_OBJ_TYPE_GARBAGE)
		continue;

	    /* Name field must be greater than 2 characters. */
	    if(strlen(xsw_object[x].name) < 3)
		continue;

	    /* Match all types? */
	    if(typeofobject < 0)
	    {
		/* Partial name match. */
		if(strstr(xsw_object[x].name, nameofobject) == NULL)
			continue;

		/* Got match. */
		found_object = x;
		objects_found++;
	    }
	    /* Match specific type. */
	    else
	    {
		/* Match type first. */
		if(xsw_object[x].type != typeofobject)
			continue;

                /* Partial name match. */
                if(strstr(xsw_object[x].name, nameofobject) == NULL)
                        continue;

                /* Got match. */
                found_object = x;
                objects_found++;
	    }
	}


	/* If we matched more than 1 objects, return -2. */
	if(objects_found > 1)
		return(-2);


	return(found_object);
}
