/* ************************************************************************
 *
 *	exec.c		Remote Execution Handling
 *	-----------------------------------------
 *	Functions:
 *
 *	int ExecDoHelp(long condescriptor, char *arg)
 *	int ExecDoVersion(long condescriptor, char *arg)
 *	int ExecDoWho(long condescriptor, char *arg)
 *	int ExecDoScore(long condescriptor, char *arg)
 *	int ExecDoETA(long condescriptor, char *arg)
 *	int ExecDoSaveDB(long condescriptor, char *arg)
 *	int ExecDoShutdown(long condescriptor, char *arg)
 *	int ExecDoBoot(long condescriptor, char *arg)
 *	int ExecDoSyncTime(long condescriptor, char *arg)
 *
 *	int ExecHandleInput(long condescriptor, char *input)
 *
 *	---
 *
 *	ExecHandleInput() is the front end handler which parses input string
 *	input and calls the appropriate Exec*() function.
 *	Returns non-zero on error.
 */

#include "../include/swserv.h"




int ExecDoHelp(long condescriptor, char *arg)
{
        /* Local variables. */
        char sndbuf[MAX_NET_CMD_SIZE];


	/* Print help to condescriptor. */
	sprintf(sndbuf,
 "%i Commands: boot  eta  help  id  netstat  save  score  set  shutdown  synctime\
  version\n%c",
		NET_CMD_GLOBALMESG,
		'\0'
	);
        NetSendDataToConnection(condescriptor, sndbuf, 1);

        sprintf(sndbuf,
 "%i Commands: who\n%c",
                NET_CMD_GLOBALMESG,
                '\0'
        );
        NetSendDataToConnection(condescriptor, sndbuf, 1);


	return(0);
}



int ExecDoVersion(long condescriptor, char *arg)
{
        /* Local variables. */ 
        char sndbuf[MAX_NET_CMD_SIZE];
  

        /* Print version to condescriptor. */
        sprintf(sndbuf,
                "%i %s - Version %s.\n%c",
                NET_CMD_GLOBALMESG,
		SWSERV_NAME,
		SWSERV_VERSION,
                '\0'
        );   
        NetSendDataToConnection(condescriptor, sndbuf, 1); 

	return(0);
}



int ExecDoWho(long condescriptor, char *arg)
{
        /* Local variables. */
        char sndbuf[MAX_NET_CMD_SIZE];
	long object_num;
	long object_con;
	long connected_players;
	long x, y;
	char stringa[MAX_NET_CMD_SIZE]; 


	/* Strip spaces on arg and copy to stringa. */
	strncpy(stringa, StringStripSpaces(arg), MAX_NET_CMD_SIZE);
	stringa[MAX_NET_CMD_SIZE - 1] = '\0';


	/* Print all connections? */
	if(strlen(stringa) < 1)
	{
	    connected_players = 0;
	    for(x = 0; x < total_connections; x++)
	    {
		/* Check if connection has valid socket. */
		if(connection[x].socket < 0)
		    continue;

                /* Check if connection is logged in. */
		object_num = connection[x].object_num;
                if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
                    continue;

		sprintf(sndbuf, "%i %s(#%i) %s\n%c",
		    NET_CMD_GLOBALMESG,
		    xsw_object[object_num].name,
		    object_num,
		    StringFormatTime(time(NULL) - connection[x].contime),
		    '\0'
	        );
		NetSendDataToConnection(condescriptor, sndbuf, 1);

		connected_players++;
	    }


	    /* Print footer. */ 
            sprintf(sndbuf, "%i %i players connected.\n%c",
                    NET_CMD_GLOBALMESG,
                    connected_players,
                    '\0'
            );
            NetSendDataToConnection(condescriptor, sndbuf, 1);
	}
	/* Print specific player. */
	else
	{
	    object_num = MatchObjectByName(stringa, XSW_OBJ_TYPE_PLAYER);
	    if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
	    {
                sprintf(sndbuf,
                    "%i who: %s: No such player.\n%c",
                    NET_CMD_GLOBALMESG,
                    stringa,
                    '\0'
                );
                NetSendDataToConnection(condescriptor, sndbuf, 1);
                return(-1);
	    }

            /* Is object_num connected? */
            for(x = 0, object_con = -1; x < total_connections; x++)
            {
                if(connection[x].socket < 0)
                    continue;

                if(connection[x].object_num == object_num)
                {
                    object_con = x;
                    break;
                }
            }
 
            /* Check if we found of a connection for the object. */
            if( (object_con < 0) || (object_con >= MAX_CONNECTIONS) )
            {
                sprintf(sndbuf,
                    "%i who: %s: Not connected.\n%c",
                    NET_CMD_GLOBALMESG,
                    stringa,
                    '\0'
                );
                NetSendDataToConnection(condescriptor, sndbuf, 1);
                return(-1);
            } 


	    /* Object found and is online, print it. */
            sprintf(sndbuf, "%i %s(#%i) %s\n%c",
                NET_CMD_GLOBALMESG,
                xsw_object[object_num].name,
                object_num,
                StringFormatTime(time(NULL) - connection[x].contime),
                '\0'
            );
            NetSendDataToConnection(condescriptor, sndbuf, 1);
	}



	return(0);
}



int ExecDoNetstat(long condescriptor, char *arg)
{
	/* Local variables. */
        long object_num;
	long object_con;
        char sndbuf[MAX_NET_CMD_SIZE];
	char stringa[256];
	long connections;
	long x;


	/* Copy arg to stringa. */
	strncpy(stringa, StringStripSpaces(arg), 256);
	stringa[256 - 1] = '\0';


	/* Get current number of connections. */
	for(x = 0, connections = 0; x < total_connections; x++)
	{
	    if(connection[x].socket < 0)
		continue;

	    connections++;
	}


	/* No argument, print server network status. */
	if(strlen(stringa) < 1)
	{
	    sprintf(stringa,
"Uptime: %s  Connections: %i(%i)",
		StringFormatTime(time(NULL) - uptime),
		connections,
		highest_connection
	    );

            sprintf(sndbuf, "%i %s\n%c",
                    NET_CMD_GLOBALMESG,
		    stringa,
                    '\0'
            );
            NetSendDataToConnection(condescriptor, sndbuf, 1);
	}

	/* Print network statistics for a player. */
	else
	{
            object_num = MatchObjectByName(stringa, XSW_OBJ_TYPE_PLAYER);
            if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
            {
                sprintf(sndbuf,
                    "%i netstat: %s: No such player.\n%c",
                    NET_CMD_GLOBALMESG,
                    stringa,
                    '\0'
                );
                NetSendDataToConnection(condescriptor, sndbuf, 1);
                return(-1);
            }

            /* Is object_num connected? */
            for(x = 0, object_con = -1; x < total_connections; x++)
            {
                if(connection[x].socket < 0)
                    continue;

                if(connection[x].object_num == object_num)
                {
                    object_con = x;
                    break;
                }
            }
                
            /* Check if we found of a connection for the object. */
            if( (object_con < 0) || (object_con >= MAX_CONNECTIONS) )
            {
                sprintf(sndbuf,
                    "%i netstat: %s: Not connected.\n%c",
                    NET_CMD_GLOBALMESG,
                    stringa,
                    '\0'
                );
                NetSendDataToConnection(condescriptor, sndbuf, 1);
                return(-1);
            }   

 
            /* Object found and is online, print it. */
            sprintf(sndbuf,
 "%i %s(#%i):  Sent: %i  Recieved: %i  Errors sent: %i  Errors recv: %i\n%c",
                NET_CMD_GLOBALMESG,
                xsw_object[object_num].name,
                object_num,
		 /* Relative to client, variables swaped! */
                connection[object_con].bytes_recieved,
		connection[object_con].bytes_sent,
                connection[object_con].errors_recieved,
                connection[object_con].errors_sent,
                '\0'
            );
            NetSendDataToConnection(condescriptor, sndbuf, 1);
	}


	return(0);
}



int ExecDoScore(long condescriptor, char *arg)
{
        /* Local variables. */
	long object_num;
	char stringa[MAX_NAME_LENGTH];
        char sndbuf[MAX_NET_CMD_SIZE];


	/* Copy arg to stringa. */
	strncpy(stringa, StringStripSpaces(arg), MAX_NAME_LENGTH);
	stringa[MAX_NAME_LENGTH - 1] = '\0';


	/* If no argument, assume connection's object. */
	if(strlen(stringa) < 1)
	    object_num = connection[condescriptor].object_num;
	else
	    object_num = MatchObjectByName(stringa, XSW_OBJ_TYPE_PLAYER);


	/* Make sure object_num is valid. */
	if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
	{
            sprintf(sndbuf, "%i %s: No such player.\n%c",
                    NET_CMD_GLOBALMESG,
		    stringa,
                    '\0'
            );
            NetSendDataToConnection(condescriptor, sndbuf, 1);

	    return(-1);
	}


	/* Print scores. */
	sprintf(sndbuf,
 "%i %s(#%i): GPL: %.2lf  RMU: %.2lf(%.2lf)  DmgGiv: %.2lf  DmgRec: %.2lf\
  Kills: %i\n%c",
		NET_CMD_GLOBALMESG,
		xsw_object[object_num].name,
		object_num,
		xsw_object[object_num].score.gpl,
		xsw_object[object_num].score.rmu,
		xsw_object[object_num].score.rmu_max,
		xsw_object[object_num].score.damage_givin,
		xsw_object[object_num].score.damage_recieved,
		xsw_object[object_num].score.kills,
		'\0'
	);
        NetSendDataToConnection(condescriptor, sndbuf, 1);




	return(0);
}



int ExecDoETA(long condescriptor, char *arg)
{
        /* Local variables. */
        long object_num;
        char sndbuf[MAX_NET_CMD_SIZE];

 
        object_num = connection[condescriptor].object_num;

 
	sprintf(sndbuf,
 "%i Antimatter: %.4lf  Throttle: %i  Velocity_Max: %.4lf\n%c",
	    NET_CMD_GLOBALMESG,
	    xsw_object[object_num].antimatter,
            xsw_object[object_num].throttle,
            xsw_object[object_num].velocity_max,
	    '\0'
	);
        NetSendDataToConnection(condescriptor, sndbuf, 1);


	return(0);
}



int ExecDoSaveDB(long condescriptor, char *arg)
{
        /* Local variables. */
        long object_num;
	long x;
        char sndbuf[MAX_NET_CMD_SIZE];
	char stringa[256];
	int status;


        /* Check if object's permission allows save. */
        if(
    xsw_object[connection[condescriptor].object_num].permission.uid >
                ACCESS_UID_SAVE
        )
        {
            sprintf(sndbuf,
                "%i set: Access level %i: Permission denied.\n%c",
                NET_CMD_GLOBALMESG,
		ACCESS_UID_SAVE,
                '\0'
            ); 
            NetSendDataToConnection(condescriptor, sndbuf, 1);
            return(-1);
        }


	/* Warn all connections that save is in progress. */
        sprintf(sndbuf,
            "%i Server: Database save in progress...\n%c",
            NET_CMD_GLOBALMESG,
            '\0'
        );
	for(x = 0; x < total_connections; x++)
	{
	    if(connection[x].socket < 0)
		continue;

	    NetSendDataToConnection(x, sndbuf, 1);
	}

	/* Log who requested save. */
        sprintf(stringa, "%s(#%i): Requesting database save.",
            xsw_object[connection[condescriptor].object_num].name,
            connection[condescriptor].object_num
        );
        LogAppendLineFormatted(primary_log_fullname, stringa);



	/* Save the database. */
	status = DBSaveToFile(db_out_fullname);

	if(status < 0)
	{
	    /* Print error to connection. */
            sprintf(sndbuf,
                "%i Server: Error: DBSaveToFile(): Error code %i.\n%c",
                NET_CMD_GLOBALMESG,
		status,
                '\0'
            );
            NetSendDataToConnection(condescriptor, sndbuf, 1);

	    /* Log error. */
            sprintf(stringa,
		"Server: Error: DBSaveToFile(): Error code %i.",
		status
            );
            LogAppendLineFormatted(primary_log_fullname, stringa);


	    return(-1);
	}
	else
	{
	    /* Notify all connections of sucessful save. */
            sprintf(sndbuf,
               "%i Server: Database save finished!\n%c",
               NET_CMD_GLOBALMESG,
               '\0'
            );
            for(x = 0; x < total_connections; x++)
            {
                if(connection[x].socket < 0)
                    continue;

                NetSendDataToConnection(x, sndbuf, 1);  
            }

	    /* Log sucessful save. */ 
            sprintf(stringa, "Server: Successfully saved database: `%s'",
	        db_out_fullname
            );  
            LogAppendLineFormatted(primary_log_fullname, stringa);
	}

	return(0);
}



int ExecDoShutdown(long condescriptor, char *arg)
{
        /* Local variables. */
        long object_num;
        char sndbuf[MAX_NET_CMD_SIZE];
        char stringa[256];
        
        
        /* Check if object's permission allows save. */
        if(
    xsw_object[connection[condescriptor].object_num].permission.uid >
                ACCESS_UID_SHUTDOWN
        )
        {          
            sprintf(sndbuf,   
                "%i set: Access level %i: Permission denied.\n%c",
                NET_CMD_GLOBALMESG,
		ACCESS_UID_SHUTDOWN,
                '\0'
            );
            NetSendDataToConnection(condescriptor, sndbuf, 1);
            return(-1);
        }


	/* Log who requested shutdown. */
        sprintf(stringa,
            "%s(#%i): Requesting server shutdown.",
	    xsw_object[connection[condescriptor].object_num].name,
	    connection[condescriptor].object_num
	);
	LogAppendLineFormatted(primary_log_fullname, stringa);


	/* Shutdown server. */
	swserv_shutdown();

	return;
}



int ExecDoBoot(long condescriptor, char *arg)
{
	/* Local variables. */
	long x;
        long object_num;
        char sndbuf[MAX_NET_CMD_SIZE];
        char stringa[256];
	long object_con;


        /* Check if object's permission allows boot. */
        if(
    xsw_object[connection[condescriptor].object_num].permission.uid >
                ACCESS_UID_BOOT
        )
        {
            sprintf(sndbuf,
                "%i set: Access level %i: Permission denied.\n%c",
                NET_CMD_GLOBALMESG,
		ACCESS_UID_BOOT,
                '\0'
            );
            NetSendDataToConnection(condescriptor, sndbuf, 1);
            return(-1);
        }


	/* Print usage? */
	if(strlen(arg) < 1)
	{
            sprintf(sndbuf,
                "%i Usage: `boot <player>'\n%c",
                NET_CMD_GLOBALMESG,
                '\0'
            );
            NetSendDataToConnection(condescriptor, sndbuf, 1);
            return(-1);
	}
 

	/* Match the object_num to be booted. */
	object_num = MatchObjectByName(arg, XSW_OBJ_TYPE_PLAYER);


	/* Was a match found? */
	if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
	{
            sprintf(sndbuf,
                "%i boot: %s: No such player.\n%c",
                NET_CMD_GLOBALMESG,
	        arg,
                '\0'
            );
            NetSendDataToConnection(condescriptor, sndbuf, 1);
            return(-1);
	}


        /* Is that object connected? */
	for(x = 0, object_con = -1; x < total_connections; x++)
	{
	    if(connection[x].socket < 0)
		continue;

	    if(connection[x].object_num == object_num)
	    {
	        object_con = x;
		break;
	    }
	}


	/* Check if we found of a connection for the object. */
	if( (object_con < 0) || (object_con >= MAX_CONNECTIONS) )
	{
            sprintf(sndbuf,
                "%i boot: %s: Not connected.\n%c",
                NET_CMD_GLOBALMESG,
                arg,
                '\0'
            );
            NetSendDataToConnection(condescriptor, sndbuf, 1);
	    return(-1);
	}


        sprintf(sndbuf,
            "%i Booted: %s(#%i) on connection %i.\n%c",
            NET_CMD_GLOBALMESG,
            xsw_object[object_num].name,
            object_num,
            object_con,
            '\0'
        );
        NetSendDataToConnection(condescriptor, sndbuf, 1);


	/* Log boot. */
        sprintf(stringa,
            "%s(#%i): Booted: %s(#%i) on connection %i.",
            xsw_object[connection[condescriptor].object_num].name,
            connection[condescriptor].object_num,
            xsw_object[object_num].name,
	    object_num,
            object_con
        );
        LogAppendLineFormatted(primary_log_fullname, stringa);
 

	/* Boot connection object_con. */
	NetCloseConnection(object_con);


	return(0);
}



int ExecDoSyncTime(long condescriptor, char *arg)
{
        /* Local variables. */
	long x;
        char sndbuf[MAX_NET_CMD_SIZE];
        char stringa[256];


        /* Check if object's permission allows sync time. */
        if(
    xsw_object[connection[condescriptor].object_num].permission.uid >
                ACCESS_UID_SYNCTIME
        )
        {
            sprintf(sndbuf,
                "%i synctime: Access level %i: Permission denied.\n%c",
                NET_CMD_GLOBALMESG,
		ACCESS_UID_SYNCTIME,
                '\0'
            );
            NetSendDataToConnection(condescriptor, sndbuf, 1);
            return(-1);
        }




	/* Log who requested timming reset. */
        sprintf(stringa,
            "%s(#%i): Requesting synctime (resetting timmers).",
            xsw_object[connection[condescriptor].object_num].name,
            connection[condescriptor].object_num
        );
        LogAppendLineFormatted(primary_log_fullname, stringa);


	/* Warn all connections about timming reset. */
        sprintf(sndbuf,
            "%i Server: Timming reset.\n%c",
            NET_CMD_GLOBALMESG,
            '\0'
        );
        for(x = 0; x < total_connections; x++)
        {
            if(connection[x].socket < 0)
                continue;
                
            NetSendDataToConnection(x, sndbuf, 1);
        }


	/* Reset all timmers to 0. */
	swserv_reset_timming();



	return(0);
}



int ExecDoID(long condescriptor, char *arg)
{
        /* Local variables. */
        long object_num;
        char stringa[MAX_NAME_LENGTH];
        char sndbuf[MAX_NET_CMD_SIZE];
                
                
        /* Copy arg to stringa. */
        strncpy(stringa, StringStripSpaces(arg), MAX_NAME_LENGTH);
        stringa[MAX_NAME_LENGTH - 1] = '\0';


        /* If no argument, assume connection's object. */
        if(strlen(stringa) < 1)
            object_num = connection[condescriptor].object_num;
        else
            object_num = MatchObjectByName(stringa, XSW_OBJ_TYPE_PLAYER);
                

        /* Make sure object_num is valid. */
        if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
        {
            sprintf(sndbuf, "%i %s: No such player.\n%c",
                    NET_CMD_GLOBALMESG,
                    stringa,
                    '\0'
            );
            NetSendDataToConnection(condescriptor, sndbuf, 1);
        
            return(-1);
        }
            
            
        /* Print ID. */
        sprintf(sndbuf,
                "%i %s(#%i): uid: %i  gid: %i\n%c",
                NET_CMD_GLOBALMESG,
                xsw_object[object_num].name,
                object_num,
                xsw_object[object_num].permission.uid,
                xsw_object[object_num].permission.gid,
                '\0'
        );
        NetSendDataToConnection(condescriptor, sndbuf, 1);
        
        
        
        
	return(0);
}



int ExecHandleInput(long condescriptor, char *input)
{
	/* Local variables. */
        int x, y, z;
        char *strptr;
	long object_num;

        char sndbuf[MAX_NET_CMD_SIZE];

        char stringa[MAX_NET_CMD_SIZE];
        char stringb[MAX_NET_CMD_SIZE];
        char arg[MAX_NET_CMD_SIZE];
        char command[MAX_NET_CMD_SIZE];

        int status;


        /* Input command must be atleast 1 character long. */
        if(strlen(input) < 1)
                return(-1);

	/* Make sure condescriptor is valid. */
	if( (condescriptor < 0) || (condescriptor >= MAX_CONNECTIONS) )
		return(-1);

	/* Make sure condescriptor is connected. */
	if(connection[condescriptor].socket < 0)
		return(-1);

	/* Make sure condescriptor is logged in. */
	object_num = connection[condescriptor].object_num;
	if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
	{
/* print not logged in error message. */		


		return(-1);
	}

        /* Copy input into stringa. */
        strncpy(stringa, input, MAX_NET_CMD_SIZE);

	/* Sanitize stringa. */
        strncpy(stringb, StringStripSpaces(stringa), MAX_NET_CMD_SIZE);
        strncpy(stringa, stringb, MAX_NET_CMD_SIZE);

        stringa[MAX_NET_CMD_SIZE - 1] = '\0';

        if( (strptr = strchr(stringa, '\n')) != NULL)
                *strptr = '\0';
        if( (strptr = strchr(stringa, '\r')) != NULL)
                *strptr = '\0';



        /* ******************************************************** */
        /* ***                Begin Parsing                     *** */
        if( (strptr = strchr(stringa, ' ')) != NULL)
        {
                strncpy(arg, (strptr + 1), MAX_NET_CMD_SIZE);
		arg[MAX_NET_CMD_SIZE - 1] = '\0';
                *strptr = '\0';
                strncpy(command, stringa, MAX_NET_CMD_SIZE);
		command[MAX_NET_CMD_SIZE - 1] = '\0';
        }
        else
        {
                strncpy(arg, "", MAX_NET_CMD_SIZE);
                strncpy(command, stringa, MAX_NET_CMD_SIZE);
                command[MAX_NET_CMD_SIZE - 1] = '\0';
        }


	/* Log this. */
        sprintf(stringa, "%s(#%i): Executed: Command: `%s'  Argument: `%s'",
		xsw_object[object_num].name,
		object_num,
		command,
		arg
	);
        LogAppendLineFormatted(primary_log_fullname, stringa);


        /* ****************************************************** */
        /* ***                   Handle Command               *** */

	/* Help. */
	if( (0 == strcmp(command, "help")) ||
	    (0 == strcmp(command, "hel")) ||
            (0 == strcmp(command, "he")) ||
            (0 == strcmp(command, "h")) ||
            (0 == strcmp(command, "?"))
	)
	{
	    status = ExecDoHelp(condescriptor, arg);
	}
        /* Version. */
        else if( (0 == strcmp(command, "version")) ||
            (0 == strcmp(command, "ver"))
        )
        {
	    status = ExecDoVersion(condescriptor, arg);
	}
	/* Set. */
        else if( (0 == strcmp(command, "set")) ||
            (0 == strcmp(command, "se")) ||
            (0 == strcmp(command, "s"))
        )
        {
            status = ExecDoSet(condescriptor, arg);
        }
	/* Who. */
        else if( (0 == strcmp(command, "WHO")) ||
            (0 == strcmp(command, "who"))
        )
	{
            status = ExecDoWho(condescriptor, arg);
	}
	/* Netstat. */
        else if( 0 == strcmp(command, "netstat") )
        {
            status = ExecDoNetstat(condescriptor, arg);
        }
	/* Score. */
        else if( 0 == strcmp(command, "score") )
        {          
	    status = ExecDoScore(condescriptor, arg);
	}
	/* ETA. */
        else if( 0 == strcmp(command, "eta") )
        {
	    status = ExecDoETA(condescriptor, arg);
	}
	/* Save. */
        else if( 0 == strcmp(command, "save") )
        {
            status = ExecDoSaveDB(condescriptor, arg);
        }
	/* Shutdown. */
        else if( 0 == strcmp(command, "shutdown") )
        {
	    status = ExecDoShutdown(condescriptor, arg);
	}
	/* Boot. */
        else if( 0 == strcmp(command, "boot") )
        {
            status = ExecDoBoot(condescriptor, arg);
        }
	/* Sync time. */
        else if( (0 == strcmp(command, "synctime")) ||
		 (0 == strcmp(command, "sync"))
	)
        {
	    status = ExecDoSyncTime(condescriptor, arg);
        }
        /* ID. */
        else if(0 == strcmp(command, "id"))
        {
            status = ExecDoID(condescriptor, arg);
        }

	/* Unknown command. */
        else
        {
	    status = -1;

            sprintf(sndbuf, "%i %s: no such command or macro.\n%c",
                NET_CMD_GLOBALMESG, command, '\0');
            NetSendDataToConnection(condescriptor, sndbuf, 1);
        }



        return(status);
}
