/* **********************************************************************
 *
 *	update.c	Update Objects
 *	------------------------------
 *	Functions:
 *
 *	int UpdateObject(unsigned long object_num)
 *
 *	---
 *
 *	UpdateObject() calls all other functions here to
 *	update given object number object_num.   If object_num
 *	has a xsw_object[] type of XSW_OBJ_TYPE_ERROR or
 *	XSW_OBJ_TYPE_GARBAGE, nothing will be done.
 *	Returns 0 if the object was updated, 1 if nothing was done,
 *	and -1 for error.
 *
 */

#include "../include/xsw.h"  



int UpdateHeading(long object_num)
{
	/* Local variables. */
        double x, y;
	double rel_heading;
	int nullzone_min, nullzone_max;
	long temp_object;


	/* Skip static objects. */
        if(xsw_object[object_num].type == XSW_OBJ_TYPE_STATIC)
           return(0);



	/* Is object intercepting something? (Includes lplayer_object) */
	if(xsw_object[object_num].intercepting_object >= 0)
	{
	    /* Get rel_heading to the object we're intercepting. */
/*
	    temp_object = xsw_object[object_num].intercepting_object;
	    x = xsw_object[temp_object].x - xsw_object[object_num].x;
	    y = xsw_object[temp_object].y - xsw_object[object_num].y;
	    rel_heading = MuLocationtoHeading(x, y);
	    rel_heading = rel_heading - xsw_object[object_num].heading;
*/
	    /* rel_heading must be within the turnrate of object_num. */
/*
	    if(rel_heading > (100 * xsw_object[object_num].turnrate))
		rel_heading = 100 * xsw_object[object_num].turnrate;
	    else if(rel_heading < (-100 * xsw_object[object_num].turnrate))
		rel_heading = -100 * xsw_object[object_num].turnrate;
*/
	    /* Set the new heading on object_num. */
/*
	    xsw_object[object_num].heading += rel_heading;
*/
	}
	/* Update heading for local player object. */
	else if(object_num == lplayer_object)
	{
		/* Get nullzone_min and nullzone_max. */
		nullzone_min = (joystick_data.nullzone / 2) * -1;
		nullzone_max = joystick_data.nullzone / 2;

#ifdef LINUX
		/* Get joystick turn value, -100 to 100 */
		x = JSGetTurn(joystick_fd);
#endif
		/* If not within null zone, then do turn. */
        	if( (x < nullzone_min) || (x > nullzone_max) )
	        {
        	   xsw_object[object_num].heading +=
                	x * xsw_object[object_num].turnrate;
	        }
	}


        /* Sanitize heading. */
        while(xsw_object[object_num].heading >= 6.2831853)
           xsw_object[object_num].heading -= 6.2831853;
        while(xsw_object[object_num].heading < 0)
           xsw_object[object_num].heading += 6.2831853;


	return(0);
}



int UpdateThrottle(long object_num)
{
        /* Skip static objects. */
        if(xsw_object[object_num].type == XSW_OBJ_TYPE_STATIC)
           return(0);


        /* Update heading for local player object. */
	if(object_num == lplayer_object)
	{
#ifdef LINUX
	    xsw_object[object_num].throttle =
		JSGetThrottle(joystick_fd, js_throttle_mode);
#endif

	    /* If the throttle mode is bidirectional and throttle value */
	    /* is negative, then reverse thrust. */
	    if(js_throttle_mode == 1)
	    {
		if(xsw_object[object_num].throttle < 0)
		{
		    xsw_object[object_num].thrust_rev_state = 1;
		    xsw_object[lplayer_object].thrust_dir = 0;
		    xsw_object[object_num].throttle *= -1;
		}
		else
		{
		    xsw_object[object_num].thrust_rev_state = 0;
		    xsw_object[lplayer_object].thrust_dir = 3.1415927;
		}
	    }
	}

	return(0);
}



int UpdateThrust(long object_num)
{
	/* Local variables. */
	double x;


        /* Skip static objects. */
        if(xsw_object[object_num].type == XSW_OBJ_TYPE_STATIC)
           return(0);


	/* ******************************************************** */
	/* ***            Begin Calculating Thrust              *** */

	/* Thrust is calculated by taking the coefficent of the  */
	/* throttle and multiplying that with the thrust_power.  */
	x = xsw_object[object_num].throttle;
	x = ( x / 100 ) * xsw_object[object_num].thrust_power;

	xsw_object[object_num].thrust = x;


	/* Thrust output should be 0 if there is no antimatter left. */
	if(xsw_object[object_num].antimatter <= 0)
		xsw_object[object_num].thrust = 0;


	/* Make sure velocity is not greater than velocity_max. */
	if(xsw_object[object_num].velocity > xsw_object[object_num].velocity_max)
	{
		xsw_object[object_num].velocity = xsw_object[object_num].velocity_max;
	}


	/* Velocity drag coeff, default #defined to 0.009. */
/*	if(xsw_object[object_num].velocity >= WARP_THRESHOLD) */
	   xsw_object[object_num].velocity -=
		xsw_object[object_num].velocity * WARP_DRAG_COEF;

/*	else if(xsw_object[object_num].velocity >= TRANSWARP_THRESHOLD) */
/*	   xsw_object[object_num].velocity -= TRANSWARP_DRAG_COEF;      */


	/* If velocity is such a small number, then set it 0. */
	if(xsw_object[object_num].velocity < 0.00003)
		xsw_object[object_num].velocity = 0;

	/* Thrust cannot be negative. */
	if(xsw_object[object_num].thrust < 0)
                xsw_object[object_num].thrust = 0; 



	return(0);
}



int UpdateLocation(long object_num)
{
	/* Delta coordinates for... */
	double thrust_dir, thrust_mag;
	double x_vel, y_vel, z_vel;	/* Velocity. */
	double x_thr, y_thr, z_thr;	/* Thrust. */
	double x_res, y_res, z_res;	/* Resultant. */


	/* Skip static objects. */
        if(xsw_object[object_num].type == XSW_OBJ_TYPE_STATIC)
                return(0);


	/* Stream weapons have location of their owner. */
	if(xsw_object[object_num].type == XSW_OBJ_TYPE_STREAMWEAPON)
	{
	    if((xsw_object[object_num].owner > -1) &&
	       (xsw_object[object_num].owner < MAX_OBJECTS))
	    {
	        xsw_object[object_num].x =
		    xsw_object[xsw_object[object_num].owner].x;
		xsw_object[object_num].y =
		    xsw_object[xsw_object[object_num].owner].y;
	    }
	    return(0);
	}



	/* ***************************************************** */
	/* ***          Begin Moving the Object              *** */

	/* Sanitize xsw_object[object_num].velocity_heading. */
	while(xsw_object[object_num].velocity_heading >= 6.2831853)
		xsw_object[object_num].velocity_heading -= 6.2831853;
	while(xsw_object[object_num].velocity_heading < 0)
		xsw_object[object_num].velocity_heading += 6.2831853;


	/* Calculate movement from velocity (momentum). */
	/* In quadrant 1? */
	if(xsw_object[object_num].velocity_heading < 1.5707963)
	{
	   x_vel = xsw_object[object_num].velocity *
		sin(xsw_object[object_num].velocity_heading);
	   y_vel = xsw_object[object_num].velocity *
		cos(xsw_object[object_num].velocity_heading);
	   z_vel = 0;
	}
	/* In Quadrant 2? */
	else if(xsw_object[object_num].velocity_heading < 3.1415927 )
	{
	   x_vel = xsw_object[object_num].velocity *
		cos(xsw_object[object_num].velocity_heading - 1.5707963);
           y_vel = xsw_object[object_num].velocity *
                sin(xsw_object[object_num].velocity_heading - 1.5707963);
	   y_vel *= -1;
           z_vel = 0;
	}
	/* In Quadrant 3? */
	else if(xsw_object[object_num].velocity_heading < 4.712389 )
	{
           x_vel = xsw_object[object_num].velocity *
                sin(xsw_object[object_num].velocity_heading - 3.1415927);
	   x_vel *= -1;
           y_vel = xsw_object[object_num].velocity *
                cos(xsw_object[object_num].velocity_heading - 3.1415927);
	   y_vel *= -1;
           z_vel = 0;
	}
        /* In Quadrant 4? */
        else
        {
           x_vel = xsw_object[object_num].velocity *
                cos(xsw_object[object_num].velocity_heading - 4.712389);
           x_vel *= -1;
           y_vel = xsw_object[object_num].velocity *
                sin(xsw_object[object_num].velocity_heading - 4.712389);
           z_vel = 0;
        }


        /* Calculate movement from thrust. */
	thrust_dir = xsw_object[object_num].heading +
		xsw_object[object_num].thrust_dir + 3.1415927;
	while(thrust_dir >= 6.2831853)
		thrust_dir -= 6.2831853;
	while(thrust_dir < 0)
		thrust_dir += 6.2831853;


	thrust_mag = xsw_object[object_num].thrust;
	if( (thrust_mag + xsw_object[object_num].velocity ) >
		xsw_object[object_num].velocity_max)
	{
	   thrust_mag = xsw_object[object_num].velocity_max -
		xsw_object[object_num].velocity;
	}

        /* In quadrant 1? */
        if(thrust_dir < 1.5707963)
        {
           x_thr = thrust_mag * sin(thrust_dir);
           y_thr = thrust_mag * cos(thrust_dir);
           z_thr = 0;  
        }
        /* In Quadrant 2? */
        else if(thrust_dir < 3.1415927 )
        {
           x_thr = thrust_mag * cos(thrust_dir - 1.5707963);
           y_thr = thrust_mag * sin(thrust_dir - 1.5707963);
           y_thr *= -1;
           z_thr = 0;
        }
        /* In Quadrant 3? */
        else if(thrust_dir < 4.712389 )
        {
           x_thr = thrust_mag * sin(thrust_dir - 3.1415927);
           x_thr *= -1;
           y_thr = thrust_mag * cos(thrust_dir - 3.1415927);
           y_thr *= -1;
           z_thr = 0;
        }
        /* In Quadrant 4? */
        else
        {
           x_thr = thrust_mag * cos(thrust_dir - 4.712389);
           x_thr *= -1;
           y_thr = thrust_mag * sin(thrust_dir - 4.712389);
           z_thr = 0;
        }


	/* Find the resultant. */
	x_res = x_vel + x_thr;
	y_res = y_vel + y_thr;
	z_res = z_vel + z_thr;


        /* Record new velocity and velocity_heading on object. */
        xsw_object[object_num].velocity_heading = MuLocationtoHeading(
                x_res,
                y_res
        );  
        xsw_object[object_num].velocity = Mu3DDistance(
                x_res,
                y_res,
                z_res  
        );


	/* Move object. */
	xsw_object[object_num].x += x_res;
	xsw_object[object_num].y += y_res;
	xsw_object[object_num].z += z_res;


	return(0);
}



int UpdateAge(long object_num)
{
        /* Age object. */
        if(xsw_object[object_num].lifespan > -1)
        {
           xsw_object[object_num].age =
                cur_millitime - xsw_object[object_num].birth_time;
        }


	return(0);
}



int UpdateMortality(long object_num)
{
	/* Local variables. */
	char stringa[256];


        /* Animation Cycle Check:                                 */
        /* If this is an animated object, check if all the frame  */
        /* cycles has cycled.   If they have, then recycl object. */
        if(xsw_object[object_num].type == XSW_OBJ_TYPE_ANIMATED)
        {   
            if(xsw_object[object_num].animation.cycle_count >=
                 xsw_object[object_num].animation.cycle_times)
            {
                 DBRecycleObject(object_num);
        
                 if(print_event_bool)
                 {
                    sprintf(stringa, "Animated object #%i has died.", object_num);
                    MesgWinAddMessage(stringa, dark_grey_pixel);
                 }
                 return(0);
            }
        }



	/* Aging Check:                                      */
	/* Check if the object's age is older than lifespan, */
	/* If it is, then recycle it.                        */
        if(xsw_object[object_num].lifespan < 0)
                return(0);

        if(xsw_object[object_num].age >
            xsw_object[object_num].lifespan )
        {
            DBRecycleObject(object_num);

            /* Notify an object has died? */
            if(print_event_bool)
            {
                sprintf(stringa, "Object #%i has died.", object_num);
                MesgWinAddMessage(stringa, dark_grey_pixel);
            }
            return(0);
        }


	return(0);
}



int UpdateWeaponsLock(long object_num)
{
	/* Local variables. */
	double x, y;
	double cur_range;
	long locked_object;


	/* Is object_num locked onto anything? */
	if(
	   (xsw_object[object_num].locked_on < 0) ||
	   (xsw_object[object_num].locked_on >= MAX_OBJECTS)
	)
		return(0);


	/* Get locked_object. */
	locked_object = xsw_object[object_num].locked_on;

	/* Get delta distances between objects. */
        x = xsw_object[locked_object].x - xsw_object[object_num].x;
        y = xsw_object[locked_object].y - xsw_object[object_num].y;

	/* Get effective scanner range from object_num to locked_object. */
        cur_range = xsw_object[object_num].scanner_range *
                          DBGetObjectVisibility(locked_object);

	/* See if delta distance is out of effective scanner range. */
        if(Mu3DDistance(x, y, 0) > cur_range)
	{
           xsw_object[object_num].locked_on = -1;
	}


	return(0);
}



int UpdateIntercept(long object_num)
{
        /* Local variables. */
        double x, y;
        double cur_range;
        long intercepting_object;
 

        /* Is object_num intercepting anything? */
        if(
           (xsw_object[object_num].intercepting_object < 0) ||
           (xsw_object[object_num].intercepting_object >= MAX_OBJECTS)
        )
                return(0);

	/* Use MatchInterceptByNumber() to see if object is in range. */
	intercepting_object = xsw_object[object_num].intercepting_object;
	if(MatchInterceptByNumber(object_num, intercepting_object) < 0)
	{
		xsw_object[object_num].intercepting_object = -1;
	}


	return(0);
}



int UpdatePowerCore(long object_num)
{
	/* Increase power in the object's power core. */
        if(xsw_object[object_num].power <
              xsw_object[object_num].power_max)
        {
           xsw_object[object_num].power +=
              xsw_object[object_num].core_efficency;

           if(xsw_object[object_num].power >
                   xsw_object[object_num].power_max)
           {
                xsw_object[object_num].power = xsw_object[object_num].power_max;
           }
        }


	return(0);
}



int UpdateAntimatter(long object_num)
{
        /* Skip static objects. */
        if(xsw_object[object_num].type == XSW_OBJ_TYPE_STATIC)
                return(0);

        /* Consume Antimatter based on Thrust */
        xsw_object[object_num].antimatter -= xsw_object[object_num].thrust;
        if(xsw_object[object_num].antimatter < 0)
                xsw_object[object_num].antimatter = 0;


	return(0);
}



int UpdateCollisionCheck(long object_num)
{
	/* Local variables. */
	long object_count;
	char stringa[256];
	double heading_variance;
	double distance;
	int result;


        /* ****************************************************** */
        /* ***                Colission Check                 *** */

	/* Colission check if object_num is a regular weapon. */
        if(xsw_object[object_num].type == XSW_OBJ_TYPE_WEAPON)
        {

	    /* Begin checking for contact. */
            for(object_count = 0; object_count < total_objects; object_count++)
            {
		/* Skip garbage and error objects. */
                if(xsw_object[object_count].type <= XSW_OBJ_TYPE_GARBAGE)
                        continue;

                /* Skip stream weapons. */
                if(xsw_object[object_count].type == XSW_OBJ_TYPE_STREAMWEAPON)
                        continue;

                /* Skip XSW_OBJ_TYPE_AREA. */
                if(xsw_object[object_count].type == XSW_OBJ_TYPE_AREA)
                        continue;

		/* Skip XSW_OBJ_TYPE_ANIMATED. */
                if(xsw_object[object_count].type == XSW_OBJ_TYPE_ANIMATED)
                        continue;

                /* Don't hit its owner. */
                if(xsw_object[object_num].owner == object_count)
                        continue;
        
                /* Don't hit itself. */
                if(object_num == object_count)
                        continue;

                /* Did it hit something? */
                if(Mu3DInContact(object_num, object_count) > 0)
                {
                    /* Create explosion2(). */
                    DBCreateExplosion2(
                       xsw_object[object_count].x,
                       xsw_object[object_count].y
                    );
                    DBRecycleObject(object_num);

		    /* Print event message? */
		    if(print_event_bool)
                    {
                        sprintf(stringa,
                           "Weapons fire object %i hit target object %i.",
			   object_num,
			   object_count
			);
                        MesgWinAddMessage(stringa,white_pix);
		    }

                    return(1);
                }
            }            
        }



        /* Colission check if object_num is a stream weapon. */
        if(xsw_object[object_num].type == XSW_OBJ_TYPE_STREAMWEAPON)
        {

	    /* Stream weapon object must have power. */
	    if(xsw_object[object_num].power <= 0)
		return(0);

	    /* Begin checking for contact. */
            for(object_count = 0; object_count < total_objects; object_count++)
            {          
                /* Skip garbage and error objects. */
                if(xsw_object[object_count].type <= XSW_OBJ_TYPE_GARBAGE)
                        continue;

		/* Skip other stream weapons (and itself). */
		if(xsw_object[object_count].type == XSW_OBJ_TYPE_STREAMWEAPON)
			continue;

                /* Skip XSW_OBJ_TYPE_AREA. */
                if(xsw_object[object_count].type == XSW_OBJ_TYPE_AREA)    
                        continue;

                /* Skip XSW_OBJ_TYPE_ANIMATED. */
                if(xsw_object[object_count].type == XSW_OBJ_TYPE_ANIMATED)
                        continue;

                /* Don't hit its owner. */
                if(xsw_object[object_num].owner == object_count)
                        continue;


                /* Did it hit something? */
		distance = Mu3DDistance(
			xsw_object[object_count].x - xsw_object[object_num].x,
			xsw_object[object_count].y - xsw_object[object_num].y,
			0
		);
		distance *= 1000;	/* distance must be in XSW Screen units. */
		heading_variance = atan(
			xsw_object[object_count].size / distance
		);
		result = Mu3DInVectorContact(
			object_num,
			object_count,
			xsw_object[object_num].heading,
			heading_variance,
			xsw_object[object_num].size + xsw_object[object_count].size
		);
		if(result > 0)
		{
                    DBCreateExplosion2(
                       xsw_object[object_count].x,
                       xsw_object[object_count].y
                    );

		    /* Instead of recycling the stream weapon object, */
		    /* we set its power to 0.                         */
		    xsw_object[object_num].power = 0;

                    /* Print event message? */
                    if(print_event_bool)
                    {
                        sprintf(stringa,
                           "Stream weapon object %i hit target object %i.",
                           object_num,
                           object_count
                        );
                        MesgWinAddMessage(stringa,white_pix);
		    }
		    return(1);
		}
	    }
	}


	return(0);
}



int UpdateObject(long object_num)
{
	/* Local variables. */
	int result;


	/* ************************************************** */
	/* ***          Error/Validility Checking         *** */
        /* Check if object_num is in range. */
        if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
                return(-1);

	/* Skip XSW Objects type error or garbage. */
	if(xsw_object[object_num].type <= XSW_OBJ_TYPE_GARBAGE)
                return(0);


	/* ************************************************** */
	/* ***           Begin Updating Object            *** */

	/* Update age. (Client dosen't update this.) */
/*	UpdateAge(object_num); */

	/* Update mortality. (Client dosen't update this.) */
/*	UpdateMortality(object_num); */

	/* Update power core. (Client dosen't update this.) */
/*	UpdatePowerCore(object_num); */

	/* Update heading. */
	UpdateHeading(object_num);

	/* Update throttle. */
	UpdateThrottle(object_num);

	/* Update thrust. */
	UpdateThrust(object_num);

	/* Update location (movement). */
	UpdateLocation(object_num);

	/* Update weapons lock. (Client dosen't update this.) */
/*	UpdateWeaponsLock(object_num); */

	/* Update (consume) antimatter. (Client dosen't update this.) */
/*	UpdateAntimatter(object_num); */

	/* Update check for collision. (Client dosen't update this.) */
/*	result = UpdateCollisionCheck(object_num); */
	/* Positive results means object_num was recycled. */
/*	if(result > 0)
		return(0);
*/

	return(0);
}
