/* **********************************************************************
 *
 *	math.c		Math Utility Functions
 *	--------------------------------------
 *	Functions:
 *
 *	double MuPolarRotX(double input_angle, double input_radius)
 *      double MuPolarRotY(double input_angle, double input_radius)
 *	double MuLocationtoHeading(double delta_x, double delta_y)      
 *	double Mu3DDistance(double delta_x, double delta_y, double delta_z)
 *	int Mu3DInContact(long object_num1, long object_num2)
 *      int Mu3DInVectorContact(long object_num1, long object_num2,
 *		double heading, double heading_variance, double range)
 *
 *	---
 *
 *	Math convience functions, all trigeometric functions use
 *	the clock quadrant as referance and *not* the standard position
 *	quadrant.    All angles must be in radians!   All distances
 *	are in XSW 'real' units unless indicated otherwise.
 *
 *	MuPolarRotX() returns the x value for a point in polar
 *	coordinates.   input_angle must be in radians.
 *
 *	MuPolarRotY() is the same as MuPolarRotX() except it returns
 *      the y value.
 *
 *	MuLocationtoHeading() returns the direction, in radians,
 *	of the calculated distance of delta_x and delta_y.
 *	delta_x and delta_y can be negative or positive.
 *      Returns the 'to' direction in radians.
 *
 *	Mu3DDistance() returns the distance of two point's delta
 *      distances.
 *
 *	Mu3DInContact() returns 1 if XSW Objects object_num1 and
 *	object_num2 are in contact, by calculating their distances
 *	apart and their sizes.
 *
 */

#include "../include/xsw.h"  



double MuPolarRotX(double input_angle, double input_radius)
{
	/* Local variables. */
	double x_rtn = 0;


	/* Sanitize input_angle. */
	while(input_angle >= 6.2831853)
		input_angle -= 6.2831853;

	while(input_angle < 0)   
                input_angle += 6.2831853;


	/* input_angle in quadrant 1? */
	if(input_angle < 1.5707963)
	{
		x_rtn = input_radius * sin(input_angle);
	}
	/* input_angle in quadrant 2? */
	else if(input_angle < 3.1415927 )
	{
		x_rtn = input_radius * cos(input_angle - 1.5707963);
	}
        /* input_angle in quadrant 3? */
        else if(input_angle < 4.712389 )
        {
                x_rtn = input_radius * sin(input_angle - 3.1415927);
		x_rtn *= -1;
        }
        /* input_angle in quadrant 4? */ 
        else
        {
                x_rtn = input_radius * cos(input_angle - 4.712389);
                x_rtn *= -1;
        }


	return(x_rtn);
}



double MuPolarRotY(double input_angle, double input_radius)
{
        /* Local variables. */
        double y_rtn = 0;
        
        
        /* Sanitize input_angle. */
        while(input_angle >= 6.2831853)
                input_angle -= 6.2831853;
        
        while(input_angle < 0)
                input_angle += 6.2831853;
         
         
        /* input_angle in quadrant 1? */
        if(input_angle < 1.5707963)
        {   
                y_rtn = input_radius * cos(input_angle);
	}
        /* input_angle in quadrant 2? */
        else if(input_angle < 3.1415927 )
        {
                y_rtn = input_radius * sin(input_angle - 1.5707963);
                y_rtn *= -1;
        }
        /* input_angle in quadrant 3? */ 
        else if(input_angle < 4.712389 )
        {
                y_rtn = input_radius * cos(input_angle - 3.1415927);
                y_rtn *= -1;
        }
        /* input_angle in quadrant 4? */
        else
        {
                y_rtn = input_radius * sin(input_angle - 4.712389);
        }       
        
                
        return(y_rtn);
}



double MuLocationtoHeading(double delta_x, double delta_y)
{
	/* Local variables. */
	double x,y;
	double heading_rtn;


	/* If delta_x and delta_y are 0, return 0. */
	if( (delta_x == 0) && (delta_y == 0) )
	{
		return(0);
	}


	/* Check which quadrant. */
	/* Quadrant 1? */
	if( (delta_x >= 0) && (delta_y > 0) )
	{
		x = delta_x;
		y = delta_y;
		heading_rtn = atan(x/y);
	}
	/* Quadrant 2? */
	else if( (delta_x > 0) && (delta_y <= 0) )
        {
		x = delta_x;
		y = delta_y;
		y *= -1;
		heading_rtn = atan(y/x);
		heading_rtn += 1.5707963;
        }
	/* Quadrant 3? */
	else if( (delta_x <= 0) && (delta_y < 0) )
        {
		x = delta_x;
		y = delta_y;
		x *= -1;
		y *= -1;
		heading_rtn = atan(x/y);
		heading_rtn += 3.1415927;
        }
	/* Quadrant 4. */
	else
	{
		x = delta_x;
                y = delta_y;
                x *= -1;
		heading_rtn = atan(y/x);
		heading_rtn += 4.712389;
	}

	return(heading_rtn);
}


double Mu3DDistance(double delta_x, double delta_y, double delta_z)
{
	/* Local variables. */
	double hyp;	/* Tempory Hypotenuse. */
	double distance_rtn;


	/* Make sure all values are positive. */
	if(delta_x < 0)
		delta_x *= -1;
	if(delta_y < 0)
                delta_y *= -1;
	if(delta_z < 0)
                delta_z *= -1;


	hyp = sqrt(
		(delta_x * delta_x) + (delta_y * delta_y)
	);
	distance_rtn = sqrt(
		(hyp * hyp) + (delta_z * delta_z)
	);

	return(distance_rtn);
}


int Mu3DInContact(long object_num1, long object_num2)
{
	/* Local variables. */
	double distance;
	double size1, size2;

	/* Both objects must have a size greater than 0. */
	if(xsw_object[object_num1].size <= 0)
		return(0);
        if(xsw_object[object_num2].size <= 0)
                return(0);

	/* Sizes are in pixels. */
	size1 = xsw_object[object_num1].size;
	size2 = xsw_object[object_num2].size;

	/* Get the distance in real units. */
	distance = Mu3DDistance(
	   xsw_object[object_num1].x - xsw_object[object_num2].x,
	   xsw_object[object_num1].y - xsw_object[object_num2].y,
	   0
	);

	/* Convert distance from real units to pixels. */
	distance *= 1000;

	/* Is distance less than the two sizes? */
	if(distance < (size1 + size2))
		return(1);
	else
		return(0);
}



int Mu3DInVectorContact(long object_num1, long object_num2, double heading,
	double heading_variance, double range)
{
        /* Local variables. */
        double distance;
	double heading_result;


	/* range cannot be negative. */
	if(range < 0)
		range *= -1;

	/* Sanitize heading. */
        while(heading >= 6.2831853)
                heading -= 6.2831853;
        while(heading < 0)
                heading += 6.2831853;

	/* heading_variance must be positive. */
	if(heading_variance < 0)
		heading_variance *= -1;



	/* ******************************************************** */
	/* ***                 Begin Checking                   *** */

        /* Get the distance in real units. */
        distance = Mu3DDistance(
           xsw_object[object_num2].x - xsw_object[object_num1].x,
           xsw_object[object_num2].y - xsw_object[object_num1].y,
           0
        );
        /* Convert distance from real units to pixels. */
        distance *= 1000;

        /* If distance greater than range, return false. */
        if(distance > range)
                return(0);


	/* At this point, it is within range.                       */
	/* Now see if it is within heading with heading_variance.   */
	heading_result = MuLocationtoHeading(
           xsw_object[object_num2].x - xsw_object[object_num1].x,
           xsw_object[object_num2].y - xsw_object[object_num1].y
	);

	if(
		(heading_result >= (heading - heading_variance))
		&&
                (heading_result <= (heading + heading_variance))
	)
	{
		return(1);
	}


	return(0);
}



int Mu3DInRange(long object_num1, long object_num2, double distance)
{
        /* Local variables. */
        double distance_apart;
        double size1, size2;

	/* distance must be positive. */
	if(distance < 0)
		return(0);

        /* Sizes are in pixels. */
        size1 = xsw_object[object_num1].size;
        size2 = xsw_object[object_num2].size;
                
        /* Get the distance in real units. */
        distance_apart = Mu3DDistance(
           xsw_object[object_num1].x - xsw_object[object_num2].x,
           xsw_object[object_num1].y - xsw_object[object_num2].y,
           0
        );


        /* Convert distance_apart from real units to pixels. */
	distance_apart *= 1000;

	/* See if distance apart is closer than distance. */
	if((distance_apart - (size1 + size2)) <= distance)
		return(1);
	else
		return(0);

}

