/* *********************************************************************
 *
 *	match.c		Matching Functions
 *	----------------------------------
 *	Functions:
 *
 *	long MatchObjectWithWindow(Window w)
 *	long MatchWeaponsLock(long ref_object, long start_object, double range)
 *	long MatchInterceptByNumber(long ref_object, long target_object)
 *
 *	---
 *
 *	MatchObjectWithWindow() returns the XSW Object number that has a Window
 *	ID w or -1 for not found.
 *
 *	MatchWeaponsLock() returns the next XSW Object number that is within
 *	range range of object ref_object.   ref_object should be the object
 *	that is doing the weapons lock, and may never be -1.
 *	start_object maybe -1 or any object number less than MAX_OBJECTS.
 *	range should be the the maximum scanner range of ref_object.
 *	Returns the next XSW Object number of a valid (nongarbage or nonerror)
 *	object withing range range of ref_object starting from start_object,
 *	or returns -1 if no match can be made.
 *
 *	MatchInterceptByNumber() checks if target_object is within range
 *	of ref_object by type and distance criteria as follows:
 *	If the target_object is of type XSW_OBJ_TYPE_GARBAGE or XSW_OBJ_TYPE_ERROR
 *	then -1 is always returned.  If the target_object is of type
 *	XSW_OBJ_TYPE_STATIC then the same number for target_object is returned.
 *	If target_object type is none of the above, then a scanner range check
 *	will be done to see if target_object is within the scanner range of
 *	ref_object, if it is, then the same number for target_object is returned,
 *	or else -1.
 *
 */

#include "../include/xsw.h"  



long MatchObjectWithWindow(Window w)
{
	long x;
	int found_something = 0;

	for(x = 0; x < total_objects; x++)
	{
	   /* Skip errors and garbage. */
           if(xsw_object[x].type <= XSW_OBJ_TYPE_GARBAGE)
                continue;

	   /* See if the Window ID's match. */
	   if(xsw_object[x].window == w)
	   {
		found_something = 1;
		break;
	   }
	}

	if(found_something > 0)
		return(x);
	else
		return(-1);
}



long MatchWeaponsLock(long ref_object, long start_object, double range) 
{
	/* Local variables. */
	long object_count;
        double x, y, z;
	long found_object = -1;
	double cur_range = 0;


	/* If start_object is -1, set it to 0. */
	if(start_object < 0)
		start_object = 0;


	/* ref_object must be a valid object. */
	if(ref_object < 0)
		return(-1);

	/* range must be positive. */
	if(range < 0)
		range = 1;


	/* Begin search. */
	for( object_count = start_object + 1; object_count < total_objects;
	   object_count++ )
	{
	   /* Skip garbage and error objects. */
	   if(xsw_object[object_count].type <= XSW_OBJ_TYPE_GARBAGE)
		continue;

	   /* Don't match ref_object. */
	   if(object_count == ref_object)
		continue;

	   /* See if object is in range. */
	   x = xsw_object[object_count].x - xsw_object[ref_object].x;
	   y = xsw_object[object_count].y - xsw_object[ref_object].y;
	   cur_range = range * DBGetObjectVisibility(object_count);
	   if(Mu3DDistance(x, y, 0) > cur_range)
		continue;

	   /* Checks passed, this is the object we want. */
	   found_object = object_count;
	   break;
	}


	return(found_object);
}



long MatchInterceptByNumber(long ref_object, long target_object)
{
	/* Local variables. */
	double x, y;


	/* target_object value must be valid. */
	if( (target_object < 0) || (target_object >= MAX_OBJECTS) )
		return(-1);


	/* ref_object and target_object cannot be the same. */
	if(ref_object == target_object)
		return(-1);


	/* Skip garbage and error. */
	if(xsw_object[target_object].type <= XSW_OBJ_TYPE_GARBAGE)
		return(-1);


	/* Static objects always return itself (true). */
	if(xsw_object[target_object].type == XSW_OBJ_TYPE_STATIC)
		return(target_object);


	/* If none of the above, then do scanner range check. */
	x = xsw_object[target_object].x - xsw_object[ref_object].x;
        y = xsw_object[target_object].y - xsw_object[ref_object].y;
	if(Mu3DDistance(x, y, 0) < xsw_object[ref_object].scanner_range)
		return(target_object);


	/* All else return false. */
	return(-1);
}
