#!/usr/bin/env python

import sys, string, os
import signal, getpass
import locale, codecs
import executor
import re
import logging
import traceback, StringIO

import n1common, Getline
from n1common import _

# Constants
EXIT_COMMAND = u'exit'

CODE_INTERNAL_ERROR = 5
CODE_SUCCESS = 0

USE_CACAO = True

def main(argv, (terminal_language, terminal_encoding)):
    # See main()'s caller for SIGINT information.

    # Default cacao return status.
    status = None

    try:
        # Session figures out the encoding.
        session = Session(language=terminal_language,
                          encoding=terminal_encoding)

        # Process args
        filename, args = process_args(argv, session)

        # Encode/decode the streams.

        # If the lookup has problems (e.g. 'iso885915'), try playing with
        # n1common.enc_alias(enc) instead.
        reader, writer = codecs.lookup(session.encoding)[2:4]
        sys.stdin  = reader(sys.__stdin__, 'strict')
        sys.stdout = writer(sys.__stdout__, 'replace')
        sys.stderr = writer(sys.__stderr__, 'replace')
        logging.info('stdin and stdout set to "%s" encoding.'
                     % session.encoding)

        try:
            # Create executor.
            if USE_CACAO:
                cacao = executor.CacaoExecutor()
            else:
                cacao = executor.Executor()

            cacao.start(session)

            session.get_tabdata = TabdataCacao(session, cacao)

            # Reader class supporting get()
            if args:
                line_reader = CommandReader(session, args)
            elif filename:
                line_reader = FileReader(session, filename)
            elif not sys.stdin.isatty():
                line_reader = StreamReader()
            else:
                line_reader = InteractiveN1Reader(session)

            # Start executing!
            while True:
                # Read lines, ignore blanks and comments
                while True:
                    try:
                        if not cacao.ready:
                            raise IOError, _('Not ready for writing.')

                        command = line_reader.get()
                        if is_valid_line(command):
                            break
                    except Getline.UnbalancedQuotes:
                        print _('command.malformed.unbalanced.quotes')

                #print repr(command)
                command = command.strip()

                # Quit loop when done
                if command == EXIT_COMMAND:
                    break

                prompts = []
                if line_reader.isatty():
                    # Prompt for '?' user fields.
                    descs = n1common.map_commands_to_nodes(
                                session.get_tabdata(),
                                n1common.tokenize(command))

                    if descs:
                        """
                        try:
                            old_sig = signal.getsignal(signal.SIGINT)
                            signal.signal(signal.SIGINT, signal.SIG_DFL)
                            try:
                        """
                        prompts = n1common.get_prompts(descs)
                        prompts = map(lambda s: s.decode(session.encoding),
                                      prompts)

                        # TODO: Command should end up with '?' over
                        #       sensitive data, which is passed in via
                        #       prompts list.

                        command = n1common.join_descs(descs, prompts)
                        """
                            finally:
                                print "<ctrl>-C"
                                signal.signal(signal.SIGINT, old_sig)

                        except KeyboardInterrupt:
                            print "here"
                            break
                        """

                output = execute_command(cacao, command, prompts, session)

                if output:
                    print output
        finally:
            # Clean up when done.
            try:
                status = cacao.stop()
            except:
                status = CODE_INTERNAL_ERROR

    except N1SHSuccess:
        status = CODE_SUCCESS
    except IOError, e:
        print _('n1sh.lost.connection')
        try:
            logging.error('Reason: %s' % e)
        except UnicodeEncodeError, e:
            logging.error('Exception raised %s exception'
                          % e.__class__.__name__)
        status = CODE_INTERNAL_ERROR
    except executor.SessionConfigError, e:
        print _('n1sh.session.config.error')
        try:
            logging.error('Reason: %s' % e)
        except UnicodeEncodeError, e:
            logging.error('Exception raised %s exception'
                          % e.__class__.__name__)
        status = CODE_INTERNAL_ERROR
    except executor.Executor.Error, e:
        print _('n1sh.no.connection')
        try:
            logging.error('Reason: %s' % e)
        except UnicodeEncodeError, e:
            logging.error('Exception raised %s exception'
                          % e.__class__.__name__)
        status = CODE_INTERNAL_ERROR
    except N1SHError, e:
        print e
        status = CODE_INTERNAL_ERROR
    except (UnicodeDecodeError, UnicodeEncodeError,
            UnicodeError, UnicodeTranslateError,
            locale.Error), e:
        try:
            print _('n1sh.no.locale')
        except UnicodeEncodeError:
            print u'Unknown locale.'

        logging.error('Reason: [%s] %s'
                      % (e.__class__.__name__, str(e).encode(terminal_encoding, 'replace')))
        #sio = StringIO.StringIO()
        #traceback.print_exc(file=sio)
        #logging.error('Uncaught Exception:\n' + sio.getvalue())
        status = CODE_INTERNAL_ERROR

    except:
        logging.error('Uncaught Exception; start trace', exc_info=True)
        print u'Unknown error.'
        status = CODE_INTERNAL_ERROR

    return status

#
# Local exception.
class N1SHError(Exception):
    """Generic error exception to use locally."""

class N1SHSuccess(Exception):
    """Generic success exception to use locally for good exit conditions."""


# Store run information.

class Session:
    """Store session information."""

    def __init__(self, role=u'', output=u'text',
                 language=None, encoding=None):
        self.locale = locale.getlocale(locale.LC_MESSAGES)[0]

        self.encoding = encoding or locale.getpreferredencoding()
        self.role = role            # Cacao role
        self.output = output        # Output format
        self._tabdata = None

    def get_tabdata(self):
        return self._tabdata

    def invalidate_tabdata(self):
        self._tabdata = None

    def __str__(self):
        out = []
        for a in ( 'locale', 'encoding', 'role', 'output' ):
            out.append('%s = %s' % (a, str(getattr(self, a))))
        return '\n'.join(out)


#
# Reader classes are used to gather commands for execution.
# See Strategy Pattern.

class BaseReader:
    class NoMoreLines(Exception):
        """Reader has no more lines to read."""

    def __init__(self):
        logging.info('Using reader "%s"' % self.__class__.__name__)

    def get(self):
        """Main hook to overload.  Return 'exit' when done."""
        return EXIT_COMMAND

    def isatty(self):
        """Determine if there's somebody at the keyboard."""
        return sys.__stdin__.isatty()


class InteractiveN1Reader(BaseReader):
    """Return a line from the interactive Getline."""

    def __init__(self, session):
        import N1Getline

        BaseReader.__init__(self)
        self._prompt = u'N1-ok> '
        self._session = session

        self._n1 = N1Getline.n1line()

        print_license_short()
        print

    def get(self):
        self._n1.set_tabdata(self._session.get_tabdata())

        line = ''
        try:
            line = self._n1.getline(self._prompt)

            # Hide sensitive information.
            sline = n1common.hide_sensitive(line, self._session)

            self._n1.add_history(sline)
        except KeyboardInterrupt:
            print
        except self._n1.EOFInterrupt:
            line = EXIT_COMMAND
            print line

        return line


class CommandReader(BaseReader):
    """Return the non-interactive command."""

    def __init__(self, session, args):
        BaseReader.__init__(self)
        self.commands = []
        line = ' '.join(args).decode(session.encoding)

        bits = line.split()

        if bits[:2] == [ 'set', 'session' ]:
            print _('n1sh.noninteractive.session')
        else:
            self.commands.append(line)

    def get(self):
        try:
            return self.commands.pop(0)
        except IndexError:
            return EXIT_COMMAND

    def isatty(self):
        # TODO: Temp fix to disable getting of commandtree every time.
        return False


class FileReader(BaseReader):
    """Return lines from a file.  Strips comments and blank lines."""

    def __init__(self, session, filename):
        BaseReader.__init__(self)
        self.lines = []
        try:
            f = codecs.open(filename, 'r', session.encoding)
            self.lines = f.readlines()
            f.close()
        except IOError:
            print _('n1sh.no.file', filename)
            #self.lines = [ EXIT_COMMAND ]

    def get(self):
        try:
            line = self.lines.pop(0).rstrip()
            return line
        except IndexError:
            return EXIT_COMMAND

    def isatty(self):
        return False


class StreamReader(BaseReader):
    """Return lines from stdin."""

    def __init__(self):
        BaseReader.__init__(self)
        self.instream = sys.stdin

    def get(self):
        line = self.instream.readline()
        if not line:
            return EXIT_COMMAND

        return line.rstrip()

# End Readers


class TabdataCacao:
    """Collect tab-completion data and return it as one structure."""

    def __init__(self, session, cacao):
        self._session = session
        self._cacao = cacao

    def __call__(self):
        if self._session._tabdata is None:
            self._session._tabdata = self._get_tabdata()
        return self._session._tabdata

    def _get_tabdata(self):
        from N1Getline import TokenNode, CommandHandler, print_commands
        import xml, copy
        from StringIO import StringIO

        all_cmds = TokenNode('tree')

        if self._cacao:
            handler = CommandHandler(all_cmds)

            if not USE_CACAO:
                filename = 'n1sh.xml'
                try:
                    xml.sax.parse(filename, handler)
                except IOError:
                    raise executor.Executor.Error, \
                        "Can't open Command Set Descriptor file \"%s\"." % \
                          filename
            else:
                result = self._cacao.execute('<<text @get commandtree')
                commandtree = '<?xml version="1.0" encoding="%s"?>\n%s' \
                              % (self._cacao.ENCODING,
                                 result.output.encode(self._cacao.ENCODING))
                #fo = open('/tmp/n1sh.xml', 'w')
                #fo.write(commandtree)
                #fo.close()
                try:
                    xml.sax.parse(StringIO(commandtree), handler)
                except xml.sax._exceptions.SAXException, e:
                    raise IOError, _('Cannot parse command tree.\n--\n%s\n--'
                                     % commandtree)

            all_cmds = handler.get_command_tree()
            #print_commands(all_cmds)

        # To add additional commands, modify CommandtreeFilter.java

        return all_cmds


def process_args(argv, session):
    """Parse commandline arguments and configure session."""

    from getopt import getopt, GetoptError

    filename = None

    try:
        short_opts = 've:f:r:o:'
        long_opts = [
            'execute=', 'file=', 'role=', 'output=',
            'version'
        ]
        opts, args = getopt(argv, short_opts, long_opts)

        for o, a in opts:
            if o in ('-v', '--version'):
                print_license_long()
                raise N1SHSuccess, 'Print license, long version'

            elif o in ('-e', '--execute'):
                args.insert(0, a)

            elif o in ('-f', '--file'):
                if args:
                    raise GetoptError('commands not allowed with option %s' \
                                      % o)
                filename = a

            elif o in ('-r', '--role'):
                session.role = a

            elif o in ('-o', '--output'):
                session.output = a

    except GetoptError, e:
        raise N1SHError, _('Invalid command line: {0}', str(e))

    return filename, args


def execute_command(cacao, command, prompts, session):
    """Execute command and do any additional processing."""

    commands = n1common.tokenize(command)

    #
    # Process command: execute locally or send to executor.

    if commands in ( ['?'], ['help'] ):
        from StringIO import StringIO
        tabdata = session.get_tabdata()

        matches = tabdata.tokens.values() + tabdata.attrs.values()
        o = StringIO()
        n1common.possible_completions(matches, '', o)
        result = executor.Result(o.getvalue())
    elif commands[:2] == ['connect', 'server'] and len(commands) == 3:
        descs = n1common.map_commands_to_nodes(session.get_tabdata(),
                                               commands)
        if descs:
            try:
                server = descs[2].token
                user = getpass.getuser()

                logging.info('scpath server "%s"; user "%s"' % (server, user))
                result = cacao.execute( \
                        '<<text @get serialconsolepath "%s" "%s"' \
                        % (server, user) )

                scpath = result.output
                if result.status != 0:
                    return scpath
                    """
                    print scpath
                    raise N1SHError, 'scpath = "%s"; status = %d' \
                                     % (scpath, result.status)
                    """

                logging.info('Connecting to server "%s" via scirpt "%s"' \
                             % (server, scpath))

                status = os.system( \
                        '/opt/sun/n1gc/bin/n1shconsole.exp "%s" "%s"' \
                        % (scpath, user) )
                if status != 0:
                    raise N1SHError, 'os.system("%s") returned %d' \
                                     % (scpath, status)

                result = executor.Result('')
            except N1SHError, e:
                result = executor.Result(_('n1sh.no.serial.console'),
                                         CODE_INTERNAL_ERROR)
                logging.error('Reason: %s' % str(e))
        else:
            # Command is not in commandset.  Send to cacao for the
            # default error message.
            result = to_executor(cacao, command, prompts)


    elif commands[:1] == ['/echo']:
        if prompts:
            result = executor.Result(prompts.pop(0))
        else:
            result = executor.Result(' '.join(commands[1:]))
    else:
        result = to_executor(cacao, command, prompts)

    logging.info('Exit Status: %d' % result.status)

    #
    # If command was successful, do post processing

    if result.status == 0:
        # Reset tabdata for "set session ..."
        if commands[:2] == ['set', 'session']:
            commands = commands[2:]
            commands.reverse()
            orole = session.role
            while commands:
                try:
                    attr = commands.pop()
                    val = commands.pop()
                except IndexError:
                    break
                oldval = getattr(session, attr)
                setattr(session, attr, val)
                logging.info('session.%s from "%s" to "%s"'
                             % (attr, oldval, val))

            if orole != session.role:
                session.invalidate_tabdata()

        # Reset tabdata for "set module ..."
        if commands[:2] in (['set', 'module'],):
            session.invalidate_tabdata()

    return result.output


#
# Helper functions

def to_executor(executor, command, prompts):
    # TODO: Revisit once CommandParserCommand can reassemble
    #       prompts correctly.
    #result = cacao.execute(command, prompts)
    command_parts = command.split(' ')
    clear_text = ''
    if command_parts[0].startswith('<<'):
        clear_text = ' '.join(command_parts[:3])
        command = ' '.join(command_parts[3:])
    else:
        clear_text = ' '.join(command_parts[:2])
        command = ' '.join(command_parts[2:])
    result = executor.execute(clear_text, [ command ])
    return result


def is_valid_line(line):
    """Verify line is not empty or a comment."""

    tokens = n1common.tokenize(line)  # Make sure line is tokenizable.
    return line.rstrip() and not line.lstrip().startswith('#')


#
# //TODO: get licenses from message catalog

def print_license_short():
    print _('hss.ui.cli.license.short')


def print_license_long():
    print _('hss.ui.cli.license.long')


if __name__ == '__main__':

    status = main(sys.argv[1:])

    sys.exit(status)



