#!/usr/bin/env python

import sys, os
import popen2, select
import codecs
import logging
from datetime import datetime

from n1common import _


class SessionConfigError(Exception):
    """
    Error establishing "session" after the connection has been
    set up.  Could happen if the user fails to authenticate.
    """


class Result:
    """Hold result of cacao command."""

    def __init__(self, output, status = 0):
        self.output = output          # The output string
        self.status = status          # The return code

    def __str__(self):
        return '*** Start (status = %d) ***\n%s\n*** End (status = %d) ***' \
             % (self.status, self.output, self.status)


class Executor:
    """
    Do-nothing base class.  Just return commands.

    Simple example:

        e = Executor()

        e.start(Session())  # Base Executor does nothing with session
        result = execute('do something')
        print result
        e.stop()
    """

    class Error(Exception): pass

    def __init__(self):
        # Is cacao ready for reading/writing?
        self.ready = False

    def __del__(self):
        try:
            self.stop()
        except:
            pass

    def start(self, session):
        """Initialize the Executor connection."""

        self.ready = True

    def stop(self):
        """
        Shutdown connection (if there is one) and return exit code.

        If the subprocess has trouble shutting down, should return a None.
        """

        self.ready = False
        return 0

    def execute(self, cmd, args = []):
        """Execute and return a Result object."""

        if args:
            cmd = '>>%d %s' % ( len(args), cmd )

        out = [ '### Executing: %s' % cmd ]

        i = 0
        for a in args:
            i += 1
            out.append('%2d) %s' % (i, a))

        return Result('\n'.join(out))


class CacaoExecutor(Executor):
    """
    Maintain state for the CacaoExecutor pipe.
    """

    ENCODING = 'UTF-8'   # Internal encoding.  Output is Unicode.
    SENTINEL = '<__SENTINEL__%s__>'
    CACAOCMD = 'com.sun.hss.ui.commands.cacao:CommandParser'

    CACAO_CONSTANTS = {
        'Linux' : ( '/opt/sun/cacao/private/bin',
                    '/etc/opt/sun/cacao/security' ),
        'SunOS' : ( '/opt/SUNWcacao/lib/tools',
                    '/etc/opt/SUNWcacao/security' ),
    }

    def __init__(self):
        Executor.__init__(self)
        self._session = None
        self._cacao = None

        #TODO: Redo this to use platform.properties
        ostype = os.uname()[1]

        # Set up run variables
        bindir, secdir = self.CACAO_CONSTANTS[os.uname()[0]]
        host = 'localhost'
        port = '10163'

        # Build commandline
        self._cacao_cmd = \
            '%(bindir)s/cacaocsc -d %(secdir)s/nss/wellknown -f %(secdir)s/password -h %(host)s -p %(port)s' \
            % locals()

        # Streams to and from the child process.
        self._to = None
        self._from = None

        self._last_status = None


    def to(self, s):
        # cacaocsc uses '~' escape character, so it must be escaped
        s = s.replace('~', '~~')

        #logging.info('<< %s' % s)
        self._to.write('%s\n' % s)
        self._to.flush()


    def from_line(self):
        """Read until newline."""

        bits = []
        while True:
            ready = select.select([self._from], [], [])

            """
            if self._from not in ready[0]:
                sys.stdout.write('.')
                continue
            """

            bit = self._from.read(1)

            #print repr(bit),
            #sys.stdout.flush()

            if bit == '\n':
                break
            if not bit:
                raise IOError, "Unable to read."

            bits.append(bit)

        line = ''.join(bits)

        maxline = 120
        l = '<line longer than %d chars>' % (maxline)
        if len(line) < maxline:
            l = repr(line)
        #logging.info('>> ' + l)

        return line


    def start(self, session):
        """Initialize the CacaoExecutor connection."""
        if self.ready:
            return

        logging.info('Session.locale: %s; session.encoding: %s'
                     % (session.locale, session.encoding))

        # Create pipe
        """
        Note on supported languages:
        All supported languages are forced into UTF-8 encoding
        at the moment.  If there is ever a new language added that
        doesn't support UTF-8, this is where it needs to be fixed.
        """
        cacaocmd = 'LC_MESSAGES=%s.%s %s' % (session.locale, self.ENCODING,
                                             self._cacao_cmd)
        logging.debug('CACAOCSC command: ' + cacaocmd)
        self._cacao = popen2.Popen3(cacaocmd, True, 0)

        self.ready = True  # Mark as ready so that self.stop() doesn't
                           # ignore it in case of exception.

        # Make sure it's ready to write.
        ready = select.select([], [self._cacao.tochild], [], 0)
        if self._cacao.tochild not in ready[1]:
            raise Executor.Error, _('Not ready for writing.')

        reader = codecs.getreader(self.ENCODING)
        self._from = reader(self._cacao.fromchild)

        writer = codecs.getwriter(self.ENCODING)
        self._to = writer(self._cacao.tochild)

        # Initialize pipe
        #self.to('com.sun.cacao:setLocale %s' % (session.locale,))
        #self.to('com.sun.cacao:setEncoding %s' % (self.ENCODING,))
        self.to('%s <<text set session role "%s" output "%s"' % \
            (self.CACAOCMD, session.role, session.output))
        err = self.get_output()

        if err:
            raise SessionConfigError, err


    def stop(self):
        if not self.ready:
            return

        self.to('com.sun.cacao:exit')
        self._cacao.tochild.close()
        wait = self._cacao.wait()

        wait = wait >> 8  # Need only the high 8 bits

        if wait != 0:
            # There was a problem...
            return None

        self.ready = False

        return self._last_status


    def execute(self, cmd, args = []):
        # Make sure we can still write.
        ready = select.select([], [self._to], [], 0)
        if self._to not in ready[1]:
            raise IOError, _('Unable to send command.')

        # Send command and collect output.
        numlines_prefix = ''
        if args:
            numlines_prefix = '>>%d' % len(args)

        cacao_cmd = '%s %s %s' % (self.CACAOCMD, numlines_prefix, cmd)
        self.to(cacao_cmd)
        for a in args:
            self.to(a)
        output = self.get_output()

        # TODO: send the exit status with the results somehow.
        self.to('%s <<text @get exitstatus' % (self.CACAOCMD))
        try:
            self._last_status = int(self.get_output())
        except (TypeError, ValueError):
            self._last_status = None

        return Result(output, self._last_status)


    def get_output(self):
        """Read output until sentinel is reached."""

        t = datetime.now()
        sentinel = self.SENTINEL % str(t) #(t.strftime('%F_%H:%M:%S.'))

        # Mark to know when to stop reading.
        self.to('com.sun.cacao:echo %s' % sentinel)

        all_out = []
        while True:
            line = self.from_line()

            if line == sentinel:
                break

            all_out.append(line)

        return '\n'.join(all_out)




if __name__ == '__main__':
    import Getline
    import locale
    prompt = Getline.Getline()

    class FakeSession:
        role = ''
        output = 'text'
        locale, encoding \
            = locale.setlocale(locale.LC_MESSAGES, '').split('.', 1)


    cacao = CacaoExecutor()
    session = FakeSession()

    sys.stdout = (codecs.getwriter(session.encoding))(sys.__stdout__)

    try:
        cacao.start(session)
        while True:
            cmd = prompt.getline()

            if cmd.strip() == 'exit':
                break

            result = cacao.execute(cmd)
            print result.output
            print 'Status:', result.status
    except KeyboardInterrupt:
        pass
    except IOError, e:
        print 'Lost connection to commandline service:'
        print e
    except Executor.Error:
        print "Couldn't connect!"

    exit = cacao.stop()
    if exit == None:
        exit = -1

    print 'exit: %d' % (exit)

    sys.exit(exit)




