-- ----------------------------------------------------------------
--
-- Schema creation script
--
-- Created on May 24, 2001
-- version: $Id: rox_create_schema.sql,v 1.281 2006/01/31 09:55:37 br195241 Exp $
-- ----------------------------------------------------------------

-- To create any hard-coded DB entries, follow the procedure below
-- object IDs are hardcoded based on the following conventions:
-- IIIIIIIIIIII-TTTTTTTTTTTTT-SSSSS-RRRRRRRRRR
-- IP: machine IP of user creating data (system date uses 010010001024)
-- time: epoch (0)
-- sequence: monotonically increasing for objects of a particular type
-- random: denotes object type:
--   user                1
--   group               2
--   permission          3
--   host type           4
--   host                5
--   ap instance         6
--   host set            7
--   host search         8
--   execution plan      9
--   event              10
--   task               11
--   target             12
--   folder             13
--   category           28
--   resource           30
--   session var desc   40
--   comp def           51
--   comp               52
--   system service     53
--   comp var set       54
--   host type var      55
--   host type criteria 56
--   plugin             70

--
-- version table describes version of schema
--
CREATE TABLE ROX_VERSION (
C_VERSION    INT4,
C_TYPE       VARCHAR(56)
);

-- BEGIN PLUGINDB SCHEMA --

CREATE TABLE ROX_PLUGIN (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_NAME	VARCHAR(64) NOT NULL CONSTRAINT c_plugin_unique_name UNIQUE,
C_DESCRIPTION	VARCHAR(1024),
C_MENU_ORDER    VARCHAR(64) NOT NULL,
C_VENDOR        VARCHAR(1024) NOT NULL,
VERSION	        INT NOT NULL,
C_MENU_ITEM     VARCHAR(64),
C_TOOL_TIP      VARCHAR(1024),
C_CERTIFICATE_TYPE VARCHAR(64)
);

CREATE TABLE ROX_PLUGIN_DEPENDENCY_LINK(
OBJECTID	VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT	INT NULL,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_PARENT_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_plugin_ref_parentplugin REFERENCES ROX_PLUGIN(OBJECTID),
C_CHILD_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_plugin_ref_childplugin REFERENCES ROX_PLUGIN(OBJECTID),
CONSTRAINT c_plugin_dependency_link_unique
	PRIMARY KEY (C_PARENT_ID, C_CHILD_ID)
);

-- END PLUGINDB SCHEMA --


--
-- system data table indicates if system data has been loaded
--
CREATE TABLE ROX_SYSTEM_DATA (
C_DATA_LOAD_TIME TIMESTAMP
);


-- BEGIN CATDB SCHEMA --

CREATE TABLE ROX_CATEGORY (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_NAME	VARCHAR(32) NOT NULL CONSTRAINT c_category_unique_name UNIQUE,
C_DESCRIPTION	VARCHAR(1024) NOT NULL
);

-- END CATDB SCHEMA --


-- BEGIN USERDB SCHEMA --

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.userdb.PersistentPermissionData

CREATE TABLE ROX_PERMISSION (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_NAME	VARCHAR(256) NULL,
C_CLASS_NAME	VARCHAR(256) NOT NULL,
C_ACTIONS	VARCHAR(256) NULL,
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
UNIQUE (C_NAME, C_CLASS_NAME, C_ACTIONS)
 );

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.userdb.PersistentUserData
-- hashed pwd len = (32+32+1)*2

CREATE TABLE ROX_USER (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_USERNAME	VARCHAR(32) NOT NULL UNIQUE,
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
C_PASSWORD	VARCHAR(130) NULL,
C_ACTIVE        BOOLEAN NOT NULL DEFAULT TRUE,
C_SESSION_VAR_DETECT        BOOLEAN NOT NULL DEFAULT TRUE,
C_LOGIN_CONFIG	VARCHAR(130) NOT NULL DEFAULT 'internal'
 );

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.userdb.PersistentGroupData

CREATE TABLE ROX_GROUP (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_NAME	VARCHAR(32) NOT NULL UNIQUE,
C_TYPE	VARCHAR(256) NULL,
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
C_DESCRIPTION	VARCHAR(1024) NULL
 );

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.userdb.PersistentGroupMember

CREATE TABLE ROX_GROUP_MEMBER (
OBJECTID	VARCHAR(56) NULL,
C_UPDATE_COUNT	INT,
C_GROUP_ID	VARCHAR(56) NOT NULL REFERENCES ROX_GROUP(OBJECTID),
C_USER_ID	VARCHAR(56) NOT NULL REFERENCES ROX_USER(OBJECTID),
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
PRIMARY KEY (C_GROUP_ID, C_USER_ID)
 );

CREATE INDEX I_ROX_GROUP_MEMBER_USER ON ROX_GROUP_MEMBER (
   C_USER_ID
);

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.userdb.PersistentParentChildGroup

CREATE TABLE ROX_PARENT_CHILD_GROUP (
OBJECTID	VARCHAR(56) NULL,
C_UPDATE_COUNT	INT,
C_PARENT_GROUP_ID	VARCHAR(56) NOT NULL REFERENCES ROX_GROUP(OBJECTID),
C_CHILD_GROUP_ID	VARCHAR(56) NOT NULL REFERENCES ROX_GROUP(OBJECTID),
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
PRIMARY KEY (C_PARENT_GROUP_ID, C_CHILD_GROUP_ID)
 );

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.userdb.PersistentGroupGraph

CREATE TABLE ROX_GROUP_GRAPH (
OBJECTID	VARCHAR(56) NULL,
C_UPDATE_COUNT	INT,
C_ANCESTOR_GROUP_ID	VARCHAR(56) NOT NULL REFERENCES ROX_GROUP(OBJECTID),
C_DESCENDANT_GROUP_ID	VARCHAR(56) NOT NULL REFERENCES ROX_GROUP(OBJECTID),
C_COUNT	INT NOT NULL,
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
PRIMARY KEY (C_ANCESTOR_GROUP_ID, C_DESCENDANT_GROUP_ID),
CONSTRAINT CYCLIC_GROUPS CHECK
 (C_ANCESTOR_GROUP_ID != C_DESCENDANT_GROUP_ID OR C_COUNT = 1)
 );

CREATE INDEX I_ROX_GROUP_GRAPH_DESC_CNT ON ROX_GROUP_GRAPH (
   C_DESCENDANT_GROUP_ID, C_COUNT
);

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.userdb.PersistentGrantedPermission

CREATE TABLE ROX_GRANTED_PERMISSION (
OBJECTID	VARCHAR(56) NULL,
C_UPDATE_COUNT	INT,
C_GROUP_ID	VARCHAR(56) NOT NULL REFERENCES ROX_GROUP(OBJECTID),
C_PERMISSION_ID	VARCHAR(56) NOT NULL REFERENCES ROX_PERMISSION(OBJECTID),
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
PRIMARY KEY (C_GROUP_ID, C_PERMISSION_ID)
 );

CREATE INDEX I_ROX_GRANTED_PERMISSION_GRPID ON ROX_GRANTED_PERMISSION (
   C_GROUP_ID
);

CREATE TABLE ROX_SESSION_VAR_DESC (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_NAME	VARCHAR(32) NOT NULL CONSTRAINT c_session_var_unique_name UNIQUE,
C_DESCRIPTION	VARCHAR(1024) NOT NULL,
C_SECURE BOOL DEFAULT TRUE,
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1
 );

CREATE TABLE ROX_SESSION_VAR_SET (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_USERID VARCHAR(56) NOT NULL REFERENCES ROX_USER(OBJECTID),
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
CONSTRAINT c_session_var_unique_userid UNIQUE(C_USERID)
 );

CREATE TABLE ROX_SESSION_VAR (
OBJECTID	VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_VAR_SET_ID VARCHAR(56) NOT NULL REFERENCES ROX_SESSION_VAR_SET(OBJECTID),
C_NAME	VARCHAR(32) NOT NULL CONSTRAINT c_session_var_name_ref REFERENCES ROX_SESSION_VAR_DESC(C_NAME),
C_VALUE	TEXT NOT NULL,
C_SECURE BOOL NOT NULL,
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
CONSTRAINT c_session_var_unique_set UNIQUE(C_VAR_SET_ID,C_NAME)
 );

CREATE FUNCTION rox_group_relations_count(varchar, varchar)
	   RETURNS int4
	   AS 'SELECT c_count FROM rox_group_graph
			 WHERE c_ancestor_group_id = $1 AND c_descendant_group_id = $2;'
LANGUAGE 'sql';

-- END USERDB SCHEMA --


-- BEGIN USERDB INITIAL DATA --

-- create admin user
INSERT INTO ROX_USER (OBJECTID, C_USERNAME, C_PASSWORD)
  VALUES('010010001024-0000000000000-00001-0000000001',
		 'admin', 'd2abaa37a7c3db1137d385e1d8c15fd2');

INSERT INTO ROX_USER (OBJECTID, C_USERNAME, C_PASSWORD, C_ACTIVE)
  VALUES('010010001019-0000000000000-00001-0000000001', '(unknown user)', 'password', false);

-- admin group
INSERT INTO ROX_GROUP (OBJECTID, C_NAME, C_DESCRIPTION, C_TYPE)
  VALUES('010010001024-0000000000000-00001-0000000002',
		 'admin', 'group containing CR system administrators',
		 'system');

-- registered group
INSERT INTO ROX_GROUP (OBJECTID, C_NAME, C_DESCRIPTION, C_TYPE)
  VALUES('010010001024-0000000000000-00002-0000000002',
		 'registered', 'group containing all registered CR users',
		 'system');

-- universal group
INSERT INTO ROX_GROUP (OBJECTID, C_NAME, C_DESCRIPTION, C_TYPE)
  VALUES('010010001024-0000000000000-00003-0000000002',
		 'universal', 'group containing the anonymous user',
		 'system');

-- group graph table info.  registered is child of universal.
-- admin descendants
INSERT INTO ROX_GROUP_GRAPH
  (C_ANCESTOR_GROUP_ID, C_DESCENDANT_GROUP_ID, C_COUNT)
  VALUES('010010001024-0000000000000-00001-0000000002',
		 '010010001024-0000000000000-00001-0000000002', 1);
INSERT INTO ROX_GROUP_GRAPH
  (C_ANCESTOR_GROUP_ID, C_DESCENDANT_GROUP_ID, C_COUNT)
  VALUES('010010001024-0000000000000-00001-0000000002',
		 '010010001024-0000000000000-00002-0000000002', 0);
INSERT INTO ROX_GROUP_GRAPH
  (C_ANCESTOR_GROUP_ID, C_DESCENDANT_GROUP_ID, C_COUNT)
  VALUES('010010001024-0000000000000-00001-0000000002',
		 '010010001024-0000000000000-00003-0000000002', 0);

-- registered descendants
INSERT INTO ROX_GROUP_GRAPH
  (C_ANCESTOR_GROUP_ID, C_DESCENDANT_GROUP_ID, C_COUNT)
  VALUES('010010001024-0000000000000-00002-0000000002',
		 '010010001024-0000000000000-00001-0000000002', 0);
INSERT INTO ROX_GROUP_GRAPH
  (C_ANCESTOR_GROUP_ID, C_DESCENDANT_GROUP_ID, C_COUNT)
  VALUES('010010001024-0000000000000-00002-0000000002',
		 '010010001024-0000000000000-00002-0000000002', 1);
INSERT INTO ROX_GROUP_GRAPH
  (C_ANCESTOR_GROUP_ID, C_DESCENDANT_GROUP_ID, C_COUNT)
  VALUES('010010001024-0000000000000-00002-0000000002',
		 '010010001024-0000000000000-00003-0000000002', 0);

-- universal descendants
INSERT INTO ROX_GROUP_GRAPH
  (C_ANCESTOR_GROUP_ID, C_DESCENDANT_GROUP_ID, C_COUNT)
  VALUES('010010001024-0000000000000-00003-0000000002',
		 '010010001024-0000000000000-00001-0000000002', 0);
INSERT INTO ROX_GROUP_GRAPH
  (C_ANCESTOR_GROUP_ID, C_DESCENDANT_GROUP_ID, C_COUNT)
  VALUES('010010001024-0000000000000-00003-0000000002',
		 '010010001024-0000000000000-00002-0000000002', 1);
INSERT INTO ROX_GROUP_GRAPH
  (C_ANCESTOR_GROUP_ID, C_DESCENDANT_GROUP_ID, C_COUNT)
  VALUES('010010001024-0000000000000-00003-0000000002',
		 '010010001024-0000000000000-00003-0000000002', 1);

-- add direct parent/child relationships
INSERT INTO ROX_PARENT_CHILD_GROUP (C_PARENT_GROUP_ID, C_CHILD_GROUP_ID)
  VALUES('010010001024-0000000000000-00003-0000000002',
		 '010010001024-0000000000000-00002-0000000002');

-- add group member links (admin user member of admin group)
INSERT INTO ROX_GROUP_MEMBER (C_USER_ID, C_GROUP_ID)
  VALUES('010010001024-0000000000000-00001-0000000001',
		 '010010001024-0000000000000-00001-0000000002');

-- userdb read permission
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME, C_ACTIONS)
  VALUES('010010001024-0000000000000-00001-0000000003',
		 'com.raplix.rolloutexpress.persist.PersistPermission',
		 'userdb.*', 'read');

-- userdb write permission
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME, C_ACTIONS)
  VALUES('010010001024-0000000000000-00002-0000000003',
		 'com.raplix.rolloutexpress.persist.PersistPermission',
		 'userdb.*', 'write');

-- host read permission
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME, C_ACTIONS)
  VALUES('010010001024-0000000000000-00003-0000000003',
		 'com.raplix.rolloutexpress.persist.PersistPermission',
		 'hostdb.host.*', 'read');

-- host write permission
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME, C_ACTIONS)
  VALUES('010010001024-0000000000000-00004-0000000003',
		 'com.raplix.rolloutexpress.persist.PersistPermission',
		 'hostdb.host.*', 'write');

-- host type read permission
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME, C_ACTIONS)
  VALUES('010010001024-0000000000000-00005-0000000003',
		 'com.raplix.rolloutexpress.persist.PersistPermission',
		 'hostdb.type.HostType', 'read');

-- host type write permission
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME, C_ACTIONS)
  VALUES('010010001024-0000000000000-00006-0000000003',
		 'com.raplix.rolloutexpress.persist.PersistPermission',
		 'hostdb.type.HostType', 'write');

-- rule read permission
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME, C_ACTIONS)
  VALUES('010010001024-0000000000000-00007-0000000003',
		 'com.raplix.rolloutexpress.persist.PersistPermission',
		 'rule.Rule', 'read');

-- rule write permission
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME, C_ACTIONS)
  VALUES('010010001024-0000000000000-00008-0000000003',
		 'com.raplix.rolloutexpress.persist.PersistPermission',
		 'rule.Rule', 'write');

-- all hosts diff run permission
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME, C_ACTIONS)
  VALUES('010010001024-0000000000000-00013-0000000003',
		 'com.raplix.rolloutexpress.difference.DifferencePermission',
		 'allhosts', NULL);

-- diff read permission
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME, C_ACTIONS)
  VALUES('010010001024-0000000000000-00014-0000000003',
		 'com.raplix.rolloutexpress.persist.PersistPermission',
		 'differencedb.DifferenceSettings', 'read');

-- diff write permission
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME, C_ACTIONS)
  VALUES('010010001024-0000000000000-00015-0000000003',
		 'com.raplix.rolloutexpress.persist.PersistPermission',
		 'differencedb.DifferenceSettings', 'write');


-- system service ref read permission
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME, C_ACTIONS)
  VALUES('010010001024-0000000000000-00017-0000000003',
		 'com.raplix.rolloutexpress.persist.PersistPermission',
		 'SystemServiceRef', 'read');

-- system service ref write permission
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME, C_ACTIONS)
  VALUES('010010001024-0000000000000-00018-0000000003',
		 'com.raplix.rolloutexpress.persist.PersistPermission',
		 'SystemServiceRef', 'write');

-- component type ref read permission
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME, C_ACTIONS)
  VALUES('010010001024-0000000000000-00019-0000000003',
		 'com.raplix.rolloutexpress.persist.PersistPermission',
		 'ComponentTypeRef', 'read');

-- component type ref write permission
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME, C_ACTIONS)
  VALUES('010010001024-0000000000000-00020-0000000003',
		 'com.raplix.rolloutexpress.persist.PersistPermission',
		 'ComponentTypeRef', 'write');

-- plugin read permission
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME, C_ACTIONS)
  VALUES('010010001024-0000000000000-00027-0000000003',
		 'com.raplix.rolloutexpress.persist.PersistPermission',
		 'plugindb.Plugin', 'read');

-- plugin write permission
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME, C_ACTIONS)
  VALUES('010010001024-0000000000000-00028-0000000003',
		 'com.raplix.rolloutexpress.persist.PersistPermission',
		 'plugindb.Plugin', 'write');

-- All Folders all actions permission
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME, C_ACTIONS)
  VALUES('010010001024-0000000000000-00029-0000000003',
		 'com.raplix.rolloutexpress.systemmodel.userdb.FolderPermission',
		 'folder:*:hostset:*', '*');

-- Root Folder
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME, C_ACTIONS)
 VALUES('010010001024-0000000000000-00030-0000000003',
        'com.raplix.rolloutexpress.systemmodel.userdb.FolderPermission',
        'folder:010010001024-0000000000000-00000-0000000013', 'owner');
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME, C_ACTIONS)
 VALUES('010010001024-0000000000000-00031-0000000003',
        'com.raplix.rolloutexpress.systemmodel.userdb.FolderPermission',
        'folder:010010001024-0000000000000-00000-0000000013', 'write');
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME, C_ACTIONS)
 VALUES('010010001024-0000000000000-00032-0000000003',
        'com.raplix.rolloutexpress.systemmodel.userdb.FolderPermission',
        'folder:010010001024-0000000000000-00000-0000000013', 'autorun');
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME, C_ACTIONS)
 VALUES('010010001024-0000000000000-00033-0000000003',
        'com.raplix.rolloutexpress.systemmodel.userdb.FolderPermission',
        'folder:010010001024-0000000000000-00000-0000000013', 'checkin');

-- Task Permission all tasks.
INSERT INTO ROX_PERMISSION (OBJECTID, C_CLASS_NAME, C_NAME)
 VALUES('010010001024-0000000000000-00038-0000000003',
        'com.raplix.rolloutexpress.systemmodel.userdb.TaskPermission',
		  'owner.*');



-- add granted permission links
-- (read permissions granted to registered group,
--  all others granted to admin group)
INSERT INTO ROX_GRANTED_PERMISSION (C_GROUP_ID, C_PERMISSION_ID)
  VALUES('010010001024-0000000000000-00002-0000000002',
		 '010010001024-0000000000000-00001-0000000003');
INSERT INTO ROX_GRANTED_PERMISSION (C_GROUP_ID, C_PERMISSION_ID)
  VALUES('010010001024-0000000000000-00001-0000000002',
		 '010010001024-0000000000000-00002-0000000003');
INSERT INTO ROX_GRANTED_PERMISSION (C_GROUP_ID, C_PERMISSION_ID)
  VALUES('010010001024-0000000000000-00002-0000000002',
		 '010010001024-0000000000000-00003-0000000003');
INSERT INTO ROX_GRANTED_PERMISSION (C_GROUP_ID, C_PERMISSION_ID)
  VALUES('010010001024-0000000000000-00001-0000000002',
		 '010010001024-0000000000000-00004-0000000003');
INSERT INTO ROX_GRANTED_PERMISSION (C_GROUP_ID, C_PERMISSION_ID)
  VALUES('010010001024-0000000000000-00002-0000000002',
		 '010010001024-0000000000000-00005-0000000003');
INSERT INTO ROX_GRANTED_PERMISSION (C_GROUP_ID, C_PERMISSION_ID)
  VALUES('010010001024-0000000000000-00001-0000000002',
		 '010010001024-0000000000000-00006-0000000003');
INSERT INTO ROX_GRANTED_PERMISSION (C_GROUP_ID, C_PERMISSION_ID)
  VALUES('010010001024-0000000000000-00002-0000000002',
		 '010010001024-0000000000000-00007-0000000003');
INSERT INTO ROX_GRANTED_PERMISSION (C_GROUP_ID, C_PERMISSION_ID)
  VALUES('010010001024-0000000000000-00001-0000000002',
		 '010010001024-0000000000000-00008-0000000003');
INSERT INTO ROX_GRANTED_PERMISSION (C_GROUP_ID, C_PERMISSION_ID)
  VALUES('010010001024-0000000000000-00001-0000000002',
		 '010010001024-0000000000000-00013-0000000003');
INSERT INTO ROX_GRANTED_PERMISSION (C_GROUP_ID, C_PERMISSION_ID)
  VALUES('010010001024-0000000000000-00002-0000000002',
		 '010010001024-0000000000000-00014-0000000003');
INSERT INTO ROX_GRANTED_PERMISSION (C_GROUP_ID, C_PERMISSION_ID)
  VALUES('010010001024-0000000000000-00001-0000000002',
		 '010010001024-0000000000000-00015-0000000003');
INSERT INTO ROX_GRANTED_PERMISSION (C_GROUP_ID, C_PERMISSION_ID)
  VALUES('010010001024-0000000000000-00002-0000000002',
		 '010010001024-0000000000000-00017-0000000003');
INSERT INTO ROX_GRANTED_PERMISSION (C_GROUP_ID, C_PERMISSION_ID)
  VALUES('010010001024-0000000000000-00001-0000000002',
		 '010010001024-0000000000000-00018-0000000003');
INSERT INTO ROX_GRANTED_PERMISSION (C_GROUP_ID, C_PERMISSION_ID)
  VALUES('010010001024-0000000000000-00002-0000000002',
		 '010010001024-0000000000000-00019-0000000003');
INSERT INTO ROX_GRANTED_PERMISSION (C_GROUP_ID, C_PERMISSION_ID)
  VALUES('010010001024-0000000000000-00001-0000000002',
		 '010010001024-0000000000000-00020-0000000003');
INSERT INTO ROX_GRANTED_PERMISSION (C_GROUP_ID, C_PERMISSION_ID)
  VALUES('010010001024-0000000000000-00002-0000000002',
		 '010010001024-0000000000000-00027-0000000003');
INSERT INTO ROX_GRANTED_PERMISSION (C_GROUP_ID, C_PERMISSION_ID)
  VALUES('010010001024-0000000000000-00001-0000000002',
		 '010010001024-0000000000000-00028-0000000003');
INSERT INTO ROX_GRANTED_PERMISSION (C_GROUP_ID, C_PERMISSION_ID)
  VALUES('010010001024-0000000000000-00001-0000000002',
		 '010010001024-0000000000000-00029-0000000003');
INSERT INTO ROX_GRANTED_PERMISSION (C_GROUP_ID, C_PERMISSION_ID)
 VALUES('010010001024-0000000000000-00001-0000000002',
        '010010001024-0000000000000-00030-0000000003'); 
 INSERT INTO ROX_GRANTED_PERMISSION (C_GROUP_ID, C_PERMISSION_ID)
   VALUES('010010001024-0000000000000-00001-0000000002',
        '010010001024-0000000000000-00038-0000000003');

-- hidden session var for detecting password changes
INSERT INTO ROX_SESSION_VAR_DESC(OBJECTID,C_NAME,C_DESCRIPTION,C_SECURE)
 VALUES('010010001024-0000000000000-00040-0000000001',
 'sys:detectpasschange','detects password changes',true);


-- END USERDB INITIAL DATA --

-- BEGIN HOSTDB SCHEMA --

CREATE TABLE ROX_HOST_TYPE (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_NAME	VARCHAR(97) NOT NULL CONSTRAINT c_hosttype_unique_name UNIQUE,
C_PLUGIN_ID     VARCHAR(56) NULL DEFAULT NULL
           CONSTRAINT c_hosttype_ref_plugin REFERENCES ROX_PLUGIN(OBJECTID),
C_DESCRIPTION	VARCHAR(1024) NOT NULL
);

CREATE TABLE ROX_HOST_TYPE_VAR_DECL (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_HOST_TYPE_ID  VARCHAR(56) NOT NULL REFERENCES ROX_HOST_TYPE(OBJECTID),
C_VAR_NAME	VARCHAR(512) NOT NULL,
C_DEFAULT_VALUE	VARCHAR(1024) NOT NULL,
C_ORDER	INT NOT NULL,
UNIQUE(C_HOST_TYPE_ID, C_VAR_NAME),
UNIQUE(C_HOST_TYPE_ID, C_ORDER)
);

CREATE TABLE ROX_HOST (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_NAME	VARCHAR(512) NOT NULL CONSTRAINT c_host_unique_name UNIQUE,
C_DESCRIPTION	VARCHAR(1024) NOT NULL,
C_HOST_TYPE_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_host_ref_hosttype REFERENCES ROX_HOST_TYPE(OBJECTID),
C_PARENT_HOST_ID  VARCHAR(56) NULL DEFAULT NULL
	CONSTRAINT c_host_ref_parent REFERENCES ROX_HOST(OBJECTID),
C_INSTALLED_COMPONENT_ID VARCHAR(56) NULL DEFAULT NULL,
C_OS_NAME	VARCHAR(128) NULL,
C_OS_VERSION	VARCHAR(128) NULL,
C_OS_ARCH	VARCHAR(128) NULL,
C_RA_HOME_DIR	VARCHAR(1024) NULL,
C_RA_DATA_DIR	VARCHAR(1024) NULL,
C_RA_CONFIG_DIR	VARCHAR(1024) NULL,
C_RA_TMP_DIR	VARCHAR(1024) NULL,
C_RA_BUILD_NUMBER	VARCHAR(64) NULL,
C_RA_VERSION_NUMBER	VARCHAR(32) NULL,
C_RA_FILE_SEPARATOR	VARCHAR(5) NULL,
C_RA_PATH_SEPARATOR	VARCHAR(5) NULL,
C_MACHINE_INFO_STAMP	TIMESTAMP NULL,
UNIQUE(OBJECTID, C_HOST_TYPE_ID)
);

CREATE TABLE ROX_HOST_VAR_OVERRIDE (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_HOST_TYPE_ID  VARCHAR(56) NOT NULL REFERENCES ROX_HOST_TYPE(OBJECTID),
C_HOST_ID  VARCHAR(56) NOT NULL,
C_VAR_NAME	VARCHAR(512) NOT NULL,
C_OVERRIDE_VALUE	VARCHAR(1024) NOT NULL,
CONSTRAINT c_host_ref_var FOREIGN KEY (C_HOST_TYPE_ID, C_VAR_NAME)
   REFERENCES ROX_HOST_TYPE_VAR_DECL(C_HOST_TYPE_ID, C_VAR_NAME),
FOREIGN KEY (C_HOST_ID, C_HOST_TYPE_ID)
   REFERENCES ROX_HOST(OBJECTID, C_HOST_TYPE_ID)
);

CREATE INDEX I_ROX_HOST_VAR_OVER_HOST_NAME ON ROX_HOST_VAR_OVERRIDE (
  C_HOST_ID, 
  C_VAR_NAME
);
CREATE INDEX I_ROX_HOST_VAR_OVER_HOST ON ROX_HOST_VAR_OVERRIDE (
  C_HOST_ID
);

CREATE TABLE ROX_APP_INSTANCE (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_TYPE          SMALLINT NOT NULL,
C_CONNECTION_TYPE          SMALLINT NOT NULL DEFAULT 0,
C_PORT	INT NOT NULL,
C_IP_ADDRESS	INT NOT NULL,
C_IP_ADDRESS_STRING	VARCHAR(15) NOT NULL,
C_PARAMETERS	TEXT NOT NULL DEFAULT '',
C_HOST_ID  VARCHAR(56) NOT NULL REFERENCES ROX_HOST(OBJECTID),
C_PARENT_ID  VARCHAR(56) NULL
	CONSTRAINT c_app_ref_parent REFERENCES ROX_APP_INSTANCE(OBJECTID),
CONSTRAINT c_app_unique_port_ip UNIQUE(C_PORT, C_IP_ADDRESS),
UNIQUE(C_HOST_ID, C_TYPE)
);

CREATE INDEX I_ROX_APP_INSTANCE_C_HOST_ID ON ROX_APP_INSTANCE(C_HOST_ID);

CREATE TABLE ROX_HOST_SET (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_NAME	VARCHAR(97) NOT NULL CONSTRAINT c_hostset_unique_name UNIQUE,
C_PLUGIN_ID     VARCHAR(56) NULL DEFAULT NULL
           CONSTRAINT c_hostset_ref_plugin REFERENCES ROX_PLUGIN(OBJECTID),
C_DESCRIPTION	VARCHAR(1024) NOT NULL,
C_IS_SUPPORTED BOOL DEFAULT TRUE
);

CREATE TABLE ROX_HOST_SET_TO_HOST_LINK (
OBJECTID	VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT	INT NULL,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_PARENT_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_host_ref_hostset REFERENCES ROX_HOST_SET(OBJECTID),
C_CHILD_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_hostset_ref_host REFERENCES ROX_HOST(OBJECTID),
CONSTRAINT c_set_host_link_unique
	PRIMARY KEY (C_PARENT_ID, C_CHILD_ID)
);

CREATE INDEX I_HOST_SET_TO_HOST_LINK_CHILD ON ROX_HOST_SET_TO_HOST_LINK(C_CHILD_ID);

CREATE TABLE ROX_HOST_SEARCH (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_NAME	VARCHAR(97) NOT NULL CONSTRAINT c_hostsearch_unique_name UNIQUE,
C_PLUGIN_ID     VARCHAR(56) NULL DEFAULT NULL
           CONSTRAINT c_hostsearch_ref_plugin REFERENCES ROX_PLUGIN(OBJECTID),
C_DESCRIPTION	VARCHAR(1024) NOT NULL,
C_INCLUDE_MS    BOOL,
C_INCLUDE_LD    BOOL,
C_INCLUDE_RA    BOOL,
C_INCLUDE_NONE    BOOL,
C_INCLUDE_PHYSICAL    BOOL,
C_INCLUDE_VIRTUAL    BOOL
);

CREATE TABLE ROX_NAME_CRITERIA (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_COMPARISON_TYPE        SMALLINT NOT NULL,
C_VALUE_PATTERN_STRING        VARCHAR(1024) NOT NULL,
C_COMP_VALUE_PATTERN_SQL        VARCHAR(1024) NOT NULL,
C_HOST_SEARCH_ID  VARCHAR(56) NOT NULL REFERENCES ROX_HOST_SEARCH(OBJECTID)
);

CREATE TABLE ROX_DESCRIPTION_CRITERIA (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_COMPARISON_TYPE        SMALLINT NOT NULL,
C_VALUE_PATTERN_STRING        VARCHAR(1024) NOT NULL,
C_COMP_VALUE_PATTERN_SQL        VARCHAR(1024) NOT NULL,
C_HOST_SEARCH_ID  VARCHAR(56) NOT NULL REFERENCES ROX_HOST_SEARCH(OBJECTID)
);

CREATE TABLE ROX_HOST_TYPE_CRITERIA (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_COMPARISON_TYPE        SMALLINT NOT NULL,
C_VALUE_PATTERN_STRING        VARCHAR(1024) NOT NULL,
C_COMP_VALUE_PATTERN_SQL        VARCHAR(1024) NOT NULL,
C_HOST_SEARCH_ID  VARCHAR(56) NOT NULL REFERENCES ROX_HOST_SEARCH(OBJECTID)
);

CREATE TABLE ROX_IP_ADDRESS_CRITERIA (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_COMPARISON_TYPE        SMALLINT NOT NULL,
C_VALUE_PATTERN_STRING        VARCHAR(1024) NOT NULL,
C_COMP_VALUE_PATTERN_SQL        VARCHAR(1024) NOT NULL,
C_HOST_SEARCH_ID  VARCHAR(56) NOT NULL REFERENCES ROX_HOST_SEARCH(OBJECTID)
);

CREATE TABLE ROX_PARENT_CRITERIA (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_COMPARISON_TYPE        SMALLINT NOT NULL,
C_VALUE_PATTERN_STRING        VARCHAR(1024) NOT NULL,
C_COMP_VALUE_PATTERN_SQL        VARCHAR(1024) NOT NULL,
C_HOST_SEARCH_ID  VARCHAR(56) NOT NULL REFERENCES ROX_HOST_SEARCH(OBJECTID)
);

CREATE TABLE ROX_OS_NAME_CRITERIA (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_COMPARISON_TYPE        SMALLINT NOT NULL,
C_VALUE_PATTERN_STRING        VARCHAR(1024) NOT NULL,
C_COMP_VALUE_PATTERN_SQL        VARCHAR(1024) NOT NULL,
C_HOST_SEARCH_ID  VARCHAR(56) NOT NULL REFERENCES ROX_HOST_SEARCH(OBJECTID)
);

CREATE TABLE ROX_OS_VERSION_CRITERIA (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_COMPARISON_TYPE        SMALLINT NOT NULL,
C_VALUE_PATTERN_STRING        VARCHAR(1024) NOT NULL,
C_COMP_VALUE_PATTERN_SQL        VARCHAR(1024) NOT NULL,
C_HOST_SEARCH_ID  VARCHAR(56) NOT NULL REFERENCES ROX_HOST_SEARCH(OBJECTID)
);

CREATE TABLE ROX_OS_ARCH_CRITERIA (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_COMPARISON_TYPE        SMALLINT NOT NULL,
C_VALUE_PATTERN_STRING        VARCHAR(1024) NOT NULL,
C_COMP_VALUE_PATTERN_SQL        VARCHAR(1024) NOT NULL,
C_HOST_SEARCH_ID  VARCHAR(56) NOT NULL REFERENCES ROX_HOST_SEARCH(OBJECTID)
);

CREATE TABLE ROX_VAR_CRITERIA (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_COMPARISON_TYPE        SMALLINT NOT NULL,
C_VAR_NAME        VARCHAR(512) NOT NULL,
C_VALUE_PATTERN_STRING        VARCHAR(1024) NOT NULL,
C_COMP_VALUE_PATTERN_SQL        VARCHAR(1024) NOT NULL,
C_HOST_SEARCH_ID  VARCHAR(56) NOT NULL REFERENCES ROX_HOST_SEARCH(OBJECTID)
);

CREATE TABLE ROX_HOST_SET_TO_SEARCH_LINK (
OBJECTID	VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT	INT NULL,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_PARENT_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_search_ref_hostset REFERENCES ROX_HOST_SET(OBJECTID),
C_CHILD_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_hostset_ref_search REFERENCES ROX_HOST_SEARCH(OBJECTID),
CONSTRAINT c_set_search_link_unique
	PRIMARY KEY (C_PARENT_ID, C_CHILD_ID)
);

CREATE TABLE ROX_HOST_SET_TO_HOST_SET_LINK (
OBJECTID	VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT	INT NULL,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_PARENT_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_hostset_child_ref_parent REFERENCES ROX_HOST_SET(OBJECTID),
C_CHILD_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_hostset_parent_ref_child REFERENCES ROX_HOST_SET(OBJECTID),
CONSTRAINT c_set_set_link_unique
	PRIMARY KEY (C_PARENT_ID, C_CHILD_ID)
);


-- END HOSTDB SCHEMA --

-- BEGIN FOLDER SCHEMA --

CREATE TABLE ROX_FOLDER (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL DEFAULT 1,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_NAME	VARCHAR(512) NOT NULL,
C_DESCRIPTION	VARCHAR(1024) NOT NULL DEFAULT '',
C_FULL_PATH  VARCHAR(513) NOT NULL UNIQUE,
C_PARENT_FOLDER_ID  VARCHAR(56) NULL CONSTRAINT c_folder_ref_parent REFERENCES ROX_FOLDER(OBJECTID),
C_GROUP_ID  VARCHAR(56) NULL CONSTRAINT c_folder_ref_group REFERENCES ROX_GROUP(OBJECTID),
C_PLUGIN_ID  VARCHAR(56) NULL CONSTRAINT c_folder_ref_plugin REFERENCES ROX_PLUGIN(OBJECTID)
);
 
INSERT INTO ROX_FOLDER (OBJECTID, C_NAME, C_FULL_PATH, C_GROUP_ID)
  VALUES('010010001024-0000000000000-00000-0000000013', '', '/', '010010001024-0000000000000-00001-0000000002' );


CREATE TABLE ROX_FOLDER_TO_FOLDER_LINK(
OBJECTID	VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT	INT NULL,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_PARENT_ID  VARCHAR(56) NOT NULL,
C_CHILD_ID  VARCHAR(56) NOT NULL,
CONSTRAINT c_folder_dependency_link_unique
	PRIMARY KEY (C_PARENT_ID, C_CHILD_ID)
);


-- END FOLDER SCHEMA --




-- BEGIN RSRC SCHEMA --

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.resource.Resource

CREATE TABLE ROX_RESOURCE (
OBJECTID	VARCHAR(56) PRIMARY KEY,
VERSION	INT,
C_DESCRIPTION	VARCHAR(1024),
C_HIERARCHICAL	BOOLEAN NOT NULL DEFAULT FALSE,
C_TYPE	VARCHAR(64) NOT NULL,
C_SOURCE_HOSTID	VARCHAR(56) REFERENCES ROX_HOST(OBJECTID),
C_CONFIGURABLE	BOOLEAN NOT NULL DEFAULT FALSE,
USER_ID	VARCHAR(56) REFERENCES ROX_USER(OBJECTID),
C_TIMESTAMP	TIMESTAMP,
C_SIZE	BIGINT,
C_NAME	VARCHAR(1024),
ROOT_ID	VARCHAR(56) REFERENCES ROX_RESOURCE(OBJECTID),
C_HASPERMISSIONS	BOOLEAN NOT NULL DEFAULT FALSE,
C_UPDATE_COUNT	INT,
C_CUSTOMPARENTID	VARCHAR(56) REFERENCES ROX_RESOURCE(OBJECTID),
C_PATH_LOGIC_IDX	VARCHAR(32) NOT NULL,
C_PLATFORM	VARCHAR(56) REFERENCES ROX_HOST_SET(OBJECTID),
C_SOURCEPATH	VARCHAR(1024),
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_PLUGIN_ID  VARCHAR(56) NULL DEFAULT NULL CONSTRAINT c_resource_ref_plugin REFERENCES ROX_PLUGIN(OBJECTID),
UNIQUE (C_NAME,C_CUSTOMPARENTID,VERSION)
 );


CREATE TABLE ROX_RESOURCE_TO_CAT_LINK (
OBJECTID	VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT	INT NULL,
C_CHILD_ID 	VARCHAR(56) NOT NULL CONSTRAINT c_rsrc_link_resource REFERENCES ROX_RESOURCE(OBJECTID),
C_PARENT_ID 	VARCHAR(56) NOT NULL CONSTRAINT c_rsrc_link_category REFERENCES ROX_CATEGORY(OBJECTID),
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
CONSTRAINT c_resource_link_unique PRIMARY KEY (C_PARENT_ID, C_CHILD_ID)
);


-- END RSRC SCHEMA --


-- BEGIN CONFIGGEN SCHEMA

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.config.PersistentVariableSettings

-- note: var_set length computed as:
--  64 (typeName) + 512 (identifier) + 2048 (value) + 3 (separators)
CREATE TABLE ROX_GEN_VAR_SETTINGS (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL,

C_NAME	VARCHAR(32) NOT NULL,
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
C_VAR_SETTINGS	VARCHAR(2627)[] NOT NULL

 );

-- END CONFIGGEN SCHEMA

-- BEGIN COMPONENTDB SCHEMA


-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.componentdb.ComponentImpl

CREATE TABLE ROX_COMPONENT (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL,
C_TIMESTAMP	TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
USER_ID	VARCHAR(56) NULL  REFERENCES ROX_USER(OBJECTID),
ROOT_ID	VARCHAR(56) NOT NULL REFERENCES ROX_COMPONENT(OBJECTID),
VERSION	INT NOT NULL,
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
C_DESCRIPTION	VARCHAR(1024) NOT NULL DEFAULT '',
C_SOFTWARE_VENDOR	VARCHAR(256),
C_AUTHOR	VARCHAR(256),
C_LOCAL_INSTALL_PATH	VARCHAR(1024) NULL,
C_NAME	VARCHAR(512) NOT NULL,
C_PATHID	VARCHAR(56) NOT NULL DEFAULT '010010001024-0000000000000-00000-0000000013' CONSTRAINT c_component_ref_folder REFERENCES ROX_FOLDER(OBJECTID),
C_MODIFIER SMALLINT NOT NULL DEFAULT 0, -- NONE
C_ACCESS SMALLINT NOT NULL DEFAULT 0, -- PUBLIC
C_LABEL	VARCHAR(32) NOT NULL DEFAULT '',
C_IGNORE_PATHS	VARCHAR(1024)[],
C_PLATFORM	VARCHAR(97) NOT NULL DEFAULT 'system#any'
	 CONSTRAINT c_comp_ref_host_set_name REFERENCES ROX_HOST_SET(C_NAME),
C_LIMIT_TO_HOSTSET	VARCHAR(97) NULL
	 CONSTRAINT c_comp_limit_to_hostset REFERENCES ROX_HOST_SET(C_NAME),
C_HOST_TYPE VARCHAR(97) NULL
     CONSTRAINT c_targetable_comp_host_type REFERENCES ROX_HOST_TYPE(C_NAME),
C_EXTENDS_TYPE_NAME VARCHAR(129) NULL,
C_XML_CONTENT	TEXT,
C_SRC_INFO_RAW_HOST_ID  VARCHAR(56) NULL
	CONSTRAINT c_comp_ref_host REFERENCES ROX_HOST(OBJECTID),
C_SRC_INFO_RAW_STR  VARCHAR(1024) NULL,
C_IS_SIMPLE BOOLEAN NOT NULL DEFAULT false,
C_CONFIG_RSRC_VALIDATED BOOLEAN NOT NULL DEFAULT false,
C_LOCAL_RESOURCE_ID  VARCHAR(56) NULL
	CONSTRAINT c_comp_ref_rsrc REFERENCES ROX_RESOURCE(OBJECTID),
C_PLUGIN_ID VARCHAR(56) NULL REFERENCES ROX_PLUGIN(OBJECTID),
CONSTRAINT c_comp_unique_name UNIQUE(C_NAME,C_PATHID,VERSION)
 );


-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.componentdb.ComponentToComponentLink

CREATE TABLE ROX_COMPONENT_TO_COMP_LINK (
OBJECTID	VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT	INT NULL,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_PARENT_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_child_ref_component REFERENCES ROX_COMPONENT(OBJECTID),
C_CHILD_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_component_ref_child REFERENCES ROX_COMPONENT(OBJECTID),
CONSTRAINT c_comp_comp_link_unique
	PRIMARY KEY (C_PARENT_ID, C_CHILD_ID)
);

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.componentdb.ComponentToComponentNameRef

CREATE TABLE ROX_COMPONENT_TO_COMP_NAME_REF (
OBJECTID VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT INT NULL,
C_VISIBILITY SMALLINT NOT NULL DEFAULT 1,
C_PARENT_ID VARCHAR(56) NOT NULL
    CONSTRAINT c_comp_name_ref_component REFERENCES ROX_COMPONENT(OBJECTID),
C_CHILD_NAME VARCHAR(512) NOT NULL,
C_CHILD_PATH VARCHAR(512) NOT NULL DEFAULT '/',
CONSTRAINT c_comp_comp_name_ref_unique
	UNIQUE (C_PARENT_ID, C_CHILD_NAME, C_CHILD_PATH)
);

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.componentdb.ComponentToCategoryLink

CREATE TABLE ROX_COMPONENT_TO_CAT_LINK (
OBJECTID	VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT	INT NULL,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_PARENT_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_cat_ref_component REFERENCES ROX_CATEGORY(OBJECTID),
C_CHILD_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_component_ref_cat REFERENCES ROX_COMPONENT(OBJECTID),
CONSTRAINT c_comp_cat_link_unique
	PRIMARY KEY (C_PARENT_ID, C_CHILD_ID)
);


-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.componentdb.ComponentVariableSettings
-- note: override_settings length computed as:
-- 512 (identifier) + 1024 (value) + 1 (separator)

CREATE TABLE ROX_COMP_VAR_SETTINGS (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL,

C_NAME	VARCHAR(32) NOT NULL,
C_COMPONENT_ID	VARCHAR(56) NOT NULL
	CONSTRAINT c_comp_vars_ref_component REFERENCES ROX_COMPONENT(OBJECTID),
C_OVERRIDE_SETTINGS	VARCHAR(1537)[] NOT NULL,
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
CONSTRAINT c_comp_vars_unique_name UNIQUE (C_COMPONENT_ID, C_NAME)

);

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.componentdb.CompVarSettingsToComponentNameRef

CREATE TABLE ROX_COMPVARSET_TO_COMP_NAME_REF (
OBJECTID VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT INT NULL,
C_VISIBILITY SMALLINT NOT NULL DEFAULT 1,
C_PARENT_ID VARCHAR(56) NOT NULL
    CONSTRAINT c_cvs_name_ref_component REFERENCES ROX_COMP_VAR_SETTINGS(OBJECTID),
C_CHILD_NAME VARCHAR(512) NOT NULL,
C_CHILD_PATH VARCHAR(512) NOT NULL DEFAULT '/',
CONSTRAINT c_cvs_comp_name_ref_unique
    UNIQUE (C_PARENT_ID, C_CHILD_NAME, C_CHILD_PATH)
);

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.componentdb.SystemServiceRefImpl

CREATE TABLE ROX_SYSTEM_SERVICE_REF (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT NOT NULL,
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
C_NAME	VARCHAR(129) NOT NULL CONSTRAINT c_service_ref_unique_name UNIQUE,
C_DESCRIPTION	VARCHAR(1024) NOT NULL,
C_INST_COMP_PATHID     VARCHAR(56) NOT NULL CONSTRAINT c_systemservice_ref_folder REFERENCES ROX_FOLDER(OBJECTID),
C_INST_COMP_NAME	VARCHAR(512) NOT NULL,
C_INST_COMP_VERSION     INT NOT NULL,
C_INST_COMP_INSTALL_PATH     VARCHAR(1024) NULL,
C_PLUGIN_ID VARCHAR(56) NULL REFERENCES ROX_PLUGIN(OBJECTID),
 CONSTRAINT c_service_ref_ref_comp
 		FOREIGN KEY (C_INST_COMP_NAME, C_INST_COMP_PATHID, C_INST_COMP_VERSION)
        REFERENCES ROX_COMPONENT(C_NAME, C_PATHID, VERSION) INITIALLY DEFERRED,
 CONSTRAINT c_service_ref_unique_comp 
   UNIQUE (C_INST_COMP_NAME, C_INST_COMP_PATHID)
  );
 
 -- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.componentdb.ComponentTypeRefImpl
 
 CREATE TABLE ROX_COMPONENT_TYPE_REF (
 OBJECTID	VARCHAR(56) PRIMARY KEY,
 C_UPDATE_COUNT	INT NOT NULL,
 C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
 C_NAME	VARCHAR(129) NOT NULL CONSTRAINT c_comp_type_ref_unique_name UNIQUE,
 C_DESCRIPTION	VARCHAR(1024) NOT NULL,
 C_COMP_PATHID	VARCHAR(56) NOT NULL CONSTRAINT c_componenttype_ref_folder REFERENCES ROX_FOLDER(OBJECTID),
 C_COMP_NAME	VARCHAR(512) NOT NULL,
 C_COMP_VERSION     INT NOT NULL,
 C_ORDER         VARCHAR(64) NOT NULL,
 C_GROUP	 VARCHAR(64) NOT NULL,
 C_INDENT_LEVEL  INT NOT NULL,
 C_PLUGIN_ID VARCHAR(56) NULL REFERENCES ROX_PLUGIN(OBJECTID),
  CONSTRAINT c_component_type_ref_comp
  		FOREIGN KEY (C_COMP_NAME, C_COMP_PATHID, C_COMP_VERSION)
        REFERENCES ROX_COMPONENT(C_NAME, C_PATHID, VERSION) INITIALLY DEFERRED
  );

ALTER TABLE ROX_COMPONENT
  ADD CONSTRAINT c_comp_ref_comptype_name
  FOREIGN KEY (C_EXTENDS_TYPE_NAME) REFERENCES ROX_COMPONENT_TYPE_REF(C_NAME);

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.componentdb.CompTypeRefToCompTypeRefLink

CREATE TABLE ROX_COMPTYPE_TO_COMPTYPE_LINK (
OBJECTID	VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT	INT NULL,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_PARENT_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_desc_comptype_ref_comptype REFERENCES ROX_COMPONENT_TYPE_REF(OBJECTID),
C_CHILD_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_anc_comptype_ref_comptype REFERENCES ROX_COMPONENT_TYPE_REF(OBJECTID),
CONSTRAINT c_comptype_comptype_link_unique
	PRIMARY KEY (C_PARENT_ID, C_CHILD_ID)
);

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.componentdb.CompTypeRefToComponentLink

CREATE TABLE ROX_COMPTYPE_TO_COMP_LINK (
OBJECTID	VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT	INT NULL,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_PARENT_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_comp_ref_comptype REFERENCES ROX_COMPONENT_TYPE_REF(OBJECTID),
C_CHILD_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_comptype_ref_comp REFERENCES ROX_COMPONENT(OBJECTID),
CONSTRAINT c_comptype_comp_link_unique
	PRIMARY KEY (C_PARENT_ID, C_CHILD_ID)
);


-- END COMPONENTDB SCHEMA --


-- BEGIN PLANDB SCHEMA --

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.plandb.ExecutionPlan

CREATE TABLE ROX_EXECUTION_PLAN (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT,
C_TIMESTAMP	TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
USER_ID	VARCHAR(56) REFERENCES ROX_USER(OBJECTID),
C_NAME	VARCHAR(512) NOT NULL,
C_PATHID	VARCHAR(56) NOT NULL DEFAULT '010010001024-0000000000000-00000-0000000013' CONSTRAINT c_plan_ref_folder REFERENCES ROX_FOLDER(OBJECTID),
C_DESCRIPTION VARCHAR(1024) NOT NULL DEFAULT '',
C_STEPS	TEXT,
C_PLUGIN_ID VARCHAR(56) NULL REFERENCES ROX_PLUGIN(OBJECTID),
ROOT_ID	VARCHAR(56) REFERENCES ROX_EXECUTION_PLAN(OBJECTID) INITIALLY DEFERRED,
VERSION	INT,
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
CONSTRAINT c_plan_unique_name UNIQUE(C_NAME,C_PATHID,VERSION)
 );


-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.plandb.PlanToCategoryLink

CREATE TABLE ROX_PLAN_TO_CAT_LINK (
OBJECTID	VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT	INT NULL,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_PARENT_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_cat_ref_plan REFERENCES ROX_CATEGORY(OBJECTID) INITIALLY DEFERRED,
C_CHILD_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_plan_ref_cat REFERENCES ROX_EXECUTION_PLAN(OBJECTID) INITIALLY DEFERRED,
CONSTRAINT c_plan_cat_link_unique
	PRIMARY KEY (C_PARENT_ID, C_CHILD_ID)
);

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.plandb.ExecutionPlanToComponentNameRef

CREATE TABLE ROX_PLAN_TO_COMP_NAME_REF (
OBJECTID VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT INT NULL,
C_VISIBILITY SMALLINT NOT NULL DEFAULT 1,
C_PARENT_ID VARCHAR(56) NOT NULL
    CONSTRAINT c_comp_name_ref_plan REFERENCES ROX_EXECUTION_PLAN(OBJECTID),
C_CHILD_NAME VARCHAR(512) NOT NULL,
C_CHILD_PATH VARCHAR(512) NOT NULL DEFAULT '/',
CONSTRAINT c_plan_comp_name_ref_unique
    UNIQUE (C_PARENT_ID, C_CHILD_NAME, C_CHILD_PATH)
);

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.plandb.ExecutionPlanToExecutionPlanNameRef

CREATE TABLE ROX_PLAN_TO_PLAN_NAME_REF (
OBJECTID VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT INT NULL,
C_VISIBILITY SMALLINT NOT NULL DEFAULT 1,
C_PARENT_ID VARCHAR(56) NOT NULL
    CONSTRAINT c_plan_name_ref_plan REFERENCES ROX_EXECUTION_PLAN(OBJECTID),
C_CHILD_NAME VARCHAR(512) NOT NULL,
C_CHILD_PATH VARCHAR(512) NOT NULL DEFAULT '/',
CONSTRAINT c_plan_plan_name_ref_unique
    UNIQUE (C_PARENT_ID, C_CHILD_NAME, C_CHILD_PATH)
);

-- END PLANDB SCHEMA --


-- BEGIN RULES SCHEMA --

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.event.rule.PersistentCriteriaMetaData

CREATE TABLE ROX_CRITERIA (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT,
C_MESSAGE_PATTERN	VARCHAR(256) NULL,
C_TASK_EVENT_TYPE INT NOT NULL,
C_EVENT_TYPES	VARCHAR(256)[] NULL,
C_HOST_SET	VARCHAR(56) NULL
		CONSTRAINT c_criteria_ref_hostset REFERENCES ROX_HOST_SET(OBJECTID),
C_HOST_STATUS INT NOT NULL,
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
C_SEVERITIES	INT[] NULL
 );

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.event.rule.CriteriaToHostLink

CREATE TABLE ROX_CRITERIA_TO_HOST_LINK (
OBJECTID    VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT  INT NULL,
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
C_PARENT_ID  VARCHAR(56) NOT NULL
    CONSTRAINT c_host_ref_criteria REFERENCES ROX_CRITERIA(OBJECTID),
C_CHILD_ID  VARCHAR(56) NOT NULL
    CONSTRAINT c_criteria_ref_host REFERENCES ROX_HOST(OBJECTID),
CONSTRAINT c_criteria_host_link_unique
    PRIMARY KEY (C_PARENT_ID, C_CHILD_ID)
);

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.event.rule.PersistentActionMetaData

CREATE TABLE ROX_ACTION (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT,
C_EMAIL_ADDRESS	VARCHAR(1024) NULL,
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
C_ACTION_TYPE	VARCHAR(256) NOT NULL
 );

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.event.rule.PersistentRuleMetaData
-- TBD: how to specify foreign key constraint on array?

CREATE TABLE ROX_RULE (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT	INT,
C_CRITERIA	VARCHAR(56)[] NOT NULL,
C_NAME	VARCHAR(32) NOT NULL UNIQUE,
C_IS_ACTIVE	BOOLEAN NOT NULL,
C_ACTIONS	VARCHAR(56)[] NOT NULL,
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
C_DESCRIPTION	VARCHAR(1024) NULL
 );

-- END RULES SCHEMA --
-- BEGIN TARGET SCHEMA --
-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.executor.target.Target

CREATE TABLE ROX_TARGET (
OBJECTID		VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT		INT,
C_LOGICAL_HOST_ID	VARCHAR(56) NOT NULL REFERENCES ROX_HOST(OBJECTID),
C_ACTUAL_HOST_ID	VARCHAR(56) NOT NULL REFERENCES ROX_HOST(OBJECTID),
C_PHYSICAL_HOST_ID	VARCHAR(56) NOT NULL REFERENCES ROX_HOST(OBJECTID),
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
CONSTRAINT c_target_unique_hosts
	UNIQUE(C_LOGICAL_HOST_ID, C_ACTUAL_HOST_ID, C_PHYSICAL_HOST_ID)
);


CREATE INDEX I_ROX_TARGET_LOGICAL ON ROX_TARGET(C_LOGICAL_HOST_ID);
CREATE INDEX I_ROX_TARGET_ACTUAL ON ROX_TARGET(C_ACTUAL_HOST_ID);
CREATE INDEX I_ROX_TARGET_PHYSICAL ON ROX_TARGET(C_PHYSICAL_HOST_ID);

-- END TARGET SCHEMA --

-- BEGIN TASK_INFO SCHEMA --
-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.executor.TaskInfo

CREATE TABLE ROX_TASK_INFO (
OBJECTID		VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT		INT,
C_PLAN_ID		VARCHAR(56) NOT NULL REFERENCES ROX_EXECUTION_PLAN(OBJECTID) INITIALLY DEFERRED,
C_RUN_LEVEL		TEXT NOT NULL,
C_STATUS		TEXT DEFAULT 'Not Started',
C_PLAN_TIMEOUT		INT NOT NULL,
C_NATIVE_CALL_TIMEOUT	INT NOT NULL,
C_ROX_USER	        VARCHAR(56) NOT NULL REFERENCES ROX_USER(OBJECTID),
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_TASKSTART		TIMESTAMP,
C_TASKCOMPLETE          TIMESTAMP,
C_PREFLIGHTFAILED	BOOL,
C_ABORTREQUESTED	BOOL DEFAULT false,
C_ERRORCOUNT		INT DEFAULT 0,
C_WARNINGCOUNT		INT DEFAULT 0,
C_PLAN_THROTTLE     	INT DEFAULT 0,
C_DETAILED_PREFLIGHT 	BOOL DEFAULT false
);

CREATE INDEX I_ROX_TASK_INFO_PLANID on ROX_TASK_INFO(C_PLAN_ID);

-- END TASK_INFO SCHEMA --

-- BEGIN STEP_INFO SCHEMA --
-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.executor.StepInfo
CREATE TABLE ROX_STEP_INFO (
OBJECTID		VARCHAR(56) PRIMARY KEY,
C_TARGETID		VARCHAR(56) NOT NULL,
C_TASKID		VARCHAR(56),
C_EXECUTIONPLANID	VARCHAR(56),
C_PARENTSTEPID		VARCHAR(56),
C_STEPMAP		INT,
C_HOSTSTEPMAP		INT,
C_STEPTYPE		INT DEFAULT -1,
C_NESTINGLEVEL		INT DEFAULT 0,
C_TARGETCOUNT		INT DEFAULT 0,
C_HOSTSSTARTED          INT DEFAULT 0,
C_HOSTSCOMPLETED	INT DEFAULT 0,
C_HOSTSFAILED		INT DEFAULT 0,
C_SEVERITY		INT,
C_MAPPABLE		BOOL NOT NULL DEFAULT true,
C_SKIPPED               BOOL NOT NULL DEFAULT false,
C_MANDATORYSTEP         BOOL DEFAULT false,
C_HANDLED               BOOL NOT NULL DEFAULT false,
C_RUNLEVEL		TEXT NOT NULL,
C_STATUS		TEXT NOT NULL DEFAULT 'Not Started',
C_LABEL		    TEXT,
C_STARTDATE		TIMESTAMP,
C_COMPLETEDATE		TIMESTAMP,
C_DESCRIPTION		TEXT,
C_EXCEPTIONDESCRIPTION	TEXT,
C_UPDATE_COUNT		INT,
C_VISIBILITY		SMALLINT NOT NULL DEFAULT 1
);

CREATE INDEX I_ROX_STP_INF_TSK_STP_HST_RL ON ROX_STEP_INFO(C_TASKID, C_STEPMAP, C_HOSTSTEPMAP, C_RUNLEVEL);
CREATE INDEX I_ROX_STP_INF_PRNT ON ROX_STEP_INFO(C_PARENTSTEPID);
CREATE INDEX I_ROX_STEP_INFO_C_TARGETID ON ROX_STEP_INFO(C_TARGETID);

-- END STEP_INFO SCHEMA --

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.executor.task.ExecNativeOutput
CREATE TABLE ROX_EXEC_NATIVE_INFO (
OBJECTID		VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT		INT,
C_STDOUT		TEXT NULL,
C_OUT_START_IDX		BIGINT NULL,
C_STDERR		TEXT NULL,
C_ERR_START_IDX   BIGINT NULL,
C_EXIT_CODE       VARCHAR(11) NULL,
C_TARGET_ID	VARCHAR(56) NOT NULL REFERENCES ROX_TARGET(OBJECTID) INITIALLY DEFERRED,
C_TASK_ID 	VARCHAR(56) NOT NULL REFERENCES ROX_TASK_INFO(OBJECTID) INITIALLY DEFERRED,
C_STEP_ID		VARCHAR(56) NOT NULL REFERENCES ROX_STEP_INFO(OBJECTID) INITIALLY DEFERRED,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
UNIQUE(C_TASK_ID,C_TARGET_ID,C_STEP_ID)
);
CREATE INDEX I_ROX_EXEC_NATIVE_INFO_STEPID ON ROX_EXEC_NATIVE_INFO(C_STEP_ID);
CREATE INDEX I_ROX_EXEC_NATIVE_INFO_TGTID ON ROX_EXEC_NATIVE_INFO(C_TARGET_ID);

-- BEGIN INSTALL DB SCHEMA --

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.installdb.PersistentInstalledComponent

CREATE TABLE ROX_INSTALLED_COMPONENT (
OBJECTID		VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT		INT,
C_COMPONENT_ID		VARCHAR(56) NOT NULL REFERENCES ROX_COMPONENT(OBJECTID),
C_INSTALL_PATH	        VARCHAR(1024) NOT NULL,
C_TARGET_ID		VARCHAR(56) NOT NULL REFERENCES ROX_TARGET(OBJECTID) INITIALLY DEFERRED,
C_VARIABLE_SETTINGS_ID		VARCHAR(56) NOT NULL REFERENCES ROX_GEN_VAR_SETTINGS(OBJECTID),
C_TASK_ID		VARCHAR(56) NOT NULL REFERENCES ROX_TASK_INFO(OBJECTID) INITIALLY DEFERRED,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_PLAN_ID		VARCHAR(56) NOT NULL REFERENCES ROX_EXECUTION_PLAN(OBJECTID),
C_INSTALL_NAME		VARCHAR(512) DEFAULT 'default',
C_INSTALLDATE	        TIMESTAMP,
C_UNINSTALLDATE	        TIMESTAMP,
C_INSTALL_STATUS        INT NOT NULL DEFAULT 0,
C_PARENT_ICOMP_ID        VARCHAR(56) NULL 
        CONSTRAINT c_icomp_ref_parent_icomp REFERENCES ROX_INSTALLED_COMPONENT(OBJECTID),
C_ROOT_ICOMP_ID          VARCHAR(56) NOT NULL 
        CONSTRAINT c_icomp_ref_root_icomp REFERENCES ROX_INSTALLED_COMPONENT(OBJECTID),
C_REACHABLE_ICOMP_ID     VARCHAR(56) NOT NULL 
        CONSTRAINT c_icomp_ref_reach_icomp REFERENCES ROX_INSTALLED_COMPONENT(OBJECTID),
C_PARENT_ICOMP_REF_NAME  VARCHAR(32) NULL
 );

CREATE INDEX I_ROX_INST_COMP_TARGETID ON ROX_INSTALLED_COMPONENT(C_TARGET_ID, C_INSTALL_STATUS, C_INSTALL_PATH);
CREATE INDEX I_ROX_INST_COMP_COMPID ON ROX_INSTALLED_COMPONENT(C_COMPONENT_ID, C_INSTALL_STATUS, C_TARGET_ID);
CREATE INDEX I_ROX_INST_COMP_PARENT ON ROX_INSTALLED_COMPONENT(C_PARENT_ICOMP_ID);
CREATE INDEX I_ROX_INST_COMP_REACHABLE ON ROX_INSTALLED_COMPONENT(C_REACHABLE_ICOMP_ID);
CREATE INDEX I_ROX_INST_COMP_ROOT ON ROX_INSTALLED_COMPONENT(C_ROOT_ICOMP_ID);
CREATE INDEX I_ROX_INST_COMP_VARS ON ROX_INSTALLED_COMPONENT(C_VARIABLE_SETTINGS_ID);
CREATE INDEX I_ROX_INST_COMP_TASKID  ON ROX_INSTALLED_COMPONENT(C_TASK_ID);
CREATE INDEX I_ROX_INST_COMP_PLANID ON ROX_INSTALLED_COMPONENT(C_PLAN_ID);

ALTER TABLE ROX_HOST
  ADD CONSTRAINT c_host_ref_instcomp_id
  FOREIGN KEY (C_INSTALLED_COMPONENT_ID) REFERENCES ROX_INSTALLED_COMPONENT(OBJECTID);

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.installdb.InstalledCompToInstalledCompLink

CREATE TABLE ROX_INST_COMP_TO_INST_COMP_LINK (
OBJECTID	VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT	INT NULL,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_PARENT_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_desc_icomp_ref_icomp REFERENCES ROX_INSTALLED_COMPONENT(OBJECTID),
C_CHILD_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_anc_icomp_ref_icomp REFERENCES ROX_INSTALLED_COMPONENT(OBJECTID),
CONSTRAINT c_icomp_icomp_link_unique
	PRIMARY KEY (C_PARENT_ID, C_CHILD_ID)
);
CREATE INDEX I_INSTCOMP_LINK_CHILD ON ROX_INST_COMP_TO_INST_COMP_LINK(C_CHILD_ID);

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.installdb.PersistentInstalledResource

CREATE TABLE ROX_INSTALLED_RESOURCE (
OBJECTID		VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT		INT,
C_INSTALLED_COMPONENT_ID		VARCHAR(56) NOT NULL REFERENCES ROX_INSTALLED_COMPONENT(OBJECTID),
C_RESOURCE_ID		VARCHAR(56) NOT NULL REFERENCES ROX_RESOURCE(OBJECTID),
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
C_INSTALL_PATH	        VARCHAR(1024) NOT NULL
 );

CREATE INDEX I_ROX_INST_RSRC_INSTALLEDCOMPID ON ROX_INSTALLED_RESOURCE(C_INSTALLED_COMPONENT_ID);
CREATE INDEX I_ROX_INST_RSRC_INSTALLPATH ON ROX_INSTALLED_RESOURCE(C_INSTALL_PATH);

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.installdb.Snapshot

CREATE TABLE ROX_SNAPSHOT (
OBJECTID		VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT		INT,
C_INSTALLED_COMPONENT_ID		VARCHAR(56) NOT NULL REFERENCES ROX_INSTALLED_COMPONENT(OBJECTID),
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
C_NAME	        VARCHAR(512) NOT NULL,
C_INSTALL_ORDER	        INT NOT NULL,
C_STATUS	        INT DEFAULT 1,
C_CREATOR_TYPE_NAME     VARCHAR(64) NULL REFERENCES ROX_COMPONENT_TYPE_REF(C_NAME),
CONSTRAINT c_snapshot_unique_order
	UNIQUE(C_INSTALLED_COMPONENT_ID, C_INSTALL_ORDER)
 );

CREATE INDEX I_ROX_SNAPSHOT_INSTCOMPID ON ROX_SNAPSHOT(C_INSTALLED_COMPONENT_ID);

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.installdb.SnapshotToHostLink

CREATE TABLE ROX_SNAPSHOT_TO_HOST_LINK (
OBJECTID	VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT	INT NULL,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_PARENT_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_host_ref_snap REFERENCES ROX_SNAPSHOT(OBJECTID),
C_CHILD_ID  VARCHAR(56) NOT NULL
	CONSTRAINT c_snap_ref_host REFERENCES ROX_HOST(OBJECTID),
CONSTRAINT c_snap_host_link_unique
	PRIMARY KEY (C_PARENT_ID, C_CHILD_ID)
);

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.installdb.DependencyImpl

CREATE TABLE ROX_DEPENDENCY (
OBJECTID			VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT			INT,
C_NAME	        		VARCHAR(512) NOT NULL,
C_DEPENDANT_INST_COMP_ID	VARCHAR(56) NOT NULL REFERENCES ROX_INSTALLED_COMPONENT(OBJECTID),
C_DEPENDANT_COMP_TYPE_NAME      VARCHAR(64) NULL REFERENCES ROX_COMPONENT_TYPE_REF(C_NAME),	
C_DEPENDEE_INST_COMP_ID		VARCHAR(56) NOT NULL REFERENCES ROX_INSTALLED_COMPONENT(OBJECTID),
C_DEPENDEE_COMP_TYPE_NAME       VARCHAR(64) NULL REFERENCES ROX_COMPONENT_TYPE_REF(C_NAME),
C_INST_COMP_TARGETER		TEXT NOT NULL,
C_VISIBILITY    		SMALLINT NOT NULL DEFAULT 1,
CONSTRAINT c_dependency_unique
	UNIQUE(C_NAME, C_DEPENDANT_INST_COMP_ID)
 );

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.systemmodel.installdb.DependencyToComponentNameRef

CREATE TABLE ROX_DEPENDENCY_TO_COMP_NAME_REF (
OBJECTID VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT INT NULL,
C_VISIBILITY SMALLINT NOT NULL DEFAULT 1,
C_PARENT_ID VARCHAR(56) NOT NULL
    CONSTRAINT c_comp_name_ref_dep REFERENCES ROX_DEPENDENCY(OBJECTID),
C_CHILD_NAME VARCHAR(512) NOT NULL,
C_CHILD_PATH VARCHAR(512) NOT NULL DEFAULT '/',
CONSTRAINT c_dep_comp_name_ref_unique
    UNIQUE (C_PARENT_ID, C_CHILD_NAME, C_CHILD_PATH)
);

-- END INSTALL DB SCHEMA --

-- BEGIN DIFF SCHEMA --

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.difference.differencedb.DifferenceSettings

 CREATE TABLE ROX_DIFFERENCE_SETTINGS (
 OBJECTID	VARCHAR(56) PRIMARY KEY,
 C_UPDATE_COUNT	INT,
 C_DESCRIPTION	VARCHAR(1024),
 C_DIFF_LEVEL	INT,
 C_DIFF_STYLE	INT,
 C_SRC_DIR	VARCHAR(1024),
 C_DST_HOST	VARCHAR(56)
		CONSTRAINT c_diff_dst_ref_host REFERENCES ROX_HOST(OBJECTID),
 C_DST_HOST_SET	VARCHAR(56)
		CONSTRAINT c_diff_dst_ref_hostset REFERENCES ROX_HOST_SET(OBJECTID),
 C_SRC_HOST	VARCHAR(56)
		CONSTRAINT c_diff_src_ref_host REFERENCES ROX_HOST(OBJECTID),
 C_NAME	VARCHAR(256),
 C_IGNORE_PATHS	VARCHAR(1024)[],
 C_INST_COMPONENT_ID	VARCHAR(56) REFERENCES ROX_INSTALLED_COMPONENT(OBJECTID),
 C_DST_DIR	VARCHAR(1024),
 C_INST_RESOURCE_ID	VARCHAR(56) REFERENCES ROX_INSTALLED_RESOURCE(OBJECTID),
 C_DIFF_SCOPE	INT,
 C_TIMEOUT_SECS INT,
 C_INCLUDE_SUBDIRECTORIES BOOLEAN,
 C_SRC_PREPARE   TEXT,
 C_DST_PREPARE   TEXT,
 C_SRC_CLEANUP   TEXT,
 C_DST_CLEANUP   TEXT,
 C_USE_MINI_PLANS  BOOLEAN DEFAULT false,
 C_DST_USE_SRC_P   BOOLEAN DEFAULT true,
 C_DST_USE_SRC_C   BOOLEAN DEFAULT true,
 C_FOLLOW_LINKS   BOOLEAN DEFAULT true,
 ROOT_ID	VARCHAR(56) REFERENCES ROX_DIFFERENCE_SETTINGS(OBJECTID),
 VERSION	INT,
 C_TIMESTAMP	TIMESTAMP,
 USER_ID	VARCHAR(56) REFERENCES ROX_USER(OBJECTID),
 C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
 UNIQUE(C_NAME,VERSION)
 );

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.difference.Difference

 CREATE TABLE ROX_DIFFERENCE (
 OBJECTID	VARCHAR(56) PRIMARY KEY,
 C_UPDATE_COUNT	INT,
 C_DEST_ORDER	INT,
 C_DEST_VAL	TEXT,
 C_SOURCE_ORDER	INT,
 C_SOURCE_VAL	TEXT,
 C_TYPE	INT,
 C_IDENTIFIER	VARCHAR(256),
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
 C_TRUE_CLASS	VARCHAR(256)
  );

CREATE TABLE ROX_DIFFSET_TO_CAT_LINK (
OBJECTID	VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT	INT NULL,
C_CHILD_ID 	VARCHAR(56) NOT NULL CONSTRAINT c_diffsetcat_link_differencesettings REFERENCES ROX_DIFFERENCE_SETTINGS(OBJECTID),
C_PARENT_ID 	VARCHAR(56) NOT NULL CONSTRAINT c_diffsetcat_link_category REFERENCES ROX_CATEGORY(OBJECTID),
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
CONSTRAINT c_diffsetcat_link_unique PRIMARY KEY (C_PARENT_ID, C_CHILD_ID)
);

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.difference.differencedb.DifferenceSettingsToComponentNameRef

CREATE TABLE ROX_DIFFSET_TO_COMP_NAME_REF (
OBJECTID VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT INT NULL,
C_VISIBILITY SMALLINT NOT NULL DEFAULT 1,
C_PARENT_ID VARCHAR(56) NOT NULL
    CONSTRAINT c_comp_name_ref_diffset REFERENCES ROX_DIFFERENCE_SETTINGS(OBJECTID),
C_CHILD_NAME VARCHAR(512) NOT NULL,
C_CHILD_PATH VARCHAR(512) NOT NULL DEFAULT '/',
CONSTRAINT c_diffset_comp_name_ref_unique
    UNIQUE (C_PARENT_ID, C_CHILD_NAME, C_CHILD_PATH)
);

-- END DIFF SCHEMA --

-- BEGIN SUBPLAN_INFO SCHEMA --
-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.executor.SubplanInfo

CREATE TABLE ROX_SUBPLAN_INFO (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_TASKID        VARCHAR(56) NOT NULL CONSTRAINT c_subplan_taskinfo REFERENCES ROX_TASK_INFO(OBJECTID) INITIALLY DEFERRED,
C_SUBPLAN_NAME	VARCHAR(512) NOT NULL,
C_SUBPLAN_ID	VARCHAR(56) NOT NULL REFERENCES ROX_EXECUTION_PLAN(OBJECTID) INITIALLY DEFERRED,
C_COMPONENT_IDS	VARCHAR(56)[],
C_VARSET_IDS	VARCHAR(56)[],
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
C_UPDATE_COUNT	INT,
C_SUBPLAN_INDEX SMALLINT,
UNIQUE (C_TASKID, C_SUBPLAN_INDEX)
 );

CREATE INDEX I_ROX_SUBPLAN_INFO_SUBPLAN_ID ON ROX_SUBPLAN_INFO(C_SUBPLAN_ID);

CREATE TABLE ROX_SUBPLAN_TO_TARGET_LINK (
OBJECTID	VARCHAR(56) NULL DEFAULT '',
C_UPDATE_COUNT	INT NULL,
C_VISIBILITY	SMALLINT NOT NULL DEFAULT 1,
C_PARENT_ID     VARCHAR(56) NOT NULL CONSTRAINT c_subplan_target_subplaninfo REFERENCES ROX_SUBPLAN_INFO(OBJECTID) INITIALLY DEFERRED,
C_CHILD_ID      VARCHAR(56) NOT NULL CONSTRAINT c_subplan_target_target REFERENCES ROX_TARGET(OBJECTID) INITIALLY DEFERRED,
CONSTRAINT c_subplan_to_target_unique PRIMARY KEY (C_PARENT_ID, C_CHILD_ID)
);

CREATE INDEX I_SUBPLAN_TO_TGT_LINK_CHILDID ON ROX_SUBPLAN_TO_TARGET_LINK(C_CHILD_ID);

-- END ROX_SUBPLAN_INFO SCHEMA --

-- BEGIN TARGET_STATUS SCHEMA --
-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.event.TargetStatus
CREATE TABLE ROX_TARGET_STATUS (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_TARGET_ID 	VARCHAR(56) NOT NULL REFERENCES ROX_TARGET(OBJECTID) INITIALLY DEFERRED,
C_TASK_ID	VARCHAR(56) NOT NULL REFERENCES ROX_TASK_INFO(OBJECTID) INITIALLY DEFERRED,
C_STATUS	INT NOT NULL,
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
C_UPDATE_COUNT	INT
 );
CREATE INDEX I_ROX_TARGET_STATUS_TARGETID ON ROX_TARGET_STATUS(C_TARGET_ID);
CREATE INDEX I_ROX_TARGET_STATUS_TASKID ON ROX_TARGET_STATUS(C_TASK_ID, C_TARGET_ID);
-- END TARGET_STATUS SCHEMA --

-- BEGIN EVENT SCHEMA --

-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.event.ROXEventFactory

CREATE TABLE ROX_EVENTS (
OBJECTID		VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT		INT,
C_EVENTTYPE		VARCHAR(256),
C_TARGETID		VARCHAR(56),
C_TASKID		VARCHAR(56),
C_PATH			VARCHAR(1024),
C_SUBSYSTEM		VARCHAR(256),
C_CONTROLSERVICE	VARCHAR(256),
C_SEVERITY		INT,
C_SUBPLANINDEX		INT,
C_DIFFERENCEID		VARCHAR(56),
C_HOSTID		VARCHAR(56),
C_EXCEPTIONDESCRIPTION	VARCHAR(1024),
C_PLANID		VARCHAR(56),
C_USERID		VARCHAR(56),
C_USERNAME		VARCHAR(256),
C_INST_COMPONENTID	VARCHAR(56),
C_INST_RESOURCEID	VARCHAR(56),
C_EVENTDATE		TIMESTAMP,
C_MESSAGE		VARCHAR(512),
C_DIFFERENCEJOBID	VARCHAR(56),
C_DIFFERENCESETTINGSID	VARCHAR(56),
C_STEPID		VARCHAR(56),
C_TYPE			INT,
C_PREFLIGHT		BOOL,
C_PROGRESS		FLOAT4,
C_MAP			INT,
C_HOSTMAP		INT,
C_SKIPPED               BOOL,
C_MAPPABLE		BOOL,
C_HOSTNAME		VARCHAR(512),
C_HOSTIP		VARCHAR(20),
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1,
C_NODETYPE		VARCHAR(256),
C_HANDLED               BOOL
 );

CREATE INDEX I_ROX_EVENTS_EVENTTYPE ON ROX_EVENTS(C_EVENTTYPE);
CREATE INDEX I_ROX_EVENTS_INST_COMPONENTID ON ROX_EVENTS(C_INST_COMPONENTID);
CREATE INDEX I_ROX_EVENTS_INST_RESOURCEID ON ROX_EVENTS(C_INST_RESOURCEID);
CREATE INDEX I_ROX_EVENTS_TASKID ON ROX_EVENTS(C_TASKID);
CREATE INDEX I_ROX_EVENTS_EVENTDATE ON ROX_EVENTS(C_EVENTDATE);
CREATE INDEX I_ROX_EVENTS_STEPID ON ROX_EVENTS(C_STEPID);
CREATE INDEX I_EVENTS_DIFFSETTINGSID ON ROX_EVENTS(C_DIFFERENCESETTINGSID);

-- END EVENT SCHEMA --

-- BEGIN DIFFERENCE_JOB SCHEMA --
-- SQL Table Generation from Classmap for Class: com.raplix.rolloutexpress.difference.DifferenceJob

CREATE TABLE ROX_DIFFERENCE_JOB (
OBJECTID		VARCHAR(56) PRIMARY KEY,
C_UPDATE_COUNT		INT,
C_VISIBILITY		SMALLINT NOT NULL DEFAULT 1,
C_DIFFERENCESETTINGS_ID	VARCHAR(56) NOT NULL REFERENCES ROX_DIFFERENCE_SETTINGS(OBJECTID),
C_DIFFERENCESTART	TIMESTAMP NOT NULL,
C_DIFFERENCECOMPLETE	TIMESTAMP
);

CREATE INDEX I_ROX_DIFF_JOB_DIFFSETTINGS_ID ON ROX_DIFFERENCE_JOB(C_DIFFERENCESETTINGS_ID);
CREATE INDEX I_ROX_DIFF_JOB_DIFFSTART ON ROX_DIFFERENCE_JOB(C_DIFFERENCESTART);
CREATE INDEX I_ROX_DIFF_JOB_DIFFCOMPLETE ON ROX_DIFFERENCE_JOB(C_DIFFERENCECOMPLETE);

-- END DIFFERENCE_JOB SCHEMA --

CREATE FUNCTION rox_events_after_insert() returns opaque AS '
begin
  if((new.c_eventtype = \'com.raplix.rolloutexpress.event.ROXDifferenceStartEvent\') and
	 (new.c_preflight = false )) then
	INSERT INTO rox_difference_job (objectid,c_differencesettings_id,c_differencestart) VALUES(new.c_differencejobid,new.c_differencesettingsid,new.c_eventdate);
  end if;

  if((new.c_eventtype = \'com.raplix.rolloutexpress.event.ROXDifferenceCompleteEvent\') and
	 (new.c_preflight = false)) then
	UPDATE rox_difference_job SET c_differencecomplete = new.c_eventdate
	  WHERE new.c_differencejobid = rox_difference_job.objectid;
  end if;

  return new;
end;' LANGUAGE 'plpgsql';

CREATE TRIGGER t_rox_events_after_insert AFTER INSERT
	   ON rox_events FOR EACH ROW
	   EXECUTE PROCEDURE rox_events_after_insert();

-- END EVENT SCHEMA

CREATE TABLE ROX_AUTOGENERATED_PLAN (
OBJECTID	VARCHAR(56) PRIMARY KEY,
C_COMPONENTID	VARCHAR(56) NOT NULL REFERENCES ROX_COMPONENT(OBJECTID),
C_PLANID VARCHAR(56) NOT NULL REFERENCES ROX_EXECUTION_PLAN(OBJECTID) CONSTRAINT c_autogen_unique_planid UNIQUE,
C_PROCEDURENAME VARCHAR(512),
C_PROCEDURETARGET VARCHAR(1024),
C_PROCEDURETYPE TEXT NOT NULL,
C_STEPCOUNT	INT NOT NULL,
C_UPDATE_COUNT	INT NOT NULL,
C_VISIBILITY    SMALLINT NOT NULL DEFAULT 1);

CREATE INDEX I_ROX_AUTOGENERATED_PLAN_COMPID ON ROX_AUTOGENERATED_PLAN(C_COMPONENTID);
CREATE INDEX I_ROX_AUTOGENERATED_PLAN_PROCNM ON ROX_AUTOGENERATED_PLAN(C_PROCEDURENAME);
CREATE INDEX I_ROX_AUTOGENERATED_PLAN_PROCTG ON ROX_AUTOGENERATED_PLAN(C_PROCEDURETARGET);
CREATE INDEX I_ROX_AUTOGENERATED_PLAN_PROCTP ON ROX_AUTOGENERATED_PLAN(C_PROCEDURETYPE);

