#************************************************************ 
#
# Component      CNF  
#
# Synopsis       Reliable Boot Server
#
#
# Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
# Use is subject to license terms.
#
#
#************************************************************
#
# #ident	"@(#)RBS.pm	1.13	06/01/23 SMI"
#
#************************************************************

#=head1 Reliable Boot Server
#
#=over 4
#
#=cut

use strict;
package RBS ;

use VAR;
use LOG ;
use COMMON;

my $E_dhcp_configuration_file = "/etc/inet/dhcpsvc.conf" ;
my $E_dhcp_inittab            = "/etc/dhcp/inittab";
my $E_dhcp_symbol             = "NhCgtpAddr";

my $G_status = "OK" ;

my $G_nic0;
my $G_nic1;


#--------------------------------------------------------------
#
#  rbs_installed
#
#--------------------------------------------------------------

sub rbs_installed {
  
  my $version ;
  my $name ;
  my $basedir ;
  
  if (OS::is_feature_installed($RBS_FEATURE)) {
    return $TRUE ;
  }
  return $FALSE;
}

#--------------------------------------------------------------
#
#  get_parameter
#
#  1 : parameter name
#
#  return value of the parameter
#
#--------------------------------------------------------------

sub get_parameter
  {
    my ($L_parameter) = @_ ;

    my $result;
    $result = qx{/usr/bin/grep \"^$L_parameter=\" $E_dhcp_configuration_file} ;
    chomp($result);

    LOG::printMsg(3, "dhcpsvc.conf grep: %s", $result) ;
    if ($result eq "")
      {
	return "";
      }

    $result =~ s/#.*// ;
    $result =~ m/[\s\t]*([a-zA-Z0-9_]*)[\s\t]*=[\s\t]*([^\s\t]*)/ ;
    if ($1 ne $L_parameter)
      {
	return "" ;
      }

    LOG::printMsg(3, "%s value = %s", $L_parameter, $2) ;

    return $2 ;
}

#--------------------------------------------------------------
#
#  check_parameter
#
#  1 : parameter name
#  2 : parameter value
#
#  return value of the parameter
#
#--------------------------------------------------------------

sub check_parameter
  {
    my ($L_parameter, $L_value) = @_ ;

    my $value = get_parameter($L_parameter) ;
    if ($L_value eq "")
      {

#=item <parameter> not defined
#
#    The specified parameter must be defined in the 
#    /etc/inet/dhcpsvc.conf file.
#
#=cut

	LOG::printError("%s not defined", $L_parameter) ;
	$G_status = "FAILED" ;
	return ;
      }

    if ($L_value ne $value)
      {

#=item <parameter>: <current> instead of <expected>
#
#    Please set the correct value for the specified parameter in
#    the /etc/inet/dhcpsvc.conf file.
#
#=cut

	LOG::printError("%s: %s instead of %s", 
			$L_parameter,  $value, $L_value) ;
	$G_status = "FAILED" ;
      }
}

#--------------------------------------------------------------
#
#  check_dhcpsvc
#
#--------------------------------------------------------------

sub check_dhcpsvc {

  my $L_status = "OK";
  if (! OS::check_file($E_dhcp_configuration_file))
    {
      $L_status = "FAILED";
      LOG::printResult($L_status, "check %s",$E_dhcp_configuration_file) ;
      return ;
    } 
  
  #
  # parameter checking
  # 
  check_parameter("DAEMON_ENABLED", "TRUE") ;
  check_parameter("RUN_MODE", "server") ;
  check_parameter("RESOURCE", "SUNWnhrbs") ;
  check_parameter("CONVER", 1) ;
  LOG::printResult($G_status, "general parameter in %s", $E_dhcp_configuration_file) ;
  

  $L_status = "OK" ;  
  my $L_value = get_parameter("PATH") ;
  if ($L_value eq "")
    {
      
#=item PATH not defined
#
#    PATH is not defined in the /etc/inet/dhcpsvc.conf file.
#
#=cut
      
	LOG::printError("PATH not defined") ;
	$L_status = "FAILED";
      }
    else
      {	if (! -d $L_value)
	  {

#=item PATH does not reference a valid file (<file>)
#
#    Please check that the directory referenced by PATH in the 
#    /etc/inet/dhcpsvc.conf file exists, or set PATH with the correct
#    path name.
#
#=cut

	    LOG::printError("PATH does not reference a valid directory (%s)", $L_value) ;
	    $L_status = "FAILED";
	  }
      }

  LOG::printResult($L_status, "PATH in %s", $E_dhcp_configuration_file) ;

  $L_status = "OK";
  my $interfaces = get_parameter("INTERFACES");

  if ($interfaces eq "") {

#=item no interface defined in /etc/inet/dhcpsvc.conf
#
#    The DHCP network interfaces must be defined with the
#    INTERFACES parameter.
#
#=cut

	    LOG::printError("INTERFACES is missing") ;
	    $L_status = "FAILED";
  } else {
    
    ($G_nic0, $G_nic1) = split(",", $interfaces);
    my $physical_interface;
    
    if (! defined($G_nic0) || 
	(! COMMON::exist_physical_interface($G_nic0, \$physical_interface))) {

#=item nic0 <nic> for dhcp access does not exist
#
#    The specified interface does not exist. Please check
#    the /etc/inet/dhcpsvc.conf file to configure an existing
#    interface or configure the mentioned interface.
#
#=cut

      LOG::printError("nic0 %s for dhcp access does not exist", $G_nic0);
      $L_status = "FAILED";
    }
    
    if (($E_variables{USE_CGTP} eq "TRUE") && 
	(! defined($G_nic1) 
	 || (! COMMON::exist_physical_interface($G_nic1, \$physical_interface)))
       ) {
      
#=item nic1 <nic> for dhcp access does not exist
#
#    The specified interface does not exist. Please check
#    the /etc/inet/dhcpsvc.conf file to configure an existing
#    interface or configure the mentioned interface.
#
#=cut

      LOG::printError("nic1 %s for dhcp access does not exist", $G_nic1);
      $L_status = "FAILED";
    }
  }

  LOG::printResult($L_status, "INTERFACES in %s", $E_dhcp_configuration_file) ;
  
}

#--------------------------------------------------------------
#
#  check_inittab
#
#--------------------------------------------------------------

sub check_inittab {

  my $L_status = "OK";

  if (! OS::check_file($E_dhcp_inittab))
    {
      $L_status = "FAILED";
      LOG::printResult($L_status, "check %s",$E_dhcp_inittab) ;
      return ;
    } 

  # check if NhCgtpAddr is defined in /etc/dhcp/inittab
  my $result;
  chomp($result = qx{/usr/bin/grep "^$E_dhcp_symbol" $E_dhcp_inittab}) ;
  
  LOG::printMsg(3, "/etc/dhcp/inittab grep: %s***", $result) ;  
  $result =~ s/#.*// ;
  $result =~ m/[\s\t]*([a-zA-Z0-9_]*)[\s\t]*/ ;
  if ($1 ne $E_dhcp_symbol)
    {

#=item symbol <symbol> not defined in /etc/dhcp/inittab
#
#    The specified symbol is required to allow information to
#    be retrieved from the DHCP requests on a diskless node.
#
#=cut

	LOG::printError("symbol %s not defined in %s", $E_dhcp_symbol, $E_dhcp_inittab) ;
	$L_status = "FAILED" ;
      }
  
  LOG::printResult($L_status, "check %s",$E_dhcp_inittab) ;
}

#--------------------------------------------------------------
#
#  check_dhcpfiles
#
#--------------------------------------------------------------

sub check_dhcpfiles {

  my $L_status = "OK";

  my $file;
  my $netmask;
  my $ip;
  my $subnet;

  if (CMM::get_actual_role() ne "MASTER") {
    return ;
  }

  #
  # check the context (no error since it exists specific test
  #
  my $dir = get_parameter("PATH");
  if ($dir eq "") {
    return ;
  }
  if (! -d $dir) {
    return ;
  }
  my $resource = get_parameter("RESOURCE");
  if ($resource eq "") {
    return ;
  }

  #
  # macro file
  #
  $L_status = "OK";
  $file = "$dir/${resource}1_dhcptab";
  if (! OS::check_file($file))
    {
      $L_status = "FAILED";
    } 

  LOG::printResult($L_status, "dhcptab present") ;

  #
  # configuration for nic0 subnet
  #
  $L_status = "OK";
  if (defined($G_nic0) && COMMON::exist_physical_interface($G_nic0)) {
    $netmask = COMMON::get_interface_info($G_nic0, "NETMASK");
    $ip = COMMON::get_interface_info($G_nic0, "INET");
    $subnet = COMMON::convertDottedToBitmask($ip) & hex($netmask);
    $subnet = COMMON::convertBitmaskToDotted($subnet);
    $subnet =~ tr/\./_/;
  
    $file = "$dir/${resource}1_$subnet";
    if (! OS::check_file($file)) {
      $L_status = "FAILED";
    } 
  }

  LOG::printResult($L_status, "dhcp configuration for nic0 present") ;

  $L_status = "OK";
  if (($E_variables{USE_CGTP} eq "TRUE") &&
      (defined($G_nic1)) && COMMON::exist_physical_interface($G_nic1)) {
    #
    # configuration for nic1 subnet
    #
    $netmask = COMMON::get_interface_info($G_nic1, "NETMASK");
    $ip = COMMON::get_interface_info($G_nic1, "INET");
    $subnet = COMMON::convertDottedToBitmask($ip) & hex($netmask);
    $subnet = COMMON::convertBitmaskToDotted($subnet);
    $subnet =~ tr/\./_/;
    
    $file = "$dir/${resource}1_$subnet";
    if (! OS::check_file($file)) {
	$L_status = "FAILED";
      } 
    
    LOG::printResult($L_status, "dhcp configuration for nic1 present") ;
  }
}

#--------------------------------------------------------------
#
#  check_configuration
#
#--------------------------------------------------------------

sub check_configuration
  {
    if (OS::guess_node_role ne "DISKFULL") {
      return ;
    }
    
    if (! rbs_installed) {
      return ;
    }

    LOG::printMsg(0,"");
    LOG::printMsg(0,"Reliable Boot Server checking");
    LOG::printMsg(0,"");

    check_dhcpsvc;
    check_inittab;
    check_dhcpfiles;

  }

#--------------------------------------------------------------
#
#  display
#
#      display information regarding Reliable Boot Server
#
#--------------------------------------------------------------

sub display
 { 
   return;
   printLine("") ;
   printLine("Reliable Boot Server") ;
   printLine("") ;    
 } 

#--------------------------------------------------------------
#
#  main
#
#--------------------------------------------------------------

#=back
#
#=cut

{
}
