#************************************************************ 
#
# Component      CNF  
#
# Synopsis       Operating System and Packages
#
#
# Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
# Use is subject to license terms.
#
#
#************************************************************
#
# #ident	"@(#)OS.pm	1.50	06/02/03 SMI"
#
#************************************************************

#=head1 Operating System and Packages
#
#    This section also contains descriptions for error than can be detected
#    in circumstances that are not related to installation.
#
#=over 4
#
#=cut

package OS ;

use strict;
use VAR;
use LOG;

my $G_category = "OS" ;

my @G_empty_array = ();

# for compare_version
my $OK      = 1;
my $NOK     = 2;
my $LESS    = 3;
my $GREATER = 4;

my $NHAS_VERSION  = "2.1.2";
my $SNDR_VERSION  = "3.1.31";
my $JSNMP_VERSION = "5.0";
my $J3RT_VERSION  = "1.3.1";
my $JDRT_VERSION  = "5.0";

# location
my $LOCATION_LOCAL  = 1;
my $LOCATION_SHARED = 2;

# feature
my $CGTP_FEATURE = 1;

# comparison
my $AT_LEAST_COMPARISON = 1;
my $EXACT_COMPARISON = 2;

# support
my $OBSOLETE=1;
my $DEPRECATED=2;
my $SUPPORTED=3;
my $READY=4;

my @G_packages_S8 = 
  (
   {
    name                => "SUNWnhtp8",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $TRUE,
    feature             => $CGTP_FEATURE,
   } ,

   {
    name                => "SUNWnhtu8",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $TRUE,
    feature             => $CGTP_FEATURE,
   } ,

);

my @G_packages_S9 = 
  (
   {
    name                => "SUNWnhtp9",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $TRUE,
    feature             => $CGTP_FEATURE,
   } ,

   {
    name                => "SUNWnhtu9",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $TRUE,
    feature             => $CGTP_FEATURE,
   } ,

);
my @G_packages_S10 = 
  (
   {
    name                => "SUNWnhtp0",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $TRUE,
    feature             => $CGTP_FEATURE,
   } ,

   {
    name                => "SUNWnhtu0",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $TRUE,
    feature             => $CGTP_FEATURE,
   } ,

);

my @G_packages_common = 
  (
   # ADM packages
   {
    name                => "SUNWnhadm",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $TRUE,
    feature             => $ADM_FEATURE,
   } ,

   # CMM packages
   {
    name                => "SUNWnhhb",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $TRUE,
    feature             => $CMM_FEATURE,
   } ,
   {
    name                => "SUNWnhcdt",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $TRUE,
    feature             => $CMM_FEATURE,
   } ,
   {
    name                => "SUNWnhcmd",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $TRUE,
    feature             => $CMM_FEATURE,
   } ,
   {
    name                => "SUNWnhcma",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $TRUE,
    feature             => $CMM_FEATURE,
   } ,
   {
    name                => "SUNWnhcmb",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $TRUE,
    feature             => $CMM_FEATURE,
   } ,

   # SNDR packages
   {
    name                => "SUNWscmr",
    location            => $LOCATION_LOCAL,
    version             => $SNDR_VERSION,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $FALSE,
    feature             => $SNDR_FEATURE,
   } ,
   {
    name                => "SUNWscmu",
    location            => $LOCATION_LOCAL,
    version             => $SNDR_VERSION,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $FALSE,
    feature             => $SNDR_FEATURE,
   } ,
   {
    name                => "SUNWspsvr",
    location            => $LOCATION_LOCAL,
    version             => $SNDR_VERSION,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $FALSE,
    feature             => $SNDR_FEATURE,
   } ,
   {
    name                => "SUNWspsvu",
    location            => $LOCATION_LOCAL,
    version             => $SNDR_VERSION,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $FALSE,
    feature             => $SNDR_FEATURE,
   } ,
   {
    name                => "SUNWrdcr",
    location            => $LOCATION_LOCAL,
    version             => $SNDR_VERSION,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $FALSE,
    feature             => $SNDR_FEATURE,
   } ,
   {
    name                => "SUNWrdcu",
    location            => $LOCATION_LOCAL,
    version             => $SNDR_VERSION,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $FALSE,
    feature             => $SNDR_FEATURE,
   } ,

   # RNFS packages
   {
    name                => "SUNWnhfsa",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $TRUE,
    feature             => $RNFS_FEATURE,
   } ,
   {
    name                => "SUNWnhfsb",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $TRUE,
    feature             => $RNFS_FEATURE,
   } ,

   # PMD packages
   {
    name                => "SUNWnhpma",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $TRUE,
    feature             => $PMD_FEATURE,
   } ,
   {
    name                => "SUNWnhpmb",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $TRUE,
    feature             => $PMD_FEATURE,
   } ,
   {
    name                => "SUNWnhpmm",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $TRUE,
    feature             => $PMD_FEATURE,
   } ,
   # SUNWnhpmn useful only if SNDR is installed (feature=SNDR)
   {
    name                => "SUNWnhpmn",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $FALSE,
    feature             => $SNDR_FEATURE,
   } ,
   {
    name                => "SUNWnhpms",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $TRUE,
    feature             => $PMD_FEATURE,
   } ,

   # NSM packages
   {
    name                => "SUNWnhnsa",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $FALSE,
    feature             => $NSM_FEATURE,
   } ,
   {
    name                => "SUNWnhnsb",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $FALSE,
    feature             => $NSM_FEATURE,
   } ,

   # EAM packages
   {
    name                => "SUNWnheaa",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $FALSE,
    feature             => $EAM_FEATURE,
   } ,
   {
    name                => "SUNWnheab",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $FALSE,
    feature             => $EAM_FEATURE,
   } ,

   # JDK packages
   {
    name                => "SUNWj3rt",
    location            => $LOCATION_LOCAL,
    version             => $J3RT_VERSION,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $TRUE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $FALSE,
    feature             => $JDK_FEATURE,
   } ,

   # NMA packages + JDMK packages
   {
    name                => "SUNWnhmas",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $FALSE,
    feature             => $NMA_FEATURE,
   } ,
   {
    name                => "SUNWnhmaj",
    location            => $LOCATION_SHARED,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $FALSE,
    feature             => $NMA_FEATURE,
   } ,
   {
    name                => "SUNWnhmal",
    location            => $LOCATION_SHARED,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $FALSE,
    feature             => $NMA_FEATURE,
   } ,
   {
    name                => "SUNWnhmad",
    location            => $LOCATION_SHARED,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $FALSE,
    feature             => $NMA_FEATURE,
   } ,

   {
    name                => "SUNWjsnmp",
    location            => $LOCATION_LOCAL,
    version             => $JSNMP_VERSION,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $FALSE,
    feature             => $NMA_FEATURE,
   } ,
   {
    name                => "SUNWjdrt",
    location            => $LOCATION_SHARED,
    version             => $JDRT_VERSION,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $FALSE,
    feature             => $NMA_FEATURE,
   } ,

   # RBS packages
   {
    name                => "SUNWnhrbb",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $FALSE,
    feature             => $RBS_FEATURE,
   } ,
   {
    name                => "SUNWnhrbs",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $FALSE,
    feature             => $RBS_FEATURE,
   } ,

   # WDT packages
   {
    name                => "SUNWnhwdt",
    location            => $LOCATION_LOCAL,
    version             => $NHAS_VERSION,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    comparison          => $AT_LEAST_COMPARISON,
    mandatory           => $FALSE,
    feature             => $WDT_FEATURE,
   } ,
);

# current table
my @G_packages;

my @G_patches_S8U7B8 =
  (
   # Solaris patches
   {
    name                => "108727-22",
    location            => $LOCATION_LOCAL,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    mandatory           => $TRUE,
    feature             => $NONE_FEATURE,
   } ,
   {
    name                => "111958-02",
    location            => $LOCATION_LOCAL,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $TRUE,
    mandatory           => $TRUE,
    feature             => $NONE_FEATURE,
   } ,

   # SNDR patch
   {
    name                => "116710-03",
    location            => $LOCATION_LOCAL,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    mandatory           => $TRUE,
    feature             => $SNDR_FEATURE,
   } ,  
 );

my @G_patches_S8PSR1 =
  (
   # Solaris patches
   {
    name                => "108727-22",
    location            => $LOCATION_LOCAL,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    mandatory           => $TRUE,
    feature             => $NONE_FEATURE,
   } ,

   # SNDR patch
   {
    name                => "116710-03",
    location            => $LOCATION_LOCAL,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    mandatory           => $TRUE,
    feature             => $SNDR_FEATURE,
   } ,  
  );

my @G_patches_S8PSR3 =
  (
   # SNDR patch
   {
    name                => "116710-03",
    location            => $LOCATION_LOCAL,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    mandatory           => $TRUE,
    feature             => $SNDR_FEATURE,
   } ,  
  ) ;

my @G_patches_S9U1 = 
  (
   # Solaris patches
   {
    name                => "113318-03",
    location            => $LOCATION_LOCAL,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    mandatory           => $TRUE,
    feature             => $NONE_FEATURE,
   } ,

   # CGTP patches
   {
    name                => "112233-03",
    location            => $LOCATION_LOCAL,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    mandatory           => $TRUE,
    feature             => $CGTP_FEATURE,
   } ,

   # SNDR patch
   {
    name                => "116710-03",
    location            => $LOCATION_LOCAL,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    mandatory           => $TRUE,
    feature             => $SNDR_FEATURE,
   } ,  
);

my @G_patches_S9HWPL3 = 
  (
   # Solaris patches
   {
    name                => "113318-03",
    location            => $LOCATION_LOCAL,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    mandatory           => $TRUE,
    feature             => $NONE_FEATURE,
   } ,

   # CGTP patches
   {
    name                => "112233-03",
    location            => $LOCATION_LOCAL,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    mandatory           => $TRUE,
    feature             => $CGTP_FEATURE,
   } ,
   {
    name                => "112902-06",
    location            => $LOCATION_LOCAL,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    mandatory           => $TRUE,
    feature             => $CGTP_FEATURE,
   } ,
   {
    name                => "112904-01",
    location            => $LOCATION_LOCAL,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    mandatory           => $TRUE,
    feature             => $CGTP_FEATURE,
   } ,
   {
    name                => "112917-01",
    location            => $LOCATION_LOCAL,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $TRUE,
    mandatory           => $TRUE,
    feature             => $CGTP_FEATURE,
   } ,
   {
    name                => "112918-01",
    location            => $LOCATION_LOCAL,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $TRUE,
    mandatory           => $TRUE,
    feature             => $CGTP_FEATURE,
   } ,
   {
    name                => "112919-01",
    location            => $LOCATION_LOCAL,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $TRUE,
    mandatory           => $TRUE,
    feature             => $CGTP_FEATURE,
   } ,

   # SNDR patch
   {
    name                => "116710-03",
    location            => $LOCATION_LOCAL,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    mandatory           => $TRUE,
    feature             => $SNDR_FEATURE,
   } ,  
 );

my @G_patches_S9_GT_U5 =
  (

   # SNDR patch
   {
    name                => "116710-03",
    location            => $LOCATION_LOCAL,
    men                 => $TRUE,
    diskless            => $FALSE,
    dataless            => $FALSE,
    mandatory           => $TRUE,
    feature             => $SNDR_FEATURE,
   } ,  
  ) ; 

my @G_patches_S10 =
  (
  ) ; 


my @G_patches_common =
  (
  );

#
# list of the supported releases
# 
# exact = $TRUE means that the string must match exactly
# otherwise wild character are authorized
#
my @G_supported_release = 
  (
   { 
    idstring    => "Solaris 8 2/02 s28s_u7wos_08a SPARC",
    exact       => $TRUE,
    support     => $SUPPORTED,
    pkg_list    => \@G_packages_S8,
    patch_list  => \@G_patches_S8U7B8,
    sndr40      => $FALSE,
   } ,
   { 
    idstring    => "Solaris 8 HW 12/02 s28s_hw1wos_06a SPARC",
    exact       => $TRUE,
    support     => $OBSOLETE,
    pkg_list    => \@G_packages_S8,
    patch_list  => \@G_patches_S8PSR1,
    sndr40      => $FALSE,
   } ,
   {
    idstring    => "Solaris 8 HW 7/03 s28s_hw3wos_05a SPARC",
    exact       => $TRUE,
    support     => $SUPPORTED,
    pkg_list    => \@G_packages_S8,
    patch_list  => \@G_patches_S8PSR3,
    sndr40      => $FALSE,
   } ,
   { 
    idstring    => "Solaris 9 s9_58shwpl3 SPARC",
    exact       => $TRUE,
    support     => $OBSOLETE,
    pkg_list    => \@G_packages_S9,
    patch_list  => \@G_patches_S9HWPL3,
    sndr40      => $FALSE,
   } ,
   { 
    idstring    => "Solaris 9 9/02 s9s_u1wos_08b SPARC",
    exact       => $TRUE,
    support     => $OBSOLETE,
    pkg_list    => \@G_packages_S9,
    patch_list  => \@G_patches_S9U1,
    sndr40      => $FALSE,
   } ,
   { 
    idstring    => "Solaris 9 4/04 s9s_u6wos_08a SPARC",
    exact       => $TRUE,
    support     => $DEPRECATED,
    pkg_list    => \@G_packages_S9,
    patch_list  => \@G_patches_S9_GT_U5,
    sndr40      => $FALSE,
   } ,
   { 
    idstring    => "Solaris 9 9/04 s9s_u7wos_09 SPARC",
    exact       => $TRUE,
    support     => $SUPPORTED,
    pkg_list    => \@G_packages_S9,
    patch_list  => \@G_patches_S9_GT_U5,
    sndr40      => $FALSE,
   } ,
   { 
    idstring    => "Solaris 9 9/05 s9s_u8wos_05 SPARC",
    exact       => $TRUE,
    support     => $SUPPORTED,
    pkg_list    => \@G_packages_S9,
    patch_list  => \@G_patches_S9_GT_U5,
    sndr40      => $FALSE,
   } ,
   { 
    idstring    => "Solaris 10",
    exact       => $FALSE,
    support     => $READY,
    pkg_list    => \@G_packages_S10,
    patch_list  => \@G_patches_S10,
    sndr40      => $TRUE,
   } ,
);


#
#  Patch list depending on the kernel patch
#
#  0: name of the patch
#  1: local (LOCAL), shared (SHARED)
#  2: on master-eligible (YES or NO)
#  3: on diskless (YES or NO)
#
# Note that, on Solaris 8, the CGTP patch is dependent on the kernel patch
# not on the Solaris version
#
my @G_patches_108528_13 = 
  (
   # CGTP patch
   {
    name                => "112281-03",
    location            => $LOCATION_LOCAL,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    mandatory           => $TRUE,
    feature             => $CGTP_FEATURE
   }
);


my @G_patches_108528_18 = 
  (
   # CGTP patch
   {
    name                => "115419-01",
    location            => $LOCATION_LOCAL,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    mandatory           => $TRUE,
    feature             => $CGTP_FEATURE
   }
);

my @G_patches_108528_21 = 
  (
   # CGTP patch
   {
    name                => "116036-03",
    location            => $LOCATION_LOCAL,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    mandatory           => $TRUE,
    feature             => $CGTP_FEATURE
   }
);

my @G_patches_108528_22 = 
  (
   # CGTP patch
   {
    name                => "116036-03",
    location            => $LOCATION_LOCAL,
    men                 => $TRUE,
    diskless            => $TRUE,
    dataless            => $TRUE,
    mandatory           => $TRUE,
    feature             => $CGTP_FEATURE
   }
);

#
# list of the supported kernel patch
#
my @G_supported_kernel_patch = 
  (
   { 
    idstring    => "Generic_108528-13",
    pkg_list    => \@G_empty_array,
    patch_list  => \@G_patches_108528_13,
   } ,
   { 
    idstring    => "Generic_108528-18",
    pkg_list    => \@G_empty_array,
    patch_list  => \@G_patches_108528_18,
   } ,
   { 
    idstring    => "Generic_108528-21",
    pkg_list    => \@G_empty_array,
    patch_list  => \@G_patches_108528_21,
   } ,
   { 
    idstring    => "Generic_108528-22",
    pkg_list    => \@G_empty_array,
    patch_list  => \@G_patches_108528_22,
   } ,
   { 
    # Solaris 9
    idstring    => "Generic",
    pkg_list    => \@G_empty_array,
    patch_list  => \@G_empty_array,
   } ,
);

# current table
my @G_patches;

#
# package acessibility
#
my $G_swdb_local_dir = "/SUNWcgha/local/export/services" ;
my $G_swdb_services = "/SUNWcgha/services" ;
my $G_swdb_mounting_point = "/SUNWcgha/swdb" ;
my $G_swdb_accessible = $FALSE;
my $G_swdb_remote_access = $FALSE; ;  
my $G_swdb_dir = "" ;
my $G_swdb_basedir = "" ;
my $G_swdb_path_confshare = "var/sadm/system/admin" ;
my $G_swdb_file_confshare = "INST_RELEASE" ;

#
# installed patch list
#
my %G_local_patch  = ();
my %G_shared_patch = ();
my $G_local_patch_loaded = $FALSE;
my $G_shared_patch_loaded = $FALSE;

#--------------------------------------------------------------
#
# set_os_env
#
#    determine the OS version and set all the related value
#
#--------------------------------------------------------------

sub set_os_env {

  LOG::printMsg(0,"-----------------------------------------") ;
  LOG::printMsg(0,"Retrieving OS information") ;
  LOG::printMsg(0,"-----------------------------------------") ;
  LOG::printMsg(0,"") ;


  my $release_file = "/etc/release";
  my $idx=-1;
  
  my $release = qx{/usr/bin/grep "Solaris" $release_file};
  $release =~ s/^[\s\t]*//;
  $release =~ s/[\s\t]*$//;
  
  # is it a supported release ?
  my $found = $FALSE;
  for my $i (0..$#G_supported_release) {
    if ($G_supported_release[$i]{exact}) {
      if ($G_supported_release[$i]{idstring} eq $release) {
	$idx = $i;
	$found = $TRUE;
      }
    } else {
      my $id_string =$G_supported_release[$i]{idstring};
      if ($release =~ m/$id_string/) {
	$idx = $i;
	$found = $TRUE;
      }
    }
  }

  my $support = $G_supported_release[$idx]{support};
  my $ok = $TRUE;

  $E_use_sndr40 = $G_supported_release[$idx]{sndr40};

  if ($support != $SUPPORTED) {

    $ok = $FALSE;
    if ($support == $OBSOLETE) {

      LOG::printError("Support of %s OBSOLETE: NOT FOR PRODUCTION", $release);
      $ok = $TRUE;

    } elsif ($support == $DEPRECATED) {

      LOG::printError("Support of %s deprecated (more recent release recommanded)", $release);

    } elsif ($support == $READY) {

      LOG::printError("Support of %s FOR EVALUATION ONLY: NOT FOR PRODUCTION", $release);

    }
  }
  if ($ok) {
    LOG::printResult("OK", "Solaris release checking") ;
  } else {
    LOG::printResult("FAILED", "Solaris release checking") ;
  }

  if ($found) {
    my $ref = $G_supported_release[$idx]{pkg_list};
    push @G_packages, @$ref;
    
    $ref = $G_supported_release[$idx]{patch_list};
    push @G_patches, @$ref;
    
  } else {

    LOG::printWarning("Unknown Solaris release (%s): only standard packages and patches will be checked", $release);
    
  }
  
  # retrieve the kernel patch release
  chomp(my $solaris_release = qx{/usr/bin/uname -r}) ;
  chomp(my $kernel_patch  = qx{/usr/bin/uname -v}) ;

  # is it a supported kernel patch release ?
  # only check on Solaris 8
  if ($solaris_release eq "5.8") {
    $found = $FALSE;
    for my $i (0..$#G_supported_kernel_patch) {
      if ($G_supported_kernel_patch[$i]{idstring} eq $kernel_patch) {
	$idx = $i;
	$found = $TRUE;
      }
    }
    
    if ($found) {
      my $ref = $G_supported_kernel_patch[$idx]{pkg_list};
      push @G_packages, @$ref;
      
      $ref = $G_supported_kernel_patch[$idx]{patch_list};
      push @G_patches, @$ref;
      
    } else {
      
      LOG::printWarning("Unsupported Solaris kernel patch (%s): only standard packages and patches will be checked", $kernel_patch);
      
    }
  }

  push @G_packages, @G_packages_common;
  push @G_patches, @G_patches_common;

}
#--------------------------------------------------------------
#
#  determine_swdb
#
#  shared package info database is
#  . only accessible from one node during installation
#    (when mounting points are not yet defined)
#  . accessible from all nodes after reboot (mounting point
#    has been defined on all nodes)
#
#--------------------------------------------------------------

sub determine_swdb {

  # is the mounting point defined ?
  my $result = qx{/usr/sbin/mount | /usr/bin/grep $G_swdb_mounting_point} ;
  
  # if the mounting point is not present, we cannot access the shared
  # database: several versions of shared packages possible then we 
  # cannot select one in particular
  
  if ($result ne "") {
    
    #
    # determine the base dir
    #
    $result = qx{/usr/sbin/mount | /usr/bin/grep $G_swdb_services};
    if ($result eq "") {
      LOG::printError("Cannot determine the current BASE_DIR for shared packages") ;
      return ;
    }
    # extract the remote sub-directory name as the basedir
    if ($result !~ m{$G_swdb_local_dir/([^\s\t]*)}) {
      LOG::printError("Cannot determine the current BASE_DIR for shared packages") ;
      return ;
    }
    $G_swdb_basedir = "/$1";
    $G_swdb_accessible = $TRUE;
    $G_swdb_remote_access = $TRUE;
    $G_swdb_dir = $G_swdb_mounting_point ;
  }
  
}

#--------------------------------------------------------------
#
#  guess_node_role
#
#    try to guess the node role by testing the mounting point
#    of the root partition since CMM can't be used
#    and nhfs.conf may be not already configured
#
#--------------------------------------------------------------

sub guess_node_role
  {
    my $result;
    chomp($result = qx{mount -v | /usr/bin/grep " on / type "}) ;
    $result =~ m/ on \/ type ([a-z]*) /;
    if ($1 eq "ufs")
      {
	my $version ;
	my $name ;
	my $basedir ;
	
	my $ok = get_package("SUNWnhfsa", $LOCATION_LOCAL, 
				 \$version, \$name, \$basedir) ;
	if ($ok) {
	  return "DISKFULL" ;
	} else {
	  return "DATALESS"
	}
      }

    return "DISKLESS" ;
  }

#--------------------------------------------------------------
#
#  compare_version
#
#--------------------------------------------------------------

sub compare_version {

  my ($actual, $expected, $option) = @_ ;

  my $version;
  
  if ($option == $EXACT_COMPARISON) {
    foreach $version (split(",",$expected)) {
      if ($actual eq $version) {
	return $OK;
	}
    }
    return $NOK ;

  } else {
    
    if ($option == $AT_LEAST_COMPARISON) {
      
      my @part_current = split('\.', $actual) ;
      my @part_expected = split('\.', $expected) ;
      my $atleast = $TRUE ;
      my $end = $FALSE ;
      my $exp_size = scalar(@part_expected) ;
      my $cur_size = scalar(@part_current) ;
      my $i = 0 ;
      while (($i < $exp_size) and (! $end)) {
	if ($i < $cur_size) {
	  if ($part_current[$i] > $part_expected[$i]) {
	    $end = $TRUE ;
	  }
	  else {
	    if ($part_current[$i] < $part_expected[$i]) {
	      $atleast = $FALSE ;
	      $end = $TRUE ;
	    }
	  }
	}
	else {
	  $atleast = $FALSE ;
	  $end = $TRUE ;
	}
	$i++ ;
      }
      
      if (! $atleast) {
	return $LESS;
      }
      return $OK;
    }
  }
}

#--------------------------------------------------------------
#
#  get_package
#
#    get info about a specified package
#
#    1 : in: name of the package
#    2 : in: location
#
#    3 : out: version of the package
#    4 : out: complete name
#    5 : out: base directory
#
#    return :
#    . OK if the package exists
#    . NA if the package is not part of this node (Not Applicable)
#    . NOK if the package doesn't exist
#
#--------------------------------------------------------------

sub get_package {
  
  my ($L_package, $L_location, $R_version, $R_name, $R_basedir) = @_ ;
  
  my $command = "" ;
  my $dir = "" ;
  
  # basedir may be not present in package information so it should be initialized
  $$R_basedir = "" ; 
  
  if ($L_location == $LOCATION_SHARED) {
    $dir = "$G_swdb_dir" ;
    $command = "-R $G_swdb_dir" ;
  } elsif ($L_location != $LOCATION_LOCAL) {
    LOG::printError("Invalid location (%s) for package %s", $L_location, $L_package) ;
    exit 1;
  }
  
  my $L_result = system("/usr/bin/pkginfo $command $L_package > /dev/null 2>&1");
  
  if ($L_result != 0) {
    return $FALSE ;
  }
  
  my @L_output = qx{/usr/bin/pkginfo $command -l $L_package} ;
  
  foreach my $L_line (@L_output) {
    chomp($L_line) ;
    if ($L_line =~ m/ERROR:/) {
      return $FALSE ;
    }
    if ($L_line =~ m/VERSION:[\s\t]*([0-9.]*)/) {
      $$R_version = $1 ;
    }
    if ($L_line =~ m/NAME:[\s\t]*(.*)/) {
      $$R_name = $1 ;
    }
    if ($L_line =~ m/BASEDIR:[\s\t]*(.*)/) {
      $$R_basedir = $1 ;
    }
  }
  
  return $TRUE ;  
}

#--------------------------------------------------------------
#
#  is_feature_installed
#
#    check if at least one package of a feature is installed
#    return TRUE or FALSE
#
#--------------------------------------------------------------

sub is_feature_installed {

  my ($feature) = @_;

  my $name;
  my $location;
  my $version;
  my $basedir;

  foreach my $package (@G_packages) {
    if (@$package{feature} eq $feature) {
      $name = @$package{name},
      $location = @$package{location};
      if (get_package($name, $location, \$version, \$name, \$basedir)) {
	return $TRUE;
      }
    }
  }

  return $FALSE;

}

#--------------------------------------------------------------
#
#  check_package
#
#    check if a package is installed and if its version
#    is correct
#
#    1 : name of the package
#    2 : location
#    3 : version to be tested (can be a list if type=EXACT
#    4 : type of comparison
#        EXACT   : must match the exact version
#        AT_LEAST: minimal version
#
#--------------------------------------------------------------

sub check_package {
  
  my ($L_package, $L_location, $L_version, $L_comp, $L_mandatory) = @_ ;
  
  my $L_status = "OK" ;
  my $L_current_version ;
  my $L_current_name ;
    my $L_basedir ;
  
  my $ok = get_package($L_package, $L_location, 
			   \$L_current_version, \$L_current_name, \$L_basedir) ;
  
  if (! $ok) {
    if ($L_mandatory) {
      

#=item package <name> not installed.
#
#    Please install the specified package.
#
#=cut

      LOG::printError("package %s not installed", $L_package) ;
      my $status = "FAILED";
      LOG::printResult($status, "Package %s", $L_package) ;
    } else {
      LOG::printWarning("optional package %s not installed", $L_package) ;
    }
    return ;
  }	
  
  my $status = compare_version($L_current_version, $L_version, $L_comp) ;
  
  if ($status == $NOK) {
    
#=item package <name>: version <current> instead of <expected>
#
#    Please install the correct version of the specified package.
#
#=cut

    LOG::printError("package %s: version %s instead of %s",
		    $L_package, $L_current_version, $L_version) ;
    $L_status = "FAILED" ;
    
  } else {
    
    if ($status == $LESS) {
      
#=item package <name>: version <current> (at least <minimum> required)
#
#    Please install the correct version of the specified package.
#
#=cut

      LOG::printError("package %s: version %s (at least %s required)", 
		      $L_package, $L_current_version, $L_version) ;
      $L_status = "FAILED" ;
    }
  }
  
  LOG::printResult($L_status, "Package %s", $L_package) ;
}


#--------------------------------------------------------------
#
#  check_package_array
#
#    check if a array of package is installed
#
#    1: package array
#    2: MEN or NMEN
#
#--------------------------------------------------------------

sub check_package_array {
  
  my ($pkgarr, $role) = @_ ;
  
  my $field ;
  
  if ($role eq "MEN") {
    $field = "men" ;
  } elsif ($role eq "DISKLESS") {
    $field = "diskless" ;
  } else {
    $field = "dataless" ;
  }
  
  #
  # check local packages if it makes sense
  #
  
  LOG::printMsg(0,"") ;
  LOG::printMsg(0,"Local packages") ;
  LOG::printMsg(0,"") ;
  foreach my $pkg (@$pkgarr) {
    if ((@$pkg{$field}) && (@$pkg{location} == $LOCATION_LOCAL))
      {
	check_package(@$pkg{name}, 
		      @$pkg{location}, 
		      @$pkg{version},
		      @$pkg{comparison},
		      @$pkg{mandatory});
      }
  }

  if (! $G_swdb_accessible) {
    LOG::printMsg(0,"") ;
    LOG::printMsg(0,"Shared packages currently not accessible") ;
    LOG::printMsg(0,"") ;
    return ;
  }
  
  #
  # check shared packages
  #
  LOG::printMsg(0,"") ;
  LOG::printMsg(0,"Shared packages (on %s)", $G_swdb_dir) ;
  LOG::printMsg(0,"") ;
  foreach my $pkg (@$pkgarr) {
    if (@$pkg{location} == $LOCATION_SHARED) {
      check_package(@$pkg{name}, 
		    @$pkg{location}, 
		    @$pkg{version},
		    @$pkg{comparison},
		    @$pkg{mandatory});
    }
  }
}

#--------------------------------------------------------------
#
#  load_patch
#
#    load patch list into an array
#
#    1 : array name
#    2 : location
#
#--------------------------------------------------------------

sub load_patch {

  my ($arrayref, $location) = @_;

  my $optcmd = "";

  if ($location == $LOCATION_SHARED) {
    $$arrayref = \%G_shared_patch;
    if ($G_shared_patch_loaded) {
      return ;
    } else {
      $optcmd = "-R $G_swdb_dir" ;
      $G_shared_patch_loaded = $TRUE;
    }
  } elsif ($location == $LOCATION_LOCAL) {
    $$arrayref = \%G_local_patch;
    if ($G_local_patch_loaded) {
      return ;
    } else {
      $G_local_patch_loaded = $TRUE;
    }
  } else {
    LOG::printMsg("Invalid location (%s)", $location) ;
    exit 1;
  }

  my @L_result = qx{/usr/sbin/patchadd $optcmd -p | /usr/bin/cut -f2 -d' '} ;
  foreach my $line (@L_result) {
    if (chomp($line) ne "") {
      if ($line =~ m/([0-9]*)-([0-9]*)/) {
	my $patchid = $1;
	my $release = $2;
	if (defined($$arrayref->{$patchid})) {
	  # keep the higher release number
	  if ($release > $$arrayref->{$patchid}) {
	    $$arrayref->{$patchid} = $release;
	  }
	} else {
	  # new patch id: store the current release
	  $$arrayref->{$patchid} = $release;
	}
      }
    }
  }
}

#--------------------------------------------------------------
#
#  get_patch
#
#    get info about a specified patch
#
#    1 : in: name of the package
#    2 : in: location
#
#    return TRUE if the patch exists
#
#--------------------------------------------------------------

sub get_patch {

  my ($patch, $location, $refcompare, $refpatch) = @_ ;
  
  my $arrayref;
  my $optcmd = "" ;
  
  load_patch(\$arrayref, $location);

  $patch =~ m/([0-9]*)-([0-9]*)/ ;
  my $patchid = $1;
  my $release = $2;
  
  my $i = 0;
  if (defined($arrayref->{$patchid})) {
    my $current_release = $arrayref->{$patchid};
    if ($current_release == $release) {
      $$refcompare = $OK;
    } elsif ($current_release < $release) {
      $$refcompare = $LESS;
    } else {
      $$refcompare = $GREATER;
    }
    $$refpatch = "$patchid-$current_release";
    return $TRUE;
  }
  
  return $FALSE ;
}

#--------------------------------------------------------------
#
#  check_patch
#
#    check if a patch is installed and if its version
#    is correct
#
#    1 : name of the patch
#    2 : location
#    3 : to check or not
#    4 : feature to be present
#
#--------------------------------------------------------------

sub check_patch {
  
  my ($L_patch, $L_location, $L_tocheck, $L_feature) = @_ ;
  
  if (! $L_tocheck) {
    return ;
  }
  
  my $L_status = "OK" ;
  
  if (($L_location == $LOCATION_SHARED) && (! $G_swdb_accessible)) {
    return ;
  }
  
  # check if the specified feature is installed
  if ($L_feature != $NONE_FEATURE) {
    if (! is_feature_installed($L_feature)) {
      # do not check
      return ;
    }
  }

  my $compare;
  my $patch;
  if (! get_patch($L_patch, $L_location, \$compare, \$patch)) {

#=item Patch <patch> not installed
#
#    The specified patch must be installed.
#
#=cut

    LOG::printError("Patch %s not installed", $L_patch) ;
    $L_status = "FAILED";
  } else {
    if ($compare == $LESS) {
      LOG::printError("Patch %s missing: %s installed", $L_patch, $patch);
    $L_status = "FAILED";
    } elsif ($compare == $GREATER) {
      LOG::printWarning("Patch %s: more recent version (%s) installed", $L_patch, $patch);
    }
  }
  
  LOG::printResult($L_status, "Patch %s", $L_patch) ;
}

#--------------------------------------------------------------
#
#  check_file
#
#    1 : name of the file to check (exists, readable)
#
#--------------------------------------------------------------

sub check_file {
  
  my ($filename) = @_ ;
  
  if (! -e $filename) {
    
#=item <file> does not exist
#
#    This may be caused by a wrong pathname.
#
#=cut

    LOG::printError("%s does not exist", $filename) ;
    return $FALSE ;
  }
  
  if (! -r $filename) {

#=item cannot read <file>: permission denied
#
#    Please check that the root account can access the specified file.
#
#=cut

    LOG::printError("can't read %s: permission denied", $filename) ;
    return $FALSE ;
  }
  
  return $TRUE ;
}

#--------------------------------------------------------------
#
#  check_service
#
#    check if a service is defined in /etc/services
#
#    1 : name of the service
#    2 : protocol (tcp/udp)
#
#--------------------------------------------------------------

sub check_service {

  my ($L_service,$L_protocol) = @_ ;
  
  my $L_status = "OK" ;
  my $protocol;
  my $wrong_protocol;
  my $save_protocol;
  my $found;
  
  LOG::printMsg(2,"Check service %s, protocol %s",
		$L_service, $L_protocol) ;
  
  #
  # crfs service must be defined in /etc/services
  # a service can appear twice with different protocol
  #
  $L_status = "OK" ;
  if (check_file("/etc/services")) {
    open(FH,"</etc/services") or die "/etc/services: $!";
    $found = $FALSE ;
    $wrong_protocol = $FALSE ;
    while (<FH>) {
      my $line;
      chomp($line = $_) ;
      $line =~ s/#.*// ;
      if ($line =~ m/[\s\t]*([^\s\t]*)[\s\t]*([0-9]*)\/([a-z]*)/) {
	my $service = $1 ;
	my $port = $2 ;
	my $protocol = $3 ;
	if ($service eq $L_service) {
	  if ($protocol eq $L_protocol) {
	    $found = $TRUE ;
	    $wrong_protocol = $FALSE ;
	  } else {
	    $wrong_protocol = $TRUE ;
	    $save_protocol = $protocol;
	  }
	}
      }
    }
    
    if ($found) {
      $L_status = "OK" ;
    } else {
      if ($wrong_protocol) {

#=item <service> service: <current> instead of <expected>
#
#    Please correct the protocol for the specified service
#    in /etc/services.
#
#=cut

	LOG::printError ("%s service: %s instead of %s", $L_service, $save_protocol, $L_protocol);
	$L_status = "FAILED" ;
      } else {

#=item <service> service must be defined in /etc/services
#
#    Please add an entry for the specified service in /etc/services.
#    Otherwise, daemons using an unspecified service might not work
#    correctly.
#
#=cut
 
	LOG::printError("%s service must be defined in /etc/services", $L_service) ;
	$L_status = "FAILED" ;
      }
    }
  }
  
  close (FH) ;
  
  LOG::printResult($L_status, "%s defined in /etc/services",
		   $L_service) ;
}

#--------------------------------------------------------------
#
#  check_installation
#
#      check information related to the Solaris OS
#
#--------------------------------------------------------------

sub check_installation {

  my $L_status = "OK" ;
  
  my $i;
  
  LOG::printMsg(0, "") ;
  LOG::printMsg(0, "OS and Software checking") ;
  LOG::printMsg(0, "") ;
  
  my $bits = qx{/usr/bin/isainfo -b} ;
  if ($bits != "64") {

#=item Invalid kernel mode: <bits>-bit instead of 64-bit mode
#
#    Please install the correct Solaris version.
#
#=cut

    LOG::printError("Invalid kernel mode: %s-bit instead of 64-bit mode",
		    $bits) ;
    $L_status = "FAILED";
  }
  LOG::printResult($L_status,"64-bit kernel mode" ) ;
  
  #
  #  check various packages
  #
  
  determine_swdb ;
  
  if (guess_node_role eq "DISKFULL") {
    check_package_array(\@G_packages, "MEN") ;
  } elsif (guess_node_role eq "DISKLESS") {
    check_package_array(\@G_packages, "DISKLESS") ;
  } else {
    check_package_array(\@G_packages, "DATALESS") ;
  }
  
  #
  #  check various patches
  #
  LOG::printMsg(0, "") ;
  LOG::printMsg(0, "Patches") ;
  LOG::printMsg(0, "") ;
  LOG::printMsg(0, "Can take a long time ...") ;
  LOG::printMsg(0, "") ;
  for ($i = 0 ; $i < scalar(@G_patches) ; $i++) {
    if (guess_node_role eq "DISKFULL") {
      check_patch($G_patches[$i]{name}, 
		  $G_patches[$i]{location},
		  $G_patches[$i]{men},
		  $G_patches[$i]{feature}) ;
    } elsif (guess_node_role eq "DISKLESS") {
      check_patch($G_patches[$i]{name}, 
		  $G_patches[$i]{location},
		  $G_patches[$i]{diskless},
		  $G_patches[$i]{feature}) ;
    } else {
      check_patch($G_patches[$i]{name}, 
		  $G_patches[$i]{location},
		  $G_patches[$i]{dataless},
		  $G_patches[$i]{feature}) ;
    }
  }
  
  #
  # check if shared file for patchabitlity support is present 
  # 
  if ($G_swdb_accessible) {
    $L_status = "FAILED" ;
    if (-f "$G_swdb_dir/$G_swdb_path_confshare/$G_swdb_file_confshare") {
      $L_status = "OK" ;
    } else {
      
#=File for patchability of shared packages not present.
#
#    Please use the "nhadm confshare" command to create this file.
#
#=cut

      LOG::printError("File for patchability of shared packages not present.") ;
    }
    LOG::printResult($L_status, "INST_RELEASE file for shared packages") ;
  }
}

#--------------------------------------------------------------
#
#  display_package_array
# 
#      display information regarding package array
#
#--------------------------------------------------------------

sub display_package_array {
  
  my ($pkgarr, $role) = @_ ;
  
  my $field ;
  my $L_name;
  my $L_location ;
  my $L_version ;
  my $L_basedir ;
  
  if ($role eq "MEN") {
    $field = "men" ;
  } elsif ($role eq "DISKLESS") {
    $field = "diskless" ;
  } else {
    $field = "dataless" ;
  }
  
  determine_swdb ;
  
  #
  # display local packages
  #
  LOG::printMsg(0,"") ;
  LOG::printMsg(0,"Local packages") ;
  LOG::printMsg(0,"") ;
  foreach my $pkg (@$pkgarr) {
    if ((@$pkg{$field}) && (@$pkg{location} == $LOCATION_LOCAL)) {
      my $L_package = @$pkg{name};
      my $L_location = @$pkg{location};
      my $ok = get_package($L_package, $L_location, 
			   \$L_version, \$L_name, \$L_basedir) ;
      if ($ok) {
	printLine("    package %s", $L_package) ;
	printLine("        name:        %s", $L_name) ;
	printLine("        version:     %s", $L_version) ;
	if ($L_basedir ne "") {
	  printLine("        basedir:     %s", $L_basedir) ;
	}
      } else {
	printLine("    package %s -> not installed", $L_package) ;
      }
    }
  }

  if (! $G_swdb_accessible) {
    LOG::printMsg(0,"") ;
    LOG::printMsg(0,"Shared packages currently not accessible") ;
    LOG::printMsg(0,"") ;
    return ;
  }
  
  #
  # display shared packages
  #
  LOG::printMsg(0,"") ;
  LOG::printMsg(0,"Shared packages (on %s)", $G_swdb_dir) ;
  LOG::printMsg(0,"") ;
  foreach my $pkg (@$pkgarr) {
    if (@$pkg{location} == $LOCATION_SHARED) {
      my $L_package = @$pkg{name};
      my $L_location = @$pkg{location};
      my $ok = get_package($L_package, $L_location, 
			   \$L_version, \$L_name, \$L_basedir) ;
      if ($ok) {
	printLine("    package %s", $L_package) ;
	printLine("        name:        %s", $L_name) ;
	printLine("        version:     %s", $L_version) ;
	if ($L_basedir ne "") {
	  printLine("        basedir:     %s", $L_basedir) ;
	}
      } else {
	printLine("    package %s -> not installed", $L_package) ;
      }
    }
  }
}

#--------------------------------------------------------------
#
#  display
#
#      display information regarding OS
#
#--------------------------------------------------------------

sub display {

  printLine("") ;
  printLine("System information") ;
  printLine("") ;
  
  chomp(my $hardware  = qx{/usr/bin/uname -i}) ;
  my $release = qx{/usr/bin/grep "Solaris" /etc/release};
  $release =~ s/^[\s\t]*//;
  $release =~ s/[\s\t]*$//;
  chomp($release);
  
  printLine("    node name:  %s", $E_node_name) ;
  printLine("    Hardware:   %s", $hardware) ;
  printLine("    OS release: %s", $release) ;
  
  if (guess_node_role eq "DISKFULL") {
    display_package_array(\@G_packages, "MEN") ;
  } elsif (guess_node_role eq "DISKLESS") {
    display_package_array(\@G_packages, "DISKLESS") ;
  } else {
    display_package_array(\@G_packages, "DATALESS") ;
  }
}

#--------------------------------------------------------------
#
#  configure_shared
#
#--------------------------------------------------------------

sub configure_shared {
  
  my ($dir) = @_ ;
  
  if ($dir eq "") {
    #
    # the default directory is used if accessible
    #
    determine_swdb ;
    if (! $G_swdb_accessible) {
      LOG::printError("Directory for shared packages not accessible") ;
      return ;
    }
    $dir = $G_swdb_dir ;
  }
  
  my $result = qx{/usr/bin/mkdir -p $dir/$G_swdb_path_confshare 2>&1} ;
  if ( $? ne 0 ) {
    LOG::printMsg(0,"Error when executing mkdir") ;
    LOG::printMsg(0,$result) ;
    exit 1 ;
  }
  
  my $filename = "$dir/$G_swdb_path_confshare/$G_swdb_file_confshare";
  
  $result = qx{/usr/bin/cp /$G_swdb_path_confshare/$G_swdb_file_confshare $filename 2>&1} ;
  if ( $? ne 0 ) {
    LOG::printMsg(0,"Error when executing copy") ;
    LOG::printMsg(0,$result) ;
    exit 1 ;
  }
  
  LOG::printMsg(0, "File %s created", $filename) ;
}

#--------------------------------------------------------------
#
#  main
#
#--------------------------------------------------------------

#=back
#
#=cut

{
}
