#************************************************************ 
#
# Component       CNF  
#
# Synopsis        Network management
#
#
# Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
# Use is subject to license terms.
#
#
#************************************************************
#
# #ident	"@(#)NETWORK.pm	1.3	06/01/23 SMI"
#
#************************************************************

#=head1 Network Management
#
#=over 4
#
#=cut

package NETWORK ;

use strict;
use VAR;
use LOG;

my @interface_list = ();
my @netstat_list = ();

my $loaded = $FALSE;

#==============================================================================
#
#  NETWORK INTERFACE MANAGEMENT
#
#==============================================================================

#------------------------------------------------------------------------------
#
# load_interface_list
#
#------------------------------------------------------------------------------

sub load_interface_list {
  
  #
  # get the complete list of network devices with their characteristics
  #
  
  LOG::printMsg(2, "    execution '/sbin/ifconfig -a") ;
  my @result = qx'/sbin/ifconfig -a';

  my $interface_desc;
  my $already_one = $FALSE;
  
  foreach my $line (@result) {
    chomp($line) ;
    LOG::printMsg(2,"    analyzing %s", $line) ;

    #
    # check if the line contains an ethernet device reference
    # if true, store the previous interface description
    # and re-initialize a new one
    #
    if (($line =~ m/^([a-z0-9]*): /o) ||
	($line =~ m/^([a-z0-9]*:[0-9]+): /o)) {

      # this is the beginning of an interface description
      if ($already_one) {
	push @interface_list, $interface_desc;
      }
      $interface_desc = {} ;
      $interface_desc->{name} = $1;

      $interface_desc->{physical} = $TRUE;
      if ($line =~ m/^[a-z0-9]*:[0-9]+: /o) {
	$interface_desc->{physical} = $FALSE;
      }
      $already_one = $TRUE;
    }

    # analyze each list and store information
      
    
    if ($line =~ /flags=[0-9a-z]*<([^>]*)>/o) {
      $interface_desc->{flags} = $1 ;
      # interpret some flags
      $interface_desc->{up} = $FALSE;
      $interface_desc->{running} = $FALSE;
      $interface_desc->{failover} = $TRUE;
      $interface_desc->{deprecated} = $FALSE;
      foreach my $flag (split(',', $interface_desc->{flags})) {
	if ($flag eq "UP") {
	  $interface_desc->{up} = $TRUE;
	}
 	if ($flag eq "RUNNING") {
	  $interface_desc->{running} = $TRUE;
	}
 	if ($flag eq "NOFAILOVER") {
	  $interface_desc->{failover} = $FALSE;
	}
 	if ($flag eq "DEPRECATED") {
	  $interface_desc->{deprecated} = $TRUE;
	}
      }
    }

    if ($line =~ /mtu ([0-9]*)/o) {
      $interface_desc->{mtu} = $1;
    }
    if ($line =~ /index ([0-9]*)/o) {
      $interface_desc->{index} = $1;
    }
    if ($line =~ /inet ([0-9]*\.[0-9]*\.[0-9]*\.[0-9]*)/o) {
      $interface_desc->{ipversion} = $IP_VERSION_V4;
      $interface_desc->{IP} = $1;
    }
    if ($line =~ /inet6 ([0-9a-z:\/]*)/o) {
      $interface_desc->{ipversion} = $IP_VERSION_V6;
      $interface_desc->{IP} = $1;
    }
    if ($line =~ /netmask ([0-9a-f]*)/o) {
      $interface_desc->{netmask} = $1;
    } 
    if ($line =~ /broadcast ([0-9]*\.[0-9]*\.[0-9]*\.[0-9]*)/o) {
      $interface_desc->{broadcast} = $1;
    }
    if ($line =~ /ether ([0-9a-f:]*)/o) {
      $interface_desc->{ether} = $1;
    }

  }

  if ($already_one) {
    push @interface_list, $interface_desc;
  }
}

#------------------------------------------------------------------------------
#
#  exist_interface
#
#------------------------------------------------------------------------------

sub exist_interface {

  my ($interface, $ip_version) = @_ ;

  if (! defined($ip_version)) {
    $ip_version = $IP_VERSION_V4;
  }

  foreach my $interface_desc (@interface_list) {
    if (($interface_desc->{name} eq $interface)
	&& ($interface_desc->{ipversion} == $ip_version)) {
      return $TRUE;
    }
  }

  return $FALSE;

}

#------------------------------------------------------------------------------
#
#  convert_IP_address
#
#  remove trailing zeros
#
#------------------------------------------------------------------------------

sub convert_IP_address {

  my ($ip, $ipversion) = @_;


  # remove possible mask: xxxx/yy (keep only xxxx)
  $ip =~ m{([^/]*)};
  my $ipinput = $1;

  my $sep = ".";
  if ($ipversion == $IP_VERSION_V6) {
    $sep = ":";
  }

  my $ipconverted = "";
  my $first = $TRUE;
  foreach my $elem (split("\\$sep",$ipinput)) {
    # remove trailing zero
    $elem =~ m/[0]*(.*)/;
    if (! $first) {
      $ipconverted .= "$sep";
    }
    $first = $FALSE;
    $ipconverted .= "$1";
  }

  return $ipconverted;
}


#------------------------------------------------------------------------------
#
#  find_IP_interface
#
#------------------------------------------------------------------------------

sub find_IP_interface {

  my ($ip, $ip_version, $ref_interface) = @_ ;

  if (! defined($ip_version)) {
    $ip_version = $IP_VERSION_V4;
  }

  my $ipconverted = convert_IP_address($ip, $ip_version);

  foreach my $interface_desc (@interface_list) {
    if ($interface_desc->{ipversion} == $ip_version) {
      if ($ipconverted eq convert_IP_address($interface_desc->{IP}, 
					     $ip_version)) {
	$$ref_interface = $interface_desc->{name};
	return $TRUE;
      }
    }
  }

  return $FALSE;
}

#------------------------------------------------------------------------------
#
#  get_interface_info
#
#------------------------------------------------------------------------------

sub get_interface_info {

  my ($interface, $ip_version, $info) = @_ ;

  if (! defined($ip_version)) {
    $ip_version = $IP_VERSION_V4;
  }

  foreach my $interface_desc (@interface_list) {
    if (($interface_desc->{name} eq $interface)
	&& ($interface_desc->{ipversion} == $ip_version)) {
      return $interface_desc->{$info};
    }
  }

  return "";

}

#==============================================================================
#
#  ROUTE INTERFACE MANAGEMENT
#
#==============================================================================

#------------------------------------------------------------------------------
#
# load_netstat_list
#
#------------------------------------------------------------------------------

sub load_netstat_list {

  my @result = qx{netstat -rn} ;


  # currently process only IPv4 routes

  foreach my $line (@result) {
    chomp($line) ;
    if ($line =~ m/([0-9\.]+)[\s\t]+([0-9\.]+)[\s\t]+([A-Z]*)[\s\t]+([0-9]+)[\s\t]+([0-9]+)[\s\t]+([0-9a-z:]*)/) {
      LOG::printMsg(2,"destination=%s gateway=%s flags=%s ref=%s use=%s interface=%s", $1, $2, $3, $4, $5, $6) ;

      my $netstat_desc = {};
      $netstat_desc->{destination} = $1;
      $netstat_desc->{gateway} = $2;
      $netstat_desc->{flags} = $3;
      $netstat_desc->{ref} = $4;
      $netstat_desc->{use} = $5;
      $netstat_desc->{interface} = $6;
      
      push @netstat_list, $netstat_desc;
    }
  }
}

#------------------------------------------------------------------------------
#
#  exist_route
#
#------------------------------------------------------------------------------

sub exist_route {

  my ($destination, $gateway) = @_;

  foreach my $netstat_desc (@netstat_list) {
    if (($netstat_desc->{destination} eq $destination) 
	&& ($netstat_desc->{gateway} eq $gateway)) {
      return $TRUE ;
    }
  }

  return $FALSE;
}

#------------------------------------------------------------------------------
#
# load
#
#------------------------------------------------------------------------------
 
sub load {

  if ($loaded) {
    return ;
  }

  $loaded = $TRUE;
  load_interface_list();
  load_netstat_list();
}

#------------------------------------------------------------------------------
#
#  main
#
#------------------------------------------------------------------------------

#=back
#
#=cut

{
}
