#************************************************************ 
#
# Component       CNF  
#
# Synopsis        Cluster Membership Monitor
#
# Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
# Use is subject to license terms.
#
#
#************************************************************
#
# #ident	"@(#)CONF.pm	1.32	06/06/22 SMI"
#
#************************************************************

#=head1 Foundation Services Configuration File
#
#=over 4
#
#=cut

package CONF ;

use strict;
use VAR;
use LOG;

my $G_prop_error = $FALSE;
my $G_fatal      = $FALSE;

# control type
my $CONTROL_STRING=1;
my $CONTROL_BOOLEAN=2;

# Domain names
my $DOMAIN_CLUSTER = "Cluster";
my $DOMAIN_NODE    = "Node";
my $DOMAIN_CMM     = "Cmm";
my $DOMAIN_RNFS    = "Rnfs";
my $DOMAIN_PMD     = "Pmd";
my $DOMAIN_PROBE   = "Probe";
my $DOMAIN_EAM     = "Eam";
my $DOMAIN_NSM     = "Nsm";
my $DOMAIN_WDT     = "Watchdog";
my $DOMAIN_SCHED   = "Scheduling";

# Scope
my $SCOPE_PUBLIC  = 1;
my $SCOPE_PRIVATE = 2;

my @G_properties = 
  (

   # Node.Type (must be the fist in the list)
   {
    name               => "type",
    domain             => ($DOMAIN_NODE),
    men_mandatory      => $TRUE,
    diskless_mandatory => $TRUE,
    dataless_mandatory => $TRUE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "NODE_TYPE",
   } ,

   # Node.DomainId
   {
    name               => "DomainId",
    domain             => ($DOMAIN_NODE),
    men_mandatory      => $TRUE,
    diskless_mandatory => $TRUE,
    dataless_mandatory => $TRUE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "DOMAIN_ID",
   } ,

   # Node.Nic0
   {
    name               => "nic0",
    domain             => ($DOMAIN_NODE),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => "hme0",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "NIC0",
   } ,

   # Node.Nic1
   {
    name               => "nic1",
    domain             => ($DOMAIN_NODE),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $CGTP_FEATURE,
    multiple           => $FALSE,
    default            => "hme1",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "NIC1",
   } ,

   # Node.NicCGTP
   {
    name               => "NicCGTP",
    domain             => ($DOMAIN_NODE),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $CGTP_FEATURE,
    multiple           => $FALSE,
    default            => "cgtp0",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "NIC_CGTP",
   } ,

   # Node.UseCGTP
   {
    name               => "UseCGTP",
    domain             => ($DOMAIN_NODE),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => "TRUE",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PUBLIC,
    field              => "USE_CGTP",
   } ,

   # Node.RNFS.Installed
   {
    name               => "RNFS.Installed",
    domain             => ($DOMAIN_NODE),
    men_mandatory      => $TRUE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PUBLIC,
    field              => "RNFS_INSTALLED",
   } ,

   # Node.NodeID
   {
    name               => "NodeID",
    domain             => ($DOMAIN_NODE),
    men_mandatory      => $TRUE,
    diskless_mandatory => $TRUE,
    dataless_mandatory => $TRUE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "NODE_ID",
   } ,

   # Node.NodesGroup
   {
    name               => "NodesGroup",
    domain             => ($DOMAIN_NODE),
    men_mandatory      => $TRUE,
    diskless_mandatory => $TRUE,
    dataless_mandatory => $TRUE,
    component          => $SMCT_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "SMCT_NODE_GROUP",
   } ,

   # Node.Direct-Link.Serial.Device
   {
    name               => "Direct-Link.Serial.Device",
    domain             => ($DOMAIN_NODE),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "DIRECT_LINK_DEVICE",
   } ,

   # Node.Direct-Link.Serial.Speed
   {
    name               => "Direct-Link.Serial.Speed",
    domain             => ($DOMAIN_NODE),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "DIRECT_LINK_SPEED",
   } ,

   # Node.DebugLevel
   {
    name               => "DebugLevel",
    domain             => ($DOMAIN_NODE),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "NODE_DEBUG_LEVEL",
   } ,

   # Node.External.FloatingAddress
   {
    name               => "External.FloatingAddress",
    domain             => ($DOMAIN_NODE),
    men_mandatory      => $TRUE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $EAM_FEATURE,
    multiple           => $TRUE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => \@E_external_floating_address,
   } ,

   # Node.External.IPMPGroup
   {
    name               => "External.IPMPGroup",
    domain             => ($DOMAIN_NODE),
    men_mandatory      => $TRUE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $EAM_FEATURE,
    multiple           => $TRUE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => \@E_external_ipmp_group,
   } ,

   # Cluster.Master.Id
   {
    name               => "Master.Id",
    domain             => ($DOMAIN_CLUSTER),
    men_mandatory      => $TRUE,
    diskless_mandatory => $TRUE,
    dataless_mandatory => $TRUE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "MASTER_ID",
   } ,

   # Cluster.Direct-Link.Backend
   {
    name               => "Direct-Link.Backend",
    domain             => ($DOMAIN_CLUSTER),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "DIRECT_LINK_BACKEND",
   } ,

   # Cluster.Direct-Link.Heartbeat
   {
    name               => "Direct-Link.Heartbeat",
    domain             => ($DOMAIN_CLUSTER),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "DIRECT_LINK_HEARTBEAT",
   } ,

   # Cluster.DataManagementPolicy
   {
    name               => "DataManagementPolicy",
    domain             => ($DOMAIN_CLUSTER),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "DATA_MGT_POLICY",
   } ,

   # Cmm.IsEligible
   {
    name               => "IsEligible",
    domain             => ($DOMAIN_CMM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $CMM_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PUBLIC,
    field              => "IS_ELIGIBLE",
   } ,

   # Cmm.LocalConfig.Dir
   {
    name               => "LocalConfig.Dir",
    domain             => ($DOMAIN_CMM),
    men_mandatory      => $TRUE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $CMM_FEATURE,
    multiple           => $FALSE,
    default            => "/etc/opt/SUNWcgha",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "LOCAL_CONFIG_DIR",
   } ,

   # Cmm.MiniConfig.Dir
   {
    name               => "MiniConfig.Dir",
    domain             => ($DOMAIN_CMM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $CMM_FEATURE,
    multiple           => $FALSE,
    default            => "/etc/opt/SUNWcgha",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "MINI_CONFIG_DIR",
   } ,

   # Cmm.MiniConfig.File
   {
    name               => "MiniConfig.File",
    domain             => ($DOMAIN_CMM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $CMM_FEATURE,
    multiple           => $FALSE,
    default            => "target.conf",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "MINI_CONFIG_FILE",
   } ,

   # Cmm.CurrentNodeName
   {
    name               => "CurrentNodeName",
    domain             => ($DOMAIN_CMM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $CMM_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "CURRENT_NODE_NAME",
   } ,

   # Cmm.Daemonize
   {
    name               => "Daemonize",
    domain             => ($DOMAIN_CMM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $CMM_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PRIVATE,
    field              => "CMM_DAEMONIZE",
   } ,

   # Cmm.MaxOutTime
   {
    name               => "MaxOutTime",
    domain             => ($DOMAIN_CMM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $CMM_FEATURE,
    multiple           => $FALSE,
    default            => 80,
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "CMM_MAX_OUT_TIME",
   } ,

   # Cmm.UseCGTP
   {
    name               => "UseCGTP",
    domain             => ($DOMAIN_CMM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $CMM_FEATURE,
    multiple           => $FALSE,
    default            => "TRUE",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PRIVATE,
    field              => "CMM_USE_CGTP",
   } ,

   # Cmm.UseProbe
   {
    name               => "UseProbe",
    domain             => ($DOMAIN_CMM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $CMM_FEATURE,
    multiple           => $FALSE,
    default            => "TRUE",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PRIVATE,
    field              => "CMM_USE_PROBE",
   } ,

   # Cmm.UseIFCheckMode
   {
    name               => "UseIFCheckMode",
    domain             => ($DOMAIN_CMM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $CMM_FEATURE,
    multiple           => $FALSE,
    default            => "TRUE",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PRIVATE,
    field              => "NO_IF_CHECK_MODE",
   } ,

   # Cmm.Election.MaxEligible
   {
    name               => "Election.MaxEligible",
    domain             => ($DOMAIN_CMM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $CMM_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "MAX_ELIGIBLE",
   } ,

   # Cmm.Election.Duration
   {
    name               => "Election.Duration",
    domain             => ($DOMAIN_CMM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $CMM_FEATURE,
    multiple           => $FALSE,
    default            => 100,
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "ELECTION_DURATION",
   } ,

   # Cmm.Probe.DetectionDelay
   {
    name               => "Probe.DetectionDelay",
    domain             => ($DOMAIN_CMM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $CMM_FEATURE,
    multiple           => $FALSE,
    default            => 900,
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "PROBE_DETECTION_DELAY",
   } ,

   # Cmm.Probe.HeartBeatNumber
   {
    name               => "Probe.HeartBeatNumber",
    domain             => ($DOMAIN_CMM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $CMM_FEATURE,
    multiple           => $FALSE,
    default            => 3,
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "PROBE_HEARTBEAT_NUMBER",
   } ,

   # Cmm.DebugLevel
   {
    name               => "DebugLevel",
    domain             => ($DOMAIN_CMM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $CMM_FEATURE,
    multiple           => $FALSE,
    default            => 0,
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "CMM_DEBUG_LEVEL",
   } ,

   # Cmm.Nic0
   {
    name               => "Nic0",
    domain             => ($DOMAIN_CMM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $CMM_FEATURE,
    multiple           => $FALSE,
    default            => "hme0",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "CMM_NIC0",
   } ,

   # Cmm.Nic1
   {
    name               => "Nic1",
    domain             => ($DOMAIN_CMM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $CMM_FEATURE,
    multiple           => $FALSE,
    default            => "hme1",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "CMM_NIC1",
   } ,

   # Cmm.NicCGTP
   {
    name               => "NicCGTP",
    domain             => ($DOMAIN_CMM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $CMM_FEATURE,
    multiple           => $FALSE,
    default            => "cgtp0",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "CMM_NIC_CGTP",
   } ,

   # Cmm.MasterLoss.Detection
   {
    name               => "MasterLoss.Detection",
    domain             => ($DOMAIN_CMM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $CMM_FEATURE,
    multiple           => $FALSE,
    default            => "TRUE",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PUBLIC,
    field              => "MASTER_LOSS_DETECTION",
   } ,

   # Probe.DebugLevel
   {
    name               => "DebugLevel",
    domain             => ($DOMAIN_PROBE),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => 0,
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "PROBE_DEBUG_LEVEL",
   } ,

   # Probe.SelfCheck
   {
    name               => "SelfCheck",
    domain             => ($DOMAIN_PROBE),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => "TRUE",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PRIVATE,
    field              => "PROBE_SELF_CHECK",
   } ,

   # Probe.FirstCheck
   {
    name               => "FirstCheck",
    domain             => ($DOMAIN_PROBE),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "PROBE_FIRST_CHECK",
   } ,

   # Probe.Daemonize
   {
    name               => "Daemonize",
    domain             => ($DOMAIN_PROBE),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => "TRUE",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PRIVATE,
    field              => "PROBE_DAEMONIZE",
   } ,

   # Probe.UseCGTP
   {
    name               => "UseCGTP",
    domain             => ($DOMAIN_PROBE),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => "TRUE",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PRIVATE,
    field              => "PROBE_USE_CGTP",
   } ,

   # Probe.Nic0
   {
    name               => "Nic0",
    domain             => ($DOMAIN_PROBE),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => "hme0",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "PROBE_NIC0",
   } ,

   # Probe.Nic1
   {
    name               => "Nic1",
    domain             => ($DOMAIN_PROBE),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => "hme1",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "PROBE_NIC1",
   } ,

   # Rnfs.DebugLevel
   {
    name               => "DebugLevel",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $FALSE,
    default            => 0,
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "RNFS_DEBUG_LEVEL",
   } ,

   # Rnfs.StatdAlternatePath
   {
    name               => "StatdAlternatePath",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $TRUE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "STATD_ALTERNATE_PATH",
   } ,

   # Rnfs.Slice
   {
    name               => "Slice",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $TRUE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $TRUE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => \@E_slice_string,
   } ,

   # Rnfs.Share 
   {
    name               => "Share",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $TRUE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $TRUE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => \@E_share_string,
   } ,

   # Rnfs.UsePmd
   {
    name               => "UsePmd",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $FALSE,
    default            => "TRUE",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PRIVATE,
    field              => "RNFS_USE_PMD",
   } ,

   # Rnfs.SyncType
   {
    name               => "SyncType",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $FALSE,
    default            => "FS",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "SYNC_TYPE",
   } ,

   # Rnfs.DiskFailureFailover
   {
    name               => "DiskFailureFailover",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $FALSE,
    default            => "TRUE",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PUBLIC,
    field              => "DISK_FAILURE_FAILOVER",
   } ,

   # Rnfs.CheckReplicatedSlices
   {
    name               => "CheckReplicatedSlices",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $FALSE,
    default            => "FALSE",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PUBLIC,
    field              => "WRITE_CHECK",
   } ,

   # Rnfs.CheckReplicatedSlicesInterval
   {
    name               => "CheckReplicatedSlicesInterval",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $FALSE,
    default            => 10,
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "WRITE_CHECK_INTERVAL",
   } ,

   # Rnfs.EnableSync
   {
    name               => "EnableSync",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $FALSE,
    default            => "TRUE",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PUBLIC,
    field              => "SYNC_FLAG",
   } ,

   # Rnfs.SerializeSync
   {
    name               => "SerializeSync",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $FALSE,
    default            => "FALSE",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PUBLIC,
    field              => "SERIALIZE_SYNC",
   } ,

   # Rnfs.Mode
   {
    name               => "Mode",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $FALSE,
    default            => "SNDR",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "RNFS_MODE",
   } ,

   # Rnfs.FencingType
   {
    name               => "FencingType",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $FALSE,
    default            => "None",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "RNFS_FENCING_TYPE",
   } ,

   # Rnfs.SCSIFencingDisk
   {
    name               => "SCSIFencingDisk",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $TRUE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => \@E_SCSI_fencing_disk,
   } ,

   # Rnfs.SCSIFailfastTimeout
   {
    name               => "SCSIFailfastTimeout",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $FALSE,
    default            => 500,
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "RNFS_SCSI_FAILFAST_TIMEOUT",
   } ,

   # Rnfs.SCSIFailOverFailfastTimeout
   {
    name               => "SCSIFailoverFailfastTimeout",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $FALSE,
    default            => 500,
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "RNFS_SCSI_FAILOVER_FAILFAST_TIMEOUT",
   } ,

   # Rnfs.SCSIFencingSleepDelay
   {
    name               => "SCSIFencingSleepDelay",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $FALSE,
    default            => 2000,
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "RNFS_SCSI_FENCING_SLEEP_DELAY",
   } ,

   # Rnfs.SCSIFastFencingSleepDelay
   {
    name               => "SCSIFastFencingSleepDelay",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $FALSE,
    default            => 100,
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "RNFS_SCSI_FAST_FENCING_SLEEP_DELAY",
   } ,

   # Rnfs.SCSIFastFencingTimeout
   {
    name               => "SCSIFastFencingTimeout",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $FALSE,
    default            => 2000,
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "RNFS_SCSI_FAST_FENCING_TIMEOUT",
   } ,

   # Rnfs.SCSIFencingDisksWatchDelay
   {
    name               => "SCSIFencingDisksWatchDelay",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $FALSE,
    default            => 2000,
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "RNFS_SCSI_FENCING_DISKS_WATCH_DELAY",
   } ,

   # Rnfs.SCSIFencingDisksFailureFailover
   {
    name               => "SCSIFencingDisksFailureFailover",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $FALSE,
    default            => "FALSE",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PRIVATE,
    field              => "RNFS_SCSI_FENCING_DISKS_FAILURE_FAILOVER",
   } ,

   # RNFS.RdcBitmapMode
   {
    name               => "RdcBitmapMode",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "RNFS_RDC_BITMAP_MODE",
   } ,

   # RNFS.WaitForDevicesOnTakeSleepDelay
   {
    name               => "WaitForDevicesOnTakeSleepDelay",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $FALSE,
    default            => 10000,
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "RNFS_WAIT_FOR_DEVICES_ON_TAKE_SLEEP_DELAY",
   } ,

   # RNFS.WaitForDevicesOnTakeTimeout
   {
    name               => "WaitForDevicesOnTakeTimeout",
    domain             => ($DOMAIN_RNFS),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $RNFS_FEATURE,
    multiple           => $FALSE,
    default            => 10000,
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "RNFS_WAIT_FOR_DEVICES_ON_TAKE_TIMEOUT",
   } ,

   # Nsm.Exec.Masterdir
   {
    name               => "Exec.Masterdir",
    domain             => ($DOMAIN_NSM),
    men_mandatory      => $TRUE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NSM_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "NSM_MASTER_DIR",
   } ,

   # Nsm.Exec.ViceMasterdir
   {
    name               => "Exec.ViceMasterdir",
    domain             => ($DOMAIN_NSM),
    men_mandatory      => $TRUE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NSM_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "NSM_VICEMASTER_DIR",
   } ,

   # Nsm.Log.Masterdir
   {
    name               => "Log.Masterdir",
    domain             => ($DOMAIN_NSM),
    men_mandatory      => $TRUE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NSM_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "NSM_LOG_MASTER_DIR",
   } ,

   # Nsm.Log.ViceMasterdir
   {
    name               => "Log.ViceMasterdir",
    domain             => ($DOMAIN_NSM),
    men_mandatory      => $TRUE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NSM_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "NSM_LOG_VICEMASTER_DIR",
   } ,

   # Nsm.Option.Fifo
   {
    name               => "Option.Fifo",
    domain             => ($DOMAIN_NSM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NSM_FEATURE,
    multiple           => $FALSE,
    default            => "",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "NSM_OPTION_FIFO",
   } ,

   # Nsm.Option.Trace
   {
    name               => "Option.Trace",
    domain             => ($DOMAIN_NSM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NSM_FEATURE,
    multiple           => $FALSE,
    default            => "FALSE",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PRIVATE,
    field              => "NSM_OPTION_TRACE",
   } ,

   # Nsm.Option.Debug 
   {
    name               => "Option.Debug",
    domain             => ($DOMAIN_NSM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NSM_FEATURE,
    multiple           => $FALSE,
    default            => "FALSE",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PRIVATE,
    field              => "NSM_OPTION_DEBUG",
   } ,

   # Nsm.Option.Print
   {
    name               => "Option.Print",
    domain             => ($DOMAIN_NSM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NSM_FEATURE,
    multiple           => $FALSE,
    default            => "FALSE",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PRIVATE,
    field              => "NSM_OPTION_PRINT",
   } ,

   # Nsm.Option.Heap
   {
    name               => "Option.Heap",
    domain             => ($DOMAIN_NSM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NSM_FEATURE,
    multiple           => $FALSE,
    default            => "FALSE",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PRIVATE,
    field              => "NSM_OPTION_HEAP",
   } ,

   # Nsm.Option.Rt
   {
    name               => "Option.Rt",
    domain             => ($DOMAIN_NSM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NSM_FEATURE,
    multiple           => $FALSE,
    default            => "FALSE",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PRIVATE,
    field              => "NSM_OPTION_RT",
   } ,

   # Nsm.Option.Log
   {
    name               => "Option.Log",
    domain             => ($DOMAIN_NSM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NSM_FEATURE,
    multiple           => $FALSE,
    default            => "FALSE",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PRIVATE,
    field              => "NSM_OPTION_LOG",
   } ,

   # Nsm.Option.NoCgha
   {
    name               => "Option.NoCgha",
    domain             => ($DOMAIN_NSM),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NSM_FEATURE,
    multiple           => $FALSE,
    default            => "FALSE",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PRIVATE,
    field              => "NSM_OPTION_NOCGHA",
   } ,

   # PMD.FailureAction
   {
    name               => "FailureAction",
    domain             => ($DOMAIN_PMD),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $PMD_FEATURE,
    multiple           => $FALSE,
    default            => "KernelPanic",
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PRIVATE,
    field              => "PMD_FAILURE_ACTION",
   } ,

   # WATCHDOG.PattingIntvl
   {
    name               => "PattingIntvl",
    domain             => ($DOMAIN_WDT),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $WDT_FEATURE,
    multiple           => $FALSE,
    default            => 2000,
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "WATCHDOG_PATTING_INTVL",
   } ,

   # WATCHDOG.ShutdownTimeout
   {
    name               => "ShutdownTimeout",
    domain             => ($DOMAIN_WDT),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $WDT_FEATURE,
    multiple           => $FALSE,
    default            => 5000,
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "WATCHDOG_SHUTDOWN_TIMEOUT",
   } ,

   # WATCHDOG.OsTimeout
   {
    name               => "OsTimeout",
    domain             => ($DOMAIN_WDT),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $WDT_FEATURE,
    multiple           => $FALSE,
    default            => 3000,
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "WATCHDOG_OS_TIMEOUT",
   } ,

   # WATCHDOG.NhasWatchdog
   {
    name               => "NhasWatchdog",
    domain             => ($DOMAIN_WDT),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $WDT_FEATURE,
    multiple           => $FALSE,
    default            => "FALSE",
    control            => $CONTROL_BOOLEAN,
    scope              => $SCOPE_PUBLIC,
    field              => "WATCHDOG_ENABLE",
   } ,

   # SCHEDULING.OTHER.BasePriority
   {
    name               => "OTHER.BasePriority",
    domain             => ($DOMAIN_SCHED),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => 0,
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "SCHED_OTHER_BASE",
   } ,

   # SCHEDULING.FIFO.BasePriority
   {
    name               => "FIFO.BasePriority",
    domain             => ($DOMAIN_SCHED),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => 0,
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "SCHED_FIFO_BASE",
   } ,

   # SCHEDULING.RR.BasePriority
   {
    name               => "RR.BasePriority",
    domain             => ($DOMAIN_SCHED),
    men_mandatory      => $FALSE,
    diskless_mandatory => $FALSE,
    dataless_mandatory => $FALSE,
    component          => $NONE_FEATURE,
    multiple           => $FALSE,
    default            => 0,
    control            => $CONTROL_STRING,
    scope              => $SCOPE_PUBLIC,
    field              => "SCHED_RR_BASE",
   }

 ) ;

my $G_node_type_property = 0;


#--------------------------------------------------------------
#
# init properties
#
#--------------------------------------------------------------

sub init_properties {

  # build the full name
  for (my $i = 0 ; $i < scalar(@G_properties) ; $i++) {
    $G_properties[$i]{fullname} = $G_properties[$i]{domain} . ".". $G_properties[$i]{name};
  }

}


#--------------------------------------------------------------
#
# process one parameter
#
#--------------------------------------------------------------

sub process_one_parameter {

  my ($parameter, $value) = @_;
  my $idx;

  my $upparameter = $parameter;
  $upparameter =~ tr/a-z/A-Z/;

  # a property match when equal to 
  #   <domain>.<name> or NODE.<name> or CLUSTER.<name>
  my $found = $FALSE;
  for (my $i = 0 ; ($i < scalar(@G_properties)) && (! $found) ; $i++) {
    my $property = $G_properties[$i]{name};

    my $domain_list = ($G_properties[$i]{domain});
    foreach my $domain ($domain_list) {
      my $fullname = "$domain.$property";
      $fullname =~ tr/a-z/A-Z/;
      if ($G_properties[$i]{multiple}) {
	# we remove the rank before comparing (no rank means no need to compare)
	if ($upparameter =~ m/(.*)\.([0-9]+)/) {
	  if ($1 eq $fullname) {
	    $found = $TRUE;
	    $idx = $i;
	    last ;
	  }
	}
      } else {
	if ($upparameter eq $fullname) {
	  $found = $TRUE;
	  $idx = $i;
	  last ;
	}
      }
    }
  }

  if (! $found) {
    LOG::printWarning("%s: unknown property", $parameter);
    return $FALSE;
  }

  # check the value type
  if ($G_properties[$idx]{control} == $CONTROL_BOOLEAN) {
    $value =~ tr/a-z/A-Z/;
    if (($value ne "TRUE") && ($value ne "FALSE")) {
      LOG::printError("%s: value must be TRUE or FALSE", $parameter);
      $G_properties[$idx]{status} = "defined";
      $G_fatal = $TRUE;
      return $FALSE;
    }
  }

  # check that the parameter is not yet defined
  if (defined ($G_properties[$idx]{status}) 
      && (! $G_properties[$idx]{multiple})) {
    LOG::printWarning("%s: multiple defined", $parameter);
    return $TRUE;
  }

  if (! $G_properties[$idx]{multiple}) {
    # eliminated useless spaces
    $value =~ m/[\s\t]*([^\s\t]*)/;
    $E_variables{$G_properties[$idx]{field}} = $1;
  } else {
    my $array = $G_properties[$idx]{field};
    push @$array, $value;
  }

  $G_properties[$idx]{status} = "defined";
  return $TRUE;

exit 1;
}

#--------------------------------------------------------------
#
# process when all variables have been loaded
#
# . check mandatory properties
# . manage default values
#
#--------------------------------------------------------------

sub process_end {

  my $L_status = "OK";
  my $mandatory;

  # process the default value : to be sure that Cmm.isEligible is set
  # required to check mandatory properties (depend on eligibility)
  for (my $i = 0 ; $i < scalar(@G_properties) ; $i++) {
    # for simple property not defined, set the default value
    if ((! $G_properties[$i]{multiple}) &&
	(! defined ($G_properties[$i]{status}))) {
      $E_variables{$G_properties[$i]{field}} = $G_properties[$i]{default};
    }
  }

  # retrieve the node type to be able to perform the right checks later
  if (defined ($G_properties[$G_node_type_property]{status})) {
    $E_variables{NODE_TYPE} =~ tr/a-z/A-Z/;
    if (($E_variables{NODE_TYPE} ne "DISKFULL") &&
	($E_variables{NODE_TYPE} ne "DISKLESS") &&
	($E_variables{NODE_TYPE} ne "DATALESS")) {

#=item Incorrect value for Node.Type in <file>
#
#    Node.Type must be equal to Diskfull, Diskless or Dataless
#
#=cut

      printError("Incorrect value for Node.Type in %s", 
		 $E_fs_configuration_file);
      $L_status = "FAILED";
      $G_fatal = $TRUE;
    }
  } else {

#=item Node.Type definition missing in <file>
#
#    Node.Type is required and must be equal to Diskfull, Diskless
#    or Dataless
#
#=cut

    printError("Node.Type definition missing in %s", 
	       $E_fs_configuration_file);
    $L_status = "FAILED";
    $G_fatal = $TRUE;
  } 

  if (! $G_fatal) {
    # check mandatory properties

    if ($E_variables{NODE_TYPE} eq "DISKFULL") {
      $mandatory = "men_mandatory";
    } elsif ($E_variables{NODE_TYPE} eq "DISKLESS") {
      $mandatory = "diskless_mandatory";
    } else {
      $mandatory = "dataless_mandatory";
      
    }

    for (my $i = 0 ; $i < scalar(@G_properties) ; $i++) {
      if (($G_properties[$i]{$mandatory}) &&
	  (! defined($G_properties[$i]{status}))) {
	
	# check if it depends on a component
	if ($G_properties[$i]{component} eq "") {
	  
	  # mandatory without feature dependencies
	  LOG::printError("%s definition required",
			  $G_properties[$i]{fullname});
	  $G_fatal = $TRUE;
	  $L_status = "FAILED";
	  
	} else {
	  
	  # mandatory when a specific feature is installed
	  
	  my $return = OS::is_feature_installed($G_properties[$i]{component});
	  if (($return eq "TRUE") &&
	      (! defined($G_properties[$i]{status}))) {
	    LOG::printError("%s definition required when the %s feature is installed",
			    $G_properties[$i]{fullname},$G_properties[$i]{component});
	    $G_fatal = $TRUE;
	    $L_status = "FAILED";
	    
	  }
	}
      }
    }
  }

  my $result = "OK";
  if (($L_status eq "FAILED") || ($G_prop_error)) {
    $result = "FAILED";
  }
  LOG::printResult($result, "Configuration file checking");
  if ($G_fatal) {
    LOG::printLine("FATAL: All others tests are skipped");
    exit 1;
  }
}

#--------------------------------------------------------------
#
# loading the Foundation Services configuration file
#
#--------------------------------------------------------------

sub load_fs_conf {

  my $parameter;
  my $value;
  my $upvalue;
  my $line;

  LOG::printMsg(0, "Loading FS configuration file %s ...", 
		$E_fs_configuration_file) ;
  
  if (! OS::check_file($E_fs_configuration_file)) {
    exit 1;
  }

  init_properties();
  
  open(FH, "<$E_fs_configuration_file") ;
  while (<FH>) {
    chomp($line =$_) ;
    # remove comments
    $line =~ /([^\#]*)/;
    $line=$1 ;
    if ($line eq "") {
      next ;
    }
    
    # retrieve parameter and its value
    if ($line =~ /[\s\t]*([^=\s\t]*)[\s\t]*=(.*)/) {
      if (! process_one_parameter($1, $2)) {
	$G_prop_error = $TRUE;
      }  
    }  
  }

  close(FH) ;

  process_end;
}

#--------------------------------------------------------------
#
# display the Foundation Services configuration data
#
#--------------------------------------------------------------

sub display {

  printLine("");
  printLine("Foundation Services configuration data");
  printLine("");;
  for my $i (0..$#G_properties) {
    if (($G_properties[$i]{scope} == $SCOPE_PUBLIC) 
	&& (! $G_properties[$i]{multiple})) {
      my $property = $G_properties[$i]{fullname};
      my $field = $G_properties[$i]{field};
      my $value = $E_variables{$field};
      printf "    %-30s = %s\n", $property , $value ;
    }
  }
}

#=back
#
#=cut

{
}
