#************************************************************ 
#
# Component       CNF  
#
# Synopsis        Network and CGTP interfaces
#
#
# Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
# Use is subject to license terms.
#
#************************************************************
#
# #ident	"@(#)CGTP.pm	1.18	06/01/23 SMI"
#
#************************************************************

#=head1 CGTP interfaces
#
#=over 4
#
#=cut

package CGTP ;

use strict;
use VAR;
use LOG;

#
#
#
my $G_category = "CGTP" ;

my @G_IP_list = ();

#------------------------------------------------------------------------------
#
#  build_IP_address
#
#------------------------------------------------------------------------------

sub build_IP_address {

  my ($node_id, $interface, $ip_version) = @_ ;
  
  if (! defined($ip_version)) {
    $ip_version = $IP_VERSION_V4;
  }

  if (! NETWORK::exist_interface($interface, $ip_version)) {
    return ("unknow IP");
  }
  
  # build the expected IP address of the node
  my $netmask = NETWORK::get_interface_info($interface, 
					    $ip_version, 
					    "netmask");
  $netmask = hex($netmask);
  my $subnet  = NETWORK::get_interface_info($interface, 
					    $ip_version, 
					    "IP");
  $subnet = COMMON::convertDottedToBitmask($subnet);
  
  my $address = COMMON::convertBitmaskToDotted
    (
     ($subnet & $netmask) + $node_id
    );
  
  return $address ;
}

#--------------------------------------------------------------
#
#  get_master_name
#
#--------------------------------------------------------------

sub get_master_name {
  
  NETWORK::load();
  
  my $master_ip;
  if ($E_variables{USE_CGTP} eq "TRUE") {
    $master_ip = build_IP_address($E_variables{MASTER_ID}, 
				  $E_variables{NIC_CGTP});
  } else {
    $master_ip = build_IP_address($E_variables{MASTER_ID},
				  $E_variables{NIC0});
  }

  my $master_name = "";
  open(FH, "/etc/hosts") ;
  while (<FH>) {
    if ($_ =~ m/^$master_ip ([^ \t]*)/) {
      $master_name = $1;
      last
    }
  }
  close(FH);
  return $master_name;
}

#------------------------------------------------------------------------------
#
# check that interfaces used by CGTP have different MAC address
#
#------------------------------------------------------------------------------

sub is_MAC_unique {

  if (NETWORK::get_interface_info($E_variables{NIC0}, $IP_VERSION_V4, "ether")
      eq (NETWORK::get_interface_info($E_variables{NIC1}, $IP_VERSION_V4, "ether"))) {
    return $FALSE;
  }
  
  return $TRUE;
}


#--------------------------------------------------------------
#
# build the list of IP addresses to be checked
#
#--------------------------------------------------------------

sub build_IP_address_list {

 my $node_role = CMM::get_actual_role();
 my $ip_desc;

 my $node_id = $E_local_node_id;

 #
 # IP addresses for the node
 #

 $ip_desc = {};
 $ip_desc->{IP} = build_IP_address($node_id, $E_variables{NIC0}) ;
 $ip_desc->{up} = $TRUE;
 $ip_desc->{meaning} = "nic0";
 $ip_desc->{ipversion} = $IP_VERSION_V4;
 push @G_IP_list, $ip_desc;

 if ($E_variables{USE_CGTP} eq "TRUE") {

   $ip_desc = {};
   $ip_desc->{IP} = build_IP_address($node_id, $E_variables{NIC1}) ;
   $ip_desc->{up} = $TRUE;
   $ip_desc->{meaning} = "nic1";
   $ip_desc->{ipversion} = $IP_VERSION_V4;
   push @G_IP_list, $ip_desc;

   $ip_desc = {};
   $ip_desc->{IP} = build_IP_address($node_id, $E_variables{NIC_CGTP}) ;
   $ip_desc->{up} = $TRUE;
   $ip_desc->{meaning} = "CGTP";
   $ip_desc->{ipversion} = $IP_VERSION_V4;
   push @G_IP_list, $ip_desc;
 }

 #
 # IP addresses for master-eligible nodes
 #
 if ($E_variables{NODE_TYPE} eq "DISKFULL") {
   
   my $up_or_not = $FALSE;
   if (CMM::get_actual_role eq "MASTER") {
     $up_or_not = $TRUE;
   }

   my $master_id = $E_variables{MASTER_ID};

   $ip_desc = {};
   $ip_desc->{IP} = build_IP_address($master_id, $E_variables{NIC0}) ;
   $ip_desc->{up} = $up_or_not;
   $ip_desc->{meaning} = "master nic0";
   $ip_desc->{ipversion} = $IP_VERSION_V4;
   push @G_IP_list, $ip_desc;
   
   if ($E_variables{USE_CGTP} eq "TRUE") {
     
     $ip_desc = {};
     $ip_desc->{IP} = build_IP_address($master_id, $E_variables{NIC1}) ;
     $ip_desc->{up} = $up_or_not;
     $ip_desc->{meaning} = "master nic1";
     $ip_desc->{ipversion} = $IP_VERSION_V4;
     push @G_IP_list, $ip_desc;
     
     $ip_desc = {};
     $ip_desc->{IP} = build_IP_address($master_id, $E_variables{NIC_CGTP}) ;
     $ip_desc->{up} = $up_or_not;
     $ip_desc->{meaning} = "master CGTP";
     $ip_desc->{ipversion} =$IP_VERSION_V4;
     push @G_IP_list, $ip_desc;
   }  
 }
}

#------------------------------------------------------------------------------
#
#  ready
#
#      return TRUE if the device specified as an argument
#      is up and running
#      assumption: existence has been previously tested
#
#------------------------------------------------------------------------------

sub ready {

  my ($device, $ip_version) = @_ ;
  
  if (! defined($ip_version)) {
    $ip_version = $IP_VERSION_V4;
  }
  
  return ((NETWORK::get_interface_info($device, $ip_version, "up"))
	  && (NETWORK::get_interface_info($device, $ip_version, "running")));
}

#--------------------------------------------------------------
#
#  check_route
#
#--------------------------------------------------------------

sub check_route {

  my $status = "OK" ;
  my $interfaceip;
  my $cgtpip;
  
  if (! NETWORK::exist_interface($E_variables{NIC_CGTP})) {
    
#=item CGTP interface is not present
#
#    Foundations Services are probably not being started.
#
#=cut
			
    LOG::printError("CGTP interface is not present") ;
    $status = "FAILED_SKIP" ;
    LOG::printResult($status, "CGTP route checking") ;
    return ;
  }
  
  
  # check that the route has been created for each node of the cluster
  
  for (my $i = 0 ; $i < $E_cluster_node_count ; $i++) {
    if ($i != $E_cluster_local_node) {
      $cgtpip = build_IP_address($E_cluster_node[$i]{node_id},
				    $E_variables{NIC_CGTP}) ;
      foreach  my $interface ($E_variables{NIC0}, $E_variables{NIC1}) {
	$interfaceip = build_IP_address($E_cluster_node[$i]{node_id},
					   $interface) ;
	if (! NETWORK::exist_route($cgtpip, $interfaceip)) {
	  
#=item Destination <CGTP ip>, gateway <interface IP>: no route defined
#
#    To be routed, CGTP protocol requires a route to be defined for each
#    interface.
#
#=cut
			
	  LOG::printError("Destination %s, gateway %s: no route defined", $cgtpip, $interfaceip) ;
	  $status = "FAILED" ;
	}
	LOG::printResult($status, "CGTP route for %s through %s",
			 $E_cluster_node[$i]{node_name}, $interface) ;
      }
    }
  }
  
  # check that there is a route also for the master
  
  my $masterip = build_IP_address($E_variables{MASTER_ID},
				  $E_variables{NIC_CGTP}) ;
  foreach my $interface ($E_variables{NIC0}, $E_variables{NIC1}) {
    $interfaceip = build_IP_address($E_variables{MASTER_ID}, $interface) ;
    if (! NETWORK::exist_route($masterip, $interfaceip)) {
      LOG::printError("Destination %s, gateway %s: no route defined", $masterip, $interfaceip) ;
      $status = "FAILED" ;
    }
    my $nodename = "master";
    LOG::printResult($status, "CGTP route for %s through %s",
		       $nodename, $interface) ;
  }
}

#------------------------------------------------------------------------------
#
#  check_diskless_configuration
#
#------------------------------------------------------------------------------

sub check_diskless_configuration {

  my $status = "OK" ;
  my $line;

  
  my @interface_list = ( $E_variables{NIC0} ) ;
  if ($E_variables{USE_CGTP} eq "TRUE") {
    @interface_list = ( $E_variables{NIC0},
			$E_variables{NIC1}
		      ); 
  }

  #
  # test the existence of /etc/hostname.<NICx>
  #
  $status = "OK" ;
  foreach my $interface (@interface_list) {

    my $hostname_file = "/etc/hostname.$interface";

    LOG::printMsg(1,"    test existence of %s", $hostname_file);
    if (! -e $hostname_file) {
	
#=item /etc/hostname.<interface> does not exists
#
#    This file is required but must be empty.
#
#=cut
      LOG::printError("%s does not exist", $hostname_file);
      $status = "FAILED" ;
      next ;
    }
    
    if (! NETWORK::exist_interface($interface)) {
      next ;
    }
    
    if (! -z $hostname_file ne "") {
      
#=item host name defined for <interface>
#
#    /etc/hostname must exist but must be empty.
#
#=cut

      LOG::printError("%s must be empty", $hostname_file);
      $status = "FAILED" ;
      next ;
    }
  }
  LOG::printResult($status, "/etc/hostname.* must be empty for each interface") ;
  
  #
  # test the existence of /etc/dhcp.<NICx>
  #
  $status = "OK" ;
  foreach my $interface (@interface_list) {

    my $dhcp_file = "/etc/dhcp.$interface";

    LOG::printMsg(1,"    test existence of %s", $dhcp_file);
    if (! -e $dhcp_file) {

#=item /etc/dhcp.<interface> does not exists
#
#    This file is required but must be empty.
#
#=cut
      LOG::printError("%s does not exist", $dhcp_file);
      $status = "FAILED" ;
      next ;
    }
    
    if (! NETWORK::exist_interface($interface)) {
      next ;
    }

    if (! -z $dhcp_file) {
	
#=item dhcp name defined for <interface>
#
#    /etc/dhcp must exist but must be empty.
#
#=cut

      LOG::printError("%s must be empty", $dhcp_file);
      $status = "FAILED" ;
      next ;
    }
  }
  LOG::printResult($status, "/etc/dhcp.* must be empty for each interface") ;

  check_dhcpinfo();
}

#------------------------------------------------------------------------------
#
#  check_with_disk_configuration
#
#------------------------------------------------------------------------------

sub check_with_disk_configuration {

  my $status = "OK" ;
  
  my @interface_list = ( $E_variables{NIC0} ) ;
  if ($E_variables{USE_CGTP} eq "TRUE") {
    @interface_list = ( $E_variables{NIC0},
			$E_variables{NIC1},
			$E_variables{NIC_CGTP}
		      ); 
  }

  $status = "OK" ;
  foreach my $interface (@interface_list) {
    
    my $hostname_file = "/etc/hostname.$interface";
    LOG::printMsg(1,"    test existence of %s", $hostname_file);
    if (! -e $hostname_file) {
      
#=item /etc/hostname.<interface> does not exists
#
#    This file is required and should contain the name of the node
#    for the specified interface.
#
#=cut
      LOG::printError("%s does not exist", $hostname_file);
      $status = "FAILED" ;
    }
  }

  LOG::printResult($status, "/etc/hostname.* must exist for each interface") ;
}
  
#--------------------------------------------------------------
#
#  check_dhcpinfo
#
#      check whether symbols are provided
#      by the DHCP server correctly
#
#--------------------------------------------------------------

sub check_dhcpinfo
  {
    return if $E_variables{USE_CGTP} ne "TRUE";

    foreach my $L_iface ($E_variables{NIC0}, $E_variables{NIC1})
      {
	next if !NETWORK::exist_interface($L_iface);
	next if !(NETWORK::get_interface_info($L_iface,
	  $IP_VERSION_V4, "flags") =~ /DHCP/);

	foreach my $L_tmp ("NhNic0Addr:NIC0", "NhNic1Addr:NIC1",
	  "NhCgtpAddr:NIC_CGTP")
	  {
	    my $L_status = "OK";
	    my ($L_symbol, $L_symiface) = split(":", $L_tmp);
	    my $L_expected = build_IP_address($E_local_node_id, 
	      $E_variables{$L_symiface});
	    my $L_value = qx"/sbin/dhcpinfo -i $L_iface $L_symbol";
	    if ($? == 0x300)
	      {

#=item Symbol <symbol> is unknown
#
#    The dhcpinfo tool returned an error to request of
#    <symbol> value. Probably this symbol is not defined
#    in /etc/dhcp/inittab.
#
#=cut

		LOG::printError("Symbol $L_symbol is unknown");
		$L_status = "FAILED";
	      }
	    elsif (! $L_value)
	      {

#=item Symbol <symbol> is not provided by the DHCP server
#
#    The DHCP server didn't provide value in DHCP response.
#    Check dhcptab on the master node and
#    /etc/default/dhcpagent on the client.
#
#=cut

		LOG::printError("Symbol $L_symbol is not provided " .
		  "by the DHCP server");
		  $L_status = "FAILED";
	      }
	    chomp($L_value);

#=item <symbol> on <interface>: invalid IP address (<found_ip> instead of <expected_ip>)
#
#    The DHCP server provided an incorrect IP address value
#    in <symbol> symbol.
#
#=cut

	    if ($L_status eq "OK" && $L_value ne $L_expected)
	      {
		LOG::printError( "$L_symbol on %s: invalid IP address"
		  . " (%s instead of %s)", $L_iface, $L_value,
		   $L_expected);
		$L_status = "FAILED";
	      }

	    LOG::printResult($L_status,
	      "DHCP symbol $L_symbol defined on %s", $L_iface);
	  }
      }
  }

#--------------------------------------------------------------
#
#  check_configuration
#
#      check network information to eliminate potential problem
#
#--------------------------------------------------------------

sub check_configuration {

  my $status = "OK" ;
  
  LOG::printMsg(0, "") ;
  if ($E_variables{USE_CGTP} eq "TRUE") {
    LOG::printMsg(0, "CGTP and network configuration") ;
  } else {
    LOG::printMsg(0, "Network configuration") ;
  }
  LOG::printMsg(0, "") ;
  

  if ($E_variables{NODE_TYPE} eq "DISKLESS") {
    check_diskless_configuration();
  } else {
    check_with_disk_configuration();
  }
  
  #
  # test the existence of /etc/hostname.<NICx>
  #
  my @interface_list = ( $E_variables{NIC0} ) ;
  if ($E_variables{USE_CGTP} eq "TRUE") {
    if ($E_variables{NODE_TYPE} eq "DISKLESS") {
      @interface_list = ( $E_variables{NIC0},
			  $E_variables{NIC1});
    } else {
      @interface_list = ( $E_variables{NIC0},
			  $E_variables{NIC1},
			  $E_variables{NIC_CGTP}) ;
    }
  }

  #
  # hostname and IP address of each interface must be defined in /etc/hosts
  #
  $status = "OK" ;
  foreach my $interface (@interface_list) {
    
    if (! NETWORK::exist_interface($interface)) {
      next ;
    }
     
    my $ip = build_IP_address($E_local_node_id, $interface) ;
    my $found = $FALSE;
    open(FH, "</etc/hosts") ;
    while (<FH>) {
      my $line;
      chomp($line = $_) ;
      $line =~ s/#.*// ;
      #
      # <IP address> <hostname> ...
      #
      if ($line =~ m/[\s\t]*([0-9.]*)[\s\t]*(.*)/) {
	if ($1 eq $ip) {
	  LOG::printMsg(3, "definition IP: %s hostname: %s being processed", $1, $2) ;
	  if ($found) {
	    
		  #=item <IP address> defined twice in /etc/hosts
		  #
#    Confirm that only one hostname is defined for the specified IP
#    address.
#
#=cut
			
	    LOG::printError("%s defined twice in /etc/hosts", $ip) ;
	    $status = "FAILED" ;
	  }
	  else {
	    $found = $TRUE;
	  }
	}
      }
    }
    if (! $found) {
	
#=item <interface> (<IP address>) has no associated host name in /etc/hosts
#
#    The <IP address> corresponding to the <interface> is not defined
#    in /etc/hosts.
#
#=cut
    
      LOG::printError("%s (%s) has no associated host name in /etc/hosts", $interface, $ip) ;
      $status = "FAILED" ;
    }
    close(FH) ;
  }
  LOG::printResult($status, "node name for each interface defined in /etc/hosts") ;

}

#--------------------------------------------------------------
#
#  check_node_access
#
#      check access To the other nodes withing the cluster
#
#--------------------------------------------------------------

sub check_node_access {

  my $status = "OK";

  if ($E_cluster_file_loaded ne "TRUE") {
    return ;
  }

  #
  # check accessibility
  #
  LOG::printMsg(0,"") ;
  LOG::printMsg(0,"Following tests perform network access through available interfaces");
  LOG::printMsg(0,"Can take a long time ...") ;
  LOG::printMsg(0,"") ;
  
  my @interface_list = ( $E_variables{NIC0} );
  if ($E_variables{USE_CGTP} eq "TRUE") {
    @interface_list = ( $E_variables{NIC0},
			$E_variables{NIC1},
			$E_variables{NIC_CGTP} 
		      );
  }

  for (my $i = 0; $i < $E_cluster_node_count; $i++) {
    
    LOG::printMsg(1,"check accessibility of node %s", $E_cluster_node[$i]{node_name}) ;
    foreach my $interface (@interface_list) {
      if (! NETWORK::exist_interface($interface)) {
	next ;
      }
      if (! ready($interface)) {
	LOG::printMsg(2,"    %s not ready: not tested", $interface);
	next ;
      }
      LOG::printMsg(2,"    %s ready: to be tested", $interface);
      $status = "OK" ;
      
      #
      #  build the address
      #
      my $address=build_IP_address($E_cluster_node[$i]{node_id}, $interface) ;
      
      LOG::printMsg(1,"    ping node %s (%s)", $E_cluster_node[$i]{node_name}, $address) ;
      my $command = "/usr/sbin/ping -i $interface $address 3" ;
      LOG::printMsg(2, "  execute: %s", $command) ;
      my $result = qx{$command} ;
      LOG::printMsg(2, "      %s", $result) ;
      if ($result =~ m/no answer/) {
	$status = "FAILED" ;
      }
      LOG::printResult($status, "%s reachable through %s", 
		       $E_cluster_node[$i]{node_name}, $interface) ;
    }
  }
}

#--------------------------------------------------------------
#
#  check_starting
#
#      check when CGTP has been started
#
#--------------------------------------------------------------

sub check_starting {

  my $status;
  
  LOG::printMsg(0, "") ;
  if ($E_variables{USE_CGTP} eq "TRUE") {
    LOG::printMsg(0, "CGTP availability") ;
  } else {
    LOG::printMsg(0, "Network availability") ;
  }
  LOG::printMsg(0, "") ;
  
  NETWORK::load();
  build_IP_address_list();
  
  #
  # check that required IP addresses are configured
  # the list depends on the usage of CGTP or not and the node type and role
  #

  foreach my $ip_desc (@G_IP_list) {
    # check that there is a interface with this IP address
    $status = "OK";
    my $interface;
    if (! NETWORK::find_IP_interface($ip_desc->{IP}, $ip_desc->{ipversion}, \$interface)) {
      LOG::printError("%s: IP address %s not configured", $ip_desc->{meaning}, $ip_desc->{IP});
      $status = "FAILED";
    }
    
    if ((NETWORK::get_interface_info(
				       $interface,
				       $ip_desc->{ip_version}, 
				       "up")) 
	!= $ip_desc->{up}) {
      if ($ip_desc->{up}) {
	LOG::printError("interface %s must be UP", $interface);
      } else {
	LOG::printError("interface %s must be DOWN", $interface);
      }
      $status = "FAILED";
    }
    
    LOG::printResult($status, "%s interface (%s on %s)", $ip_desc->{meaning}, $ip_desc->{IP}, $interface) ;
  }

  
  #
  # check that interfaces supporting CGTP have different MAC addresses
  #
  $status = "OK" ;
  if ($E_isRoot eq "TRUE") {
    if (! is_MAC_unique()) {
      $status = "FAILED" ;
    }
  }
  else {
    $status = "must be root";
  }
  LOG::printResult($status, "not the same MAC address used twice") ;

  check_node_access();

  if ($E_variables{USE_CGTP} ne "FALSE") {
    LOG::printMsg(0,"") ;
    LOG::printMsg(0,"Routing") ;
    LOG::printMsg(0,"") ;
    
    # check route
    if ($E_variables{NODE_TYPE} eq "DISKFULL") {
      $status = "OK" ;
      if (! -f "/etc/notrouter")  {
	$status = "FAILED" ;
      }
      LOG::printResult($status, "Routing mechanism disabled (/etc/notrouter exists)") ;
    }
    
    check_route ;
  }
}

#--------------------------------------------------------------
#
#  display
#
#--------------------------------------------------------------

sub display {

  my $info;
  
  printLine("") ;
  printLine("Network interfaces") ;
  printLine("") ;
  
  NETWORK::load();
  build_IP_address_list();
  
  foreach my $ip_desc (@G_IP_list) {
    my $interface;
    if (NETWORK::find_IP_interface($ip_desc->{IP}, $ip_desc->{ipversion}, \$interface)) {
      
      printLine("    IP address:            %s", $ip_desc->{IP}) ;

      if (NETWORK::get_interface_info(
				      $interface,
				      $ip_desc->{ip_version}, 
				      "up")) {
	
	printLine("    Interface:             %s", $interface) ;
	
	if (! NETWORK::exist_interface($interface)) {
	  printLine("    -> not available") ;
	  printLine("");
	  next ;
	}
		
	$info = NETWORK::get_interface_info($interface, 
					    $ip_desc->{ip_version},
					    "broadcast");
	
	printLine("    Broadcast address:     %s", $info) ;
	
	$info = NETWORK::get_interface_info($interface, 
					    $ip_desc->{ip_version},
					    "flags");
	printLine("    Flags:                 %s", $info) ;
	
	$info = NETWORK::get_interface_info($interface, 
					    $ip_desc->{ip_version},
					    "netmask");
	printLine("    Netmask:               %s", $info) ;
	
	my $physical = NETWORK::get_interface_info($interface, 
						   $ip_desc->{ip_version},
						   "physical");
	if (! $physical) {
	  printLine("") ;
	  next ;
	}
	
	$info = NETWORK::get_interface_info($interface, 
				  $ip_desc->{ip_version},
				  "ether");
	printLine("    MAC address:           %s", $info) ;
	printLine("") ;
      }
    }
  }
}

#--------------------------------------------------------------
#
#  main
#
#--------------------------------------------------------------

#=back
#
#=cut

{
}
