#!/usr/bin/perl
# -----------------------------------------------------------------------------
#  ident	"@(#)common.pl	1.53	06/01/26 SMI"
# -----------------------------------------------------------------------------
#
#  Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
#  Use is subject to license terms.
# -----------------------------------------------------------------------------

package COMMON;

use strict;
use variables;

require "$ENV{NHINSTALL_LIB}/tools/lib/conf.pl";

# variables and constants used to create hostname files
my %HOSTNAME_ARRAY = () ;

my $OPTION_NONE       = 0;
my $OPTION_DOWN       = 1;
my $OPTION_DEPRECATED = 2;
my $OPTION_FAILOVER   = 4;

#-----------------------------------------------------------
#
#  init
#
#-----------------------------------------------------------

sub init {

  if (var_defined("NH_LOG_FILE", "", \$LOGFILE_NAME)) {
    $IN_LOGFILE = $TRUE ;
  }
}

#-----------------------------------------------------------
#
#  date
#
#-----------------------------------------------------------

sub getdate {
  my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = localtime;
  my $date = sprintf("%02.02d:%02.02d:%02.02d", $hour, $min, $sec) ;
  return $date ;
}


#==============================================================================
#
#  OUTPUT MANAGEMENT FUNCTIONS
#
#==============================================================================

#-----------------------------------------------------------
#
#   general output
#
#-----------------------------------------------------------

# stdin

sub print_log {
  my ($msg) = @_ ;

  print $LOG "$msg" ;
  if ($IN_LOGFILE) {
    # open, write and close to be sure all is ok even if there is a crash later
    if (!open(LOGFILE, ">>$LOGFILE_NAME")) {
      my $date = getdate() ;
      print("$date WARNING: Can't append log file $LOGFILE_NAME: logfile option disabled\n") ;
      $IN_LOGFILE = $FALSE;
      return ;
    }
    print LOGFILE "$msg";
    close(LOGFILE);
  }
}

# stderr

sub print_err {
  my ($msg) = @_ ;

  print $LOGERR "$msg" ;
  if ($IN_LOGFILE) {
    # open, write and close to be sure all is ok even if there is a crash later
    if (!open(LOGFILE, ">>$LOGFILE_NAME")) {
      my $date = getdate() ;
      print("$date WARNING: Can't append log file $LOGFILE_NAME: logfile option disabled\n") ;
      $IN_LOGFILE = $FALSE;
      return ;
    }
    print LOGFILE "$msg";
    close(LOGFILE);
  }
}

#-----------------------------------------------------------
#
#  debug
#
#  1: name of debug variable to be tested
#  2: string to be displayed
#
#-----------------------------------------------------------

sub print_debug {
  my ($debug, $msg) = @_ ;

  if (! $debug) {
    return ;
  }

  my $string = "  DEBUG: $msg\n";
  print_log($string);
}

#-----------------------------------------------------------
#
#  display stage description message
#
#-----------------------------------------------------------

sub print_stage {
  my ($comment) = @_ ;

  my $date = getdate() ;
  print_log("$date $comment\n") ;
}

#-----------------------------------------------------------
#
#  display stage description message with node reference
#
#-----------------------------------------------------------

sub print_stage_node {
  my ($comment) = @_ ;

  my $nodename = get_external_name($NODE_ID, "") ;
  my $date = getdate() ;
  print_log("$date $comment (on node $nodename)\n") ;
}

#-----------------------------------------------------------
#
#  display recovery description message
#
#-----------------------------------------------------------

sub print_recovery_node {
  my ($comment) = @_ ;

  my $nodename = get_external_name($NODE_ID, "") ;
  my $date = getdate() ;
  print_log("$date Recovery after failure: $comment (on node $nodename)\n") ;
}

#-----------------------------------------------------------
#
#  display action description message
#
#-----------------------------------------------------------

sub print_action {
  my ($comment) = @_ ;

  my $date = getdate() ;
  print_log("$date   - $comment\n") ;
}

#-----------------------------------------------------------
#
#  display sub-action description message
#
#-----------------------------------------------------------

sub print_subaction {
  my ($comment) = @_ ;

  my $date = getdate() ;
  print_log("$date    . $comment\n") ;
}

#-----------------------------------------------------------
#
#  display subsub-action description message
#
#-----------------------------------------------------------

sub print_subsubaction {
  my ($comment) = @_ ;

  my $date = getdate() ;
  print_log("$date      $comment\n") ;
}

#-----------------------------------------------------------
#
# error displays an error message and exit
#
# 1: error message
#
#-----------------------------------------------------------

sub error {
   my ($errorinfo) = @_;

   my $date = getdate() ;
   print_err("$date Error: $errorinfo\n") ;
   exit 1;
}

#------------------------------------------------------------
#
# warning displays a message
#
#   1: message
#
#------------------------------------------------------------

sub warning {
   my ($warninginfo) = @_;

   my $date = getdate() ;
   print_log("$date Warning: $warninginfo\n") ;
}

#==============================================================================
#
#  DIRECTORY AND FILE MANAGEMENT
#
#==============================================================================

#------------------------------------------------------------
#
# Directory and file status
#
#------------------------------------------------------------

sub control_dir_is_readable {
   my ($directory) = @_;
   &error("Directory \"$directory\" does not exist or is not readable") 
   unless ( -d $directory && -x $directory  && -r $directory );   
}

sub control_dir_rxw_or_create {
   my ($directory) = @_;
   
   if ( -d $directory ) {
      &error("\"$directory\" directory exist but user don't have \"rwx\" rights on it") 
      unless ( -r $directory && -w $directory && -x $directory );
   } else {
      mkdir($directory,0777) || &error("Can not create \"$directory\" directory" );
   }
}

sub control_file_is_readable {
   my ($file) = @_;
   &error("File \"$file\" does not exist or is not readable") 
   unless ( -f $file && -r $file );
}

sub control_file_is_executable {
   my ($file) = @_;
   &error("File \"$file\" does not exist or is not executable") 
   unless ( -f $file && -x $file );
}


#==============================================================================
#
# COMMAND EXECUTION
#
#==============================================================================

#------------------------------------------------------------------------------
#
# get the node id of the vice master
#
#------------------------------------------------------------------------------

sub get_id {

  my ($role) = @_;
  my $node_ip = get_external_ip($MEN1_ID);
  
  my $command = "$RSH -n $node_ip $CMMADM -c $role";
  print_debug($DEBUG_CMD, "$command");

  my @result = qx/$command/;

  my $line;
  foreach $line (@result) {
    chomp($line);
    print_debug($DEBUG_CMD, "  $line");
    if ($line =~ m/node_id[\s\t]*=[\s\t]([0-9]*)/) {
      print_debug($DEBUG_CMD, "    node id ($role): $1");
      return $1;
    }
  }

  error("Cannot determine node id for node with role=$role");
}

#------------------------------------------------------------
#
#  prepare a command containing "`" by adding a "\" before
#
#  1: remote node
#  2: command
#
#------------------------------------------------------------

sub remote_exec_bis {
   my ($client, $command ) = @_ ;

   remote_exec($client, $command) ;
 }

#------------------------------------------------------------------------------
#
#  execute a remote command and return the status result
#  if an error occurs, the output is displayed
#
#  1: remote node
#  2: command
#
#------------------------------------------------------------------------------

sub remote_exec {
   my ($client, $command ) = @_ ;
   my $status = "OK";
   my $local_statusfile = "$WORKING_DIR/install-status.$$";
   my $remote_statusfile = "/tmp/install-status.$$";
   
   #
   # Initialize status on client side
   #
   open (LSTATUS,">$local_statusfile") || return 0;
   print LSTATUS "$status\n" ;
   close(LSTATUS);
      
   unless ( &exec_cmd( "$RCP $local_statusfile $client:$remote_statusfile" ) ) {  
        unlink($local_statusfile);
        &error ("Unable to initialize remote command status on client \"$client\" (remote node may be unreachable)");
   }
   
   unlink($local_statusfile);
      
   print_debug($DEBUG_CMD, "$RSH -n $client \"$command\"");

   #
   # Execute  remote command
   #   
   
   my $output="";
   open ( REMOTEXEC, " $RSH -n $client \"$command || echo \\\$? > $remote_statusfile \" 2>&1|" )
     || error("Can not fork command: $RSH -n $client \"$command\"");
   while (<REMOTEXEC>) {
      chop();
      print_debug($DEBUG_CMD, "  $_");

      # capture outputs to displayed if an error occurs
      $output="$output  $_\n" ;

   }
   close (REMOTEXEC) || error ("Failure when executing: $RSH -n $client \"$command\" 2>&1 (remote node may be unreachable)"); 
   
   #
   # Retrieve execution status on client side
   #   
   unless ( &exec_cmd( "$RCP $client:$remote_statusfile $local_statusfile" ) ) {
     unlink($local_statusfile);
     &error ("Unable to retrieve remote command status on client \"$client\"");
   }
   
   unless ( open (LSTATUS,$local_statusfile) ) {
      unlink($local_statusfile);
      return 0;
   }
   chop($status=<LSTATUS>);
   close(LSTATUS);
   unlink($local_statusfile);
   
   unless ( $status eq "OK" ) {
     print_log("---------------------------------------------------------\n") ;
     &warning ( "remote command \"$command\" returned status: \"$status\" on client \"$client\""
	      );
     print_log($output) ;
     print_log("---------------------------------------------------------\n") ;
     return 0;
   }
	
}
#------------------------------------------------------------------------------
#
#  execute a remote command and return the status result
#  the outpout is returned as well
#
#  1: remote node
#  2: command
#  3: ref to output array
#
#------------------------------------------------------------------------------

sub remote_exec_with_output {

   my ($client, $command, $array ) = @_ ;
   my $status = "OK";
   my $local_statusfile = "$WORKING_DIR/install-status.$$";
   my $remote_statusfile = "/tmp/install-status.$$";
   
   #
   # Initialize status on client side
   #
   open (LSTATUS,">$local_statusfile") || return 0;
   print LSTATUS "$status\n" ;
   close(LSTATUS);
      
   unless ( &exec_cmd( "$RCP $local_statusfile $client:$remote_statusfile" ) ) {  
        unlink($local_statusfile);
        &error ("Unable to initialize remote command status on client \"$client\" (remote node may be unreachable)");
   }
   
   unlink($local_statusfile);
      
   print_debug($DEBUG_CMD, "$RSH -n $client \"$command\"");

   #
   # Execute  remote command
   #   
   
   @$array=();;
   open ( REMOTEXEC, " $RSH -n $client \"$command || echo \$? > $remote_statusfile \" 2>&1|" )
     || error("Can not fork command: $RSH -n $client \"$command\"");
   while (<REMOTEXEC>) {
      chop();
      print_debug($DEBUG_CMD, "  $_");

      # capture outputs to displayed if an error occurs
      push@ $array, $_ ;

   }
   close (REMOTEXEC) || error ("Failure when executing: $RSH -n $client \"$command\" 2>&1 (remote node may be unreachable)"); 
   
   #
   # Retrieve execution status on client side
   #   
   unless ( &exec_cmd( "$RCP $client:$remote_statusfile $local_statusfile" ) ) {
     unlink($local_statusfile);
     &error ("Unable to retrieve remote command status on client \"$client\"");
   }
   
   unless ( open (LSTATUS,$local_statusfile) ) {
      unlink($local_statusfile);
      return 0;
   }
   chop($status=<LSTATUS>);
   close(LSTATUS);
   unlink($local_statusfile);

   if ($status ne "OK") {
     return $FALSE;
   }

   return $TRUE;
}

#------------------------------------------------------------------------------
#
#  execute a remote command without displaying remote errors
#  (error only if command can't be issued)
#
#  1: remote node
#  2: command
#
#------------------------------------------------------------------------------

sub remote_exec_ignore {
   my ($client, $command ) = @_ ;

   #
   # Execute  remote command
   #   
   
   print_debug($DEBUG_CMD, "$RSH -n $client \"$command\"");

   my $output="";
   open ( REMOTEXEC, " $RSH -n $client \"$command\" 2>&1|" )
     || error("Can not fork command: $RSH -n $client \"$command\" 2>$1") ;
   while (<REMOTEXEC>) {
      chop();
      print_debug($DEBUG_CMD, "  $_");
   }
   close (REMOTEXEC) || error ("Failure when executing: $RSH -n $client \"$command\" 2>&1 (remote node may be unreachable)");   
}

#------------------------------------------------------------------------------
#
#  execute a local command and display the output:
#  . if there is an error
#  . if tracing option is set
#
#  1: command
#
#------------------------------------------------------------------------------

sub exec_cmd {

  my ($command) = @_;
  
  print_debug($DEBUG_CMD, "$command");
  
  my @result = qx/$command 2>&1/;
  
  # in case of error, display output and exit
  my $status = $?;
  if ( $status ne 0 ) {
    print_log("-----------------------------------------------------------------------------\n");
    print_log("Error when executing: $command (status=$status)\n") ;
    my $line;
    foreach $line (@result)
      {
	print_log("  $line\n") ;
      }
    print_log("-----------------------------------------------------------------------------\n");
    exit 1;
  }
  
  # if it's OK, display if tracing is required

  my $line;
  foreach $line (@result)
    {
      print_debug($DEBUG_CMD, "  $line");
    }
  
  return 1;
}

#------------------------------------------------------------------------------
#
#  execute a local command and ignore the error:
#  if tracing option is set, outputs are displayed
#  1: command
#
#------------------------------------------------------------------------------

sub exec_cmd_ignore {
  my ($command) = @_;
  
  print_debug($DEBUG_CMD, "$command");
  my @result = qx/$command 2>&1/;
  
  # if it's OK, display if tracing is required
  
  my $line;
  foreach $line (@result)
    {
      print_debug($DEBUG_CMD, "  $line");
    }
  
  return 1;
}

#------------------------------------------------------------------------------
#
# remote_mount
#
# . dismount the mounting point (to prevent errors if it is already mounted)
# . mount the remote directory on the mounting point
#
# 1: node IP
# 2: exported directory to mount
# 3: mounting point
#
#------------------------------------------------------------------------------

sub remote_mount
  {
    my ($node_ip, $exported, $mountpoint) = @_ ;

    remote_exec_ignore($node_ip, "$RUMOUNT $mountpoint") ;
    remote_exec ($node_ip, "$RMKDIR -p $mountpoint ; $RMOUNT -o retry=0 -f nfs $SERVER_IP:$exported $mountpoint" ) || 
      error ( "Unable to NFS mount $exported" );
  }

#==============================================================================
#
#    SHARE
#
#==============================================================================

#------------------------------------------------------------------------------
#
#  share a directory on the installation server
#
#------------------------------------------------------------------------------

sub share {

  my ($directory) = @_;

  my @result;
  my @result_parent;
  my $i;
  my $found;
  my $status;
  my $line;
  my $command = "$SHARE -F nfs -o ro,anon=0 $directory";
  
  control_dir_is_readable($directory);
  
  
  print_debug($DEBUG_CMD, "$command");
  
  @result = qx/$command 2>&1/;
  
  # check if the error is because the parent directory is already shared
  # then check the share option of the parent (anon=0 must be set)
  if ( $? ne 0 ) {
    if ($result[0] =~ m/parent-directory \((.*)\) already shared/) {
      my $parent = $1;
      # retrieve the share information
      @result_parent = qx/$SHARE/;
      $found = 0;
      for ($i = 0 ; $i < scalar(@result_parent) ; $i++) {
	#look for the parent: 1 = directory name, 2 = option
	if ($result_parent[$i] 
	    =~ /^[^\s\t]*[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)/) {
	  if ($1 eq $parent) {
	    if ($2 !~ m/anon=0/) {
	      error("Parent directory $parent must be shared with the \"anon=0\" option");
	      exit 1;
	    } else {
	      $found = 1;
	    }
	  }
	}
      
      }
      if (! $found) {
	error("Can't retrieve information about $parent");
      }
    } else {
      # real error
      $status = $?;
      print_log("-----------------------------------------------------------------------------\n");
      print_log("Error when executing: $command (status=$status)\n") ;
      foreach $line (@result)
	{
	  print_log("  $line\n") ;
	}
      print_log("-----------------------------------------------------------------------------\n");
      exit 1;
    }
  }
  
  # if it's OK, display if tracing is required
  
  foreach $line (@result)
    {
      print_debug($DEBUG_CMD, "  $line");
    }
  
  return 1;
  
}

#==============================================================================
#
#    SLICE LIST HANDLING
#
#==============================================================================
#------------------------------------------------------------------------------
#
# look for a slice with a particular attribute concerning the current node
#
#------------------------------------------------------------------------------

sub slice_search {

  my ($nodeid, $attribute, $value, $ref_slice_idx) = @_ ;
  
  my $node_idx = getnode($nodeid);

  for my $slice_idx (@{$NODE_LIST[$node_idx]{slice}}) {
    if ($SLICE_LIST[$slice_idx]{$attribute} eq $value) {
      $$ref_slice_idx = $slice_idx;
      return $TRUE;
    }
  }
  
  return $FALSE;
}

#==============================================================================
#
#    DISK LIST HANDLING
#
#==============================================================================
#------------------------------------------------------------------------------
#
# look for a disk
#
#------------------------------------------------------------------------------

sub disk_search {

  my ($disk, $refidx) = @_;

  my $found = $FALSE;
  my $mirror = $FALSE;

  for (my $idx = 0 ; ($idx < scalar(@SHARED_DISK_LIST)) && ! $found ;
       $idx++) {
    if ($SHARED_DISK_LIST[$idx]{disk1} eq $disk) {
      $found = $TRUE;
      $$refidx = $idx;
    } elsif ($SHARED_DISK_LIST[$idx]{type} == $MIRROR_DISK) {
      if ($SHARED_DISK_LIST[$idx]{disk2} eq $disk) {
	$found = $TRUE;
	$$refidx = $idx;
      }
    }
  }
  
  return $found;
}

#==============================================================================
#
#    MOUNT/UMOUNT
#
#==============================================================================

#------------------------------------------------------------------------------
#
# mount required mounting points after a men boot
# the current status of these mounting points is unknown (after a recovery)
# they are unmounted (without taking into account errors) and re-mounted
#
#------------------------------------------------------------------------------

sub mount_men {

   my ($node_id) = @_;

   my $node_ip = get_external_ip($node_id);
   
    # ignore errors
   remote_exec_ignore($node_ip, "$RUMOUNT $SLICE_SUNWCGHA");

   if (! $CHECK_MOUNT_ERROR) {
     
     # this is the recovery
          
     remote_exec_ignore($node_ip, "$RMOUNT $SLICE_SUNWCGHA");
     
   } else {
     
     # initial case, errors are displayed
     
     remote_exec($node_ip, "$RMOUNT $SLICE_SUNWCGHA");
   }
 }

#------------------------------------------------------------------------------
#
# mount required mounting points
# the current status of these mounting points is unknown (after a recovery)
# they are unmounted (without taking into account errors) and re-mounted
#
#------------------------------------------------------------------------------

sub mount_for_diskless {

  my ($node_id) = @_;
  
  my $node_ip = get_external_ip($node_id);
  my $node_idx = getnode($node_id);
  
  my $i;
  my $exec_fct;
  
  foreach $i (@{$NODE_LIST[$node_idx]{slice}}) {
    
    if ($USE_SHARED_DISK) {
      # all the partitions located on shared disk are unmounted
      if (($SLICE_LIST[$i]{type} == $SHARED_SLICE) 
	  && ($SLICE_LIST[$i]{mounting_point} ne $SLICE_TAG_REPLICA)) {
	# ignore errors
	remote_exec_ignore($node_ip, "$RUMOUNT $SLICE_LIST[$i]{mounting_point}") ;
      }
      
    } else {
      
      # all the replicated slice are unmounted (ignore errors)
      if ($SLICE_LIST[$i]{bitmap} ne "") {
	# ignore errors
	remote_exec_ignore($node_ip, "$RUMOUNT $SLICE_LIST[$i]{mounting_point}") ;
      }
    }
  }
  
  # reboot has been partial so mounting point are not present
  
  # ignore errors when called from recovery mechanism (they may be still
  # used so error "busy" may occurs
  if (! $CHECK_MOUNT_ERROR) {
    
    $exec_fct = \&remote_exec_ignore;
    
  } else {
    
    $exec_fct = \&remote_exec;
  }
  
  # if LOGICAL_SLICE_SUPPORT is used, 
  # /export must be mounted on the physical because
  # of a smosservices bug (not needed when using DCSS)
  # so do the same for all the replicated slices as some of them can be
  # something like /export/root if the /export has been split
  
  foreach $i (@{$NODE_LIST[$node_idx]{slice}}) {

    if ($USE_SHARED_DISK) {
      if (($SLICE_LIST[$i]{type} == $SHARED_SLICE) 
	  && ($SLICE_LIST[$i]{mounting_point} ne $SLICE_TAG_REPLICA)) {
	my $devname = "/dev/dsk/$SLICE_LIST[$i]{name}";
	&$exec_fct($node_ip, "$RMOUNT $devname $SLICE_LIST[$i]{mounting_point}") ||
	  error ( "Unable to mount $devname on $SLICE_LIST[$i]{mounting_point}");
      }
      
    } else {
      
      if ($SLICE_LIST[$i]{bitmap} ne "") {
	# need to be mounted
	if (($USE_SVM) and (! $USE_DCSS)) {	  
	  my $devname = "/dev/dsk/$SLICE_LIST[$i]{name}";	  
	  &$exec_fct($node_ip, "$RMOUNT $devname $SLICE_LIST[$i]{mounting_point}") ||
	    error ( "Unable to mount $devname on $SLICE_LIST[$i]{mounting_point}");
	  
	} else { 
	  
	  &$exec_fct($node_ip, "$RMOUNT $SLICE_LIST[$i]{mounting_point}") ;
	  
	}
      }
    }
  }
}

#==============================================================================
#
#    REMOTE FILE HANDLING
#
#==============================================================================

#------------------------------------------------------------------------------
#
# test if a remote file exists
#
#------------------------------------------------------------------------------

sub exist_file {
  my ($node_ip, $remote_file) = @_;
  
  my $remote_backup_file = $remote_file . ".bak" ;
  my $result;
  
  my $cmd = "if [ -f $remote_file ] ; then echo OK ; else echo KO ; fi";
  print_debug($DEBUG_CMD, "rsh -n $node_ip \"$cmd\"");
  
  chomp($result = qx/rsh -n $node_ip \"$cmd\"/);
  print_debug($DEBUG_CMD, "  $result");
  
  if ($result eq "OK") {
    return $TRUE;
  }
  return $FALSE;
}

#------------------------------------------------------------------------------
#
# get a file from a remote host
#
#------------------------------------------------------------------------------

sub get_file {
    my ($node_ip, $remote_file, $local_file) = @_;

    exec_cmd("$RCP $node_ip:$remote_file $local_file") ||
      error("Can not get $node_ip:$remote_file > $local_file");

    if ($DEBUG_FILE) {
      print_debug($DEBUG_FILE, "---------------------------------------------");
      print_debug($DEBUG_FILE, "GET: CONTENT OF $node_ip:$remote_file");
      print_debug($DEBUG_FILE, "---------------------------------------------");
      system("cat $local_file");
      print_debug($DEBUG_FILE, "---------------------------------------------");
    }
}

#------------------------------------------------------------------------------
#
# copy a file to a remote host
#
#------------------------------------------------------------------------------

sub put_file {
    my ($node_ip, $local_file, $remote_file) = @_;

    if ($DEBUG_FILE) {
      print_debug($DEBUG_FILE, "---------------------------------------------");
      print_debug($DEBUG_FILE, "PUT: CONTENT OF $node_ip:$remote_file");
      print_debug($DEBUG_FILE, "---------------------------------------------");
      system("cat $local_file");
      print_debug($DEBUG_FILE, "---------------------------------------------");
    }

    exec_cmd("$RCP $local_file $node_ip:$remote_file ") ||
      error("Can not put $local_file > $node_ip:$remote_file");
}

#------------------------------------------------------------------------------
#
# transfer a file from a remote host to another remote host
#
#------------------------------------------------------------------------------

sub transfer_file {
    my ($node_ip_src, $node_ip_dst, $remote_file) = @_;

    my $tmp_file    = "$WORKING_DIR/trans.tmp";

    exec_cmd("$RCP $node_ip_src:$remote_file $tmp_file") ||
       error("Unable to copy $node_ip_src:$remote_file to $tmp_file");
    exec_cmd("$RCP $tmp_file $node_ip_dst:$remote_file") ||
       error("Unable to copy $tmp_file to $node_ip_dst:$remote_file");
}

#------------------------------------------------------------------------------
#
# create a backup file on the remote host
#
# store an indicator to be associated to memorize if the file
# exists
#
#------------------------------------------------------------------------------

sub save_file {

  my ($node_ip, $remote_file, $tag) = @_;
  
  my $result;
  my $remote_backup_file = $remote_file . ".bak" ;
  my $cmd = "if [ -f $remote_file ] ; then echo OK ; else echo KO ; fi";
  chomp($result = qx/rsh -n $node_ip \"$cmd\"/);
  
  if ($result eq "OK") {
    # file exists
    SEQUENCER::set_data($tag, "EXIST");
    remote_exec($node_ip, "$RCOPY $remote_file $remote_backup_file") ||
      error("Can not save $remote_file > $remote_backup_file on $node_ip");
    print_debug($DEBUG_FILE, "File $remote_file saved into $remote_backup_file on $node_ip");
  } else {
    SEQUENCER::set_data($tag, "NOT_EXIST");
    print_debug($DEBUG_FILE, "File $remote_file does not exist on $node_ip");
  }
}

#------------------------------------------------------------------------------
#
# restore a file from its backup on the remote host
#
# CAUTION !!!
#
# if the file didn't exist when saving, the new file
# is deleted to restore the previous state
#
#------------------------------------------------------------------------------

sub restore_file {
    my ($node_ip, $remote_file, $tag) = @_;

    my $remote_backup_file = $remote_file . ".bak" ;
    
    my $state = SEQUENCER::get_data($tag);
    if ($state eq "") {
      error("Restore_file: Unknown state for file $remote_file, tag $tag");
    }

    if ($state eq "EXIST") {
      remote_exec($node_ip, "$RCOPY $remote_backup_file $remote_file") ||
	error("Can not restore $remote_backup_file > $remote_file on $node_ip");
      print_debug($DEBUG_FILE, "File $remote_file restored from $remote_backup_file on $node_ip");
    } else {
      remote_exec($node_ip, "$RRM -f $remote_file") ||
      print_debug($DEBUG_FILE, "File $remote_file deleted when restoring on $node_ip");
    }
}

#------------------------------------------------------------------------------
#
# delete a backup copy on the remote host
#
# CAUTION !!!
#
# if the file didn't exist when saving, the new file
# is deleted to restore the previous state
#
#------------------------------------------------------------------------------

sub delete_backup_file {

  my ($node_ip, $remote_file, $tag) = @_;
  
  my $remote_backup_file = $remote_file . ".bak" ;
  
  my $state = SEQUENCER::get_data($tag);
  if ($state eq "") {
    error("Restore_file: Unknown state for file $remote_file, tag $tag");
  }
  
  remote_exec($node_ip, "$RRM -f $remote_backup_file");
  SEQUENCER::set_data($tag, "");
  print_debug($DEBUG_FILE, "File $remote_backup_file deleted after completion on $node_ip");
}


#------------------------------------------------------------------------------
#
# replacing a string by another in a file
#
#------------------------------------------------------------------------------

sub replace_in_file {

  my ($local_file, $old_string, $new_string) = @_;
  my $tmp_file = "$WORKING_DIR/replace.tmp";

  open(LOCAL_FILE, $local_file) 
    || COMMON::error("Unable to open $local_file");
  
  open(TMP_FILE,">$tmp_file") ||
    COMMON::error("Unable to create $tmp_file");
   
   while ( <LOCAL_FILE> ) {	
     chop();
     s/$old_string/$new_string/g;
     print TMP_FILE "$_\n";
   }
   close(LOCAL_FILE);
   close(TMP_FILE);

  exec_cmd("$CP $tmp_file $local_file");
  unlink($tmp_file);
}

#------------------------------------------------------------------------------
#
# replacing value of an entry by another value
# if the entry doesn't exist, it is added
#
#  for example, replace
#     ENTRY=VALUE
#  with
#     ENTRY=NEWVALUE
#
#------------------------------------------------------------------------------

sub replace_value_in_file {

   my ($file, $update_entry_input, $update_value_input ) = @_ ;
   my $count=0;
   my $update_entry;
   my $update_value;
   my $garbage;
     
   my $tmpfile = "$WORKING_DIR/replace.tmp";

   open (FILE, $file ) 
     || COMMON::error("Unable to open $file");
   open (FILE_TMP, ">$tmpfile" ) 
     || COMMON::error("Unable to open $tmpfile");
	
   while ( <FILE> ) {
     
     chop();
     ($update_entry, $update_value, $garbage) = split (/=/, $_) ;
     
     if ($update_entry eq $update_entry_input) {
       print FILE_TMP "${update_entry}=${update_value_input}\n";
       $count++;
     } else {
       print FILE_TMP "$_\n";
     }
     next;
     
   }
   close (FILE);
   close (FILE_TMP);
   
   if ($count <= 0) {
     open (FILE_TMP, ">>$tmpfile" ) 
       || COMMON::error("Unable to update $tmpfile");
     print FILE_TMP "$update_entry_input=$update_value_input\n";
     close (FILE_TMP);
   }
   
   COMMON::exec_cmd ( "$MV $tmpfile $file") ||
     COMMON::error ( "Unable to move $tmpfile $file");

   unlink($tmpfile);
}

#==============================================================================
#
#  MANAGEMENT OF NODE NAMES AND IP ADDRESSES
#
#==============================================================================

#------------------------------------------------------------------------------
#
# build the server name
#
#------------------------------------------------------------------------------

sub build_server_name {
 
  return "sar-c${CLUSTER}-n${SERVER_NODE}" ;
}

#------------------------------------------------------------------------------
#
# look for a node by its nodeid
#
#------------------------------------------------------------------------------

sub getnode
  {
    my ($node_id) = @_ ;

    for my $i (0..$#NODE_LIST) {
      if ($NODE_LIST[$i]{id} eq $node_id) {
	return $i;
      }
    }

    error("Internal error: getnode: $node_id can't be reached\n");
  }

#------------------------------------------------------------------------------
#
# build the node name depending on the node ID
#
#------------------------------------------------------------------------------

sub build_node_name {
 
  my ($nodeid, $interface) = @_ ;

  my $name = "";

  if ($nodeid eq $MASTER_ID) {
    $name = "master" ;
  } elsif ($nodeid eq $VICEMASTER_ID) {
    $name = "vice-master" ;
  } else {
    my $hostname = $NODE_LIST[getnode($nodeid)]{name};
    
    if (($hostname ne "") && ($hostname ne "-")) {
      $name = $hostname ;
    } else {
      if (($nodeid eq $NODE_LIST[$MEN1]{id}) || ($nodeid eq $NODE_LIST[$MEN2]{id})) {
	$name = "MEN-C${CLUSTER}-N${nodeid}" ;
      } else {
	$name = "NMEN-C${CLUSTER}-N${nodeid}" ; 
      }
    }
  }

  my $post = "" ;
  if ($interface eq "") {
    $post = "" ;
  }
  if ($interface eq "NIC0") {
    $post = $POST_NIC0 ;
  }
  if ($interface eq "NIC1") {
    $post = $POST_NIC1 ;
  }
  if ($interface eq "CGTP") {
    $post = $POST_CGTP ;
  }

  $name = $name . $post ;

  return $name ;
}

#------------------------------------------------------------------------------
#
# build the cluster IP address depending on the node ID
#
#------------------------------------------------------------------------------

sub build_node_ip {
 
  my ($node_id, $interface) = @_ ;

  # default is NIC0 (for external IP)

  if ($node_id == $MASTER_ID) {
    if ($interface eq "NIC0") {
      return computeIp($MASTER_ID, 
		       $CLUSTER_BIT_NIC0_NET, 
		       $CLUSTER_BIT_NETMASK);
    }
    if ($interface eq "NIC1") {
      return computeIp($MASTER_ID, 
		       $CLUSTER_BIT_NIC1_NET, 
		       $CLUSTER_BIT_NETMASK);
    }
    if ($interface eq "CGTP") {
      return computeIp($MASTER_ID, 
		       $CLUSTER_BIT_CGTP_NET, 
		       $CLUSTER_BIT_NETMASK);
    }
    return computeIp($MASTER_ID, 
		     $CLUSTER_BIT_NIC0_NET, 
		     $CLUSTER_BIT_NETMASK);
  }

  my $idx=getnode($node_id);

  if ($interface eq "NIC0") {
    return $NODE_LIST[$idx]{nic0Ip};
  }
  if ($interface eq "NIC1") {
    return $NODE_LIST[$idx]{nic1Ip};
  }
  if ($interface eq "CGTP") {
    return $NODE_LIST[$idx]{cgtpIp};
  }

  return $NODE_LIST[$idx]{nic0Ip};

}

#------------------------------------------------------------------------------
#
# get the external IP address of a MEN node
#
#------------------------------------------------------------------------------

sub get_external_ip {

  my ($node_id) = @_;

  my $idx = getnode($node_id);
  if ($NODE_LIST[$idx]{pubConfigured}) {
    return $NODE_LIST[$idx]{pubIp};
  } else {
    return build_node_ip($node_id, "");
  }

}

#------------------------------------------------------------------------------
#
# get the external name of a MEN node
#
#------------------------------------------------------------------------------

sub get_external_name {

  my ($node_id) = @_;

  my $idx = getnode($node_id);
  if ($NODE_LIST[$idx]{pubConfigured}) {
    return $NODE_LIST[$idx]{pubName};
  } else {
    return build_node_name($node_id, "");
  }

}

#------------------------------------------------------------------------------
#
# build node name for install
#
#------------------------------------------------------------------------------

sub build_node_name_for_install {

  my ($node_id) = @_;

  my $node = getnode($node_id);

  if ($NODE_LIST[$node]{pubConfigured}) {

    # the public name can be used
    return get_external_name($node_id);

  } else {

    # if we do not explicitely used the nic0 name,
    # an invalid entry is created into /etc/hosts file of the installed
    # host (that will stay as this file is not re-created) :
    # IP address for nic0 associated with the node name not postfixed
    # we do not update the function get_external_name because
    # it is used to display messages and for other stuff

    return build_node_name($node_id, "NIC0");
  }
}

#==============================================================================
#
#  MANAGEMENT OF NETMASKS FILE
#
#==============================================================================

#------------------------------------------------------------------------------
#
# updating a netmask file
#
#------------------------------------------------------------------------------

sub add_netmask {
  my ( $netmasks_file, $network, $mask ) = @_ ;
  my $count=0;
  my $read_network;
  my $read_mask;
  
  $count = 0;   
  if ( -f "$netmasks_file" ) {
    open (NETMASKS, $netmasks_file ) 
      || COMMON::error("Unable to open $netmasks_file");
    
    while ( <NETMASKS> ) {      
      chop();
      s/#.*//;
      ($read_network, $read_mask) = split (/[\s\t]+/, $_) ;
      next unless (defined($read_network));
      next unless (defined($read_mask));
      
      if ("$read_network" eq "$network") {
	if ("$read_mask" ne "$mask") {
	  COMMON::error("Ip \"$network\" does not match \"$mask\" as first hostname in $netmasks_file") 
	}
	$count++;
	last;
      }
    }
    close (NETMASKS);
  }

  if ( $count == 0) {
    open (NETMASKS, ">>$netmasks_file" ) || COMMON::error("Unable to update $netmasks_file");
    print  NETMASKS "$network $mask\n";
    close (NETMASKS);
  }   
}
#==============================================================================
#
#  MANAGEMENT OF HOSTS FILES
#
#==============================================================================

#------------------------------------------------------------------------------
#
# updating an host entry
#
# . look for the entry based on the IP address
# . when found, add each alias that are not yet defined and keep the already
#   alias and entries
#
#------------------------------------------------------------------------------

sub update_hosts_entry {

  my ($hosts_file, $host_ip, $alias_list, $ipversion) = @_;

  # this IP address doesn't match the IP version which the file is
  # being created for
  if (ip_version($host_ip) != $ipversion) {
    return ;
  }

  my $found_host = $FALSE ;
  my $found_alias = $FALSE;
  my $line;

  #
  # look for the entry
  #

  open(HOSTS, "<$hosts_file");
  open(HOSTS_TMP, ">$hosts_file.tmp");
  while (<HOSTS>) {
    chomp();
    s/#.*//;

    # is it a valid host entry ?
    $line = $_;
    if ($line =~ m/([0-9]*.[0-9]*.[0-9]*.[0-9]*)[\s\t]*(.*)/) {

      # $1: IP addresse
      # $2: aliases list

      # look for the searched one
      if ($1 eq $host_ip) {
	$found_host = $TRUE;

	# check for all required aliases and add the missing ones

	foreach my $expected_alias (split(/[\s\t]+/,$alias_list)) {
	  $found_alias = $FALSE;

	  foreach my $current_alias (split(/[\s\t]+/,$2)) {
	    if ($expected_alias eq $current_alias) {
	      $found_alias = $TRUE;
	    }
	  }

	  # add the alias when not found
	  if (! $found_alias) {
	    $line .= " $expected_alias";
	  }
	}
      }
      # copy the host entry (the same entry or the update one)
      print HOSTS_TMP "$line\n";
    }
  }

  if (! $found_host) {
    # add the entry
    print HOSTS_TMP "$host_ip $alias_list\n";
  }

  close (HOSTS);
  close (HOSTS_TMP);

  # move the temporary file on the final one
  exec_cmd ("$MV $hosts_file.tmp $hosts_file");
  }

#------------------------------------------------------------------------------
#
#  add alias
#
#  add an alias in the string if it's not already defined
#
#------------------------------------------------------------------------------

sub add_alias {

  my ($alias_ref, $value) = @_;

  my $elem;
  foreach $elem (split(' ', $$alias_ref)) {
    if ($elem eq $value) {
      return ;
    }
  }

  $$alias_ref = join " ", $$alias_ref, $value;
}

#------------------------------------------------------------------------------
#
#  add one entry in the host file
#
#------------------------------------------------------------------------------

sub add_hosts_file_entry {

  my ($hosts_file, $node, $current_node_id, $ipversion) = @_;
    
  my $node_id = $NODE_LIST[$node]{id};

  my $loghost= "loghost" ;
  my $alias_list = "";
    
  my $node_name = build_node_name($node_id, "") ;

  my $node_ip_if0;
  my $node_ip_if1;
  my $node_ip_cgtp0;
      
  my $node_name_if0;   
  my $node_name_if1;
  my $node_name_cgtp0;

  $node_ip_if0   = COMMON::build_node_ip($node_id, "NIC0");
  $node_name_if0 = build_node_name($node_id, "NIC0") ;

  if ($USE_CGTP) {
    $node_ip_if1   = COMMON::build_node_ip($node_id, "NIC1");
    $node_ip_cgtp0 = COMMON::build_node_ip($node_id, "CGTP");
    $node_name_if1 = build_node_name($node_id, "NIC1") ;
    $node_name_cgtp0= build_node_name($node_id, "CGTP") ;
  }


  #
  # for diskless, an alias with the generic name (without postfix) 
  # must be created for the nic0 interface (must be the first)
  # to avoid a failure while smdiskless is executing
  #

  my $domain = SEQUENCER::get_data("DOMAIN");

  # add a non-postfixed entry if none will be created
  # (a non-postfixed is required for smdiskless as the hostname is passed
  # as an argument to create the /export/root/<hostname>)

  my $add_node_name = $FALSE;
  if ($USE_CGTP) {
    if (($POST_NIC0 ne "") && ($POST_NIC1 ne "") && ($POST_CGTP ne "")) {
      $add_node_name = $TRUE;
    }
  } else {
    if ($POST_NIC0 ne "") {
      $add_node_name = $TRUE;
    }
  }
  
  if ($node_id eq $current_node_id) {
    
    $alias_list = "";

    if ($add_node_name) {
      add_alias(\$alias_list, "$node_name");
      add_alias(\$alias_list, "$node_name.$domain");
    }

    add_alias(\$alias_list, "$node_name_if0");
    add_alias(\$alias_list, "$node_name_if0.$domain") ;
    if ($NODE_LIST[$node]{type} == $DISKLESS_NODE) {
      # add "loghost" for diskless node because /etc/hosts created
      # for MEN, it's already defined because /etc/hosts only updated
      add_alias(\$alias_list, "loghost") ;
    }
    
    update_hosts_entry($hosts_file,
		       $node_ip_if0,
		       $alias_list,
		       $ipversion);
    
    if ($USE_CGTP) {
      $alias_list = "";
      add_alias(\$alias_list, "$node_name_if1");
      add_alias(\$alias_list, "$node_name_if1.$domain") ;
      update_hosts_entry($hosts_file,
			 $node_ip_if1,
			 $alias_list, 
			 $ipversion);
      
      $alias_list = "";
      add_alias(\$alias_list, "$node_name_cgtp0");
      add_alias(\$alias_list, "$node_name_cgtp0.$domain");
      update_hosts_entry($hosts_file,
			 $node_ip_cgtp0,
			 $alias_list,
			 $ipversion);
    }
  } else {
    
    $alias_list = "";

    if ($add_node_name) {
      # same reason as above
      add_alias(\$alias_list, "$node_name");
    }

    add_alias(\$alias_list, "$node_name_if0");
    update_hosts_entry($hosts_file,
		       $node_ip_if0,
		       $alias_list,
		       $ipversion);
    
    if ($USE_CGTP) {
      $alias_list = "$node_name_if1" ;
      update_hosts_entry($hosts_file,
			 $node_ip_if1,
			 $alias_list,
			 $ipversion);
      $alias_list = "$node_name_cgtp0" ;
      update_hosts_entry($hosts_file,
			 $node_ip_cgtp0,
			 $alias_list,
			 $ipversion);
    }
  }

  if ($NODE_LIST[$node]{pubConfigured}) {
    $alias_list = "";
    add_alias(\$alias_list, "$NODE_LIST[$node]{pubName}");
    update_hosts_entry($hosts_file,
		       $NODE_LIST[$node]{pubIp},
		       $alias_list,
		       $ipversion);
  }
}

#------------------------------------------------------------------------------
#
#  update the local hosts file before transferring (for MEN)
#
#  note that the update depends on the IP version
#
#------------------------------------------------------------------------------

sub update_hosts_file {
  
  my ($hosts_file, $current_node_id, $ipversion) = @_ ;
  
  my $master_name_if0;
  my $master_name_if1;
  my $master_name_cgtp0;
  my $master_ip_if0;
  my $master_ip_if1;
  my $master_ip_cgtp0;
  
  $master_name_if0 = build_node_name($MASTER_ID, "NIC0");
  $master_ip_if0 = build_node_ip($MASTER_ID, "NIC0");

  if ($USE_CGTP) {
    $master_name_if1 = build_node_name($MASTER_ID, "NIC1");
    $master_name_cgtp0 = build_node_name($MASTER_ID, "CGTP");
    $master_ip_if1 = build_node_ip($MASTER_ID, "NIC1");
    $master_ip_cgtp0 =  build_node_ip($MASTER_ID, "CGTP");
  }
    
    for my $node (0..$#NODE_LIST) {
      add_hosts_file_entry($hosts_file,
			   $node,
			   $current_node_id,
			   $ipversion);
    }
    
    update_hosts_entry($hosts_file,
		       $master_ip_if0,
		       $master_name_if0, 
		       $ipversion) ;
    if ($USE_CGTP) {
      update_hosts_entry($hosts_file,
			 $master_ip_if1,
			 $master_name_if1, 
			 $ipversion) ;
      update_hosts_entry($hosts_file,
			 $master_ip_cgtp0, 
			 $master_name_cgtp0,
			 $ipversion);
    }

  #
  # create an entry for all external access
  #
  for (my $idx = 0 ; $idx < scalar(@EXTERNAL_ACCESS_LIST) ; $idx++) {
      # remove the netmask size if present
      my $ip = $EXTERNAL_ACCESS_LIST[$idx]{IP};
      my $pos = index($ip, "/");
      if ($pos > -1) {
	$ip = substr($ip, 0, $pos);
      }
      update_hosts_entry($hosts_file,
			 $ip,
			 $EXTERNAL_ACCESS_LIST[$idx]{hostname},
			 $ipversion);
      
      if ($EXTERNAL_ACCESS_LIST[$idx]{IPv6} ne "") {
	$ip = $EXTERNAL_ACCESS_LIST[$idx]{IPv6};
	$pos = index($ip, "/");
	if ($pos > -1) {
	  $ip = substr($ip, 0, $pos);
	}
	update_hosts_entry($hosts_file,
			   $ip,
			   $EXTERNAL_ACCESS_LIST[$idx]{hostname6},
			   $ipversion);
      }
    }  
}

#------------------------------------------------------------------------------
#
#  create the local hosts file before transferring (for diskless)
#
#------------------------------------------------------------------------------

sub create_hosts_file {
  
  my ($hosts_file, $current_node_id) = @_ ;
  
  open (HOSTS, ">$hosts_file" ) || &error("Unable to open $hosts_file");
  print HOSTS "127.0.0.1 localhost\n" ;
  close (HOSTS);
  update_hosts_file($hosts_file, $current_node_id, $IP_VERSION_V4);
}

#------------------------------------------------------------------------------
#
#  create the local hosts file before transferring (for diskless)
#
#------------------------------------------------------------------------------

sub create_ipnodes_file {
    
  my ($hosts_file, $current_node_id) = @_ ;
  
  open (HOSTS, ">$hosts_file" ) || &error("Unable to open $hosts_file");
  print HOSTS "::1       localhost\n" ;
  print HOSTS "127.0.0.1 localhost\n" ;
  close (HOSTS);
    update_hosts_file($hosts_file, $current_node_id, $IP_VERSION_V6);
}

#==============================================================================
#
#  HOSTNAME FILE MANAGEMENT
#
#==============================================================================

#------------------------------------------------------------------------------
#
# determine if the physical network interface supports IPMP groups
#
# return the "group" argument if it is the case
#
#------------------------------------------------------------------------------

sub ipmp_group_to_define {

  my ($interface, $refgroupcmd) = @_;

  $$refgroupcmd = "";

  my $phys;
  my $alias;
  my $ipmp = $FALSE;

  # if it is not a physical interface, no need to check
  # as the "group" concerns only the physical interface
  if (! COMMON::is_physical_interface($interface, \$phys, \$alias)) {
    return $FALSE;
  }

  # determine if the interface is used to support an IPMP group
  for (my $idx = 0; $idx < scalar(@EXTERNAL_ACCESS_LIST); $idx++) {
    if ($EXTERNAL_ACCESS_LIST[$idx]{type} == $IPMP_EXTERNAL_ACCESS) {

      # are the interfaces on the same physical interface ?

      my $ipmp_phys;
      COMMON::is_physical_interface($EXTERNAL_ACCESS_LIST[$idx]{nic0}, 
				    \$ipmp_phys, \$alias);
      if ($phys eq $ipmp_phys) {
	$ipmp = $TRUE;
      } elsif ($EXTERNAL_ACCESS_LIST[$idx]{nic1} ne "") {
	COMMON::is_physical_interface($EXTERNAL_ACCESS_LIST[$idx]{nic1},
			     \$ipmp_phys, \$alias);
	if ($phys eq $ipmp_phys) {
	  $ipmp = $TRUE;
	}
      }

      # the argument is built here as they may have several groups
      # on this interface
      if ($ipmp) {
	$$refgroupcmd .= "group $EXTERNAL_ACCESS_LIST[$idx]{group}";
      }
    }
  }

  return $ipmp;
}

#------------------------------------------------------------------------------
#
#  reset hostname file indicator 
#
#------------------------------------------------------------------------------

sub reset_hostname {

  %HOSTNAME_ARRAY = ();
}

#------------------------------------------------------------------------------
#
#  test if an interface exists in the hostname array
#
#------------------------------------------------------------------------------

sub is_hostname_defined {

  my ($interface, $ip_version) = @_;

  my $index = "$interface/$ip_version";

  if (defined($HOSTNAME_ARRAY{$index})) {
    return $TRUE;
  }

  return $FALSE;
}

#------------------------------------------------------------------------------
#
#  register an interface in the hostname array
#
#------------------------------------------------------------------------------

sub register_hostname {

  my ($interface, $ip_version) = @_;

  my $index = "$interface/$ip_version";
  $HOSTNAME_ARRAY{$index} = $TRUE;
}

#------------------------------------------------------------------------------
#
#  create hostname file
#
#------------------------------------------------------------------------------

sub create_hostname_file {

  my ($node_id, $interface, $ip, $hostname, $option) = @_;
  
  my $node_idx = getnode($node_id);

  my $existing_hostname = $FALSE;
  my $local_file = "$WORKING_DIR/hostname.tmp";
  
  # the file name depends on the IP version
  my $remote_file;
  my $ip_version = COMMON::ip_version($ip);
  
  my $external_ip  = COMMON::get_external_ip($node_id);

  if ($ip_version == $IP_VERSION_V4) {
    $remote_file = "/etc/hostname.$interface";
  } else {
    $remote_file = "/etc/hostname6.$interface";
  }
  
  # IMPORTANT:
  # the hostname.<itf> for IPv4 are processed by /etc/rcS.d/S30network.sh
  # which makes some assumptions that disturbs the configuration
  # - if the hostname file contains one line, it adds automatically
  #   "netmask + broadcast + up"
  # - if the hostname file contains additional lines (with addif),
  #   it processes all the lines without any change
  #
  # CODING RULES FOR THE CALLER OF CREATE_HOSTNAME_FILE
  # - call the function for the interface to be set down at the end
  #
  #     the function will add an additional line containing
  #     "down" if the file wasn't existing just to get more than one line.
  #     Note that several tests have been performed: only down is considered as
  #     an additional line !
  #    . the first to make the interface down at its creation
  #    . the second "down" to create a second line preventing the script
  #      to add "up"
  #     

  # check if the file has already been created: don't take into
  # account pre-installed file, just the ones created by nhinstall
  
  if (is_hostname_defined($interface, $ip_version)) {
    
    $existing_hostname = $TRUE;
    
    COMMON::get_file($external_ip, $remote_file, $local_file);
    open(FH, ">>$local_file") ||
      COMMON::error ( "Unable to update $local_file" );
    
  } else { 
    
    open(FH, ">$local_file") ||
      COMMON::error ( "Unable to create $local_file" );
    
    register_hostname($interface, $ip_version);
  }
  
  my $group_cmd = "";
  my $identifier;

  if ($hostname ne "") {
    $identifier = $hostname;
  } else {
    $identifier = $ip;
  }
  
  if ($existing_hostname) {
    # the file already contains a definition, add a new one
    print FH "addif "
  }
  
  # define the host id
  print FH "$identifier";
  
  # if it's a physical interface supporting group, define them
  # except if hostname file already exists: ipmp group already 
  # defined on hostname file creation
  
  my $add_group_cmd = $FALSE;
  if ($NODE_LIST[$node_idx]{type} == $MEN_NODE) {
    if (ipmp_group_to_define($interface, \$group_cmd)) {
      if (! $existing_hostname) {
	$add_group_cmd = $TRUE;
      }
    }
  }
  
  # if it's an interface with group, set all the required flag
  # depending of its type: test address or floating address
  
  if ($ip_version == $IP_VERSION_V4) {
    print FH " netmask + broadcast +";
  }
  
  if ($option & $OPTION_FAILOVER) {
    print FH " failover";
  } else {
    print FH " -failover";
  }
  
  if (($ip_version == $IP_VERSION_V4) && ($option & $OPTION_DEPRECATED)) {
    # not used on IPv6
    print FH " deprecated";
  }
  
  if ($add_group_cmd) {
    print FH " $group_cmd";
  }
  
  if ($option & $OPTION_DOWN) {
    if ($existing_hostname) {
      print FH " down\n";
    } else {
      print FH " down\n";
      # just to be sure that there is at least two line
      # in order to make the "down" considered (see comment)
      print FH "down\n";
    }

  } else {
    print FH " up\n"
  }

  close(FH);

  COMMON::put_file($external_ip, $local_file, $remote_file);

  unlink($local_file);
}

#------------------------------------------------------------------------------
#
#  create hostname6 file
#
#------------------------------------------------------------------------------

sub create_hostname6_file {

  my ($node_id, $interface) = @_;

  my $index = "$interface/$IP_VERSION_V6";

  # check if the file has already been created: don't take into
  # account pre-installed file, just the ones created by nhinstall
  
  if (! is_hostname_defined($interface, $IP_VERSION_V6)) {
    my $external_ip  = COMMON::get_external_ip($node_id);    
    my $remote_file = "/etc/hostname6.$interface";
    COMMON::remote_exec($external_ip, "touch $remote_file");
    
    register_hostname($interface, $IP_VERSION_V6);
  }
}
#------------------------------------------------------------------------------
#
# Configure the intra-cluster network interface 
#
#------------------------------------------------------------------------------

sub create_internal_interface_hostname {

  my ($node_id, $nic, $interface) = @_;

  my $ip           = COMMON::build_node_ip($node_id, $nic);
  my $hostname = COMMON::build_node_name($node_id, $nic);

  create_hostname_file($node_id, $interface, $ip, $hostname, $OPTION_NONE);
}

#------------------------------------------------------------------------------
#
# Create the hostname file for a public interface
#
#------------------------------------------------------------------------------

sub create_external_interface_hostname {

  my ($node_id) = @_;

  my $node         = COMMON::getnode($node_id);

  if (! $NODE_LIST[$node]{pubConfigured}) {
    return ;
  }

  my $hostname  = $NODE_LIST[$node]{pubName};
  my $interface = $NODE_LIST[$node]{pubNic};
  my $ip        = $NODE_LIST[$node]{pubIp};

  create_hostname_file($node_id, $interface, $ip, $hostname, $OPTION_NONE);

  if ($IPV6_PUBLIC_NETWORK_ENABLE) {
    create_hostname6_file($node_id, $interface) ;
  }
}

#------------------------------------------------------------------------------
#
# Create the hostname file for the floating address
#
#------------------------------------------------------------------------------

sub create_floating_address_hostname {
  
  my ($node_id, $ip_version) = @_;
  
  # the floating address is created on the first interface
  for (my $idx = 0 ; $idx < scalar(@EXTERNAL_ACCESS_LIST) ; $idx++) {
    if ($ip_version == $IP_VERSION_V4) {
      if ($EXTERNAL_ACCESS_LIST[$idx]{type} == $IPMP_EXTERNAL_ACCESS) {
	create_hostname_file($node_id,
			     $EXTERNAL_ACCESS_LIST[$idx]{nic0},
			     $EXTERNAL_ACCESS_LIST[$idx]{IP},
			     $EXTERNAL_ACCESS_LIST[$idx]{hostname},
			     $OPTION_FAILOVER + $OPTION_DOWN
			    );
      } else {
	create_hostname_file($node_id,
			     $EXTERNAL_ACCESS_LIST[$idx]{nic0},
			     $EXTERNAL_ACCESS_LIST[$idx]{IP},
			     $EXTERNAL_ACCESS_LIST[$idx]{hostname},
			     $OPTION_DOWN
			    );
      }
    } else {
      if ($EXTERNAL_ACCESS_LIST[$idx]{IPv6} ne "") {
	create_hostname_file($node_id,
			     $EXTERNAL_ACCESS_LIST[$idx]{nic0},
			     $EXTERNAL_ACCESS_LIST[$idx]{IPv6},
			     $EXTERNAL_ACCESS_LIST[$idx]{hostname6},
			     $OPTION_FAILOVER + $OPTION_DOWN
			    );
      }
    }
  }
}

#------------------------------------------------------------------------------
#
# Create the hostname file for the IPMP (test IP).
#
#------------------------------------------------------------------------------

sub create_IPMP_test_hostname {

  my ($node_id, $ip_version) = @_;

  for (my $idx = 0 ; $idx < scalar(@EXTERNAL_ACCESS_LIST) ; $idx++) {
    
    if ($EXTERNAL_ACCESS_LIST[$idx]{type} == $IPMP_EXTERNAL_ACCESS) {
      
      if ($ip_version == $IP_VERSION_V4) {
	
	# IPv4
	
	my $testIp;
	if ($node_id == $MEN1_ID) {
	  $testIp = $EXTERNAL_ACCESS_LIST[$idx]{men1TestIp0};
	} else {
	  $testIp = $EXTERNAL_ACCESS_LIST[$idx]{men2TestIp0};
	}
	
	# the first test nic is used also to create the floating address
	create_hostname_file($node_id,
			     $EXTERNAL_ACCESS_LIST[$idx]{nic0},
			     $testIp,
			     "",
			     $OPTION_DEPRECATED
			    );
	
	if ($EXTERNAL_ACCESS_LIST[$idx]{nic1} ne "") {
	  
	  if ($node_id == $MEN1_ID) {
	    $testIp = $EXTERNAL_ACCESS_LIST[$idx]{men1TestIp1};
	  } else {
	    $testIp = $EXTERNAL_ACCESS_LIST[$idx]{men2TestIp1};
	  }
	  
	  create_hostname_file($node_id,
			       $EXTERNAL_ACCESS_LIST[$idx]{nic1},
			       $testIp,
			       "",
			       $OPTION_DEPRECATED
			      );
	}
	
      } else {
      
	# IPv6 
	if ($EXTERNAL_ACCESS_LIST[$idx]{IPv6} ne "") {
	  create_hostname_file($node_id,
			       $EXTERNAL_ACCESS_LIST[$idx]{nic0},
			       "",
			       "",
			       $OPTION_DEPRECATED
			      );
	  if ($EXTERNAL_ACCESS_LIST[$idx]{nic1} ne "") {
	    create_hostname_file($node_id,
				 $EXTERNAL_ACCESS_LIST[$idx]{nic1},
				 "",
				 "",
				 $OPTION_DEPRECATED
				);
	  } 
	}
      }
    }
  }
}

#==============================================================================
#
#  IP address handling
#
#==============================================================================

#------------------------------------------------------------------------------
#
#  ip_version
#
#    return the version of the IP address
#    (the IP address must be valid)
#
#------------------------------------------------------------------------------

sub ip_version {

  my ($ip) = @_;

  if ($ip =~ m/^([0-9]*).([0-9]*).([0-9]*).([0-9]*)$/) {
    return $IP_VERSION_V4;
  }

  return $IP_VERSION_V6;
}

#------------------------------------------------------------------------------
#
#  convertDottedToBitmask
#
#    convert a dotted form to a bit mask
#
#------------------------------------------------------------------------------

sub convertDottedToBitmask {

  my ($dotted) = @_;

  my $mask = 0;
  my $value;
  foreach $value (split('\.', $dotted)) {
    $mask = $mask * 256 + $value;
  }

  return $mask;
}

#------------------------------------------------------------------------------
#
#  convertBitmaskToDotted
#
#    convert a bit mask to a dotted form
#
#------------------------------------------------------------------------------

sub convertBitmaskToDotted {

  my ($bitmask) = @_;

  my $dotted = "";
  for (my $i=0; $i < 4; $i++) {
    if ($i == 0) {
      $dotted = sprintf("%d", $bitmask % 256);
    } else {
      $dotted = sprintf("%d.%s", $bitmask % 256, $dotted)
    }
    $bitmask = $bitmask / 256;
  }

  return $dotted;
}

#------------------------------------------------------------------------------
#
#  computeIp
#
#    compute the IP address from:
#    . the node id
#    . the subnetwork (bit mask)
#    . the netmask (bit mask)
#
#------------------------------------------------------------------------------

sub computeIp {

  my ($nodeId, $subnet, $netmask) = @_;

  my $ip = convertBitmaskToDotted(($subnet & $netmask) + $nodeId);
  
  return $ip;
}

#------------------------------------------------------------------------------
#
#  computeBroadcast
#
#    compute the default broadcast address from:
#    . the subnetwork (bit mask)
#    . the netmask (bit mask)
#
#------------------------------------------------------------------------------

sub computeBroadcast {

  my ($subnet, $netmask) = @_;

  my $hostid = 0xffffffff ^ $netmask;

  my $ip = convertBitmaskToDotted(($subnet & $netmask) + $hostid);
  
  return $ip;
}

#------------------------------------------------------------------------------
#
#  compute the post-fix file name for DHCP
# 
#------------------------------------------------------------------------------

sub getDhcpPostfix {
  my ($dotted_network) = @_;
  
  my $postfix = "";
  foreach my $byte (split('\.',$dotted_network)) {
    $postfix = "${postfix}_${byte}";
  }

  return $postfix;
}

#------------------------------------------------------------------------------
#
# build Client ID
#
#------------------------------------------------------------------------------

sub build_cid {

  my ($type, $ref) = @_ ;

  my $cid;

  if ($type eq "MAC") {

    # build the Client ID from the MAC address
    # just remove the ":" and set each value on
    # two digits
    $cid = "01" ;
    foreach my $string (split(':',$ref)) { 
      if (length($string) == 0) {
	$cid .= "00" ;
      } elsif (length($string) == 1) {
	$cid .= "0$string" ;
      } else {
	$cid .= $string;
      }
    }

    # be sure that letters are uppercase
    $cid =~ tr/a-z/A-Z/;

  } elsif ($type eq "CLIENT_ID") {

    # convert string into hexa
    # has been checked at config time, no risk of error
    my $reason; # not used
    string_hexa($ref, \$cid, \$reason);

    # be sure that letters are uppercase
    $cid =~ tr/a-z/A-Z/;

  } else {
    COMMON::error("Internal: $type as type when calling build_cid");
  }

  return $cid;
}

#==============================================================================
#
#  Environment variable handling
#
#==============================================================================

#------------------------------------------------------------------------------
#
# var_defined
#
#    read a environment variable, set a default value if optional
#
#    1: name of the variable
#    2: default value to return if it does not exist)
#    3: output variable
#
#    Note: a empty variable is considered as an undefined one
#
#    return $TRUE if the variable is defined and not empty
#
#------------------------------------------------------------------------------

sub var_defined {

  my ($name, $default, $refvar) = @_;

  if (! defined($ENV{$name})) {
    # not defined: return default
    $$refvar = $default;
    return $FALSE;
  } else  {
    $$refvar = $ENV{$name};
    if ($$refvar eq "") {
      # empty: return default
      $$refvar = $default;
      return $FALSE;
    } else {
      # defined
      return $TRUE;
    }
  }
}

#==============================================================================
#
#  /etc/vfstab handling
#
#==============================================================================

#------------------------------------------------------------------------------
#
# update mounting point in /etc/vfstab 
#
#  add a mounting point in vfstab
#
#------------------------------------------------------------------------------

sub update_vfstab_nfs {
  my ( $vfstab_file, $update_nfs_input, $update_nfs_value_input ) = @_ ;
  my $update_nfs;
  my $update_nfs_value;
  my $garbage;
  my $count = 0;
  my $line ;
  
  open (VFSTAB, $vfstab_file ) || COMMON::error("Unable to open $vfstab_file");
  open (VFSTAB_TMP, ">${vfstab_file}.tmp" ) || COMMON::error("Unable to open ${vfstab_file}.tmp");
  
  while ( <VFSTAB> ) {
    
    chop();
    $line = $_ ;
    s/#.*//;
    ($update_nfs, $garbage) = split (/\t/, $_) ;
    
    if (defined($update_nfs) && ( "$update_nfs" eq "$update_nfs_input" )) {
      print VFSTAB_TMP "$update_nfs\t$update_nfs_value_input\n";
      $count++;
    } else {
      print VFSTAB_TMP "$line\n";
    }
    next;       	        
  }
  close (VFSTAB);
  close (VFSTAB_TMP);
  
  unless ($count) {
    open (VFSTAB_TMP, ">>${vfstab_file}.tmp" ) || COMMON::error("Unable to update $vfstab_file");
    print VFSTAB_TMP "$update_nfs_input\t$update_nfs_value_input\n";
    close (VFSTAB_TMP);
  }
   
  COMMON::exec_cmd ( "$MV ${vfstab_file}.tmp ${vfstab_file}") ||
    COMMON::error ( "Unable to move ${vfstab_file}.tmp ${vfstab_file}");  
}

#==============================================================================
#
#  Numeric and hexa functions
#
#==============================================================================

#------------------------------------------------------------------------------
#
# check a hexadecimal value and its number of digits 
# (digits=0 means no control)
#
#------------------------------------------------------------------------------

sub is_hexa {

  my ($refvalue, $digits, $refreason) = @_;

  if (($digits > 0 ) && (length($$refvalue) > $digits)) {
    $$refreason = "$$refvalue: too much digits (maximum $digits)";
    return $FALSE;
  }

  if ($$refvalue !~ m/^[a-fA-F0-9]+$/) {
    $$refreason = "$$refvalue: invalid hexadecimal value";
    return $FALSE;
  }

  return $TRUE;
}

#------------------------------------------------------------------------------
#
#  encode
#
#  encode the string into hexa and return the encoded value
#
#------------------------------------------------------------------------------

sub encode {

  my ($string) = @_;
  my $encoded = "";
  my $hexa;

  foreach my $c (split(//,$string)) {
    $hexa = sprintf("%2.2x", ord($c));
    $encoded = "$encoded$hexa";
  }

  return $encoded;
}

#------------------------------------------------------------------------------
#
# convert an string into its hexa representation
#
# return FALSE if it is not a valid string
#
#------------------------------------------------------------------------------

sub string_hexa {

  my ($string, $refhexa, $refreason) = @_;

  my $c;
  my $marker_found;
  my $digit = 0;
  my $hexa;
  my $value;
  
  $$refhexa= "";
  for (my $i = 0 ; $i < length($string) ; $i++) {

    $c = substr($string, $i, 1);

    if ($c eq $HEXA_MARKER) {
      # is it the first marker ?
      if ($marker_found) {
	# this is the second marker character
	$marker_found = $FALSE;
	# encode the marker character
	$value = encode($HEXA_MARKER);
	$$refhexa = "$$refhexa$value";
      } else {
	# this is the first marker
	$marker_found = $TRUE;
      }

    } else {

      # not a marker: has a marker been found previously ?
      if ($marker_found) {
	# we are processing a hexa value
	if ($digit == 0) {
	  # first digit
	  $hexa = "$c";
	  $digit = 1;
	} elsif ($digit == 1) {
	  # second digit: check the value
	  $hexa = "$hexa$c";
	  if (! is_hexa(\$hexa, 2, $refreason)) {
	    $$refreason = "$string: $hexa: invalid hexadecimal value";
	    return $FALSE;
	  }
	  $marker_found = $FALSE;
	  $digit = 0;
	  # the last digit has been found: inster it (already encoded)
	  $$refhexa = "$$refhexa$hexa";
	}

      } else {

	# this a normal character: encode it
	$value = encode($c);
	$$refhexa = "$$refhexa$value";
      }
    }
  }
  # string terminated abnormally
  if ($marker_found) {
    $$refreason = "$string: $hexa: invalid hexadecimal value";
    return $FALSE;
  }

  return $TRUE;
}

#==============================================================================
#
#  Network interface handling
#
#==============================================================================

#------------------------------------------------------------------------------
#
# is_same_physical_interface
#
# return TRUE if both interfaces are on the same physical one
#
#------------------------------------------------------------------------------

sub is_same_physical_interface {

  my ($interface1, $interface2) = @_;

  $interface1 =~  m/^([a-zA-Z]+[0-9]+)[:]?([0-9]*)$/;
  my $phys1 = $1;
  $interface2 =~  m/^([a-zA-Z]+[0-9]+)[:]?([0-9]*)$/;
  my $phys2 = $1;
  return ($phys1 eq $phys2);
}

#------------------------------------------------------------------------------
#
# is_physical_interface
#
# return TRUE if a network interface is a physical interface
# return physical part and alias part
#
#------------------------------------------------------------------------------

sub is_physical_interface {

  my ($interface, $refphys, $refalias) = @_;

  $interface =~  m/^([a-zA-Z]+[0-9]+)[:]?([0-9]*)$/;
  $$refphys = $1;
  if ($$refphys eq $interface) {
    return $TRUE;
  }

  $$refalias = $2;
  return $FALSE;
}

#==============================================================================
#
# OTHER FUNCTIONS
#
#==============================================================================

#------------------------------------------------------------------------------
#
# get a checksum
#
#------------------------------------------------------------------------------

sub get_checksum {

  my ($filename) = @_;

  my $result = qx/$SUM $filename/;
  if ($? ne 0) {
    error("Can not compute checksum for $filename by using $SUM");
  }
  my ($checksum) = split(/[\t\s]+/, $result);

  return $checksum;
}

{
}
