#!/usr/bin/perl
# -----------------------------------------------------------------------------
#  ident	"@(#)sol_jumpstart.pl	1.11	06/06/22 SMI"
# -----------------------------------------------------------------------------
#
#  Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
#  Use is subject to license terms.
#
# -----------------------------------------------------------------------------

package SOL_JUMPSTART;

use strict;
use variables;

require "$ENV{NHINSTALL_LIB}/tools/lib/common.pl";

my $SOLARIS_DIST ;
my $NODE_TYPE;
my $JUMPSTART;
my $MOUNTPOINT = "/a/mnt_tmp" ;
my $FINISH_FILENAME;

my %DISK_ARRAY = ();

#------------------------------------------------------------------------------
#
# Work-around for dmfe driver
#
#------------------------------------------------------------------------------

sub dmfe_work_around {

   print FINISH <<"FEO";

#--------------------------------------------
# workaround for dmfe driver
#--------------------------------------------

# Bugfix for 4511259
# create the generic device 

DRV="dmfe"
BASEDIR="/a"
grep -w "\${DRV}" \${BASEDIR}/etc/name_to_major > /dev/null 2>&1
if [ \$? -eq 0 ]; then
    echo ""
    echo "--------------------------------------------"
    echo "NHAS installation: "dmfe" driver detected"
    echo "creating device (workaround for bug 4511259)"
    set `/usr/bin/grep "^\${DRV}[  ]" \$BASEDIR/etc/name_to_major`
    DRIVER_MAJ=\$2
    DEVICE=\${BASEDIR}/dev/\${DRV}
    rm -f \$DEVICE   > /dev/null 2>&1
    mknod \$DEVICE c \$DRIVER_MAJ 0 > /dev/null 2>&1
    chmod 600 \$DEVICE
    chgrp sys \$DEVICE
    chown root \$DEVICE
    echo "--------------------------------------------"
    echo ""
fi

FEO

}

#------------------------------------------------------------------------------
#
# properties setting in finish.sh or in /tftpboot
#
#------------------------------------------------------------------------------

sub set_properties {

  my $node_idx  = COMMON::getnode($NODE_ID);

  if ($NODE_LIST[$node_idx]{processor} == $SPARC_PROCESSOR) {

  print FINISH <<"FEO";
#
# set boot PROM params
#
/usr/sbin/eeprom local-mac-address?=true
/usr/sbin/eeprom auto-boot?=true
FEO

  } else {

  print FINISH <<"FEO";
#
# set boot PROM params
#
/usr/sbin/eeprom auto-boot?=true
FEO

}

  # look for an environment file in the configuration directory
  # and copy it in /tftpboot 
  if (-f "$USR_CONF_DIR/bootenv.rc") {
    my $cid = COMMON::build_cid("MAC", $NODE_LIST[$node_idx]{eth0});
    my $target_filename = "/tftpboot/$cid.bootenv.rc";
    COMMON::exec_cmd("$CP $USR_CONF_DIR/bootenv.rc $target_filename");
  }
}

#------------------------------------------------------------------------------
#
# SVM installation and configuration
#
#------------------------------------------------------------------------------

sub svm_install {

  my ($solaris_dir) = @_;
  
    COMMON::print_action("Configuring finish.sh for Solaris Volume Manager (SVM) installation");

  # installation of package: there are not located on the standard directory

  my $svm_package_subdir = COMPONENT::get_info_men_component("SVM", "DIR");
  my $svm_packages = COMPONENT::get_info_men_component("SVM", "PACKAGE");

   print FINISH <<"FEO";

$RMKDIR -p $MOUNTPOINT

# mount distribution
$RMOUNT $SERVER_IP:/$solaris_dir $MOUNTPOINT

# create admin file
$RECHO "mail=" > /a/tmp/admin
$RECHO "instance=overwrite" >> /a/tmp/admin
$RECHO "partial=nocheck" >> /a/tmp/admin
$RECHO "runlevel=quit" >> /a/tmp/admin
$RECHO "idepend=nocheck" >> /a/tmp/admin
$RECHO "rdepend=nocheck" >> /a/tmp/admin
$RECHO "space=quit" >> /a/tmp/admin
$RECHO "setuid=nocheck" >> /a/tmp/admin
$RECHO "conflict=nocheck" >> /a/tmp/admin
$RECHO "action=nocheck" >> /a/tmp/admin
$RECHO "basedir=default" >> /a/tmp/admin

# install packages
$RPKGADD -d $MOUNTPOINT/$svm_package_subdir -R /a -a /a/tmp/admin $svm_packages
# umount distribution
$RUMOUNT $MOUNTPOINT
$RRMDIR $MOUNTPOINT

FEO

}

#------------------------------------------------------------------------------
#
#  Additional packages and patches installation
#
#------------------------------------------------------------------------------

sub addon_install {

  
  # installation of package: there are not located on the standard directory

  ADDON::update_finish ($NODE_TYPE, $FINISH_FILENAME, $MOUNTPOINT);

}

#------------------------------------------------------------------------------
#
#  RARP boot method
#
#------------------------------------------------------------------------------

sub rarp_boot {

  my ($solaris_dir) = @_;

  my $cmd;
  my $node_idx  = COMMON::getnode($NODE_ID);
  my $node_ip   = COMMON::get_external_ip($NODE_ID);
  my $node_name = COMMON::build_node_name_for_install($NODE_ID);
  my $ether     = $NODE_LIST[COMMON::getnode($NODE_ID)]{eth0};

  COMMON::print_action("Configuring boot for node $node_name");

  my $arch = $NODE_LIST[$node_idx]{hardware_class};
  $cmd="$SOLARIS_DIST/Tools/add_install_client";
  $cmd .= " -i \"$node_ip\"";
  $cmd .= " -e \"$ether\"";
  $cmd .= " -s \"$SERVER_IP:$solaris_dir\"";
  $cmd .= " -c \"$SERVER_IP:${JUMPSTART}/${CLUSTER}/${NODE_ID}\"";
  $cmd .= " -p \"$SERVER_IP:${JUMPSTART}/${CLUSTER}/${NODE_ID}\"";
  $cmd .= " -n :none";
  $cmd .= " $node_name $arch";   
  COMMON::exec_cmd ($cmd) ||
    COMMON::error ("Execution of \"$cmd\" failed"); 
  
  #
  # Remplace install server name by server IP address in /etc/bootparams
  #
  $cmd="/usr/bin/perl -p -i.bak";
  $cmd .= " -e 'if ( /^$node_name/ ) {";
  $cmd .= " s/^$node_name\\s+[^:]+:/$node_name root=$SERVER_IP:/;";
  $cmd .= " }'";
  $cmd .= " /etc/bootparams";
  COMMON::exec_cmd ($cmd) ||
    COMMON::error("Can't modify \"/etc/bootparams\" file");
}

#------------------------------------------------------------------------------
#
#  DHCP boot method
#
#------------------------------------------------------------------------------

sub dhcp_boot {

  my ($solaris_dir) = @_;

  my $cmd;
  my $node_idx  = COMMON::getnode($NODE_ID);
  my $node_ip   = COMMON::get_external_ip($NODE_ID);
  my $node_name = COMMON::build_node_name_for_install($NODE_ID);
  my $ether     = $NODE_LIST[$node_idx]{eth0};
  my $arch      = $NODE_LIST[$node_idx]{hardware_class};
  my $target_os = $OS_PER_TYPE[$NODE_TYPE]{target_os};


  my $broadcast = COMMON::computeBroadcast($BIT_NETWORK, $BIT_NETMASK);
  my $router    = $DEFAULT_ROUTER_IP;
  my $subnet    = $DOTTED_NETWORK;
  my $subnetmask= $DOTTED_NETMASK;

  #
  # must be executed once
  # configure global DHCP setting if required
  #
  SEQUENCER::execute_action("configure_global_dhcp", 0);

  COMMON::print_action("Configuring DHCP for node $node_name");

  my $by_ether = $TRUE;
  my $cid = COMMON::build_cid("MAC", $ether);
  my $converted_ether = substr($cid,2,length($cid)-2);
  $converted_ether =~ tr/A-Z/a-z/;

  $cmd = ":";
  $cmd .=  "Broadcst=${broadcast}:MTU=1500:";
  $cmd .= "Router=${router}:Subnet=${subnetmask}:";
  $cmd .= "SinstNM=$SERVER_IP:";
  $cmd .= "SinstIP4=$SERVER_IP:";
  $cmd .= "SinstPTH=$solaris_dir:";
  $cmd .= "SrootIP4=${SERVER_IP}:";
  $cmd .= "SrootPTH=${solaris_dir}/$target_os/Tools/Boot:";
  $cmd .= "SrootNM=$node_name:";
  $cmd .= "SjumpsCF=\"$SERVER_IP:${JUMPSTART}/${CLUSTER}/${NODE_ID}\":";
  $cmd .= "SsysidCF=\"$SERVER_IP:${JUMPSTART}/${CLUSTER}/${NODE_ID}\":";
  if ($NODE_LIST[$node_idx]{processor} == $I386_PROCESSOR) {
    if ($by_ether) {
      $cmd .= "BootFile=nbp.${cid}:";
      $cmd .= "SbootURI=\"tftp://${SERVER_IP}/${cid}\":";
    } else {
      $cmd .= "BootFile=nbp.${node_name}:";
      $cmd .= "SbootURI=\"tftp://${SERVER_IP}/${node_name}\":";
    }
  }

  if ($by_ether) {
    COMMON::exec_cmd_ignore("$DHTADM -D -m $cid");
    COMMON::exec_cmd("$DHTADM -A -m $cid -d '$cmd'");
  } else {
    COMMON::exec_cmd_ignore("$DHTADM -D -m $node_name");
    COMMON::exec_cmd("$DHTADM -A -m $node_name -d '$cmd'");
  }
  
  my $subcmd = "-f PERMANENT+MANUAL -i $cid";
  $cmd = "$PNTADM -D $node_ip $subnet";
  COMMON::exec_cmd_ignore("$cmd'");
  $cmd = "$PNTADM -A $node_ip $subcmd -h $node_name -m $node_ip $subnet";
  COMMON::exec_cmd_ignore("$cmd'");

  # refresh dhcp daemon
  COMMON::exec_cmd_ignore("$PKILL -9 in.dhcpd");
  COMMON::exec_cmd("/usr/lib/inet/in.dhcpd");

  if ($by_ether) {
    $cmd="$SOLARIS_DIST/Tools/add_install_client";
    $cmd .= " -d";
    $cmd .= " -e \"$ether\"";
    $cmd .= " -s \"$SERVER_IP:$solaris_dir\"";
    $cmd .= " -c \"$SERVER_IP:${JUMPSTART}/${CLUSTER}/${NODE_ID}\"";
    $cmd .= " -p \"$SERVER_IP:${JUMPSTART}/${CLUSTER}/${NODE_ID}\"";
    $cmd .= " $arch";
  } else {
    $cmd =  "$SOLARIS_DIST/Tools/add_install_client";
    $cmd .= " -d";
    $cmd .= " -s \"$SERVER_IP:$solaris_dir\"";
    $cmd .= " -c \"$SERVER_IP:${JUMPSTART}/${CLUSTER}/${NODE_ID}\"";
    $cmd .= " -p \"$SERVER_IP:${JUMPSTART}/${CLUSTER}/${NODE_ID}\"";
    $cmd .= " $node_name $arch";   
  }

  COMMON::exec_cmd ($cmd);
}

#------------------------------------------------------------------------------
#
# Create jumpstart environment
#
#------------------------------------------------------------------------------
sub jumpstart_create {

  my ($solaris_dir, $profile_file, $install_svm) = @_;
  
  my $node_idx    = COMMON::getnode($NODE_ID);
  my $node_name   = COMMON::build_node_name_for_install($NODE_ID);
  my $server_name = COMMON::build_server_name();
  my $node_ip     = COMMON::get_external_ip($NODE_ID);
  
  my $cmd;
  
  #
  # Control the host configuration
  #
  
  #
  # Test the  jumpstart directory tree, create it if it does not exist
  #
  
  COMMON::control_dir_rxw_or_create ( "$JUMPSTART" );
  COMMON::control_dir_rxw_or_create ( "$JUMPSTART/$CLUSTER" );
  COMMON::control_dir_rxw_or_create ( "$JUMPSTART/$CLUSTER/$NODE_ID" );
  
  $FINISH_FILENAME = "$JUMPSTART/$CLUSTER/$NODE_ID/finish.sh";
  #
  # Create Jumpstart config files
  #
  create_sysidcfg ();
  create_profile ($profile_file);
  create_finish($solaris_dir, $install_svm);
  create_and_test_rules();   
  
  # 
  # Control and modify /etc/hosts
  #   
  add_host( $node_ip, $node_name  );
  
  # 
  # Control end modify /etc/ethers
  #
  my $ether     = $NODE_LIST[COMMON::getnode($NODE_ID)]{eth0};
  add_ether ( $ether, $node_name );
  
  #
  # Purge preexistant boot client, ignore errors silently
  #
  COMMON::exec_cmd_ignore ("$SOLARIS_DIST/Tools/rm_install_client $node_name > /dev/null 2>&1") ;
  
  #
  # Share
  #        Jumpstart environment
  COMMON::share("$JUMPSTART" );

  if ($NODE_LIST[$node_idx]{boot_method} == $RARP_METHOD) {
    rarp_boot($solaris_dir);
  } else {
    dhcp_boot($solaris_dir)
  }   
}


#------------------------------------------------------------------------------
#
# Management of /etc/hosts
#
#------------------------------------------------------------------------------

sub add_host {
   my ( $address, $name ) = @_ ;
   my $count=0;
   my $ip;
   my $host;
   my $garbage;
 
   $count = 0;   
     
   open (HOSTS, "/etc/hosts" ) || &error("Unable to open /etc/hosts");
   while ( <HOSTS> ) {
      
      chop();
      s/#.*//;
      ($ip, $host, $garbage) = split (/[\s\t]+/, $_) ;
      next unless (defined($ip));
      next unless (defined($host));
            
      if ( "$ip" eq "$address" ) {
	 COMMON::error("Ip \"$address\" does not match \"$name\" as first hostname in /etc/hosts") 
	 unless ( $host eq $name );
         $count++;
	 last;
      }
       	        
   }
   close (HOSTS);
   
   unless ( $count ) {
      open (HOSTS, ">>/etc/hosts" ) || COMMON::error("Unable to update /etc/hosts");
      print HOSTS "$address $name\n";
      close (HOSTS);
   }
   
}

#------------------------------------------------------------------------------
#
# Management of /etc/ethers files
#
#------------------------------------------------------------------------------
sub add_ether {
   
   my ( $address, $name ) = @_ ;
   my $count=0;
   my $ether;
   my $host;
      
   if ( -f "/etc/ethers" ) {
      open (ETHERS, "/etc/ethers" ) || COMMON::error("Unable to open /etc/ethers");
      while ( <ETHERS> ) {
      
         chop();
         s/#.*//;
         ($ether, $host) = split (/[\s\t]+/, $_) ;
         next unless (defined($ether));
         next unless (defined($host));
      
         if ( ethermatch ("$ether","$address") ) {  
	    COMMON::error("MAC \"$address\" does not match \"$name\" as hostname in /etc/ethers")
	    unless ( $host eq $name );
            $count++;
	    last;
         }
       	        
      }
      close (ETHERS);
   }
   
   unless ( $count ) {
      open (ETHERS, ">>/etc/ethers" ) || COMMON::error("Unable to update /etc/ethers");
      print ETHERS "$address $name\n";
      close (ETHERS);

      COMMON::exec_cmd("$PKILL in.rarpd ; /usr/sbin/in.rarpd -a") ||
   	COMMON::error("Unable to kill and restart rarpd daemon file");

   }

}


sub duplicated_ether {
   
   my ( $address ) = @_ ;
   my $count=0;
   my $ether;
   my $host;
         
   if ( -f "/etc/ethers" ) {
      open (ETHERS, "/etc/ethers" ) || COMMON::error("Unable to open /etc/ethers");
      while ( <ETHERS> ) {      
         chop();
         s/#.*//;
         ($ether, $host) = split (/[\s\t]+/, $_) ;
         next unless (defined($ether));
         next unless (defined($host));      
         $count++ if ( ethermatch($ether,$address) );         	        
      }
      close (ETHERS);
   }

   return ( $count > 1 );
      
}

sub ethermatch {

   my ( $addr1, $addr2 ) = @_;

   my $i;

   
   return 0 unless (defined($addr1));
   return 0 unless (defined($addr2));
   
   my @a = split ( /:/, $addr1 );
   my @b = split ( /:/, $addr2 );
   
   for ( $i=0; $i < 6; $i++ ) {
      return 0 unless (hex($a[$i]) == hex($b[$i])); 
   }
   
   return 1;
      
}

#------------------------------------------------------------------------------
#
# Create Sysidcfg file
#
#------------------------------------------------------------------------------
sub create_sysidcfg {

  my $node_idx = COMMON::getnode($NODE_ID);
  my $node_ip   = COMMON::get_external_ip($NODE_ID);
  my $node_name = COMMON::build_node_name_for_install($NODE_ID);
  
  my $timezone = SEQUENCER::get_data("TIMEZONE");
  
  if (COMMON::ip_version($node_ip) != COMMON::ip_version($SERVER_IP)) {
    COMMON::error("Server IP address ($SERVER_IP) is not at the same IP version than node $node_name ($node_ip)");
   }
  
  # the IPv6 configuration is performed by nhinstall
  # so jumpstart is not used for that 
  my $protocol_ipv6 = "no";
  
  open (SYSIDCFG, ">$JUMPSTART/$CLUSTER/$NODE_ID/sysidcfg") || 
    COMMON::error("Unable to create \"$JUMPSTART/$CLUSTER/$NODE_ID/sysidcfg\" file");

  if ($NODE_LIST[$node_idx]{boot_method} == $RARP_METHOD) {

    print SYSIDCFG <<"FEO";
name_service=NONE
system_locale=$LOCALE
timezone=$timezone
timeserver=$SERVER_IP
security_policy=NONE
terminal=$TERMINAL
network_interface=primary {protocol_ipv6=$protocol_ipv6
                           hostname=$node_name
                           ip_address=$node_ip
		      	   default_route=$DEFAULT_ROUTER_IP
                           netmask=$DOTTED_NETMASK}
root_password=$ENCRYPTED_PASSWORD
FEO

  } else {

    print SYSIDCFG <<"FEO";
name_service=NONE
system_locale=$LOCALE
timezone=$timezone
timeserver=$SERVER_IP
security_policy=NONE
terminal=$TERMINAL
network_interface=primary {dhcp protocol_ipv6=$protocol_ipv6}
root_password=$ENCRYPTED_PASSWORD
FEO
  }

   close(SYSIDCFG);
}

#------------------------------------------------------------------------------
#
# add filesys description in node.prof
#
#------------------------------------------------------------------------------

sub write_filesys {

  my ($name, $size, $mounting_point, $option) = @_;
  
  print FPROFILE "filesys $name $size $mounting_point $option\n"
}

#------------------------------------------------------------------------------
#
# Create jumpstart profile file
#
#------------------------------------------------------------------------------

sub create_profile {

  my ($profile_file) = @_;

  my $profile = "$JUMPSTART/$CLUSTER/$NODE_ID/node.prof" ;

  my $slice_root ;
  my $node_idx   = COMMON::getnode($NODE_ID);
  my $slice_idx;
  my $mounting_point;

  COMMON::exec_cmd( "$CP $profile_file $profile") ||
      COMMON::error("Unable to create $profile file");

  COMMON::slice_search($NODE_ID, "mounting_point", $SLICE_ROOT, \$slice_idx ) ;
  $slice_root= $SLICE_LIST[$slice_idx]{name};

  open(FPROFILE, ">>$profile") ;

  my $name = $slice_root;
  if ($NODE_LIST[$node_idx]{processor} == $I386_PROCESSOR) {
    $name = substr($slice_root, 0, length($slice_root)-2);
  }

  print FPROFILE "boot_device $name update\n" ;
  print FPROFILE "root_device $slice_root\n" ;

  # normal slice

  for $slice_idx (@{$NODE_LIST[$node_idx]{slice}}) {

    # particular case: "replica" must be replaced with "unnamed"
    $mounting_point = $SLICE_LIST[$slice_idx]{mounting_point} ;
    if ($mounting_point eq $SLICE_TAG_REPLICA) {
      $mounting_point = $SLICE_UNMOUNTED;
    }

    # shared slice are not partitioned at this stage
    if ($SLICE_LIST[$slice_idx]{type} == $LOCAL_SLICE) {
      write_filesys($SLICE_LIST[$slice_idx]{name}, 
		    $SLICE_LIST[$slice_idx]{size}, 
		    $mounting_point,
		    $SLICE_LIST[$slice_idx]{option});
    }
  }

  close FPROFILE ;

}

#----------------------------------------------------------------------------- 
#
# Create jumpstart rules file
#
#------------------------------------------------------------------------------

sub create_and_test_rules {
  
   my $output = "";

   my $node_idx= COMMON::getnode($NODE_ID);

   open (RULES, ">$JUMPSTART/$CLUSTER/$NODE_ID/rules") || 
     COMMON::error("Unable to create \"$JUMPSTART/$CLUSTER/$NODE_ID/rules\" file");
   my $node_name = COMMON::build_node_name_for_install($NODE_ID);
   print RULES "hostname $node_name - node.prof finish.sh\n";
   close (RULES);

   # check fails on x86 because of boot_device syntax
   if ($NODE_LIST[$node_idx]{processor} != $I386_PROCESSOR) {
     open (CHECK,"cd $JUMPSTART/$CLUSTER/$NODE_ID; $SOLARIS_DIST/Misc/jumpstart_sample/check |") || 
       COMMON::error ("Unable to check \"rules\" file"); 
     while ( <CHECK> ) {
       chop();
       COMMON::print_debug($DEBUG_FILE, $_);
       $output= "$output $_\n";
     }
     if (! close (CHECK)) {
       COMMON::print_log($output);
       COMMON::error("Error found while checking \"Rules\" file");
     }
   } else {
     # workaround: boot_device must be cxtxdx to boot since
     # check still looks for cxtxdxsx
     # -> create rules.ok by appending checksum
     my $rules = "$JUMPSTART/$CLUSTER/$NODE_ID/rules";
     my $rulesok = "$JUMPSTART/$CLUSTER/$NODE_ID/rules.ok";
     COMMON::exec_cmd("$CP $rules $rulesok");
     my $chksum = qx{/bin/sum $rulesok | /bin/awk '{print \$1}'};
     chomp($chksum);
     COMMON::exec_cmd("$ECHO \"# version=2 checksum=$chksum\" >> $rulesok");
   }
}

#------------------------------------------------------------------------------
#
# Create jumpstart finish.sh file
#
#------------------------------------------------------------------------------

sub create_finish {

  my ($solaris_dir, $install_svm) = @_;

  my $node_idx = COMMON::getnode($NODE_ID);
  my $node_nic0;
  my $node_nic1;
  my $node_cgtp;
  my $node_name;
  my $other_id;
  
  open (FINISH, ">$FINISH_FILENAME") || 
    COMMON::error("Unable to create $FINISH_FILENAME file");
  print FINISH "#!/bin/sh\n\n";
  
  print FINISH <<"FEO";
#
# create /etc/notrouter
#
if [ ! -f /a/etc/notrouter ] ; then
/usr/bin/touch /a/etc/notrouter
fi

#
# modify /etc/default/login
#
/usr/bin/mv /a/etc/default/login /a/etc/default/login.orig
/usr/bin/chmod 644 /a/etc/default/login.orig
/usr/bin/sed '1,\$s/^CONSOLE/#CONSOLE/' /a/etc/default/login.orig > /a/etc/default/login
/usr/bin/chmod 444 /a/etc/default/login

#
# disable power management
#
if [ ! -f /a/noautoshutdown ] ; then
/usr/bin/touch /a/noautoshutdown
fi



# prevent prompting regarding NFSv4 domain configuration
/usr/bin/touch /a/etc/.NFS4inst_state.domain

FEO

#
# only the install server is listed on /.rhosts in all cases
# moreover the file protection is set up
#
    print FINISH <<"FEO";
#
# modify /.rhosts
#
if [ -f /a/.rhosts ] ; then
/usr/bin/cp /a/.rhosts /a/.rhosts.orig
fi
echo "$SERVER_IP root" >> /a/.rhosts
/usr/bin/chmod 444 /a/.rhosts

FEO

  if (! $RESTRICT_RHOSTS) {

#
# the other MEN (if the node is a MEN) is listed on /.rhosts
#
    
    if ($NODE_TYPE == $MEN_NODE) {

      if ($NODE_ID eq $MEN1_ID) {
	$other_id = $MEN2_ID ;
      } else {
	$other_id = $MEN1_ID ;
      }
      $node_nic0 = COMMON::build_node_name($other_id, "NIC0");

      if ($USE_CGTP) {
	$node_nic1 = COMMON::build_node_name($other_id, "NIC1");
	$node_cgtp = COMMON::build_node_name($other_id, "CGTP");

	print FINISH <<"FEO";
echo "$node_nic0 root" >> /a/.rhosts
echo "$node_nic1 root" >> /a/.rhosts
echo "$node_cgtp root" >> /a/.rhosts

FEO

      } else {
	
	print FINISH <<"FEO";
echo "$node_nic0 root" >> /a/.rhosts

FEO
    
      }
    }
  }

  if ($USE_SHARED_DISK) {

   if ($NODE_TYPE == $MEN_NODE) {

      if ($NODE_ID eq $MEN1_ID) {
	$other_id = $MEN2_ID ;
      } else {
	$other_id = $MEN1_ID ;
      }
      $node_name = COMMON::get_external_name($other_id);
      
      print FINISH <<"FEO";
echo "$node_name root" >> /a/.rhosts

FEO
    
      if ($SCSI_INITIATOR_ID_DEFINED) {
	my $scsi_initiator_id;
	if ($NODE_ID == $MEN1_ID) {
	  $scsi_initiator_id = $SCSI_INITIATOR_ID_1;
	} else {
	  $scsi_initiator_id = $SCSI_INITIATOR_ID_2;
	}

	print FINISH << "FEO"
#
# add the scsi initiator id
#
/usr/bin/echo "scsi-initiator-id=$scsi_initiator_id;" >> /a/kernel/drv/mpt.conf

FEO
      }
      }
    }

  if ($install_svm) {
    svm_install($solaris_dir);
  }

  #
  # work-around for the dmfe driver (which required two reboots to
  # be conrrectly configured
  #
  if (PRODUCT::os_parameter_set($NODE_LIST[$node_idx]{type}, $DMFE_WORKAROUND_PARAMETER)) {
    dmfe_work_around();
  }

  # set properties 
  set_properties();

  # close it: re-open by addon_install
  close (FINISH);

  # install additional packages or patches during the Solaris installation
  # thru the finish.sh file
  addon_install();

  open (FINISH, ">>$FINISH_FILENAME") || 
    COMMON::error("Unable to open $FINISH_FILENAME file");

  # add hardware specific cmd
  if ($NODE_LIST[$node_idx]{processor} == $I386_PROCESSOR) {
    print FINISH <<"FEO";
sync
sync
reboot
FEO
  }

}



#------------------------------------------------------------------------------
#
#  Solaris installation
#
#------------------------------------------------------------------------------

sub prepare_jumpstart {
  
  ($NODE_ID) = @_;
  
  my $solaris_dir;
  my $target_os;
  my $profile_file;
  
  my $node_idx = COMMON::getnode($NODE_ID);
  $NODE_TYPE = $NODE_LIST[$node_idx]{type};
  
  if ($NODE_TYPE == $MEN_NODE) {  
    $solaris_dir  = $MEN_SOLARIS_DIR;
    $profile_file = $MEN_PROFILE;
  } else {
    $solaris_dir  = $DATALESS_SOLARIS_DIR;
    $profile_file = $DATALESS_PROFILE;
  }
  $target_os    = $OS_PER_TYPE[$NODE_TYPE]{target_os};
  
  if ($NODE_TYPE == $MEN_NODE) {
    COMMON::print_stage_node("Solaris installation on master-eligible node");
  } else {
    COMMON::print_stage_node("Solaris installation on dataless node");
  }
  
  my $install_svm = $FALSE;
  if ($USE_SVM && ($target_os eq "Solaris_8")) {
    if ($NODE_TYPE == $MEN_NODE) {
      $install_svm = $TRUE;
    }
  }

  # Test Solaris distribution directory
  
  COMMON::control_dir_is_readable ("$solaris_dir");
  $SOLARIS_DIST=$solaris_dir . "/$target_os";
  if (! -d $SOLARIS_DIST) {
    COMMON::error("Unable to find $target_os distribution on $solaris_dir") 
  }

  COMMON::control_dir_is_readable ("$SOLARIS_DIST");
  COMMON::control_dir_is_readable ("$SOLARIS_DIST/Tools");
  COMMON::control_dir_is_readable ("$SOLARIS_DIST/Tools/Boot");
  COMMON::control_dir_is_readable ("$SOLARIS_DIST/Misc");
  COMMON::control_dir_is_readable ("$SOLARIS_DIST/Misc/jumpstart_sample");
  
  COMMON::control_file_is_readable ("$SOLARIS_DIST/Misc/jumpstart_sample/check");
  
  COMMON::var_defined("JUMPSTART_DIR", "", \$JUMPSTART);

  COMMON::control_file_is_readable ($profile_file);
  
  COMMON::control_file_is_executable("$SOLARIS_DIST/Tools/add_install_client");
  COMMON::control_file_is_executable("$SOLARIS_DIST/Tools/rm_install_client");
  
  COMMON::control_dir_is_readable ("$SOLARIS_DIST/Tools/Boot/sbin");
  COMMON::control_file_is_executable ("$SOLARIS_DIST/Tools/Boot/sbin/bpgetfile");
    
  # Prepare a jumpstart environment
  COMMON::print_action("Preparing the Jumpstart environment");
  jumpstart_create ($solaris_dir, $profile_file, $install_svm);
}
#------------------------------------------------------------------------------
#
#  global DHCP configuration for install server
#
#------------------------------------------------------------------------------

sub configure_global_dhcp {

  my $subnet = $DOTTED_NETWORK;

  my $dhcpsvc = "/etc/inet/dhcpsvc.conf";
  my $restart_daemon = $FALSE;

  COMMON::print_action("Configuring DHCP on the installation server");
  # if /var/dhcp not present, DHCP not configured
  if (! -f $dhcpsvc) {
    COMMON::print_subaction("Configuring the DHCP service");
    COMMON::exec_cmd("$DHCPCONFIG -D -r SUNWfiles -p /var/dhcp -n");
  }

  # create the subnet table if missing
  my $postfix = COMMON::getDhcpPostfix($subnet);
  my $subnetfile = "/var/dhcp/SUNWfiles1${postfix}";

  if (! -f $subnetfile ) {
    COMMON::print_action("Creating the DHCP network tables");
    COMMON::exec_cmd("$PNTADM -C $subnet");
  }

  # add the interface if missing in dhcpsvc
  my $line = qx/$CAT $dhcpsvc | $GREP "^INTERFACES="/;
  chomp($line);

  if (length($line) <= 0) {
    # no INTERFACES defined: add it
    COMMON::exec_cmd("$ECHO \"INTERFACES=$REAL_INSTALL_INTERFACE\" >> $dhcpsvc");
    $restart_daemon = $TRUE;
  } else {
    # check if the interface is already defined
    # hme0 may appear like "=hme0", "=hme0,hme1", "=hme1,hme0", "=hme1,hme0,hme2"
    if (($line !~ m/[=,]+$REAL_INSTALL_INTERFACE,+/) && ($line !~ m/[=,]+$REAL_INSTALL_INTERFACE$/)) {
      # not defined add it
      my $id = $$;
      COMMON::exec_cmd("$CAT $dhcpsvc | $AWK '/INTERFACES=/ { print \$0 \",$REAL_INSTALL_INTERFACE\" ; next} { print \$0 }' > ${dhcpsvc}.${id}");
      COMMON::exec_cmd("$CP ${dhcpsvc}.${id} $dhcpsvc");
      unlink("${dhcpsvc}.${id}");
      $restart_daemon = $TRUE;
    }
  }

  my $platform = "";
  for my $i (0..$#INSTALL_VENDOR_TYPE) {
    if ($platform eq "") {
      $platform = $INSTALL_VENDOR_TYPE[$i] ;
    } else {
      $platform .= " " . $INSTALL_VENDOR_TYPE[$i];
    }
  }
  $platform = "\"Vendor=$platform\"" ;

  my $dhcmd;
  $dhcmd ="$RDHTADM -A -s SrootOpt -d $platform,1,ASCII,1,0";
  COMMON::exec_cmd_ignore("$dhcmd");
  $dhcmd ="$RDHTADM -A -s SrootIP4 -d $platform,2,IP,1,1";
  COMMON::exec_cmd_ignore("$dhcmd");
  $dhcmd ="$RDHTADM -A -s SrootNM -d $platform,3,ASCII,1,0";
  COMMON::exec_cmd_ignore("$dhcmd");
  $dhcmd ="$RDHTADM -A -s SrootPTH -d $platform,4,ASCII,1,0";
  COMMON::exec_cmd_ignore("$dhcmd");
  $dhcmd ="$RDHTADM -A -s SswapIP4 -d $platform,5,IP,1,0";
  COMMON::exec_cmd_ignore("$dhcmd");
  $dhcmd ="$RDHTADM -A -s SswapPTH -d $platform,6,ASCII,1,0";
  COMMON::exec_cmd_ignore("$dhcmd");
  $dhcmd ="$RDHTADM -A -s SbootFIL -d $platform,7,ASCII,1,0";
  COMMON::exec_cmd_ignore("$dhcmd");
  $dhcmd ="$DHTADM -A -s SinstIP4 -d $platform,10,IP,1,0";
  COMMON::exec_cmd_ignore("$dhcmd");
  $dhcmd ="$DHTADM -A -s SinstNM -d $platform,11,ASCII,1,0";
  COMMON::exec_cmd_ignore("$dhcmd");
  $dhcmd ="$DHTADM -A -s SinstPTH -d $platform,12,ASCII,1,0";
  COMMON::exec_cmd_ignore("$dhcmd");
  $dhcmd ="$DHTADM -A -s SsysidCF -d $platform,13,ASCII,1,0";
  COMMON::exec_cmd_ignore("$dhcmd");
  $dhcmd ="$DHTADM -A -s SjumpsCF -d $platform,14,ASCII,1,0";
  COMMON::exec_cmd_ignore("$dhcmd");
  $dhcmd ="$DHTADM -A -s SbootURI -d $platform,16,ASCII,1,0";
  COMMON::exec_cmd_ignore("$dhcmd");

  my $macro_name = "PXEClient:Arch:00000:UNDI:002001";
  my $cmd = ":BootSrvA=$SERVER_IP:";
  COMMON::exec_cmd_ignore("$DHTADM -A -m \"$macro_name\" -d '$cmd'");
}

{
}
