#!/usr/bin/perl
# -----------------------------------------------------------------------------
# ident	"@(#)men_network.pl	1.7	06/01/23 SMI"
# -----------------------------------------------------------------------------
#
# Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
# Use is subject to license terms.
#
# -----------------------------------------------------------------------------

package MEN_NETWORK;

use strict;
use variables;

#------------------------------------------------------------------------------
#
# configure the temporary nic
#
#------------------------------------------------------------------------------

sub configure_nic {

  my ($node_id, $ip_nic0, $ip_cgtp) = @_;

  # configure interface (reconfigure in case of recovery)

  my $men_ip = COMMON::get_external_ip($node_id);

  my $idx = COMMON::getnode($node_id);
  my $pubnic = $NODE_LIST[$idx]{pubNic};
  my $nic = $NODE_LIST[$idx]{nic0};

  # create the nic0 and cgtp IP addresses on the first interface
  #
  # if we are not using the public network, the nic0 IP address is used
  # for installation:
  # - nothing to do for Nic 0 @
  # - just addif CGTP @
  #
  # if we are using the public network: check if the interface used for
  # the public network is the same than the one we plan to use for new
  # addresses
  # - if it's not the same, plumb it
  # - if it's the same, just add the new @

  my $plumb_nic0 = $FALSE;
  my $addif_nic0 = $FALSE;

  if ($USE_PUBLIC_NETWORK) {
    if (COMMON::is_same_physical_interface($pubnic, $nic)) {
      $addif_nic0 = $TRUE;
    } else {
      $plumb_nic0 = $TRUE;
    }
  }

  
  # in case of recovery
  if ($plumb_nic0) {
    COMMON::remote_exec_ignore($men_ip, "$RIFCONFIG $nic unplumb");
  } elsif ($addif_nic0) {
    COMMON::remote_exec_ignore($men_ip, "$RIFCONFIG $nic removeif $ip_nic0");
  }

  if (defined($ip_cgtp)) {
    COMMON::remote_exec_ignore($men_ip, "$RIFCONFIG $nic removeif $ip_cgtp");
  }

  # normal processing
  if ($plumb_nic0) {
    COMMON::remote_exec($men_ip, "$RIFCONFIG $nic plumb $ip_nic0 netmask + broadcast + up");
  } elsif ($addif_nic0) {
    COMMON::remote_exec($men_ip, "$RIFCONFIG $nic addif $ip_nic0 netmask + broadcast + up");
  }

  if (defined($ip_cgtp)) {
    COMMON::remote_exec($men_ip, "$RIFCONFIG $nic addif $ip_cgtp netmask + broadcast + up");
  }
}


#------------------------------------------------------------------------------
#
# configure temporary network
#
# . solve a lot of problems (SVM diskset access, smdiskless, ...)
#
#------------------------------------------------------------------------------

sub men_net_configure_temporary_network {
  
  ($NODE_ID) = @_;

  COMMON::print_stage_node("Configuring network (temporary configuration)");

  my $node_ip = COMMON::get_external_ip($NODE_ID);
  my $idx = COMMON::getnode($NODE_ID);

  my $men_cgtp_ip;
  my $men_nic0_ip;
  my $men_intra_name;

  if ($USE_CGTP) {
    $men_cgtp_ip = COMMON::build_node_ip($NODE_ID, "CGTP");
    $men_nic0_ip = COMMON::build_node_ip($NODE_ID, "NIC0");
    $men_intra_name = COMMON::build_node_name($NODE_ID, "CGTP");

    configure_nic($NODE_ID, $men_nic0_ip, $men_cgtp_ip) ;


  } else {
    $men_nic0_ip = COMMON::build_node_ip($NODE_ID, "NIC0");
    $men_intra_name = COMMON::build_node_name($NODE_ID, "NIC0");
    
    configure_nic($NODE_ID, $men_nic0_ip);

  }

  # update the node name to force SVM using this name
  COMMON::remote_exec($node_ip,
		      "$RUNAME -S $men_intra_name");

  # restart rpcbind (hot restart to keep registered rpc)
  COMMON::remote_exec_ignore($node_ip, "$RPKILL -x -u 0 rpcbind");
  COMMON::remote_exec($node_ip, "$RRPCBIND -w");

  # kill rpc.metad: don't check error (may be missing)
  COMMON::remote_exec_ignore($node_ip, "$RPKILL rpc.metad");
}

#------------------------------------------------------------------------------
#
# Configure configuration files
#
#------------------------------------------------------------------------------

sub net_configure_files {

  my ($node_id) = @_;

  my $external_ip  = COMMON::get_external_ip($node_id);
  
  my $netmasks = "$WORKING_DIR/netmasks.tmp";
  my $local_hosts = "$WORKING_DIR/hosts.tmp";
  my $local_ipnodes = "$WORKING_DIR/ipnodes.tmp";
  
  my $node = COMMON::getnode($node_id);

  # retrieve /etc/netmasks 
  COMMON::get_file($external_ip, "/etc/netmasks", $netmasks);
  
  # modify locally /etc/netmasks 
  # note:
  #    add_netmask eliminate multiple definitions and incorrect definitions
  COMMON::add_netmask($netmasks, 
		      $CLUSTER_DOTTED_NIC0_NET, 
		      $CLUSTER_DOTTED_NETMASK
		     );
  
  
  if ($USE_CGTP) {
    COMMON::add_netmask($netmasks, 
			$CLUSTER_DOTTED_NIC1_NET, 
			$CLUSTER_DOTTED_NETMASK
		       );
    COMMON::add_netmask($netmasks, 
			$CLUSTER_DOTTED_CGTP_NET, 
			$CLUSTER_DOTTED_NETMASK
		       );
  }
  
  if ($NODE_LIST[$node]{pubConfigured}) {
    COMMON::add_netmask( $netmasks, 
			 $PUBLIC_DOTTED_NETWORK, $PUBLIC_DOTTED_NETMASK);
  }

  for (my $eam = 0 ; $eam < scalar(@EXTERNAL_ACCESS_LIST) ; $eam++) {
    COMMON::add_netmask($netmasks,
			$EXTERNAL_ACCESS_LIST[$eam]{dotted_network},
			$EXTERNAL_ACCESS_LIST[$eam]{dotted_netmask}
			);
  }

  # putback modified /etc/netmasks
  COMMON::put_file($external_ip, $netmasks, "/etc/netmasks");
  
  my $remote_hosts = "/etc/hosts";
  # update /etc/hosts
  COMMON::get_file($external_ip, $remote_hosts, $local_hosts);
  COMMON::update_hosts_file($local_hosts, $node_id, $IP_VERSION_V4) ;
  COMMON::put_file($external_ip, $local_hosts, $remote_hosts);

  # update /etc/inet/ipnodes
  if ($IPNODES_USED) {
    my $remote_ipnodes = "/etc/inet/ipnodes";
    COMMON::get_file($external_ip, $remote_ipnodes, $local_ipnodes);
    COMMON::update_hosts_file($local_ipnodes, $node_id, $IP_VERSION_V6) ;
    COMMON::put_file($external_ip, $local_ipnodes, $remote_ipnodes);
  }

  my $nodename;
  # update /etc/nodename
  if ($USE_CGTP) {
    $nodename = COMMON::build_node_name($node_id, "CGTP");
  } else {
    $nodename = COMMON::build_node_name($node_id, "NIC0");
  }

  my $local_nodename = "$WORKING_DIR/nodename";
  my $remote_nodename = "/etc/nodename";
  open(NODENAME, ">${local_nodename}") ||
    COMMON::error("Unable to create ${local_nodename}");
  
  print NODENAME "${nodename}\n";
  
  close(NODENAME);
  
  #
  # remote copy nodename
  #
  COMMON::put_file($external_ip, $local_nodename, $remote_nodename);
  unlink($local_nodename);

}

#------------------------------------------------------------------------------
#
# Configure network interfaces
#
#------------------------------------------------------------------------------

sub net_configure_interfaces {

  my ($node_id) = @_;

  my $external_ip  = COMMON::get_external_ip($node_id);
  
  my $node = COMMON::getnode($node_id);

  # reset the list of already defined hostname files for this node
  COMMON::reset_hostname() ;

  # configure the intra-cluster interfaces

  # create /etc/hostname.<nic0> (already exists but incorrect)
  COMMON::create_internal_interface_hostname($node_id, 
					     "NIC0", 
					     $NODE_LIST[$node]{nic0});

  if ($USE_CGTP) {
     # create /etc/hostname.<nic1>
     COMMON::create_internal_interface_hostname ($node_id,
						 "NIC1", 
						 $NODE_LIST[$node]{nic1});
     # create /etc/hostname.<cgtp>
     COMMON::create_internal_interface_hostname ($node_id,
						 "CGTP", 
						 $CGTP_INTERFACE);
   }
  
  # create /etc/hostname on public interface (if any)
  COMMON::create_external_interface_hostname($node_id);

  # IPv4
  # because of /etc/rcS.d/S30network.sh, it is recommended to
  # create the interface to be set "down" like floating IP address
  # at the end
  COMMON::create_IPMP_test_hostname($node_id, $IP_VERSION_V4);
  COMMON::create_floating_address_hostname($node_id, $IP_VERSION_V4);

  # IPv6
  # it is recommended to create test address at the beginning
  # because no IP adress associated. This avoids executing an "ifconfig addif"
  # without an IP address
  COMMON::create_IPMP_test_hostname($node_id, $IP_VERSION_V6);
  COMMON::create_floating_address_hostname($node_id, $IP_VERSION_V6);

  #
  # if installed thru DHCP, remove useless /etc/dhcp.<itf> files
  #
  if ($NODE_LIST[$node]{boot_method} == $DHCP_METHOD) {
    my $dhcpfile;
    $dhcpfile = "/etc/dhcp.$NODE_LIST[$node]{nic0}";
    COMMON::remote_exec_ignore($external_ip, "$RRM $dhcpfile");
    if ($USE_CGTP) {
      $dhcpfile = "/etc/dhcp.$NODE_LIST[$node]{nic1}";
      COMMON::remote_exec_ignore($external_ip, "$RRM $dhcpfile");
    }
    if ($USE_PUBLIC_NETWORK) {
      $dhcpfile = "/etc/dhcp.$NODE_LIST[$node]{pubnic}";
      COMMON::remote_exec_ignore($external_ip, "$RRM $dhcpfile");
    }
  }
}

#--------------------------------------------------------------
#
# network configuration: interface
#
#--------------------------------------------------------------

sub men_net_configure_files {

  ($NODE_ID) = @_;

  COMMON::print_stage_node("Network files configuration") ;

  net_configure_files ($NODE_ID);
}

#--------------------------------------------------------------
#
# network configuration: interface
#
#--------------------------------------------------------------

sub men_net_configure_interfaces {

  ($NODE_ID) = @_;

  COMMON::print_stage_node("Network interfaces configuration") ;

  net_configure_interfaces ($NODE_ID);
}

{
}







