/**
 * validate.js
 *
 * The user must ensure that the appropriately localized alert_<country>.js
 * file is loaded.
 *
 * Java versions of these functions are defined in ValidateUtil.java. If you
 * change a function definition, make sure the Java method is updated as well.
 */

function isTrue(flag)
{
    return (flag == "1" ||
            flag == "y" ||
            flag == "on" ||
            flag == "yes" ||
            flag == "true" ||
            flag == "enable" ||
            flag == "enabled");
}

function isFalse(flag)
{
    return (flag == "0" ||
            flag == "n" ||
            flag == "off" ||
            flag == "no" ||
            flag == "false" ||
            flag == "disable" ||
            flag == "disabled");
}

function isValidBoolean(flag)
{
    return (isTrue(flag) || isFalse(flag));
}

function isValidId(id) {
    idRe = /^[A-Za-z_][A-Za-z0-9\.\-_]*$/;
    return id.match(idRe);
}

function getNumOccurrences(c, s)
{
    var count = 0;

    for (var i = 0; i < s.length; i++) {
        if (s.charAt(i) == c)
            count++;
    }

    return count;
}

function hasMoreThanOneDoubleColon(s)
{
    var index = s.indexOf("::");
    if (index >= 0) {
        var s2 = s.substring(index + 1, s.length);
        if (s2.indexOf("::") >= 0)
            return true;
    }

    return false;
}

function isIpv6HexSequence(s)
{
    var hex4DigitRe = /^[A-Fa-f0-9]{0,4}$/;
    var hex4List = s.split(":");

    for (var i = 0; i < hex4List.length; i++) {
        if (!hex4List[i].match(hex4DigitRe))
            return false;
    }

    return true;
}

function isValidIpv6AddressWithMixedIpv4(s)
{
    var index = s.lastIndexOf(':');
    var ipv6Part = s.substring(0, index);
    var ipv4Part = s.substring(index + 1, s.length);

    if (!isValidIpv4Address(ipv4Part))
        return false;

    if (getNumOccurrences(':', ipv6Part) > 6)
        return false;

    if (isIpv6HexSequence(ipv6Part))
        return true;

    return false;
}

function isValidIpv6AddressNotMixedWithIpv4(s)
{
    if (getNumOccurrences(':', s) > 7)
        return false;

    if (isIpv6HexSequence(s))
        return true;

    return false;
}

function isValidIpv6Address(s)
{
    if (s == "::")
        return true;

    if (hasMoreThanOneDoubleColon(s))
        return false;

    if (s.indexOf(":") >= 0) {
        if (s.indexOf(".") >= 0)
            return isValidIpv6AddressWithMixedIpv4(s);
        return isValidIpv6AddressNotMixedWithIpv4(s);
    }

    return false;
}

function isValidIpv4Address(s)
{
    var dec3DigitRe = /^[0-9]{1,3}$/;
    var dec3List = s.split(".");

    if (dec3List.length != 4)
        return false;

    for (var i = 0; i < dec3List.length; i++) {
        if (!dec3List[i].match(dec3DigitRe) || dec3List[i] < 0 || dec3List[i] > 255)
            return false;
    }

    return true;
}

function isValidIpAddress(s)
{
    if (isIpAny(s))
        return true;

    if (isValidIpv4Address(s))
        return true;

    if (isValidIpv6Address(s))
        return true;

    return false;
}

function isIpAny(ip)
{
    return (ip == "::" ||
            ip == "0" ||
            ip == "0.0.0.0" ||
            ip == "any" ||
            ip == "ANY" ||
            ip == "INADDR_ANY");
}

function isValidPort(port)
{
    var integerRe = /^[0]+/;
    if (isInteger(port) && port > 0 && port < 65536) {
        if ( port.match (integerRe)) {
            alert ("Port number cannot start with 0.");
            return false;
        }
        return true;
    }
    return false;
}

function isValidServerName(s)
{
    var serverNameRe = /^[A-Za-z0-9\.\-_:]+$/;
    return s.match(serverNameRe);
}

function isDigit(c)
{
    var digitRe = /^[0-9]$/;
    return c.match(digitRe);
}

function isEmpty(s)
{
    return (s == null || s.length == 0);
}

function isWhitespace(s)
{
    var whitespaceRe = /^\s*$/;
    return s.match(whitespaceRe);
}

function isInteger(s)
{
    if (isWhitespace(s)) {
        // If s is whitespace, the second argument is the default return value
        if (isInteger.arguments.length > 1)
            return isInteger.arguments[1];
        return false;
    }

    var integerRe = /^[0-9]+$/;
    return s.match(integerRe);
}

function trim(s)
{
    while (s.substring(0,1) == ' ')
        s = s.substring(1);

    while (s.substring(s.length - 1, s.length) == ' ')
        s = s.substring(0, s.length-1);

    return s;
}

function validateId(field)
{
    field.value = trim(field.value);

    if (!isValidId(field.value)) {
        alertInvalidId(field.alt);
        return false;
    }

    return true;
}

function validateIpAddress(field)
{
    field.value = trim(field.value);

    if (!isValidIpAddress(field.value)) {
        alertInvalidIpAddress(field.alt);
        return false;
    }

    return true;

}

function validatePort(field)
{
    field.value = trim(field.value);

    if (!isValidPort(field.value)) {
        alertInvalidPort(field.alt);
        return false;
    }

    return true;
}

function validateServerName(field)
{
    field.value = trim(field.value);

    if (!isValidServerName(field.value)) {
        alertInvalidServerName(field.alt);
        return false;
    }

    return true;
}

function validateNumber(field, minimum, maximum)
{
    field.value = trim(field.value);

    if (!isInteger(field.value) || field.value < minimum || field.value > maximum) {
        alertInvalidNumber(field.alt, minimum, maximum);
        return false;
    }

    return true;
}

function validateURI(field)
{
    if (isEmpty(field)) {
        alertInvalidURI();
        return false;
    }

    field.value = trim(field.value);
    
    var uriExp = /^\s*[\\\s\/A-Za-z0-9]+[\\:\s\/A-Za-z0-9_\.\-\@]*\s*$/;
    if (field.value.charAt(0) != '/') {
        alertInvalidURI();
        return false;
    }

    if ( !((field.value).match(uriExp)) ) {
        alertInvalidURI();
        return false;
    }

    return true;
}

function isValidPath(field)
{
    if (isEmpty(field))
        return false;

    field.value = trim(field.value);

    var pathExp = /^\s*[\\$:\s\/A-Za-z_]+[\\$:\s\/A-Za-z0-9_\.\-]*\s*$/;
    if ( isWhitespace(field.value) )
        return false;

    if ( !((field.value).match(pathExp)) )
        return false;

    return true;
}

function validateWebappPath(field)
{
    if (!isValidPath(field)) {
        alertInvalidWebappPath();
        return false;
    }

    return true;
}

function isValidClassName(s)
{
    var classNameRe = /^[A-Za-z0-9\.\-_:]+$/;
    return s.match(classNameRe);
}

function validateDataSourceClassName(field)
{
    field.value = trim(field.value);

    if (!isValidClassName(field.value)) {
        alertInvalidClassName("Datasource Class Name");
        return false;
    }

    return true;
}

function isValidName(name) {
    nameRe = /^[A-Za-z_][A-Za-z0-9\.\-_]*$/;
    return name.match(nameRe);
}

function validateName(field)
{
    field.value = trim(field.value);

    if (!isValidName(field.value)) {
        alertInvalidName(field.alt);
        return false;
    }

    return true;
}

function validateInteger(field, name)
{
    field.value = trim(field.value);

    if (!isInteger(field.value)) {
        alertInvalidInteger(name);
        return false;
    }

    return true;
}

function isAscii(s)
{
    var i, c;

    for(i = 0; i < s.length; i++)
    {
       c = s.charCodeAt(i);
       if(c > 0x7f || c < 0)
       {
          return false;
       }
    }
    return true;
}

function validateAscii(field, name)
{
    field.value = trim(field.value);

    if (!isAscii(field.value)) {
        alertNotAscii(name);
        return false;
    }

    return true;
}

function isValidField(fieldvalue, label)
{
    fieldvalue = trim(fieldvalue);
    
    if (isEmpty(fieldvalue)) {
        alertInvalidField(label);
        return false;
    }
    
    return true;
}

function validateFamily(family)
{
    family = trim(family);

    if (!(family == "" ||
            family == "inet" ||
            family == "inet6" ||
            family == "nca")) {
        alertInvalidFamily();
        return false;
    }

    return true;
}

function validateRequiredField(field)
{
    field.value = trim(field.value);

    if (isEmpty(field.value)) {
        alertRequiredField(field.alt);
        return false;
    }

    return true;
}

