#!/usr/bin/perl -w
# $Id: moduleMgr.pl,v 1.17 2004/11/03 19:04:47 ms152511 Exp $
# moduleMgr.pl - module mgmt tool for scs
# Copyright (c) 2003 Sun Microsystems, Inc. All rights reserved
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.

use strict;

use lib '/scs/lib/perl5';
use MAPP;
use InstallMAPP;
use BDUtil;
use BDi18n;
use Progress;
use CLIids;
use SCSConfig;

#
# subroutines
#

sub printUsage
{
    my $msg = shift;
    print "\nERROR: $msg\n" if (defined($msg));

    print STDERR<<EOUSAGE;

Usage: $0 [debug] <mode> [parameters]

Modes:
------
 -l [module id]    : List modules, or list a module's installation report.
 -i <.mapp pkg>    : Install module, distribute to all eligible hosts.
 -s <.mapp pkg>    : Install module, no distribution to managed hosts.
 -u <module id>    : Uninstall module, remove from all installed hosts.
 -n <module id>    : Uninstall module, leave managed hosts components alone.
 -a <module id> <host id> [ .. <id>]  : Add module to managed hosts.
 -r <module id> <host id> [ .. <id>]  : Remove module from managed hosts.

Instead of module id, you may specify the module name as listed with -l.
EOUSAGE

    CLIids::printUsage();

    exit 0;
}

sub task_complete ($) {
    my $tag = shift @_;
    my $msg = BDi18n::getMsg($tag);
    Progress::progressComplete($msg);
}
                   
sub task_failed ($) {
    my $tag = shift;
    my $msg = BDi18n::getMsg($tag);
    Progress::progressFailed($msg);
}

#
# main
#
{
    if ($> != 0) {
        print STDERR "Must be run as root!\n";
        exit 1;
    }

    Progress::initProgress(\@ARGV);

    if (defined($ARGV[0]) && ($ARGV[0] eq 'debug')) {
        BDUtil::setDebug();
        BDUtil::printDebug('Enabled');
        shift @ARGV;
    }

    printUsage() if (!defined($ARGV[0]));

    my $action = shift @ARGV;
    my $rc = 0;

    if ($action =~ /^-[-]?([hH]|[hH]elp|\?)$/) {
        &printUsage();
        exit($rc);
    }

    # setup proxies if defined
    if (my $opt_f = SCSConfig::getFtpProxy()) {
        $ENV{ftp_proxy} = $opt_f;
        BDUtil::info('FTP proxy is  ' . $opt_f);
    }
    if (my $opt_w = SCSConfig::getHttpProxy()) {
        $ENV{http_proxy} = $opt_w;
        BDUtil::info('HTTP proxy is ' . $opt_w);
    }

    if ($action eq '-l') {
        if (defined($ARGV[0])) {
            my $moduleId = MAPP::getMAPPIdByName($ARGV[0]);
            my $appls = MAPP::getEligibleAppliances($moduleId);
            if ($appls) {
                printf "  id  %-25s  %-27s  %s\n", 'Host Name', 'Class', 'Status';
                printf "  --  %-25s  %-27s  %s\n", '---------', '-----', '------';
                foreach my $a (@{$appls}) {
                    printf "%4d  %-25s  %-27s  ", $a->{'appliance_id'},
                        $a->{'host_name'}, $a->{'build_class'};
                    if ($a->{'status'} eq 'P') {
                        print 'Installed';
                    } elsif ($a->{'reason'} ne '') {
                        print 'Failed : ' . $a->{'reason'};
                    } else {
                        print 'Not installed';
                    }
                    print "\n";
                }
            } else {
                print "No hosts currently under management are eligible for this module.\n"
            }
        } else {
            my $mapps = MAPP::getAllMAPPs();
            if ($mapps) {
                printf "  id  %-25s  %-25s  %s\n", 'Name', 'Vendor', 'Version';
                printf "  --  %-25s  %-25s  %s\n", '----', '------', '-------';
                foreach my $mapp (@{$mapps}) {
                    printf "%4d  %-25s  %-25s  %s\n", $mapp->{'mapp_id'},
                        $mapp->{'mapp_name'}, $mapp->{'vendor'}, $mapp->{'version'};
                }
            } else {
                print "No modules installed.\n";
            }
        }
    } else {
        printUsage() if (!defined($ARGV[0]));
  
        my ($moduleId, $hIds);
        CLIids::setExitOnError();
        BDi18n::setDomain('base-mgmtapp');
        # All task completion messages (progressFailed, progressComplete, etc.)
        # are processed by the respective subroutines of InstallMAPP
        if ($action eq '-i') {
            Progress::setProcessName(BDi18n::getMsg('mappInstall'));
            BDUtil::openLog('mappInstall');
            if ( InstallMAPP::installMAPP($ARGV[0])    < 0 ) {
                $rc = 1;
            }
        } elsif ($action eq '-s') {
            Progress::setProcessName(BDi18n::getMsg('mappInstallNoClients'));
            BDUtil::openLog('mappInstallNoClients');
            if ( InstallMAPP::installMAPP($ARGV[0], 1) < 0 ) {
                $rc = 1;
            }
        } elsif ($action eq '-u') {
            $moduleId = MAPP::getMAPPIdByName($ARGV[0]);
            Progress::setProcessName(BDi18n::getMsg('mappUninstall'));
            BDUtil::openLog('mappUninstall');
            if ( InstallMAPP::uninstallMAPP($moduleId)    < 0 ) {
                $rc = 1;
            }
        } elsif ($action eq '-n') {
            $moduleId = MAPP::getMAPPIdByName($ARGV[0]);
            Progress::setProcessName(BDi18n::getMsg('mappUninstallNoPurge'));
            BDUtil::openLog('mappUninstallNoPurge');
            if ( InstallMAPP::uninstallMAPP($moduleId, 1) < 0 ) {
                $rc = 1;
            }
        } elsif ($action eq '-a') {
            $moduleId = MAPP::getMAPPIdByName(shift @ARGV);
            Progress::setProcessName(BDi18n::getMsg('mappInstallOnClients'));
            BDUtil::openLog('mappInstallOnClients');
            if ( InstallMAPP::installNewClients( $moduleId,
                                                 CLIids::getHostIds(\@ARGV) ) < 0 ) {
                $rc = 1;
            }
        } elsif ($action eq '-r') {
            $moduleId = MAPP::getMAPPIdByName(shift @ARGV);
            Progress::setProcessName(BDi18n::getMsg('mappUninstallFromClients'));
            BDUtil::openLog('mappUninstallFromClients');
            if ( InstallMAPP::uninstallClients( $moduleId,
                                                CLIids::getHostIds(\@ARGV) )  < 0 ) {
                $rc = 1;
            }
        } else {
            printUsage('Invalid mode');
            $rc = 1;
        }

        BDUtil::closeLog();
    }

    exit $rc;
}

