#!/usr/bin/perl -w
# $Id: deviceMgr.pl,v 1.28 2004/12/22 01:08:50 ms152511 Exp $
# deviceMgr.pl - handle adding/deleting/upgrading hosts
# Copyright (c) 2003 Sun Microsystems, Inc. All rights reserved
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.

use strict;
use Fcntl;
use IO::File;

use lib '/scs/lib/perl5';
use AgentInterface;
use AgentMgr;
use Appliance;
use BDUtil;
use BDi18n;
use CLIips;
use Progress;
use PropertyUtil;
use SCSUtil;

my $Ssh_Keyscan_Path = PropertyUtil::getProperty('cmd.ssh-keyscan');
my $Ssh_Keygen_Path  = PropertyUtil::getProperty('cmd.ssh-keygen');

my %BATCH_RESULT = ( 'success'     =>  0,
                     'all_failed'  => -1,
                     'some_failed' => -2 );

my $WORK_UNITS_APPROVE = 3; # check fingerprint, add public key, update DB
my $WORK_UNITS_MANAGE  = $WORK_UNITS_APPROVE + 2; # approve host, get build
my $WORK_UNITS_SCAN    = 2; # parse input & generate IPs, run ssh-keyscan

#
# subroutines
#

sub printUsage ()
{
    print STDERR<<EOD;
Usage: $0 <mode> [arguments]

   mode             arguments
   ----             ---------
   manage host      <host> [ <fingerprint> ] [ <email> ] [ <desc> ]
   manage file      <filename> [ delete ]
   manage batch     <host 1> [ <host 2> ... ]
   approve host     <host> [ <fingerprint> ]
   approve batch    <host 1> [ <host 2> ... ]
   scan file        file <filename> [ delete ] [ <ip range> ... ]
   scan             <ip range> [ <ip range 2> ... ]
   del              <host 1> [ <host 2> ... ]

All the manage modes can be replaced with managen if there are to be
no control module components installed on the newly imported clients.

If the "delete" keyword is specified with any of the "file" modes, the
specified input file is deleted from disk, whether the requested
operation succeeds or fails.

EOD

    CLIips::printUsage;
}

sub doManage ($@) {
    my ($source, $newId, $rc, @lines, $line);
    my @newApplianceList = ();

    my $workUnitsPerAppliance = $WORK_UNITS_APPROVE + $WORK_UNITS_MANAGE +
        AgentMgr::getWorkPerAppliance();

    my $retCode;

    my $installMAPPs = shift @_;

    $source = shift @_;
    unless (defined $source && $source ne "") {
	return devmgr_warn(0, 'invalidSource', '');
    }

    CASE_ADD_SRC : {

	if ($source eq 'host') {
            Progress::setProcessName(BDi18n::getMsg('task_manageThisHost',
                                                    { 'dev' => $_[0] }));
	    Progress::setWorkItems($workUnitsPerAppliance);
            $newId = manageHost(@_);
            if ( $newId < 0 ) {
                $retCode = $BATCH_RESULT{'all_failed'};
            } else {
                push(@newApplianceList, $newId);
                $retCode = $BATCH_RESULT{'success'};
            }
	    last CASE_ADD_SRC;
        }

        if ( $source eq 'batch' ) {
            Progress::setWorkItems( $workUnitsPerAppliance * scalar(@_) );
            my $hosts_to_manage = scalar(@_);
            my $hosts_managed   = 0;
            foreach my $host (@_) {
                Progress::setProcessName(BDi18n::getMsg('task_manageThisHost',
                                                        { 'dev' => $host } ));
                $newId = manageHost($host);
                if ( $newId >= 0 ) {
                    push( @newApplianceList, $newId );
                    $hosts_managed++;
                }
            }

            if ( $hosts_managed == $hosts_to_manage ) {
                $retCode = $BATCH_RESULT{'success'};
            } elsif ( $hosts_managed == 0 ) {
                $retCode = $BATCH_RESULT{'all_failed'};
            } elsif ( $hosts_managed < $hosts_to_manage ) {
                $retCode = $BATCH_RESULT{'some_failed'};
            } else {
                die "Unexpected condition: hosts_to_manage=$hosts_to_manage, hosts_managed=$hosts_managed";
            }

            last CASE_ADD_SRC;
        }

	if ($source eq 'file') {
	    my ($filename,$delete_when_done,$FILE,%devs,$cnt,$failed,@keys,@fields,$key,$i);
            Progress::setProcessName(BDi18n::getMsg('importFile'));
	    $filename = shift @_;
            $delete_when_done = shift @_;
	    unless (defined $filename && $filename ne "") {
		return devmgr_warn(0, 'noFileGiven');
	    }
            if ( defined $delete_when_done &&
                 lc($delete_when_done) ne 'delete' ) {
                return devmgr_warn(0, 'error_expectedDelete', $delete_when_done);
            }

	    unless (-r $filename) {
                return devmgr_warn(0, 'cantOpenFile', $filename);
	    }

	    unless(open($FILE,$filename)) {
		return devmgr_warn(0, 'cantOpenFile', $filename);
	    }

            devmgr_info(0, 'status_parseFile', $filename);

	    while ( $_ = get_valid_line($FILE) ) {
                $_ =~ s/\s+$//g;
		BDUtil::printDebug("Line is:\n $_");
		@lines = split(/\r/,$_);
		foreach $line (@lines) {
		    @fields = split(/(?:\s+)|\|/,$line);
		    unless ( defined $fields[0] and
                             SCSUtil::isHostname($fields[0]) ) {
			devmgr_warn(0, 'invalidLine', $.);
			next;
		    }
		    $devs{$fields[0]} = join('|',@fields);
		}
	    }
	    close ($FILE);

	    @keys = keys(%devs);
	    $cnt = scalar(@keys);
	    if ($cnt < 1) {
		return devmgr_warn(1, 'noValidLines');
	    }
	    $failed = 0;

            devmgr_info(0, 'status_readLines', $cnt);

	    # delete the input file if requested
            if ( defined $delete_when_done &&
                 lc($delete_when_done) eq 'delete' ) {
                unlink($filename) or
                    return devmgr_warn( 0,
                                        'error_unlinkFailed',
                                        $filename );
            }

	    Progress::setWorkItems($cnt + 1);

	    my $maxTasks = 8;
	    my $taskCnt = 0;
	    my $curTask = 0;
	    my @taskIds = ();
	    my $parent = 0;
	    my $status = '';
	    my $slot = 0;

	    foreach $key (@keys) {

		# wait for running tasks to finish

		while ($taskCnt >= $maxTasks) {
		    sleep(10);

		    for ($slot = 0; $slot <= $#taskIds; $slot++) {
			next if ($taskIds[$slot] == 0);
			$status = Task::getTaskStatus($taskIds[$slot]);
			if ($status =~ /^[CFU]$/) {
			    $taskIds[$slot] = 0;
			    $taskCnt--;
			    $failed++ unless ($status eq "C");
			}
		    }
		}

		if ($taskCnt < $maxTasks) {
                    my $dev = (split(/\|/,$devs{$key}))[0];
		    ($curTask, $parent) =
			Task::spawnTask( BDi18n::getMsg('task_manageThisHost',
                                                        { 'dev' => $dev }),
                                         $workUnitsPerAppliance );
		    if ($parent) {
			push(@taskIds, $curTask);
			$taskCnt++;
                        devmgr_info(1, 'status_startedTask', $dev);
			sleep(2);
			next;
		    }

		    $newId = manageHost(split(/\|/,$devs{$key}));
		    if ($newId > 0) {
                        $rc = AgentMgr::installMode( [ $newId ], $installMAPPs );
                        if ( $rc >= 0 ) {
                            Task::setTaskCompleted($curTask);
                        } else {
                            Task::setTaskFailed($curTask);
                        }
                        exit(0);
		    }

		    Task::setTaskFailed($curTask);
		    exit(0);
		}
	    }

            devmgr_info(0, 'status_waitingForTasks');

	    while ($taskCnt > 0) {
		sleep(2);
		for ($slot = 0; $slot <= $#taskIds; $slot++) {
		    next if ($taskIds[$slot] == 0);
		    $status = Task::getTaskStatus($taskIds[$slot]);
		    if ($status =~ /^[CFU]$/) {
			$taskIds[$slot] = 0;
			$taskCnt--;
			$failed++ unless ($status eq "C");
		    }
		}
	    }

            devmgr_info(1, 'status_tasksComplete');

            if ( $failed == 0 ) {
                $retCode = $BATCH_RESULT{'success'};
            } elsif ( $failed == $cnt ) {
                $retCode = $BATCH_RESULT{'all_failed'};
            } elsif ( $failed > 0 ) {
                $retCode = $BATCH_RESULT{'some_failed'};
            } else {
                die "Unexpected condition: cnt=$cnt, failed=$failed";
            }

	    last CASE_ADD_SRC;
	}

	# default
	return devmgr_warn(0, 'invalidSource', $source);
    }

    $rc = AgentMgr::installMode( \@newApplianceList, $installMAPPs );
    return devmgr_warn(0, 'devmgrAgentMgrFailed') if ($rc == -1);

    return $retCode;
}

sub doApprove (@) {
    Progress::setProcessName(BDi18n::getMsg('task_approveHosts'));

    my $workUnitsPerAppliance = $WORK_UNITS_APPROVE;

    my $retCode;

    my $source = shift @_;
    unless (defined $source && $source ne "") {
        return devmgr_warn(0, 'invalidSource', '');
    }

    CASE_APPROVE_SRC : {
        if ( ( $source eq 'direct' ) || ( $source eq 'host' ) ) {
            Progress::setProcessName(BDi18n::getMsg('task_approveThisHost',
                                                    { 'dev' => $_[0] } ));
            Progress::setWorkItems($workUnitsPerAppliance);
            my $rc = approveHost(@_);
            if ( $rc < 0 ) {
                $retCode = $BATCH_RESULT{'all_failed'};
            } else {
                $retCode = $BATCH_RESULT{'success'};
            }

            last CASE_APPROVE_SRC;
        }
        
        if ( $source eq 'batch' ) {
            Progress::setWorkItems( $workUnitsPerAppliance * scalar(@_) );
            my $hosts_to_approve = scalar(@_);
            my $hosts_approved   = 0;
            foreach my $host (@_) {
                Progress::setProcessName(BDi18n::getMsg('task_approveThisHost',
                                                        { 'dev' => $host } ));
                my $rc = approveHost($host);
                $hosts_approved++ if ( $rc >= 0 );
            }

            if ( $hosts_approved == $hosts_to_approve ) {
                $retCode = $BATCH_RESULT{'success'};
            } elsif ( $hosts_approved == 0 ) {
                $retCode = $BATCH_RESULT{'all_failed'};
            } elsif ( $hosts_approved < $hosts_to_approve ) {
                $retCode = $BATCH_RESULT{'some_failed'};
            } else {
                die "Unexpected condition: hosts_to_approve=$hosts_to_approve, hosts_approved=$hosts_approved";
            }
            
            last CASE_APPROVE_SRC;
        }
    }
    
    return $retCode;
}

#
# doScan
#
# arguments:
#   One or more strings representing parseable IP ranges.
#   (Direct from the command line is fine.)
#
# return value:
#   The number of hosts scanned and successfully entered in the database.
#
sub doScan (@) {
    Progress::setProcessName(BDi18n::getMsg('task_scanHosts'));
    Progress::setWorkItems($WORK_UNITS_SCAN);

    #
    # step 1: parse input for IP addresses
    #
    my @ip_ranges = ();

    if (defined $_[0] && $_[0] eq 'file') {
        # get IP ranges from an external file
        # remove 'file' keyword
        shift @_;
        # pick off arguments
        my $filename = shift @_;
        my $delete_when_done = shift @_;

        return devmgr_warn( 0, 'error_noScanFilename' )
            unless defined( $filename );

        if (defined $delete_when_done) {
            unless ( lc($delete_when_done) eq 'delete' ) {
                # wasn't the expected keyword, process later as an IP range
                unshift @_, $delete_when_done;
            }
        }

        return devmgr_warn( 0, 'cantOpenFile', $filename )
            unless ( -r $filename );

        my $FH;
        return devmgr_warn( 0, 'cantOpenFile', $filename )
            unless ( open($FH, $filename) );

        while ( my $line = get_valid_line($FH) ) {
            push( @ip_ranges, split( /(?:\s+)|\|/, $line ) );
        }

        close($FH);

        # delete the input file if requested
        if ( defined $delete_when_done &&
             lc($delete_when_done) eq 'delete' ) {
            unlink($filename) or
                return devmgr_warn( 0,
                                    'error_unlinkFailed',
                                    $filename );
        }
    }

    # get (possibly additional) IP ranges directly from the command line
    push( @ip_ranges, @_ );

    # get IPs from IP ranges
    my @host_ips = ();
    for my $ip_range (@ip_ranges) {
        my $range_ips = CLIips::getIPs( [ $ip_range ] );
        return devmgr_warn( 0, 'error_badIPRange', $ip_range )
            if ( scalar @$range_ips == 0 );

        push( @host_ips, @$range_ips );
    }

    return devmgr_warn( 0, 'error_noValidIPs' )
        if ( scalar @host_ips == 0 );

    #
    # step 2: scan IPs for hostkeys
    #
    # FIXME: disabled for string freeze
    # devmgr_info( 1, 'status_scanningIPs', scalar(@host_ips) );
    Progress::advanceProgress(1);

    BDUtil::printDebug( 'Running ' . "$Ssh_Keyscan_Path -t rsa @host_ips 2>/dev/null" );
    my $ssh_keyscan_pipe;
    open( $ssh_keyscan_pipe, "$Ssh_Keyscan_Path -t rsa @host_ips 2>/dev/null |");
    return devmgr_warn(0, "error_keyscanPipe")
        if ( not $ssh_keyscan_pipe );

    my $scanned_hosts_count = 0;
    while ( my $scanned_host_entry = $ssh_keyscan_pipe->getline ) {
        chomp $scanned_host_entry;
        my ( $scanned_host, $scanned_type, $scanned_public_host_key ) = split( /\s+/, $scanned_host_entry, 3 );

        my $rc = processScannedHost( $scanned_host, $scanned_host_entry );

        if ( $rc < 0 ) {
            devmgr_warn( 0, 'error_scanHostFailed', $scanned_host );
        } else {
            $scanned_hosts_count++;
        }
    }

    # complete progress bar
    # appropriate message will be issued by call to devmgr_complete in main()
    Progress::updateProgress(1);

    return $scanned_hosts_count;
}

sub doDelete (@) {
    Progress::setProcessName(BDi18n::getMsg('task_remove'));
    my ($dev,$subdev,@devs,$id,$rc,$cnt);
    
    return devmgr_warn(1, 'error_noHostsToDelete')
        unless (defined $_[0] && $_[0] ne "");
    
    my $retCode = 0;
    # build the list of devices to delete
    
    foreach $dev (@_) {
	push(@devs,($dev =~ /\,/ ? split(/\,/,$dev) : $dev));
    }
    
    $cnt = scalar(@devs);
    return devmgr_warn(1, 'error_noHostsToDelete')
        if ($cnt < 1);
    
    # 1 for all cleanup scripts, then 1 each for agent mgr stuff, and 1
    # for manager deregistration and local cleanup
    Progress::setWorkItems(($cnt * 2) + 1);
    
    # now delete the devices
    my @applianceIds = ();
    foreach $dev (@devs) {
        my ($id, $ipaddr);
	if ($dev =~ /^[0-9]+$/) {
            # looks like a device ID
            my $ipaddr = Appliance::getApplianceIPaddress($dev);
            if ($ipaddr) {
                $id = $dev;
            } else {
                devmgr_warn(1, 'error_hostNotFoundForID', $dev);
                next;
            }
        } else {
            # assume this is an IP address or hostname
            $id = Appliance::getApplianceId(SCSUtil::getIPAddress($dev));
            if ($id <= 0) {
                devmgr_warn(1, 'error_hostNotFoundForIP', $dev);
                next;
            }
        }
        push(@applianceIds, $id);
    }
    
    return devmgr_warn(1, 'error_noHostsToDelete')
	if (scalar(@applianceIds) < 1);
    
    # remove modules
    AgentMgr::removeMode(\@applianceIds);
    
    my $ipList = Appliance::getApplianceIPaddressList(@applianceIds);
    foreach $dev (@{$ipList}) {
        RawAgentInterface::RemovePublicKey($dev->{'ip_address'});
        Appliance::removeAppliance($dev->{'appliance_id'});
        devmgr_info(1, 'hostDeleted', $dev->{'ip_address'});
    }
    
    return $retCode;
}

#------------------------------------------------------------
#
# Utility
#
#------------------------------------------------------------
sub devmgr_info ($;$$) {
    my $updateP = shift;
    my $tag = shift || '';
    my $parm = shift || '';

    my $msg = '';
    $msg = BDi18n::getMsg($tag, {'dev' => $parm})
        if ($tag ne '');

    Progress::advanceProgress($updateP) if ($updateP);
    Progress::setMessage($msg);
    Progress::infoEvent($msg);
    BDUtil::info($msg);
}

sub devmgr_warn ($;$$) {
    my $updateP = shift;
    my $tag = shift || '';
    my $parm = shift || '';

    my $msg = '';
    $msg = BDi18n::getMsg($tag, {'dev' => $parm})
        if ($tag ne '');

    Progress::advanceProgress($updateP) if ($updateP);
    Progress::setMessage($msg);
    Progress::warningEvent($msg);
    BDUtil::warning($msg);

    return -1;
}

sub managedFailed ($$$$) {
    my ($work, $tag, $id, $ipaddr) = @_;

    Appliance::setManagedFailed($id, BDi18n::getMsg($tag, {'dev' => $ipaddr} ));

    return devmgr_warn($work, $tag, $ipaddr);
}

sub devmgr_complete (@) {
    my $msg = BDi18n::getMsg(@_);
    # devmgr_info(0, $tag);
    Progress::progressComplete($msg);
}

sub devmgr_failed (@) {
    my $msg = BDi18n::getMsg(@_);
    # devmgr_warn(0, $tag);
    Progress::progressFailed($msg);
}

sub devmgr_warned (@) {
    my $msg = BDi18n::getMsg(@_);
    # devmgr_warn(0, $tag);
    Progress::progressWarned($msg);
}

#------------------------------------------------------------

sub decodeParam ($) {
    my $v = shift;

    return '' if ($v eq '');

    $v =~ tr/+?/  /;
    $v =~ s/%([0-9a-fA-F]{2})/pack('H2',$1)/ge; # decode %xx
    $v =~ s/\"/\\\"/g;

    return $v;
}

#
# get_valid_line
#
# Reads lines from a filehandle, discarding empty lines and 
# lines that start with #, and returns the first one.
#
sub get_valid_line($)
{
    my $FH = $_[0];

    while (<$FH>) {
        next if ( /^[ \t\r\n]*$/ ||
                  /^[ \t\r\n]*\#/ );
        return $_;
    }
    
    return undef;
}

#
# manageHost
#
sub manageHost {
    my $host = shift @_;
    return devmgr_warn( $WORK_UNITS_MANAGE, 'error_noHostsToManage' )
        unless (defined $host && $host ne "");
    $host = decodeParam($host);

    my $ipaddr = SCSUtil::getIPAddress($host);
    return devmgr_warn( $WORK_UNITS_MANAGE, 'error_resolveFailed', $host )
	unless (defined $ipaddr && $ipaddr ne "-1");
    $ipaddr = decodeParam($ipaddr);

    my $fingerprint = shift @_;
    $fingerprint = decodeParam($fingerprint)
        if (defined $fingerprint && $fingerprint ne "");
    $fingerprint = '' if ( not defined $fingerprint );

    my $email = shift @_ || '';
    my $decodedEmail = decodeParam($email);
    if ( SCSUtil::checkEmailAddress($decodedEmail) ) {
        $email = $decodedEmail;
    } else {
        unshift @_, $email;
        $email = '';
    }

    my $desc = decodeParam(join(' ',@_)) || '';

    BDUtil::info('Validated parameters, IP address = ' . $ipaddr . ', adding');
    BDUtil::printDebug("HOST   is $host");
    BDUtil::printDebug("IPADDR is $ipaddr");
    BDUtil::printDebug("FINGER is $fingerprint");
    BDUtil::printDebug("EMAIL  is $email");
    BDUtil::printDebug("DESC   is $desc");

    devmgr_info(0, 'status_manageStart', $ipaddr);

    # step 1, verify appliance is approved
    my $id;
    LOOP_MANAGE : {
        $id = Appliance::getApplianceId($ipaddr);
        if ( $id <= 0 ) {
            # no existing approved entry found, try to approve the box for management
            my $rc = approveHost( $host, $fingerprint, $email, $desc );
            return devmgr_warn( $WORK_UNITS_MANAGE - $WORK_UNITS_APPROVE,
                                'error_approveFailed', $ipaddr )
                if ( $rc < 0 );
            redo LOOP_MANAGE;
        }

        my ($rc, $hosts_ref) =  Appliance::getApplianceDetail($id);
        return devmgr_warn( $WORK_UNITS_MANAGE - $WORK_UNITS_APPROVE,
                            'error_dbQueryError', $ipaddr)
            if ( $rc < 0 );

        my $host_record_ref = $hosts_ref->[0];
        BDUtil::printDebug("Found host matching ipaddress.  Status is '$host_record_ref->{'status'}'");
        if ( $host_record_ref->{'status'} eq 'M' ) {
            # This host is already managed by this control station
            return devmgr_warn( $WORK_UNITS_MANAGE - $WORK_UNITS_APPROVE,
                                'error_alreadyManaged', $ipaddr );
        } elsif ( $host_record_ref->{'status'} eq 'S' ) {
            # This host has been autoscanned, but not approved for management
            my $rc = approveHost( $host, $fingerprint, $email, $desc );
            return devmgr_warn( $WORK_UNITS_MANAGE - $WORK_UNITS_APPROVE,
                                'error_approveFailed', $ipaddr )
                if ( $rc < 0 );
            redo LOOP_MANAGE;
        } elsif ( $host_record_ref->{'status'} eq 'F' ) {
            # Failed on a previous try, so we'll try again
            my $rc = approveHost( $host, $fingerprint, $email, $desc );
            return devmgr_warn( $WORK_UNITS_MANAGE - $WORK_UNITS_APPROVE,
                                'error_approveFailed', $ipaddr )
                if ( $rc < 0 );
            redo LOOP_MANAGE;
        } elsif ( $host_record_ref->{'status'} ne 'A' ) {
            # It's not approved, so just what is it?
            return devmgr_warn( $WORK_UNITS_MANAGE - $WORK_UNITS_APPROVE,
                                'error_dbQueryError', $ipaddr );
        }
    }

    # Step 2: fetch build information
    devmgr_info( 1, 'status_checkBuildStart', $ipaddr );
    
    # mark as managed with unknown build type so we can 
    # connect and check build information

    my $rc = Appliance::setManagedAppliance( $id, undef );
    return devmgr_warn( $WORK_UNITS_MANAGE - $WORK_UNITS_APPROVE - 1,
                        'error_dbUpdateError', $ipaddr )
        if ($rc < 0);

    my $build_type = AgentInterface::PingAgent($ipaddr);
    return managedFailed( $WORK_UNITS_MANAGE - $WORK_UNITS_APPROVE - 1,
                          'error_connFailed', $id, $ipaddr )
        if ( $build_type eq -1 );

    BDUtil::printDebug("$ipaddr is a $build_type");

    # return an error if it is a control station
    return managedFailed( $WORK_UNITS_MANAGE - $WORK_UNITS_APPROVE - 1,
                          'error_cantManageControlStation', $id, $ipaddr )
        if ($build_type =~ /^ControlStation/);

    # Step 3: update database
    
    $rc = Appliance::setManagedAppliance( $id, $build_type );
    return managedFailed( $WORK_UNITS_MANAGE - $WORK_UNITS_APPROVE - 1,
                          'error_dbUpdateError', $id, $ipaddr )
        if ($rc < 0);

    $rc =       Appliance::setEmail( $id, $email );
    return devmgr_warn( 1, 'error_dbUpdateError', $ipaddr )
        if ( $rc < 0 );
    
    $rc = Appliance::setDescription( $id, $desc  );
    return devmgr_warn( 1, 'error_dbUpdateError', $ipaddr )
        if ( $rc < 0 );
    
    # Finished managing, rest of work units for AgentMgr
    devmgr_info( 1, 'status_manageThisHostSuccess', $ipaddr );

    BDUtil::printDebug("Returning host id $id");
    return $id;
}

#
# approveHost
#
sub approveHost ($$$$$) {
    my ( $host, $fingerprint, $email, $desc ) = @_;
    my $rc;

    BDUtil::printDebug("Entering approveHost with args @_");

    my $ipaddr = SCSUtil::getIPAddress($host);
    return devmgr_warn( 3, 'error_resolveFailed', $host )
        if (not SCSUtil::isIPAddress($ipaddr));

    # get fingerprint from database if present and none was passed on the command line
    my $id = Appliance::getApplianceId($ipaddr);

    BDUtil::printDebug("Matched $ipaddr to id $id");

    if ( $id > 0 ) {
        my ( $rc, $hosts_ref ) = Appliance::getApplianceDetail($id);
        return devmgr_warn( 3, 'error_dbQueryError', $ipaddr ) if ( $rc < 0 );

        my $host_record_ref = $hosts_ref->[0];
        if ( $host_record_ref->{'status'} eq 'M' ) {
            return devmgr_warn( 3, 'error_alreadyManaged', $ipaddr );
        } elsif ( $host_record_ref->{'status'} eq 'A' ) {
            return devmgr_warn( 3, 'error_alreadyApproved', $ipaddr );
        }

        # use fingerprint from database if nothing provided on commandline to override
        $fingerprint = $host_record_ref->{'fingerprint'} if ( not defined $fingerprint or
                                                              $fingerprint eq '' );
    }

    # make sure we have a human-authenticated fingerprint available
    return devmgr_warn( 3, 'error_needFingerprint', $ipaddr)
        if ( not defined $fingerprint or
             $fingerprint eq '' );
    #
    # step 1: get fingerprint from remote host
    #
    devmgr_info(0, 'status_checkFingerprint', $ipaddr);

    my $actual_public_key = getHostPublicKey($ipaddr);
    return devmgr_warn( 3, 'error_getPublicKeyFailed', $ipaddr )
        if ( $actual_public_key eq -1 );
    
    my $actual_fingerprint = computeFingerprint($actual_public_key);
    return devmgr_warn( 3, 'error_computeFingerprintFailed', $ipaddr )
        if ( $actual_fingerprint eq -1 );

    BDUtil::printDebug("Public key from host is $actual_public_key");
    BDUtil::printDebug("Fingerprint from host is $actual_fingerprint");

    # check whether provided host key fingerprint matches the actual one
    return devmgr_warn( 3, 'error_fingerprintMismatch', $ipaddr )
        if ( $actual_fingerprint ne $fingerprint );

    #
    # step 2: add public key to SSH known hosts file
    #
    devmgr_info( 1, 'status_addPublicKey', $ipaddr );

    $rc = RawAgentInterface::AddPublicKey($actual_public_key);
    return devmgr_warn( 2, 'error_addPublicKeyFailed', $ipaddr )
        if ( $rc < 0 );

    #
    # step 3: update database
    #
    devmgr_info( 1, 'status_updateDB', $ipaddr );
    if ( $id <= 0 ) {
        # no existing entry
        $id = Appliance::addAppliance( $ipaddr,
                                       $host,
                                       # store nothing for username, password
                                       '',
                                       '',
                                       $email,
                                       $desc );
        return devmgr_warn( 1, 'error_dbUpdateError', $ipaddr )
            if ( $id <= 0 );
    } else {
        # update description and e-mail in existing entry
        if (defined $email) {
            $rc =       Appliance::setEmail( $id, $email );
            return devmgr_warn( 1, 'error_dbUpdateError', $ipaddr )
                if ( $rc < 0 );
        }

        if (defined $desc) {
            $rc = Appliance::setDescription( $id, $desc  );
            return devmgr_warn( 1, 'error_dbUpdateError', $ipaddr )
                if ( $rc < 0 );
        }
    }

    $rc = Appliance::setApprovedAppliance($id, $fingerprint);
    return devmgr_warn( 1, 'error_dbUpdateError', $ipaddr )
        if ( $rc < 0 );

    devmgr_info( 1, 'status_approveThisHostSuccess', $ipaddr );

    return $id;
}

#
# processScannedHost
#
sub processScannedHost ($$) {
    my ( $ipaddr, $publickey ) = @_;

    my $fingerprint = computeFingerprint($publickey);
    return devmgr_warn(0, 'error_computeFingerprintFailed', $ipaddr)
        if ( $fingerprint eq -1 );

    devmgr_info( 0, 'status_scannedHost', "$ipaddr ($fingerprint)" );

    # check database for existing entry
    my $id = Appliance::getApplianceId($ipaddr);
    if ( $id <= 0 ) {
        # no prior entry
        my $hostname = SCSUtil::getHostname($ipaddr);
        $hostname = $ipaddr if ($hostname eq -1);

        $id = Appliance::addScannedAppliance( $ipaddr,
                                              $hostname,
                                              $fingerprint );

        return ( $id <= 0 ) ? devmgr_warn( 0, 'error_dbUpdateError', $ipaddr ) :$id;
    } else {
        my ( $rc, $hosts_ref ) = Appliance::getApplianceDetail($id);
        return devmgr_warn( 0, 'error_dbQueryError', $ipaddr ) if ( $rc < 0 );

        my $host_record_ref = $hosts_ref->[0];
        if ( $host_record_ref->{'status'} eq 'S' or $host_record_ref->{'status'} eq 'F' ) {
            $rc = Appliance::setScannedAppliance( $id, $fingerprint );
            return devmgr_warn( 0, 'error_dbUpdateError', $ipaddr ) if ( $rc < 0 );
        }
    }

    return 0;
}

#
# getHostPublicKey
#
# Connects to a host via IP address using ssh-keyscan and fetches the
# RSA public key for the host in "<ipaddr> <key-type> <key>" format.
# Returns -1 if anything goes wrong.
#
sub getHostPublicKey ($) {
    my ($ipaddr) = @_;
    my $Ssh_Keyscan_Path = PropertyUtil::getProperty('cmd.ssh-keyscan');
    my $key = `$Ssh_Keyscan_Path -t rsa $ipaddr 2>/dev/null`;

    return -1 if ( $? != 0 );

    chomp $key;
    return $key;
}

#
# computeFingerprint
#
# Gets the fingerprint for an SSH key using ssh-keygen -l.
# Expects the key as a single string in "<ipaddr> <key-type> <key>" format.
# Returns -1 if an error occurs.
#
sub computeFingerprint ($) {
    my ($public_key) = @_;

    my ( $temp_fh, $temp_filename ) = IO::File::new_tmpfile;
    return -1 if ( not defined $temp_fh );
    my $temp_fd = fileno($temp_fh);
    $temp_fh->autoflush;

    # Clear the close-on-exec flag, so the ssh-keygen child process
    # will inherit the temporary file description and can access it through
    # /dev/fd/*.  See 'man File::Temp' under WARNING for more details.
    fcntl( $temp_fh, Fcntl::F_SETFD, 0 )
        or return -1;

    print $temp_fh "$public_key\n";

    my $fingerprint = `$Ssh_Keygen_Path -l -f /dev/fd/$temp_fd`;
    close($temp_fh);

    return -1 if ( $? != 0 );

    chomp $fingerprint;
    # isolate actual fingerprint from the keysize [0] and hostname [2]
    return (split( /\s+/, $fingerprint ))[1];
}

#
# main
#
{
    my $RetCode          = 0;
    
    # avoid "locale" warnings by undef'ing locale related env vars
    
    $ENV{'LANGUAGE'} = "" if (defined $ENV{'LANGUAGE'});
    $ENV{'LANG'} = "" if (defined $ENV{'LANG'});
    
    Progress::initProgress(\@ARGV, 'devmgr');
    BDi18n::setDomain('base-mgmt-import');
    BDUtil::openLog('devmgr');
    
    my $MODE = shift @ARGV;
    if (!defined($MODE)) {
        printUsage();
        exit(1);
    }
    
    CASE_MODE : {
        
        if ($MODE eq "debug") {
            BDUtil::setDebug();
            BDUtil::printDebug("Enabled");
            $MODE = shift @ARGV;
            $MODE = "" unless (defined($MODE));
            redo CASE_MODE;
        }
        
        if (($MODE eq "manage") || ($MODE eq 'managen')) {
            BDUtil::printDebug("MODE is '$MODE'");
            my $rc = doManage( $MODE eq 'manage', @ARGV );
            if ( $rc == $BATCH_RESULT{'all_failed'} ) {
                devmgr_failed('devmgrDevAddFailed');
                $RetCode = 1;
            } elsif ( $rc == $BATCH_RESULT{'some_failed'} ) {
                devmgr_warned('devmgrDevAddSomeFailed');
                $RetCode = 1;
            } elsif ( $rc == $BATCH_RESULT{'success'} ) {
                devmgr_complete('devmgrDevAddComplete');
            } else {
                devmgr_failed('devmgrDevAddUnexpectedResult');
            }
            last CASE_MODE;
        }
        
        if ($MODE eq "approve") {
            BDUtil::printDebug("MODE is '$MODE'");
            my $rc = doApprove(@ARGV);
            if ( $rc == $BATCH_RESULT{'all_failed'} ) {
                devmgr_failed('devmgrDevApproveFailed');
                $RetCode = 1;
            } elsif ( $rc == $BATCH_RESULT{'some_failed'} ) {
                devmgr_warned('devmgrDevApproveSomeFailed');
                $RetCode = 1;
            } elsif ( $rc == $BATCH_RESULT{'success'} ) {
                devmgr_complete('devmgrDevApproveComplete');
            } else {
                devmgr_failed('devmmgrDevApproveUnexpectedResult');
            }
            last CASE_MODE;
        }
        
        if ($MODE eq 'scan') {
            BDUtil::printDebug("MODE is '$MODE'");
            my $count = doScan(@ARGV);
            if ( $count < 0 ) {
                devmgr_failed('devmgrDevScanFailed');
                $RetCode = 1;
            } else {
                devmgr_complete( 'devmgrDevScanComplete',
                             { 'count' => $count } );
            }
            last CASE_MODE;
        }
        
        if ($MODE eq "del") {
            BDUtil::printDebug("MODE is '$MODE'");
            my $rc = doDelete(@ARGV);
            if ($rc == -1) {
                devmgr_failed('rmApplFail');
                $RetCode = 1;
            } else {
                devmgr_complete('rmApplSuccess');
            }
            last CASE_MODE;
        }
        
        # default
        BDUtil::warning("Unknown Mode: '$MODE'");
        printUsage();
        $RetCode=1;
    }
    
    BDUtil::closeLog();
    exit ($RetCode);
}
