# $Id: TreeXml.pm,v 1.3 2003/09/16 11:54:07 pp122837 Exp $

# Copyright (c) 2003 Sun Microsystems, Inc. All rights reserved
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.

package TreeXml;

BEGIN {
	require Exporter;

	use vars qw( @ISA @EXPORT @EXPORT_OK);
	@ISA    = qw( Exporter );       
	@EXPORT = qw(  
		&writeXml 
		&readXml 
		&collapse 
		&array_to_hash 
		&value_to_xml 
		&escape_value  
	);      

	
	
}

sub getListFromElement
{
  my $element = shift || '';

  return 0 if ($element eq '');
  my @reqs = ();
  if (ref($element) eq 'ARRAY')
  {
    @reqs = @{$element};
  }
  else
  {
    @reqs = ($element);
  }
  return \@reqs;
}

sub writeXml {
    my $ref = shift;
    my $filename = shift;

    # Wrap top level arrayref in a hash
    if(ref($ref) eq 'ARRAY') {
	$ref = { anon => $ref };
    }
    
    my $xml = value_to_xml($ref,'',{},'');
    
    if (!$filename eq '') {
	open(XMLOUT, ">$filename") || die "could not open file: $!";
	print XMLOUT $xml;
	close(XMLOUT);
    }

    else {
	return($xml);
    }
}

sub readXml {
    my $filename = shift;
    my $keepRoot = shift;

    unless(-f $filename) {
	warn("ERROR: Cannot find filename: $filename");
    }
		
    require XML::Parser;

    my $xp = new XML::Parser(Style => 'Tree');
    my $tree = $xp->parsefile($filename);
  
    my $ref = collapse(@{$tree->[1]});
    
    if ($keepRoot eq 1) {
	my $rootRef = {};
	$rootRef->{$tree->[0]} = $ref;
	return $rootRef;
    }

    return $ref;

}


##############################################################################
# Borrowed from XML::Simple
# Method: collapse()
#
# Takes the parse tree that XML::Parser produced from the supplied XML and
# recurses through it 'collapsing' unnecessary levels of indirection (nested
# arrays etc) to produce a data structure that is easier to work with.
#
# Elements in the original parser tree are represented as an element name
# followed by an arrayref.  The first element of the array is a hashref
# containing the attributes.  The rest of the array contains a list of any
# nested elements as name+arrayref pairs:
#
#  <element name>, [ { <attribute hashref> }, <element name>, [ ... ], ... ]
#
# The special element name '0' (zero) flags text content.
#
# This routine cuts down the noise by discarding any text content consisting of
# only whitespace and then moves the nested elements into the attribute hash
# using the name of the nested element as the hash key and the collapsed
# version of the nested element as the value.  Multiple nested elements with
# the same name will initially be represented as an arrayref, but this may be
# 'folded' into a hashref depending on the value of $keyattr.
#

sub collapse {
  #my $self = shift;  no OO for us

  my $attr  = shift;

  # Add any nested elements

  my($key, $val);
  while(@_) {
    $key = shift;
    $val = shift;

    if(ref($val)) {
      $val = collapse(@$val);
    }
    elsif($key eq '0') {
      next if($val =~ m{^\s*$}s);  # Skip all whitespace content
      if(!%$attr  and  !@_) {      # Short circuit text in tag with no attr
        return($val);
      }
    }	  
            
    # Combine duplicate attributes into arrayref if required

    if(exists($attr->{$key})) {
      if(ref($attr->{$key}) eq 'ARRAY') {
        push(@{$attr->{$key}}, stripType($val));
      }
      else {
        $attr->{$key} = [ $attr->{$key}, stripType($val) ];
      }
    }
    elsif(ref($val) eq 'ARRAY') {  # Handle anonymous arrays
      $attr->{$key} = [ $val ];
    }
    else {
	$attr->{$key} = stripType($val);
    }
  }
  
  
  # Turn arrayrefs into hashrefs 

  my $count = 0;

    while(($key,$val) = each %$attr) {
      if(ref($val) eq 'ARRAY') {
  	$attr->{$key} = array_to_hash($key, $val);
      }
      $count++;
    }

  # Fold hashes containing a single anonymous array up into just the array

  if($count == 1  and  ref($attr->{anon}) eq 'ARRAY') {
    return($attr->{anon});
  }

  return($attr)

}

sub stripType {
    my $val = shift;
    
    return $val if $TYPE;

    if (exists $val->{value}) {
	return $val->{value};
    }
    elsif (exists $val->{bool}) {
	return $val->{bool};
    }
    elsif (exists $val->{int}) {
	return $val->{int};
    }
    else {
	return $val;
    }
}

##############################################################################
# Borrowed from XML::Simple
# Method: array_to_hash()
#
# Helper routine for collapse().
# Attempts to 'fold' an array of hashes into an hash of hashes.  Returns a
# reference to the hash on success or the original array if folding is
# not possible.  Behaviour is controlled by 'keyattr' option.
#

sub array_to_hash {

  my $name     = shift;
  my $arrayref = shift;

  my $hashref  = {};

  my($i, $key, $val);

  # Handle keyattr => { .... }

    return($arrayref) unless(exists($keyattr->{$name}));
    ($key) = $keyattr->{$name};
    for($i = 0; $i < @$arrayref; $i++)  {
      if(ref($arrayref->[$i]) eq 'HASH' and exists($arrayref->[$i]->{$key})) {
	$val = $arrayref->[$i]->{$key}->{'value'}; # a bit hacky 
	$hashref->{$val} = { %{$arrayref->[$i]} };
	
	delete $hashref->{$val}->{$key};
      }
      else {
	warn "Warning: <$name> element has no '$key' key attribute" if($^W);
	return($arrayref);
      }
    }

  return($hashref);
}


##############################################################################
# Borrowed from XML::Simple
# Method: value_to_xml()
#
# Helper routine for writeXml() - recurses through a data structure building up
# and returning an XML representation of that structure as a string.
# 
# Arguments expected are:
# - the data structure to be encoded (usually a reference)
# - the XML tag name to use for this item
# - a hashref of references already encoded (to detect recursive structures)
# - a string of spaces for use as the current indent level
#

sub value_to_xml {
  
  my($ref, $name, $encoded, $indent) = @_;

  my $named = (defined($name) and $name ne '' ? 1 : 0);

  my $nl = "\n";

  if(ref($ref)) {
    die "recursive data structures not supported" if($encoded->{$ref});
    $encoded->{$ref} = $ref;
  }
  else {
    if($named) {
      return(join('',
              $indent, '<', $name, '>',
	      escape_value($ref),
              '</', $name, ">", $nl
	    ));
    }
    else {
      return("$ref$nl");
    }
  }

  # Unfold hash to array if possible

  #if(ref($ref) eq 'HASH'               # It is a hash
  #   and %$ref                         # and it's not empty
  #   and $self->{opt}->{keyattr}       # and folding is enabled
  #   and $indent                       # and its not the root element
  #) {
  #  $ref = hash_to_array($name, $ref);
  #}

  
  my @result = ();
  my($key, $value);


  # Handle hashrefs

  if(ref($ref) eq 'HASH') {
    my @nested = ();
    my $text_content = undef;
    if($named) {
      push @result, $indent, '<', $name;
    }

    if(%$ref) {
      while(($key, $value) = each(%$ref)) {
	next if(substr($key, 0, 1) eq '-');
	if(!defined($value)) {
	  unless(exists($self->{opt}->{suppressempty})
	     and !defined($self->{opt}->{suppressempty})
	  ) {
	    warn 'Use of uninitialized value';
	  }
	  $value = {};
	}
	if(ref($value)  or  $self->{opt}->{noattr}) {
	  push @nested,
	    value_to_xml($value, $key, $encoded, "$indent  ");
	}
	else {
	  $value = escape_value($value) unless($self->{opt}->{noescape});
	  if($key eq $self->{opt}->{contentkey}) {
	    $text_content = $value;
	  }
	  else {
	    push @result, ' ', $key, '="', $value , '"';
	  }
	}
      }
    }
    else {
      $text_content = '';
    }

    if(@nested  or  defined($text_content)) {
      if($named) {
        push @result, ">";
	if(defined($text_content)) {
	  push @result, $text_content;
	  $nested[0] =~ s/^\s+// if(@nested);
	}
	else {
	  push @result, $nl;
	}
	if(@nested) {
	  push @result, @nested, $indent;
	}
	push @result, '</', $name, ">", $nl;
      }
      else {
        push @result, @nested;             # Special case if no root elements
      }
    }
    else {
      push @result, " />", $nl;
    }
  }


  # Handle arrayrefs

  elsif(ref($ref) eq 'ARRAY') {
    foreach $value (@$ref) {
      if(!ref($value)) {
        push @result,
	     $indent, '<', $name, '>',
	     escape_value($value),
	     '</', $name, ">\n";
      }
      elsif(ref($value) eq 'HASH') {
	push @result, value_to_xml($value, $name, $encoded, $indent);
      }
      else {
	push @result,
	       $indent, '<', $name, ">\n",
	       value_to_xml($value, 'anon', $encoded, "$indent  "),
	       $indent, '</', $name, ">\n";
      }
    }
  }

  else {
    die "Can't encode a value of type: " . ref($ref);
  }

  return(join('', @result));
}


##############################################################################
# Borrowed from XML::Simple
# Method: escape_value()
#
# Helper routine for automatically escaping values for XMLout().
# Expects a scalar data value.  Returns escaped version.
#

sub escape_value {
  
  my($data) = @_;

  $data =~ s/&/&amp;/sg;
  $data =~ s/</&lt;/sg;
  $data =~ s/>/&gt;/sg;
  $data =~ s/"/&quot;/sg;

  return($data);
}

1;
