package Pusher;

# Copyright (c) 2003 Sun Microsystems, Inc. All rights reserved
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.


#
# $Id: Pusher.pm,v 1.11.8.1 2005/03/11 20:23:45 ms152511 Exp $
#

use strict;
use lib '/scs/lib/perl5';
use BDUtil;
use MAPP;
use RawAgentInterface;
use BDi18n;
use PropertyUtil;

my $pePath       = PropertyUtil::getProperty('cmd.pe');
my $pkgMgrPath   = PropertyUtil::getProperty('cmd.pkgMgr');
my $clientPkgDir = PropertyUtil::getProperty('dir.csadminhome');

my $SSHPORT      = 22;
my $SSHTIMEOUT   = 5 * 60; # 5 minutes

#
#
#
sub callPkgMgr
{
  my ($conn, $mode, $pkgString) = @_;

  my $cmd = $pePath . ' ' . $pkgMgrPath . ' ' . $mode . ' ' . $pkgString;
  BDUtil::info(' - Running pkgMgr with mode "' . $mode . '"');
  my ($rc, $outStr) = RawAgentInterface::RunCmd($conn, $cmd);
  if ($rc || !$outStr)
  {
    BDUtil::setWarning($outStr || BDi18n::getMsg('pkgMgrFailed'));
    return 0;
  }

  my %resultHash = ();
  my (@t, $tKey, $tVal, $line);
  foreach $line (split(/\n/, $outStr))
  {
    @t = split(/\s+/, $line);
    $tVal = shift @t;
    if ($tVal eq 'ERROR')
    {
      BDUtil::setWarning(@t);
      return 0;
    }
    $tKey = shift @t;
    $resultHash{$tKey} = $tVal;
  }

  return \%resultHash;
}

#
#
#
sub setPkgNotInstalled
{
  my ($appliance, $pkgId, $pkgName, $reason) = @_;

  if ($reason ne '')
  {
    BDUtil::warning(' - Setting appliance id = ' . $appliance->{'appliance_id'}
                  . ' to install failed for pkg ' . $pkgName
                  . ' id = ' . $pkgId . ', reason : ' . $reason);
    Progress::warningEvent(BDi18n::getMsg('failedToInstallOnAppliance',
      { 'ip' => $appliance->{'ip_address'}, 'rpmName' => $pkgName, 'reason' => $reason }));
  }
  else
  {
    BDUtil::info(' - Setting appliance id = ' . $appliance->{'appliance_id'}
               . ' to not installed for pkg ' . $pkgName . ' id = ' . $pkgId);
  }

  return BDUtil::dbError(-1)
    if (MAPP::setPkgInstallationStatus($pkgId, $appliance->{'appliance_id'}, 'F', $reason) < 0);

  return 0;
}

#
#
#
sub setInstallsFailed
{
  my ($markMappPkgFail, $badPkg, $allFailed, $appliance, $reason, $progressCnt) = @_;

  my $pkg;
  foreach my $pkgKey (keys %{$markMappPkgFail})
  {
    $pkg = $markMappPkgFail->{$pkgKey};
    if ($allFailed || ($badPkg && ($pkg->{'mapp_id'} == $badPkg->{'mapp_id'})))
    {
      return -1
        if (setPkgNotInstalled($appliance, $pkg->{'mapp_client_rpm_id'},
                               $pkg->{'rpm_name'}, $reason));
    }
    else
    {
      return -1
        if (setPkgNotInstalled($appliance, $pkg->{'mapp_client_rpm_id'},
                               $pkg->{'rpm_name'}, ''));
    }
  }

  Progress::advanceProgress($progressCnt);

  BDUtil::setError($reason);

  BDi18n::restoreLastDomain();

  return 1;
}

#
#
#
sub setUninstallsFailed
{
  my ($progressCnt, $msg) = @_;

  BDUtil::warning($msg);
  Progress::warningEvent($msg);

  Progress::advanceProgress($progressCnt);

  BDi18n::restoreLastDomain();

  return 1;
}

#-----------------------------------------------------------
#
# installRemotePkgs
#
#-----------------------------------------------------------
sub installRemotePkgs
{
  my ($appliance, $pkgList, $updateProgress, $deleteFirst) = @_;
  $updateProgress = 1 if (!defined($updateProgress));
  $deleteFirst = 0 if (!defined($deleteFirst));

  return 0 if (!(@{$pkgList}+0));
  BDi18n::setDomain('base-mgmtapp');
  my %pkgHash = ();
  my $numPkgs = 0;
  my $pkgName;
  my $pkgString = '';
  my %mappFail = ();
  my %markMappPkgFail = ();

  foreach my $pkg (@{$pkgList})
  {
    if ($pkg->{'rpm_name'} =~ /\.rpm$/)
    {
      # Get rid of the <arch>.rpm suffix
      ($pkgName) = ($pkg->{'rpm_name'} =~ /^(.*)\.[^\.]+\.rpm$/);
    }
    else
    {
      $pkgName = $pkg->{'rpm_name'};
    }
    $pkgHash{$pkgName} = $pkg;
    ++$numPkgs;
    $pkgString .= ' ' . $pkg->{'rpm_name'};
    $mappFail{$pkg->{'mapp_id'}} = 0;
    $markMappPkgFail{$pkg->{'mapp_id'}} = $pkg;
  }

  my $progressCnt = $updateProgress?$numPkgs:0;

  BDUtil::info($appliance->{'ip_address'}
             . ' (' . $appliance->{'appliance_id'} . ') : '
             . $numPkgs . ' packages to install / increment');
  #
  # Find out which packages are not already installed and push them down
  #
  BDUtil::info(' - Establishing secure connection');
  Progress::setMessage(BDi18n::getMsg('establishingConnection',
                                      {'ip' => $appliance->{'ip_address'}}));
  my $conn = RawAgentInterface::ConnStart( $appliance->{'ip_address'},
                                           $SSHPORT,
                                           $SSHTIMEOUT );

  return setInstallsFailed(\%markMappPkgFail,
                           0,
                           1,
                           $appliance,
                           BDi18n::getMsg('couldNotConnect',
                                          { 'ip' => $appliance->{'ip_address'} }),
                           $progressCnt)
    if ($conn <= 0);

  Progress::setMessage(BDi18n::getMsg('queryingAppliance',
                       { 'ip' => $appliance->{'ip_address'} }));
  my $retHash = callPkgMgr($conn, 'lock check', $pkgString);

  return setInstallsFailed(\%markMappPkgFail, 0, 1, $appliance,
                           BDUtil::getWarning(), $progressCnt)
    if (!$retHash);

  #
  # Push the needed pkgs
  #
  foreach my $pkg (keys %{$retHash})
  {
    next if ($retHash->{$pkg} > 0);
    BDUtil::info(' - ' . $pkg . ' not installed, pushing');
    Progress::setMessage(BDi18n::getMsg('pushingPkg',
                         { 'pkg' => $pkgHash{$pkg}->{'rpm_name'},
                           'ip' => $appliance->{'ip_address'} }));
    if (RawAgentInterface::SendFile($conn,
                                    $pkgHash{$pkg}->{'path_info'} . '/' .
                                    $pkgHash{$pkg}->{'rpm_name'},
                                    $clientPkgDir . '/' .
                                    $pkgHash{$pkg}->{'rpm_name'}) <= -1)
    {
      # Failed, tell the database and quit
      return setInstallsFailed(\%markMappPkgFail,
                               $pkgHash{$pkg},
                               0,
                               $appliance,
                               BDi18n::getMsg('errorPushingFile',
                                          { 'ip' => $appliance->{'ip_address'} }),
                               $progressCnt);
    }
    Progress::infoEvent(BDi18n::getMsg('pushedPkg', 
                        { 'pkg' => $pkgHash{$pkg}->{'rpm_name'},
                          'ip' => $appliance->{'ip_address'} }));
    if ($progressCnt > 1)
    {
      Progress::advanceProgress();
      --$progressCnt;
    }
  }

  #
  # Now that they're all ready, install them (or increase reference counts)
  #
  Progress::setMessage(BDi18n::getMsg('installingPkgs',
               { 'numPkgs' => $numPkgs, 'ip' => $appliance->{'ip_address'} }));
  $retHash = callPkgMgr($conn, 'unlock add', $pkgString);

  return setInstallsFailed(\%markMappPkgFail, 0, 1, $appliance,
                           BDUtil::getWarning(), $progressCnt)
    if (!$retHash);

  my @successfulPkgs = ();
  foreach my $pkg (keys %{$retHash})  # check if it worked
  {
    if ($pkg =~ /\.rpm$/)
    {
      ($pkgName) = ($pkg =~ /^(.*)\.[^\.]+\.rpm$/);
    }
    else
    {
      $pkgName = $pkg;
    }
    if ($retHash->{$pkg} == 0)
    {
      BDUtil::info(' - ' . $pkg . ' was installed');
      Progress::infoEvent(BDi18n::getMsg('pkgInstalled',
                     { 'pkg' => $pkg, 'ip' => $appliance->{'ip_address'} }));
      push (@successfulPkgs, $pkgHash{$pkgName});
    }
    elsif ($retHash->{$pkg} > 0)
    {
      BDUtil::info(' - ' . $pkg . ' is already installed (incremented count)');
      Progress::infoEvent(BDi18n::getMsg('pkgAlreadyInstalled', 
                     { 'pkg' => $pkg, 'ip' => $appliance->{'ip_address'} }));
      push (@successfulPkgs, $pkgHash{$pkgName});
    }
    else
    {
      return -1
        if (setPkgNotInstalled($appliance, $pkgHash{$pkgName}->{'mapp_client_rpm_id'},
                               $pkg, 'pkg rc = ' . ($retHash->{$pkg} * -1)));
      $mappFail{$pkgHash{$pkgName}->{'mapp_id'}} = 1;
    }
  }

  if ($progressCnt)
  {
    Progress::advanceProgress();
    --$progressCnt;
  }

  BDUtil::info(' - Closing connection');
  Progress::setMessage(BDi18n::getMsg('closingConnection',
                                  { 'ip' => $appliance->{'ip_address'} }));
  RawAgentInterface::ConnEnd($conn);

  #
  # It's all good, commit to database
  #
  BDUtil::info(' - Commiting installed pkg records to database');
  foreach my $pkg (@successfulPkgs)
  {
    return BDUtil::dbError(-1)
        if (MAPP::setPkgInstallationStatus($pkg->{'mapp_client_rpm_id'},
                                            $appliance->{'appliance_id'},
                                            'P', '') < 0);
  }

  Progress::advanceProgress($progressCnt);

  BDi18n::restoreLastDomain();

  $appliance->{'mappFail'} = \%mappFail;

  return 0;
}

#-----------------------------------------------------------
#
# runScript
#
#-----------------------------------------------------------
sub runScript
{
  my ($scriptName, $ids) = @_;
  return 0 if (!defined($ids->[0]));

  my $cmd = $scriptName . ' ' . join(' ', @{$ids});
  BDUtil::info('Running script : "' . $cmd . '"');
  my $outStr = `$cmd`;
  my $rc = $?;
  BDUtil::info('Return code = ' . $rc);
  BDUtil::info('Output = ' . $outStr);
  ## eh, need to standardize what these things will print out...
  ## BDUtil::setError($outStr) if ($rc);
  return $rc;
}

#-----------------------------------------------------------
#
# removeRemotePkgs
#
#-----------------------------------------------------------
sub removeRemotePkgs
{
  my ($ipAddress, $pkgList, $updateProgress) = @_;
  $updateProgress = 1 if (!defined($updateProgress));

  return 0 if (!(@{$pkgList}+0));
  BDi18n::setDomain('base-mgmtapp');
  my $numPkgs = ((@{$pkgList})+0);
  my $progressCnt = $updateProgress?$numPkgs:0;

  BDUtil::info($ipAddress . ' : ' . $numPkgs
             . ' packages to uninstall / decrement');
  BDUtil::info(' - Establishing secure connection');
  Progress::setMessage(BDi18n::getMsg('establishingConnection',
                                      {'ip' => $ipAddress}));
  my $conn = RawAgentInterface::ConnStart( $ipAddress,
                                           $SSHPORT,
                                           $SSHTIMEOUT );
  return setUninstallsFailed($progressCnt,
                             BDi18n::getMsg('failedToUninstallOnAppliance',
                                        {
                                         'ip' => $ipAddress,
                                         'reason' => BDi18n::getMsg('couldNotConnect',
                                                                { 'ip' => $ipAddress })
                                        }))
      if ($conn <= 0);

  Progress::setMessage(BDi18n::getMsg('cleaningUpPkgs',
                         { 'numPkgs' => $numPkgs, 'ip' => $ipAddress }));

  my $retHash = callPkgMgr($conn, 'full del', join(' ', @{$pkgList}));
  return setUninstallsFailed($progressCnt,
     BDi18n::getMsg('failedToUninstallOnAppliance',
                    { 'ip' => $ipAddress, 'reason' => BDUtil::getWarning }))
    if (!$retHash);

  foreach my $pkg (keys %{$retHash})  # check if it worked
  {
    if ($retHash->{$pkg} == 0)
    {
      BDUtil::info(' - ' . $pkg . ' was uninstalled');
      Progress::infoEvent(BDi18n::getMsg('pkgUninstalled',
                                       { 'pkg' => $pkg, 'ip' => $ipAddress }));
    }
    elsif ($retHash->{$pkg} > 0)
    {
      BDUtil::info(' - ' . $pkg . ' still needed (decremented count)');
      Progress::infoEvent(BDi18n::getMsg('pkgStillNeeded',
                                       { 'pkg' => $pkg, 'ip' => $ipAddress }));
    }
    else
    {
      return setUninstallsFailed($progressCnt,
          BDi18n::getMsg('failedToUninstallOnAppliance',
                         { 'ip' => $ipAddress,
                           'reason' => BDUtil::getWarning }));
    }
  }

  Progress::advanceProgress($progressCnt);

  BDUtil::info('Closing connection');
  Progress::setMessage(BDi18n::getMsg('closingConnection',
                                      { 'ip' => $ipAddress }));
  RawAgentInterface::ConnEnd($conn);

  BDi18n::restoreLastDomain();

  return 0;
}

1;
