#!/usr/bin/perl -w
# $Id: InstallMAPP.pm,v 1.64.2.2.4.1 2005/03/14 20:02:24 jb122832 Exp $
# InstallMAPe.pm - monule install/uninstall library
# Copyright 2003, 2004 Sun Microsystems, Inc., All rights reserved.
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms

package InstallMAPP;

use lib '/scs/lib/perl5';
use strict;
use Carp qw(cluck);

use TreeXml;
use MAPP;
use BDUtil;
use Builds;
use Pusher;
use Progress;
use BDi18n;
use SysCmd;
use PropertyUtil;
use InitUtil;

my $base_dir = PropertyUtil::getProperty('dir.base');
my $sbin_dir = PropertyUtil::getProperty('dir.sbin');

my $tomcat4_user    = 'tomcat4';
my $tomcat4_group   = 'tomcat4';
my $tomcat_init     = 'init.scs-tomcat4';
my $schedulerTool   = $sbin_dir . '/schedule.pl';
my $tmp_path        = "/tmp/mapp/unpack.$$";
my $download_path   = "/tmp/module_$$.mapp";
my $client_rpm_path = $base_dir . '/clientRpms';
my $script_path     = $base_dir . '/mappScripts';
my $tomcat4_workdir = '/var/tomcat4/work/Standalone/localhost/sdui/';

# commands
my $rm_cmd          = PropertyUtil::getProperty('cmd.rm') . ' -rf ';
my $cp_cmd          = PropertyUtil::getProperty('cmd.cp');
my $mv_cmd          = PropertyUtil::getProperty('cmd.mv');
my $chown_cmd       = PropertyUtil::getProperty('cmd.chown');
my $chmod_cmd       = PropertyUtil::getProperty('cmd.chmod');
my $mkdir_cmd       = PropertyUtil::getProperty('cmd.mkdir') . ' -p ';
my $wget_cmd        = PropertyUtil::getProperty('cmd.wget') . ' -t 2 -T 60';

#
#
#
sub downloadFile
{
  my $source = shift @_;

  Progress::setMessage(BDi18n::getMsg('downloadingMAPP',
                                      { 'source' => $source }));
  # TODO (in the original SWUpdate.pm, this fn uses cce...)
  # SWUpdate::set_proxy();

  my $cmd = $wget_cmd . ' ' . $source . ' -O ' . $download_path;
  my $rc = system($cmd);
  if ($rc || (! -r $download_path))
  {
    BDUtil::setError(BDi18n::getMsg('failedToDownload',
                                    { 'source' => $source }));
    return 0;
  }

  Progress::infoEvent(BDi18n::getMsg('downloadedMAPP',
                                      { 'source' => $source }));
  return $download_path;
}

#
#
#
sub unpackModule
{
  my ($file, $destdir) = @_;

  BDUtil::info('Starting unpackModule');
  system($mkdir_cmd . ' ' . $destdir);
  system($chmod_cmd . ' -R 700 ' . $destdir);

  my $ret_status = SysCmd::unpackGzippedTar($file, $destdir);
  return (-1, 'cantuntar') if $ret_status != 0;

  return (0);
}

#
#
#
sub getMAPPTree
{
  my $mappName = shift @_;

  if ( ! -r $mappName)
  {
    BDUtil::setError(BDi18n::getMsg('fileOpenError',
                                    { 'fileName' => $mappName }));
    return 0;
  }

  system($rm_cmd . $tmp_path);
  system($mkdir_cmd . ' ' . $tmp_path);
  if ( ! -w $tmp_path)
  {
    BDUtil::setError(BDi18n::getMsg('dirWriteError',
                                    { 'dirName' => $tmp_path }));
    return 0;
  }

  BDUtil::info('Unpacking ' . $mappName);
  my ($err, $info) = unpackModule($mappName, $tmp_path, 0);
  if ($err)
  {
    system($rm_cmd . ' ' . $tmp_path);
    BDUtil::setError(BDi18n::getMsg('unpackError',
                                       { 'mappName' => $mappName }));
    return 0;
  }

  BDUtil::info('Parsing XML specification file');
  my $tree = readXml($tmp_path . '/module-config.xml', 0);
  return 0 if (!$tree);

  return $tree;
}

#
#
#
sub getMAPPStrutsConfig
{
  my $moduleName = shift @_;

  my $fqStruts = $tmp_path . '/struts-config-' . $moduleName . '.xml';

    if (! -r $fqStruts)
    {
      BDUtil::info(BDi18n::getMsg('strutsConfigFileMissing'));
      return 0;
    }

  return $fqStruts;
}

#
#
#
sub getMAPPValidationConfig
{
  my $moduleName = shift @_;

  my $fqValidation = $tmp_path . '/validation-' . $moduleName . '.xml';

    if (! -r $fqValidation)
    {
      # OK not to have validation rules, display info though
      BDUtil::info(BDi18n::getMsg('strutsValidationFileMissing'));
      return 0;
    }

  return $fqValidation;
}

#
#
#
sub getMAPPMenuConfig
{
  my $moduleName = shift @_;

  my $fqMenu = $tmp_path . '/menu-config-' . $moduleName . '.xml';

    if (! -r $fqMenu)
    {
      BDUtil::info(BDi18n::getMsg('menuConfigFileMissing'));
      return 0;
    }

  return $fqMenu;
}

#
#
#
sub getMAPPInfo
{
  my $mappId = shift @_;

  my ($rc, $mapp) = MAPP::getMAPP($mappId);
  if ($rc)
  {
    BDUtil::dbError();
    return (0, 0);
  }
  if (!$mapp)
  {
    BDUtil::setError(BDi18n::getMsg('mappIdUnknown', { 'id' => $mappId }));
    return (0, 0);
  }

  my $mappName = $mapp->{'vendor'} . ' ' . $mapp->{'name_tag'} . ' ' . $mapp->{'version'};
  BDUtil::info('MAPP is ' . $mappName);

  return ($mapp, $mappName);
}

#
#
#
sub prepareServerPkgs
{
  my $rpmList = shift @_;

  my ($msRpm, $fqRpm, $pkgName);
  my $msRpms = TreeXml::getListFromElement($rpmList);
  system($mkdir_cmd . ' ' . $base_dir . '/msRpms');
  foreach $msRpm (@{$msRpms}) 
  {
    $pkgName = (ref($msRpm) eq 'HASH')?$msRpm->{'name'}:$msRpm;
    BDUtil::info('Preparing server pkg ' . $pkgName);
    $fqRpm = $tmp_path . '/mgmtStation/' . $pkgName;

    if (! -r $fqRpm)
    {
      BDUtil::setError(BDi18n::getMsg('ServerPkgMissing', {'pkg' => $pkgName}));
      # return 0;
    }
    else
    {
      system($cp_cmd . ' ' . $fqRpm . ' ' . $base_dir . '/msRpms');
    }
  }

  return $msRpms;
}

#
#
#
sub installServerPkgs
{
  my ($mappId, $rpmInstallList) = @_;

  my ($sPkg, $rpm, $rc, $adminFile, $resourceFile, $count);
  my $numRpms = (@{$rpmInstallList}) + 0;

  $count = 0;

  foreach $sPkg (@{$rpmInstallList})
  {
    if (ref($sPkg) eq 'HASH')
    {
      $rpm = $sPkg->{'name'};
      $adminFile = $sPkg->{'adminFile'} || '';
      $resourceFile = $sPkg->{'resourceFile'} || '';
    }
    else
    {
      $rpm = $sPkg;
      $adminFile = $resourceFile = '';
    }
    BDUtil::info('Installing server pkg ' . $rpm);
    Progress::setMessage(BDi18n::getMsg('installingServerPkg',
                                           { 'pkg' => $rpm }));
    $rc = SysCmd::installPkg($tmp_path . '/mgmtStation/' . $rpm,
                             $tmp_path . '/mgmtStation/' . $adminFile,
                             $tmp_path . '/mgmtStation/' . $resourceFile);
    if ($rc < 0)
    {
      BDUtil::setError(BDi18n::getMsg('failedInstallServerPkg',
                                         { 'pkg' => $rpm }));
      return -1;
    }
    ++$count;
    if (!$rc)
    {
      Progress::infoEvent(BDi18n::getMsg('installedServerPkg', 
                                            { 'pkg' => $rpm }));
    }
    return BDUtil::dbError(-1)
      if (MAPP::addServerPackage($mappId, $rpm) <= 0);
    Progress::updateProgress();
  }

  return 0;
}

#
#
#
sub registerTasks
{
  my ($mappId, $taskList) = @_;

  BDUtil::info('Registering tasks');
  my $tList = TreeXml::getListFromElement($taskList);
  foreach my $task (@{$tList})
  {
    MAPP::registerTask($mappId, $task);
  }
}

#
#
#
sub registerApplianceIcons
{
  my ($mappId, $viewList) = @_;

  BDUtil::info('Registering appliance views');
  my $vList = TreeXml::getListFromElement($viewList);
  foreach my $view (@{$vList})
  {
    MAPP::registerApplianceIcon($mappId, $view);
  }
  Progress::infoEvent(BDi18n::getMsg('registeredIcons'));
}

#
#
#
sub compareVersionsClean
{
	my $A = shift @_ || return -1;
	my $B = shift @_ || return 1;

	my (@A) = ($A =~ /(\.|\-|\d+|[^\.\d]+)/g);
	my (@B) = ($B =~ /(\.|\-|\d+|[^\.\d]+)/g);
	while(@A and @B) {
		$A=shift @A;
		$B=shift @B;
		if ( ($A eq '.' and $B eq '.') or
        ($A eq '-' and $B eq '-') ) {
			next;
		} elsif ( $A eq '.' and $B eq '-' ) {
			return 1;
		} elsif ( $A eq '.' or $A eq '-' ) {
			return -1;
		} elsif ( $B eq '.' or $B eq '-' ) {
			return 1;
		} elsif($A =~ /^\d+$/ and $B =~ /^\d+$/) {
			if ($A =~ /^0/ || $B =~ /^0/) {
				return $A cmp $B if $A cmp $B;
			} else {
				return $A <=> $B if $A <=> $B;
			}
		} else {
			$A = uc $A;
			$B = uc $B;
			return $A cmp $B if $A cmp $B;
		}
	}
	@A <=> @B;
}



#
#
#
sub verifyDependencies
{
  my $required = shift @_;

  my ($req, $pkgName, $version, $rev, $pkgName2, $version2, $rev2,
      $current, $cmd, $complainFor, $pkg, $type, $rc);

  my $reqs = TreeXml::getListFromElement($required);
  foreach $req (@{$reqs})
  {
    ($type, $pkg) = SysCmd::getPkgInfo($req);
    if (!$type || !$pkg)
    {
      BDUtil::setError(BDi18n::getMsg('invalidDependency'));
      return -1;
    }
    BDUtil::info('Checking if ' . $pkg . ' or newer is installed');

    if (exists($req->{'complainForPkg'}) && ($req->{'complainForPkg'} ne ''))
    {
      $complainFor = $req->{'complainForPkg'};
    }
    else
    {
      $complainFor = $pkg;
    }

    ($rc, $current, $pkgName, $version, $rev) = SysCmd::checkPkg($pkg, $type);
    if ($rc)
    {
      BDUtil::setError(BDi18n::getMsg('requiredNotInstalled',
                                      { 'pkg' => $complainFor }));
      return -1;
    }

    if (compareVersionsClean( $current, $pkg ) < 0)
    #if (compareVersions($version, $rev, $version2, $rev2) < 0)
    {
      BDUtil::setError(BDi18n::getMsg('requiredNotInstalled',
                                      { 'pkg' => $complainFor }));
      return -1;
    }

  }
  return 0;
}

#
#
#
sub validateNewMAPP
{
  my $tree = shift @_;

  BDUtil::info('Validating MAPP info');
  my $name       = $tree->{'app'}->{'name'};
  my $vendor     = $tree->{'app'}->{'vendor'};
  my $version    = $tree->{'app'}->{'version'};
  my $nameTag    = $tree->{'app'}->{'nameTag'};
  my $vendorTag  = $tree->{'app'}->{'vendorTag'};
  my $versionTag = $tree->{'app'}->{'versionTag'};
  my $updateBool = $tree->{'app'}->{'updateBool'};
  $updateBool = 'N' if (!defined($updateBool));

  if (!defined($name) || !defined($vendor) || !defined($version) ||
      !defined($nameTag) || !defined($vendorTag) || !defined($versionTag))
  {
    BDUtil::setError(BDi18n::getMsg('missingInfo'));
    return -1;
  }

  #
  # Check if there are dependencies
  #
  if (exists($tree->{'app'}->{'required'}))
  {
    if (verifyDependencies($tree->{'app'}->{'required'}))
    {
      return -1;
    }
  }

  #
  # Check if it's denied
  #
  my $deniedVersion = MAPP::getDeniedVersion($name);
  if ($deniedVersion)
  {
    if (compareVersionsClean( $version, $deniedVersion ) <= 0)
    {
      BDUtil::setError(BDi18n::getMsg('moduleTooOld'));
      return -1;
    }
  }

  #
  # Check for other versions
  #
  my $uninstallId = 0;
  my $mappEntry = MAPP::getMAPPbyName($name);
  if ($updateBool eq 'Y')
  {
    if ($mappEntry)
    {
      my $res = compareVersionsClean( $mappEntry->{'version'}, $version );
      if ( $res == 0 )
      {
        BDUtil::setError(BDi18n::getMsg('mappAlreadyInstalled'));
        return -1;
      }
      elsif ( $res > 0 )
      {
        BDUtil::setError(BDi18n::getMsg('higherVersionInstalled'));
        return -1;
      }
      else
      {
        BDUtil::info( 'This MAPP has an older version already installed, id = ' . $mappEntry->{'mapp_id'} );
        ## ahaha, we don't have an upgrade policy :)
        ##  Just uninstall old ones first for now.
        Progress::infoEvent(BDi18n::getMsg('olderVersionInstalled'));
        $uninstallId = $mappEntry->{'mapp_id'};
      }
    }
  }
  else
  {
    if ($mappEntry)
    {
      BDUtil::setError(BDi18n::getMsg('notAnUpdate'));
      return -1;
    }
  }

  return (0, $uninstallId);
}

#
#
#
sub addMAPPEntry
{
  my $tree = shift @_;

  BDUtil::info('Commiting new MAPP record to database');
  BDUtil::info('New MAPP Vendor, Name, Version = ' . $tree->{'app'}->{'vendor'}
             . ' ' .  $tree->{'app'}->{'name'} . ' ' .  $tree->{'app'}->{'version'});

  my $mappId = MAPP::addMAPP
  (
    $tree->{'app'}->{'name'},
    $tree->{'app'}->{'vendor'},
    $tree->{'app'}->{'version'},
    $tree->{'app'}->{'updateBool'} || 'N',
    $tree->{'app'}->{'initScript'} || '',
    $tree->{'app'}->{'cleanupScript'} || '',
    $tree->{'app'}->{'nameTag'},
    $tree->{'app'}->{'vendorTag'},
    $tree->{'app'}->{'versionTag'}
  );
  return BDUtil::dbError(0) if ($mappId <= 0);

  return $mappId;
}

#
#
#
sub moveModuleConfigFiles
{ # phil style heh
  my ($strutsConfigFile, $menuConfigFile, $validationConfigFile) = @_;

  # Shouldn't hard code this (at least not here anyway)
  my $sduiConfigDir = '/var/tomcat4/webapps/sdui/WEB-INF';
  
  # hmm should do something fancy here like validate the files or check to
  # make sure they don't already exist at the target... do that later heh

  # copy for now instead of move
  if ( -r $strutsConfigFile ) {
    `$cp_cmd $strutsConfigFile $sduiConfigDir`;
    `$chown_cmd $tomcat4_user:$tomcat4_user $sduiConfigDir/struts-config-*.xml`;
  }
  if ( -r $menuConfigFile ) {
    `$cp_cmd $menuConfigFile $sduiConfigDir`;
    `$chown_cmd $tomcat4_user:$tomcat4_user $sduiConfigDir/menu-config-*.xml`;
  }

  # copy over validation file if there.
  if ( -r $validationConfigFile) {
      `$cp_cmd $validationConfigFile $sduiConfigDir`;
      `$chown_cmd $tomcat4_user:$tomcat4_user $sduiConfigDir/validation-*.xml`;
  } 
  BDUtil::info('Copying module configuration files to web application');

   # always successful :)
   return 1;
}

#
#
#
sub movePkgsAndScripts
{
  my ($tree, $clientPkgs) = @_;

  my $clientPkgPath = getClientPkgPath($tree->{'app'}->{'vendor'},
                               $tree->{'app'}->{'name'}, $tree->{'app'}->{'version'});
  `$mkdir_cmd $clientPkgPath`;
  if ($clientPkgs)
  {
    BDUtil::info('Putting client pkgs in ' . $clientPkgPath);
    my $source;
    foreach my $pkg (@{$clientPkgs})
    {
      $source = $tmp_path . '/' . $pkg;
      if (system($mv_cmd . ' ' . $source . ' ' . $clientPkgPath))
      {
        BDUtil::setError(BDi18n::getMsg('clientPkgmissing',
                         { 'pkg' => $pkg, 'dirName' => $clientPkgPath }));
        return 0;
      }
    }
  }
  else
  {
    BDUtil::info('No client packages to move.');
  }

  my $scriptsPath = getScriptPath($tree->{'app'}->{'vendor'},
                     $tree->{'app'}->{'name'}, $tree->{'app'}->{'version'});
  `$mkdir_cmd $scriptsPath`;
  if ( -x $tmp_path . '/scripts')
  {
    BDUtil::info('Putting pre/post install/uninstall scripts in ' . $scriptsPath);
    system($mv_cmd . ' ' . $tmp_path . '/scripts/* ' . $scriptsPath . ' 2>/dev/null');
  }
  else
  {
    BDUtil::info('No pre/post install/uninstall scripts to move.');
  }

  return (sprintf("%s %s %s",
                  $tree->{'app'}->{'vendor'},
                  $tree->{'app'}->{'nameTag'},
                  $tree->{'app'}->{'version'}),
          $scriptsPath);
}

#
#
#
sub getPath
{
  my ($basePath, $vendor, $name, $version) = @_;
  
  return "" unless (defined($basePath) && 
		    defined($vendor) && 
		    defined($name) && 
		    defined($version));

  my $path = $basePath . '/' . $vendor . '-' . $name . '-' . $version;
  $path =~ s/ /_/g;  # Just in case
  return $path;
}

#
#
#
sub getClientPkgPath
{
  return getPath($client_rpm_path, @_);
}

#
#
#
sub getScriptPath
{
  return getPath($script_path, @_);
}

#
#
#
sub runScripts
{
  my ($path, $type) = @_;

  if (! opendir (SCRIPT_DIR, $path . '/' . $type))
  {
    BDUtil::info('No ' . $type . ' scripts to run.');
    return;
  }
  BDUtil::info('Running ' . $type . ' scripts..');
  Progress::setMessage(BDi18n::getMsg($type . '_scripts_running'));
  my @scripts = sort(readdir(SCRIPT_DIR));
  closedir(SCRIPT_DIR);
  my $rc;
  foreach my $s (@scripts)
  {
    next if ($s =~ /^\./);  # . and ..
    $rc = system($path . '/' . $type . '/' . $s);
    BDUtil::info($s . ' rc = ' . $rc);
  }
  Progress::infoEvent(BDi18n::getMsg($type . 'scripts_ran'));
}

#
# prepareClientPkgs
#
# Load the list of new client Pkgs and put them in the
# right places.
#
sub prepareClientPkgs
{
  my ($mappId, $devices, $pkgPath) = @_;

  my $devs = TreeXml::getListFromElement($devices);
  return 0 unless (defined($devs) && ref($devs) eq "ARRAY");

  my $numCRpms=0;
  my ($devType, $devClass, $cRpm, $appliances, $buildToPkgHash);
  my ($rc, $applianceId, $cRpms, $buildIdList, $source, $app);
  my %applianceToPkgsHash = ();
  my %rpmHash = ();
  my $rpmsToInstall = 0;

  foreach my $device (@{$devs}) {
      $devType = '';
      $devClass = '';
      if ((exists $device->{devType})) {
	  &BDUtil::info('Processing client pkgs for build type ' . 
			$device->{devType});
	  $devType = $device->{devType};
      } elsif ((exists $device->{devClass})) {
	  &BDUtil::info('Processing client Pkgs for build class ' . 
			$device->{devClass});
	  $devClass = $device->{devClass};
      }

      if (exists($device->{pkg})) {
	  $cRpms = TreeXml::getListFromElement($device->{pkg});
      } elsif (exists($device->{rpm})) {
	  $cRpms = TreeXml::getListFromElement($device->{rpm});
      } else {
	  &BDUtil::info('No pkgs specified for type/class');
	  next;
      }

      ($rc, $buildIdList) = Builds::getBuildList($devType, $devClass);
      return BDUtil::dbError(0) if ($rc);
      
      ($rc, $appliances) = MAPP::getManagedAppliancesByBuild($buildIdList);
      return BDUtil::dbError(0) if ($rc);
      
      foreach $cRpm (@{$cRpms}) {
	  &BDUtil::info('Processing client Pkg ' . $cRpm);
	  if (!exists($rpmHash{$cRpm})) {
	      $source = $tmp_path . '/' . $cRpm;
	      if (! -r $source) {
		BDUtil::setError(BDi18n::getMsg('clientPkgmissing',
						{ 'pkg' => $cRpm }));
		  return 0;
	      }
	      $rpmHash{$cRpm} = 1;
	  }

	  ($rc, $buildToPkgHash) =
	    MAPP::addClientPackage($buildIdList, $mappId, $cRpm, 
				   $pkgPath);
	  return BDUtil::dbError(0) if ($rc);
	  
	  if (defined($appliances) && ref($appliances) eq "ARRAY") {
	      foreach $app (@{$appliances}) {
		  $applianceId = $app->{'appliance_id'};
		  if (!exists($applianceToPkgsHash{$applianceId})) {
		    BDUtil::info(' - Making Pkg list for host Id = '
				 . $applianceId 
				 . ' (' . $app->{'ip_address'}
				 . ')');
		      # Construct a new record
		      my %newApplianceRecord = ();
		      $newApplianceRecord{'app'} = $app;
		      $newApplianceRecord{'pkgs'} = [];
		      $applianceToPkgsHash{$applianceId} = 
			  \%newApplianceRecord;
		  }
		  push(@{$applianceToPkgsHash{$applianceId}->{'pkgs'}},
		       $buildToPkgHash->{$app->{'build_id'}});
		  ++$rpmsToInstall;
	      }
	  }
      }
  }

  ############## Debugging #########
  # ## Dump of the appliance / pkg data structure ###
  # BDUtil::info('Pkgs to install:');
  # foreach my $a (keys %applianceToPkgsHash)
  # {
  #   BDUtil::info(' + Appliance Id = ' . $a . ' : '
  #        . $applianceToPkgsHash{$a}->{'app'}->{'ip_address'});
  #   foreach my $p (@{$applianceToPkgsHash{$applianceId}->{'pkgs'}})
  #   {
  #     BDUtil::info('   - ' . $p->{'rpm_name'});
  #   }
  # }
  #################################

  return (\%applianceToPkgsHash, $rpmsToInstall, [ keys %rpmHash ]);
}

#
# installClientPkgs
#
# Right here is the opportunity to multithread...
#
sub installClientPkgs
{
  my ($mappId, $applianceHash, $initScript, $noClients) = @_;

  my @successfulApplianceList = ();
  my $successfulAppliances = 0;
  my $totalAppliances = 0;
  my ($rc, $id);

  BDUtil::info('Setting appliance install statii to not installed');
  foreach $id (keys %{$applianceHash})
  {
      foreach my $pkg_ref (@{$applianceHash->{$id}->{'pkgs'}}) {
          return BDUtil::dbError(-1)
              if (!MAPP::addInstalledApplication($pkg_ref->{'mapp_client_rpm_id'},
                                                 $id, 'F', ''));
      }
  }

  return 0 if ($noClients);

  foreach $id (keys %{$applianceHash})
  {
    ++$totalAppliances;
    $rc = Pusher::installRemotePkgs($applianceHash->{$id}->{'app'},
                                    $applianceHash->{$id}->{'pkgs'},
                                    1, $mappId);
    return -1 if ($rc == -1);
    $rc = 1 if (!$rc && $applianceHash->{$id}->{'app'}->{'mappFail'}->{$mappId});
    if ($rc == 0)
    {
      push(@successfulApplianceList,
           $applianceHash->{$id}->{'app'}->{'appliance_id'});
      ++$successfulAppliances;
    }
  }

  $rc = 0;
  $rc = 1 if ($successfulAppliances < $totalAppliances);

  if ($initScript)
  {
    while ($successfulAppliances < $totalAppliances)
    {
      Progress::updateProgress();
      ++$successfulAppliances;
    }
    Progress::setMessage(BDi18n::getMsg('runningInitScript'));
    if (Pusher::runScript($initScript, \@successfulApplianceList))
    {
      Progress::warningEvent(BDi18n::getMsg('initScriptFailed'));
      $rc = 1;
      # Hrm, did it do any work at all?  Probably not...
      Progress::advanceProgress($successfulAppliances);
    }
  }

  return $rc;
}

#
#
#
sub failProgress
{
  my $mappId = shift @_ || 0;

  MAPP::removeMAPP($mappId) if ($mappId);
  system ($rm_cmd . $tmp_path);
  Progress::errorEvent(BDUtil::getError());
  Progress::progressFailed(BDi18n::getMsg('mappInstallFailed')
                              . ' : ' . BDUtil::getError());
  return -1;
}

#
#
#
sub setMAPPFailed
{
  my $mappId = shift @_;

  BDUtil::info('MAPP install failed, uninstalling installed portions');

  my ($workUnits, $mapp, $mappName, @uargs) = getUninstallMAPPWork($mappId, 1);
  doUninstallMAPP($mapp, $mappName, @uargs, 0);

  return failProgress();
}

######################################################
#
#  installMAPP
#
######################################################
sub installMAPP
{
  my $mappSource = shift @_;
  my $noClients = shift @_ || 0;

  my $fileName;
  if ((substr($mappSource, 0, 8) eq 'https://') ||
      (substr($mappSource, 0, 7) eq 'http://') ||
      (substr($mappSource, 0, 6) eq 'ftp://'))
  {
    $fileName = downloadFile($mappSource);
    return failProgress() if (!$fileName);
  }
  else
  {
    $fileName = $mappSource;
  }

  unless (-f $fileName && -r $fileName) {
      BDUtil::info("Invalid MAPP file: $fileName");
      return failProgress();
  }

  Progress::setMessage(BDi18n::getMsg('preparingForInstall'));

  BDUtil::info('Adding MAPP package file : ' . $fileName);
  BDUtil::info('  -------- Prepare Stage -------');
  my $tree = getMAPPTree($fileName);
  return failProgress() if (!$tree);
  my ($rc, $uninstallId) = validateNewMAPP($tree);
  return failProgress() if ($rc);
  my $strutsConfigFile = getMAPPStrutsConfig($tree->{'app'}->{'name'});
  #return failProgress() if (!$strutsConfigFile);
  my $validationConfigFile = getMAPPValidationConfig($tree->{'app'}->{'name'});
  my $menuConfigFile = getMAPPMenuConfig($tree->{'app'}->{'name'});
  #return failProgress() if (!$menuConfigFile);

  my $workUnits = 2;  # Prepare and Finish both 1

  my @uargs;
  if ($uninstallId)
  {
    ($rc, @uargs) = getUninstallMAPPWork($uninstallId, 0);
    $workUnits += $rc;
    $rc = doUninstallMAPP(@uargs, 1);
    return failProgress() if ($rc == -1);
  }

  my $mappId = addMAPPEntry($tree);
  return failProgress() if ($mappId <= 0);

  my $msRpmList = 0;
  if (exists ($tree->{'app'}->{'serverPkg'})
   || exists ($tree->{'app'}->{'msRpm'}))
  {
    $msRpmList = prepareServerPkgs($tree->{'app'}->{'serverPkg'}
                                     || $tree->{'app'}->{'msRpm'});
    return failProgress($mappId) if (!$msRpmList);
    $workUnits += (@{$msRpmList} + 0);  # 1 for each Server Pkg
  }

  my $clientPkgNames = 0;
  my $clientPkgHash = 0;
  my $initScript = 0;
  if (exists($tree->{'app'}->{'device'}))
  {
    my $numClientPkgsToInstall = 0;
    my $clientPkgsPath = getClientPkgPath($tree->{'app'}->{'vendor'},
                        $tree->{'app'}->{'name'}, $tree->{'app'}->{'version'});
    ($clientPkgHash, $numClientPkgsToInstall, $clientPkgNames) =
                      prepareClientPkgs($mappId, $tree->{'app'}->{'device'},
                                        $clientPkgsPath);
    return failProgress($mappId) if (!$clientPkgHash);

    if (! $noClients)
    {
      $workUnits += $numClientPkgsToInstall;

      if (exists($tree->{'app'}->{'initScript'}))
      {
        $initScript = $tree->{'app'}->{'initScript'};
        # The init script MUST do an updateProgress() for each appliance
        $workUnits += ((keys %{$clientPkgHash}) + 0);
      }
    }
  }

  BDUtil::info(' ------- Done preparing, work units = ' . $workUnits);
  Progress::setWorkItems($workUnits);

  # Update the progress bar for the 1 unit used up by the preparation
  Progress::updateProgress();

  my ($mappName, $scriptPath) = movePkgsAndScripts($tree, $clientPkgNames);
  return failProgress($mappId) if (!$mappName);

  $rc = moveModuleConfigFiles($strutsConfigFile, $menuConfigFile,
                              $validationConfigFile);
  return failProgress($mappId) if (!$rc);

  runScripts($scriptPath, 'pre_install');

  if ($msRpmList)
  {
    return setMAPPFailed($mappId)
      if (installServerPkgs($mappId, $msRpmList));
  }
  else
  {
    BDUtil::info('No server pkgs found in this module to install');
  }

  my $backupPaths = TreeXml::getListFromElement($tree->{'app'}->{'backup_path'});
  if ($backupPaths)
  {
    foreach my $bPath (@{$backupPaths})
    {
      BDUtil::info('Adding backup path : ' . $bPath);
      MAPP::addBackupPath($mappId, $bPath);
    }
  }

  $rc = 0;
  if ($clientPkgHash)
  {
    $rc = installClientPkgs($mappId, $clientPkgHash, $initScript, $noClients);
    return setMAPPFailed($mappId) if ($rc == -1);
  }

  runScripts($scriptPath, 'post_install');

  registerTasks($mappId, $tree->{'app'}->{'task'})
    if (exists($tree->{'app'}->{'task'}));

  registerApplianceIcons($mappId, $tree->{'app'}->{'appliance_icon'})
    if (exists($tree->{'app'}->{'appliance_icon'}));

  BDUtil::info('Updating Permissions');
  system($chown_cmd . " -R $tomcat4_user:$tomcat4_group $tomcat4_workdir");

  if ($msRpmList)
  {
    BDUtil::info('Reloading web container');
    BDUtil::info('Cannot restart tomcat')
	if (InitUtil::restart($tomcat_init, 10, 1));
    BDi18n::reloadTags();
  }

  BDUtil::info('Cleaning up ' . $tmp_path);
  system($rm_cmd . $tmp_path);

  Progress::advanceProgress();
  BDUtil::info('Done adding MAPP, id = ' . $mappId);
  if ($rc)
  {
    Progress::progressWarned(BDi18n::getMsg('warnedMAPP', 
                                            { 'mappName' => $mappName }));
  }
  else
  {
    Progress::progressComplete(BDi18n::getMsg('installedMAPP', 
                                              { 'mappName' => $mappName }));
  }

  MAPP::setMAPPStatus($mappId, 'AS');

  return 0;
}

######################################################
#
# uninstallMAPP
#
######################################################
sub uninstallMAPP
{
  my $mappId = shift @_;
  my $noClients = shift @_ || 0;

  my ($workUnits, $mapp, $mappName, @uargs)
                     = getUninstallMAPPWork($mappId, $noClients);

  if (!$mapp)
  {
    Progress::errorEvent(BDUtil::getError());
    Progress::progressFailed('MAPP removal failed');
    return -1;
  }

  Progress::setWorkItems(2 + $workUnits);
  Progress::advanceProgress();
  my $rc = doUninstallMAPP($mapp, $mappName, @uargs, 1, 0);
  Progress::advanceProgress();

  if ($rc > 0)
  {
    Progress::progressWarned(BDi18n::getMsg('mappUninstallWarned',
                                            { 'mappName' => $mappName }));
  }
  elsif ($rc < 0)
  {
    Progress::progressFailed(BDi18n::getMsg('mappUninstallFailed',
                                            { 'mappName' => $mappName }));
  }
  else
  {
    Progress::progressComplete(BDi18n::getMsg('mappUninstallSuccess',
                                              { 'mappName' => $mappName } ));
  }

  return 0;
}

#
#
#
sub doUninstallMAPP
{
  my ($mapp, $mappName, $numServerPkgs, $serverPkgs,
         $numClientPkgs, $clientPkgs, $useProgress, $lastEvent) = @_;
  $lastEvent = 1 if (!defined($lastEvent));

  my $mappId = -1;
  $mappId = int($mapp->{'mapp_id'}+0) 
      if (defined($mapp->{'mapp_id'}));
  
  if ($mappId <= 0) {
      cluck "ERROR: mapp id is null!";
      return -1;
  }

  BDUtil::info('Uninstalling MAPP id = ' . $mappId);
  MAPP::setMAPPStatus($mappId, 'D');

  my $scriptsPath = getScriptPath($mapp->{'vendor'}, $mapp->{'mapp_name'}, $mapp->{'version'});
  runScripts($scriptsPath, 'pre_uninstall');

  my $mappSchedules = MAPP::getMAPPSchedules($mapp->{'mapp_name'});
  if (defined($mappSchedules) && ref($mappSchedules) eq "ARRAY")
  {
    my $v;
    foreach my $s (@{$mappSchedules})
    {
      $v = $s->{'name'} . ' (' . $s->{'type'} . ', '
                        . $s->{'verbose_settings'} . ')';
      BDUtil::info('Removing schedule ' . $v);
      Progress::setMessage(BDi18n::getMsg('deletingMAPPSchedule',
                                          { 'details' => $v }));
      system($schedulerTool . ' delete ' . $s->{'schedule_id'});
      Progress::infoEvent(BDi18n::getMsg('deletedMAPPSchedule',
                                         { 'details' => $v }));
    }
  }
  else
  {
    BDUtil::info('No schedules associated with this MAPP to delete.');
  }

  BDUtil::info('Removing module configuration files from web application');
  # should probably not hard code this right here
  `$rm_cmd /var/tomcat4/webapps/sdui/WEB-INF/struts-config-$mapp->{'mapp_name'}.xml`;
  `$rm_cmd /var/tomcat4/webapps/sdui/WEB-INF/menu-config-$mapp->{'mapp_name'}.xml`;
  `$rm_cmd /var/tomcat4/webapps/sdui/WEB-INF/validation-$mapp->{'mapp_name'}.xml`;

  if ($mapp->{'cleanup_script'} ne '')
  {
    my $installedAppliances = MAPP::getInstalledAppliances($mappId);
    if ($installedAppliances)
    {
      Progress::setMessage(BDi18n::getMsg('runningCleanupScript'));
      if (Pusher::runScript($mapp->{'cleanup_script'},
                            $installedAppliances))
      {
        Progress::warningEvent(BDi18n::getMsg('cleanupScriptFailed'));
      }
      else
      {
        Progress::infoEvent(BDi18n::getMsg('cleanupScriptSuccess'));
      }
    }
  }

  my $retCode = 0;

  if ($numServerPkgs)
  {
    $retCode = uninstallServerPkgs($serverPkgs, $numServerPkgs, $useProgress);
    BDUtil::info('Reloading web container');
    BDUtil::into('Cannot restart tomcat')
	if (InitUtil::restart($tomcat_init, 10, 1));
  }

  if (($numClientPkgs) && ($retCode != -1))
  {
    my $rc = uninstallClientPkgs($clientPkgs, $numClientPkgs, $useProgress);
    $retCode = $rc if ($rc);
  }

  BDUtil::info('Removing MAPP id = ' . $mappId . ' records from database');
  $retCode = -1 if (MAPP::removeMAPP($mappId));

  runScripts($scriptsPath, 'post_uninstall'); 

  my $clientPkgPath = getClientPkgPath($mapp->{'vendor'},
                            $mapp->{'mapp_name'}, $mapp->{'version'});
  BDUtil::info('Removing ' . $clientPkgPath . ' directory.');
  system($rm_cmd . $clientPkgPath);
  BDUtil::info('Removing ' . $scriptsPath . ' directory.');
  system($rm_cmd . $scriptsPath);

  BDUtil::info('Uninstall return code : ' . $retCode);
  if (($useProgress) && ($lastEvent))
  {
    if ($retCode > 0)
    {
      Progress::warningEvent(BDi18n::getMsg('mappUninstallWarned',
                                            { 'mappName' => $mappName }));
    }
    elsif ($retCode < 0)
    {
      dbError();
      Progress::errorEvent(BDi18n::getMsg('mappUninstallFailed',
                                          { 'mappName' => $mappName }));
    }
    else
    {
      Progress::infoEvent(BDi18n::getMsg('mappUninstallSuccess',
                                         { 'mappName' => $mappName } ));
    }
  }

  return $retCode;
}

#
#
#
sub getUninstallMAPPWork
{
  my $mappId = shift @_;
  my $noClients = shift @_ || 0;

  my ($mapp, $mappName) = getMAPPInfo($mappId);
  return 0 if (!$mapp);

  my ($rc, $serverPkgs, $clientPkgs, $numServerPkgs, $numClientPkgs);
  ($rc, $serverPkgs) = MAPP::getServerPackages($mappId);
  return BDUtil::dbError(-1) if ($rc);
  return BDUtil::dbError(-1) 
      unless (defined($serverPkgs) && ref($serverPkgs) eq "ARRAY");
  $numServerPkgs = scalar(@{$serverPkgs});

  $numClientPkgs = 0;
  $clientPkgs = 0;

  unless ($noClients) {
      ($rc, $clientPkgs) = MAPP::getInstalledPackages(0, $mappId);
      return BDUtil::dbError(-1) if ($rc);
      $numClientPkgs = scalar(@{$clientPkgs})
	  if (defined($clientPkgs) && ref($clientPkgs) eq "ARRAY");
  }

  return ($numServerPkgs + $numClientPkgs, $mapp, $mappName,
          $numServerPkgs, $serverPkgs, $numClientPkgs, $clientPkgs);
}

#
#
#
sub uninstallServerPkgs
{
  my ($serverPkgs, $numServerPkgs, $useProgress) = @_;

  Progress::setMessage(BDi18n::getMsg('uninstallingServerPkg',
                                      { 'numPkgs' => $numServerPkgs }));
  my ($msg, $pkg, $rc);
  foreach $pkg (@{$serverPkgs})
  {
    $msg = $pkg->{'rpm_name'} . ' from control station';
    BDUtil::info('Uninstalling ' . $msg);
    $rc = SysCmd::uninstallPkg($base_dir . '/msRpms', $pkg->{'rpm_name'},
                               $pkg->{'pkg_type'});
    if ($rc < 0)
    {
      Progress::warningEvent(BDi18n::getMsg('failedUninstallingServerPkg',
                                            { 'pkg' => $pkg->{'rpm_name'} } ));
    }
    elsif (!$rc)
    {
      Progress::infoEvent(BDi18n::getMsg('uninstalledServerPkg',
                                            { 'pkg' => $pkg->{'rpm_name'} } ));
    } # rc = 1 means it was correctly skipped
    SysCmd::runCmd('rm', '-f ' . $base_dir . '/msRpms/' . $pkg->{'rpm_name'} . '*');
    Progress::updateProgress() if ($useProgress);
  }

  return 0;
}

#
#
#
sub uninstallClientPkgs
{
  my ($clientPkgs, $numClientPkgs, $useProgress) = @_;
  $useProgress = 1 if (!defined($useProgress));

  my $pkg;
  my @pkgList = ();
  my @pkgIdList = ();
  my @successfulAppliances = ();
  my $lastApplianceId = 0;
  my $lastApplianceIP = '';
  my $success = 0;
  my $total = 0;
  my $rc = 0;

  foreach $pkg (@{$clientPkgs})
  {
    if (($pkg->{'appliance_id'} != $lastApplianceId) && $lastApplianceId)
    {
      $rc = Pusher::removeRemotePkgs($lastApplianceIP,
                                     \@pkgList, $useProgress);
      return -1 if ($rc == -1);
      
      foreach my $pkgId (@pkgIdList) {
          return BDUtil::dbError(-1)
              if (MAPP::setPkgInstallationStatus($pkgId, $lastApplianceId,
                                                 'F', '') < 0);
      }
      
      ++$total;
      if (!$rc)
      {
        push (@successfulAppliances, $lastApplianceId);
        ++$success;
      }
      @pkgList = ($pkg->{'rpm_name'});
      @pkgIdList = ($pkg->{'mapp_client_rpm_id'});
    }
    else
    {
      push (@pkgList, $pkg->{'rpm_name'});
      push (@pkgIdList, $pkg->{'mapp_client_rpm_id'});
    }
    $lastApplianceId = $pkg->{'appliance_id'};
    $lastApplianceIP = $pkg->{'ip_address'};
  }
  if ($lastApplianceIP ne '')
  {
    $rc = Pusher::removeRemotePkgs($lastApplianceIP,
                                   \@pkgList, $useProgress);
    return -1 if ($rc == -1);

    foreach my $pkgId (@pkgIdList) {
        return BDUtil::dbError(-1)
            if (MAPP::setPkgInstallationStatus($pkgId, $lastApplianceId,
                                               'F', '') < 0);
    }
    
    ++$total;
    if (!$rc)
    {
      push (@successfulAppliances, $lastApplianceId);
      ++$success;
    }
  }

  return 1 if ($success < $total);

  return 0;
}

######################################################
#
# installNewClients
#
######################################################
sub installNewClients
{
  my ($mappId, $applianceIds) = @_;

  my ($mapp, $mappName) = getMAPPInfo($mappId);
  return newClientFail() if (!$mapp);
  my $appliances = MAPP::getApplianceBuilds($applianceIds);
  return newClientFail() if (!$appliances);
  my $schedules = MAPP::getApplianceSchedules($mapp->{'mapp_name'});

  my $workUnits = 2;
  my ($appliance, $rc, $numPkgs, $pkgs, $pkg, $alreadyInstalled, %installedHash);
  foreach $appliance (@{$appliances})
  {
    ($rc, $pkgs) = MAPP::getBuildPackages($mappId, $appliance->{'build_id'});
    return newClientFail() if ($rc);
    if ($pkgs)
    {
      ($rc, $alreadyInstalled) = MAPP::getInstalledPackages(
                      [$appliance->{'appliance_id'}], $mappId);
      return newClientFail() if ($rc);
      if ($alreadyInstalled)
      {
        # Reduce list to just needed ones.  Yes, it could be done with SQL too.
        my @newPkgList = ();
        %installedHash = ();
        foreach $pkg (@{$alreadyInstalled})
        {
          $installedHash{$pkg->{'rpm_name'}} = 1;
        }
        foreach $pkg (@{$pkgs})
        {
          next if (exists($installedHash{$pkg->{'rpm_name'}}));
          push (@newPkgList, $pkg);
        }
        $pkgs = \@newPkgList;
      }
      $numPkgs = (@{$pkgs}+0);
      if ($numPkgs)
      {
        $appliance->{'pkgs'} = $pkgs;
        $workUnits += $numPkgs;
      }
      ++$workUnits if ($mapp->{'init_script'} ne '');
    }
    else
    {
      BDUtil::info($appliance->{'ip_address'} . ' (' . $appliance->{'appliance_id'}
                      . ') is not eligible for any packages for this MAPP');
    }
  }

  Progress::setWorkItems($workUnits);
  Progress::advanceProgress();

  my @successfulApplianceList = ();
  my $successfulAppliances = 0;
  my $totalAppliances = 0;

  foreach $appliance (@{$appliances})
  {
    if (exists($appliance->{'pkgs'}))
    {
      BDUtil::info('Installing ' . $mapp->{'vendor'} . ' ' . $mapp->{'mapp_name'}
                 . ' ' . $mapp->{'version'} . ' on ' . $appliance->{'ip_address'});
      Progress::setMessage(BDi18n::getMsg('installingMAPPonAppliance',
         { 'mappName' => $mappName, 'ip' => $appliance->{'ip_address'} }));
      $rc = Pusher::installRemotePkgs($appliance, $appliance->{'pkgs'},
                                      1, $mappId);
      return -1 if ($rc == -1);
      $rc = 1 if (!$rc && $appliance->{'mappFail'}->{$mappId});
      if (!$rc)
      {
        Progress::infoEvent(BDi18n::getMsg('installedMAPPonAppliance',
           { 'mappName' => $mappName, 'ip' =>  $appliance->{'ip_address'} }));
        push (@successfulApplianceList, $appliance->{'appliance_id'});
        ++$successfulAppliances;
      }
      else
      {
        Progress::warningEvent(BDi18n::getMsg('failedMAPPinstallOnAppliance',
           { 'mappName' => $mappName, 'ip' =>  $appliance->{'ip_address'} }));
      }
      ++$totalAppliances;
    }
  }

  if ($totalAppliances && ($mapp->{'init_script'} ne ''))
  {
    Progress::advanceProgress($totalAppliances - $successfulAppliances)
      if ($totalAppliances != $successfulAppliances);
    if ($successfulAppliances)
    {
      Progress::setMessage(BDi18n::getMsg('runningInitScript'));
      if (Pusher::runScript($mapp->{'init_script'}, \@successfulApplianceList))
      {
        Progress::warningEvent(BDi18n::getMsg('initScriptFailed'));
      }
    }
  }

  if ($schedules && $successfulAppliances)
  {
    BDUtil::info('Adding appliances to schedules');
    foreach $appliance (@successfulApplianceList)
    {
      return newClientFail()
        if (MAPP::addApplianceToSchedules($appliance, $schedules));
    }
    Progress::infoEvent(BDi18n::getMsg('addedApplianceToSchedules'));
  }

  Progress::advanceProgress();
  if ($totalAppliances == $successfulAppliances)
  {
    Progress::progressComplete(BDi18n::getMsg('installedMAPPonAppliances',
                               { 'mappName' => $mappName }));
  }
  elsif ($successfulAppliances == 0)
  {
    Progress::progressFailed(BDi18n::getMsg('failedMAPPinstallOnAppliances',
                             { 'mappName' => $mappName }));
  }
  else
  {
    Progress::progressWarned(BDi18n::getMsg('warnedMAPPinstallOnAppliances',
                             { 'mappName' => $mappName }));
  }

  return 0;
}

#
#
#
sub newClientFail
{
  BDUtil::info('Failed to install MAPP on appliance');
  Progress::progressFailed(
     BDi18n::getMsg('failedMAPPinstallOnApplianceNoNames'));

  return -1;
}

######################################################
#
# uninstallClients
#
######################################################
sub uninstallClients
{
  my ($mappId, $applianceList) = @_;

  my ($mapp, $mappName) = getMAPPInfo($mappId);
  return uninstallClientsFail() if (!$mapp);

  BDUtil::info('Figuring out packages to uninstall from appliances');
  my ($rc, $clientPkgs) = MAPP::getInstalledPackages($applianceList, $mappId);
  return BDUtil::dbError(-1) if ($rc);
  my $numClientPkgs = (@{$clientPkgs}+0);

  Progress::setWorkItems(2 + $numClientPkgs);

  my $schedules = MAPP::getApplianceSchedules($mapp->{'mapp_name'}, 0);
  if ($schedules)
  {
    my %appHash = ();
    my $a;
    foreach $a (@{$clientPkgs})
    {
      $appHash{$a->{'appliance_id'}} = 1;
    }
    foreach $a (keys %appHash)
    {
      return uninstallClientsFail()
        if (MAPP::removeApplianceFromSchedules($a, $schedules));
    }
    Progress::infoEvent(BDi18n::getMsg('removedApplianceFromSchedules'));
  }

  if ($mapp->{'cleanup_script'} ne '')
  {
    Progress::setMessage(BDi18n::getMsg('runningCleanupScript'));
    if (Pusher::runScript($mapp->{'cleanup_script'},
                          $applianceList))
    {
      Progress::warningEvent(BDi18n::getMsg('cleanupScriptFailed'));
    }
    else
    {
      Progress::infoEvent(BDi18n::getMsg('cleanupScriptSuccess'));
    }
  }

  Progress::advanceProgress();

  $rc = 0;
  if ($numClientPkgs)
  {
    $rc = uninstallClientPkgs($clientPkgs, $numClientPkgs);
  }
  else
  {
    BDUtil::info('No packages to uninstall');
  }

  Progress::advanceProgress();
  if (!$rc)
  {
    Progress::progressComplete(BDi18n::getMsg('uninstalledMAPPonAppliances',
                               { 'mappName' => $mappName }));
  }
  elsif ($rc < 0)
  {
    Progress::progressFailed(BDi18n::getMsg('failedMAPPuninstallOnAppliances',
                             { 'mappName' => $mappName }));
  }
  else
  {
    Progress::progressWarned(BDi18n::getMsg('warnedMAPPuninstallOnAppliances',
                             { 'mappName' => $mappName }));
  }

  return 0;
}

#
#
#
sub uninstallClientsFail
{
  BDUtil::info('Failed to uninstall MAPP on appliances');
  Progress::progressFailed(
     BDi18n::getMsg('failedMAPPuninstallOnApplianceNoNames'));

  return -1;
}

1;
