package ImportWork;

# Copyright (c) 2003 Sun Microsystems, Inc. All rights reserved
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.


#
# $Id: ImportWork.pm,v 1.5 2004/10/21 17:04:04 ms152511 Exp $
#

use lib '/scs/lib/perl5';
use Pusher;
use BDUtil;
use MAPP;
use Progress;

#
# getInstallWork
#
# In : Reference to a list of appliance objects
#
# Out : (4 things)
# - Return code.  0 if good, non-zero for error.
# - Your original reference to list of appliance objects,
#   with an extra element: 'pkgs', a reference to a list
#   of package objects to install.
# - Reference to a list of MAPP objects, with additional
#   key 'appliances', a reference to a list of appliances
#   to run against.
# - Work units to do (calculated as sum of all packages
#   to install + number of init script iterations to run.)
#
sub getInstallWork
{
  my $appliances = shift;
  my $moduleList = shift || 0;

  my $numAppliances = (@{$appliances}+0);
  my %buildHash = ();
  my %mappHash = ();
  my ($rc, $appliance, $build, $pkg, $mapp, %myMapps);
  my $workUnits = 0;

  foreach $appliance (@{$appliances})
  {
    $buildHash{$appliance->{'build_id'}} = 0;
  }

  foreach $build (keys %buildHash)
  {
    ($rc, $buildHash{$build}) = MAPP::getPackagesByBuild($build, $moduleList);
    return dbError(-1) if ($rc);
    foreach $pkg (@{$buildHash{$build}})
    {
      if (!exists($mappHash{$pkg->{'mapp_id'}}))
      {
        $mapp = MAPP::getMAPP($pkg->{'mapp_id'});
        if ($mapp->{'init_script'} ne '')
        {
          my @newList = ();
          $mapp->{'appliances'} = \@newList;
          $mappHash{$pkg->{'mapp_id'}} = $mapp;
        }
        else
        {
          $mappHash{$pkg->{'mapp_id'}} = 0;
        }
      }
    }
  }

  if ($numAppliances > 1)
  {
    BDUtil::info('------------------------------------');
    BDUtil::info('Unique builds and associated packages:');
    foreach $build (keys %buildHash)
    {
      BDUtil::info('Build id = ' . $build . ':');
      foreach $pkg (@{$buildHash{$build}})
      {
        BDUtil::info(' - ' . $pkg->{'rpm_name'});
      }
    }
    BDUtil::info('Unique MAPPs with init scripts from package list:');
    foreach $mapp (keys %mappHash)
    {
      BDUtil::info(' - ' . $mappHash{$mapp}->{'mapp_name'})
        if ($mappHash{$mapp});
    }
  }

  foreach $appliance (@{$appliances})
  {
    $appliance->{'pkgs'} = $buildHash{$appliance->{'build_id'}};
    if ($numAppliances > 1)
    {
      BDUtil::info('Packages to install on ' . $appliance->{'ip_address'} . ' :');
      foreach $pkg (@{$appliance->{'pkgs'}})
      {
        BDUtil::info(' - ' . $pkg->{'rpm_name'});
      }
    }
    $workUnits += (@{$appliance->{'pkgs'}} + 0);
    %myMapps = ();
    foreach $pkg (@{$buildHash{$appliance->{'build_id'}}})
    {
      $myMapps{$pkg->{'mapp_id'}} = 1;
    }
    # Add to init script lists
    foreach $mapp (keys %myMapps)
    {
      if ($mappHash{$mapp})
      {
        push(@{$mappHash{$mapp}->{'appliances'}}, $appliance);
        ++$workUnits;
      }
    }
  }

  my @mappList = ();
  foreach $mapp (keys %mappHash)
  {
    if ($mappHash{$mapp})
    {
      push(@mappList, $mappHash{$mapp});
      BDUtil::info('InitScript for ' . $mappHash{$mapp}->{'mapp_name'}
            . ' will be run on ' . (@{$mappHash{$mapp}->{'appliances'}}+0)
            . ' appliances');
      $mappHash{$mapp}->{'schedules'} = MAPP::getApplianceSchedules(
                                             $mappHash{$mapp}->{'mapp_name'});
    }
  }

  return (0, $appliances, \@mappList, $workUnits);
}

#
# installPkgsOnAppliance
#
# In : appliance object, reference to list of package objects
#
# Out : return code (0 for ok, 1 for failed, -1 for unrecoverable error)
#
sub installPkgsOnAppliance
{
  my ($appliance, $pkgs) = @_;

  my ($mappId, $total, $failed);
  my $rc = 0;
  if ($pkgs)
  {
    $rc = Pusher::installRemotePkgs($appliance, $pkgs, 0);
    return -1 if ($rc == -1);

    if (exists($appliance->{'mappFail'}))
    {
      $total = $failed = 0;
      foreach $mappId (keys %{$appliance->{'mappFail'}})
      {
        ++$total;
        $failed += $appliance->{'mappFail'}->{$mappId};
      }

      if ($total == $failed)
      {
        $rc = 1;
        Progress::warningEvent(BDi18n::getMsg('allMappsFailed',
                               { 'dev' => $appliance->{'ip_address'} }));
      }
    }
  }
  else
  {
    $rc = 1;
  }

  return $rc;
}

#
# runScripts
#
# In : - Reference to list of mapps, each one is a mapp object,
#        with additional key 'appliances', a reference to a list
#        of appliance objects to run against.
#      - Script type (init | cleanup)
#
# Out : Return code (0 for ok, -1 for unrecoverable error.)
#
sub runScripts
{
  my ($mappList, $scriptType) = @_;

  my $scriptTypeMsg = BDi18n::getMsg($scriptType);
  my ($rc, $mapp, $app, @idList, $finalIdList, $mappNameMsg, $n1, $n2);
  foreach $mapp (@{$mappList})
  {
    @idList = ();
    $n1 = $n2 = 0;
    foreach $app (@{$mapp->{'appliances'}})
    {
      push(@idList, $app->{'appliance_id'});
      ++$n1;
    }
    $finalIdList = MAPP::getEligibleApplianceList($mapp->{'mapp_id'}, \@idList);
    if ($finalIdList)
    {
      $n2 = (@{$finalIdList}+0);
      $mappNameMsg = BDi18n::getSimpleMsg($mapp->{'vendor'}
                                  . ' ' . $mapp->{'name_tag'}
                                  . ' ' . $mapp->{'version'});
      Progress::setMessage(BDi18n::getMsg('runningScript',
                                          { 'scriptType' => $scriptTypeMsg,
                                            'mappName'   => $mappNameMsg,
                                            'numDevs'    => scalar @$finalIdList }));
      $rc = Pusher::runScript($mapp->{$scriptType . '_script'}, $finalIdList);
      if ($rc)
      {
        Progress::warningEvent(BDi18n::getMsg('scriptFailed',
                                              { 'scriptType' => $scriptTypeMsg,
                                                'mappName'   => $mappNameMsg }));
      }
      else
      {
        Progress::infoEvent(BDi18n::getMsg('scriptSuccess',
                                           { 'scriptType' => $scriptTypeMsg,
                                             'mappName'   => $mappNameMsg }));
      }
    }
    if ($n1 > $n2)
    {
      Progress::advanceProgress($n1 - $n2);
    }
  }

  return 0;
}

#---------------------------------------------------------------------

#
#
#
sub getRemoveWork
{
  my $appliances = shift;
  my $moduleList = shift || 0;

  my %mappHash = ();
  my $workUnits = 0;
  my ($rc, $appliance, $pkg, $mapp);
  foreach $appliance (@{$appliances})
  {
    my @myPkgs = ();
    my %myMapps = ();
    ($rc, $pkgs) = MAPP::getInstalledPackages([$appliance->{'appliance_id'}],
                                              $moduleList);
    return -1 if ($rc);
    foreach $pkg (@{$pkgs})
    {
      push (@myPkgs, $pkg->{'rpm_name'});
      ++$workUnits;
      $myMapps{$pkg->{'mapp_id'}} = $pkg->{'mapp_client_rpm_id'};
    }
    $appliance->{'pkgs'} = \@myPkgs;
    $appliance->{'mapps'} = \%myMapps;
    foreach $mapp (keys %myMapps)
    {
      if (!exists($mappHash{$mapp}))
      {
        my @newList = ();
        $mappHash{$mapp} = MAPP::getMAPP($mapp);
        $mappHash{$mapp}->{'appliances'} = \@newList;
      }
      push(@{$mappHash{$mapp}->{'appliances'}}, $appliance);
    }
  }

  my @mappList = ();
  foreach $mapp (keys %mappHash)
  {
    if ($mappHash{$mapp}->{'cleanup_script'} ne '')
    {
      push(@mappList, $mappHash{$mapp});
      $workUnits += (@{$mappHash{$mapp}->{'appliances'}}+0);
    }
  }

  BDUtil::info('Packages to remove per appliance:');
  foreach $appliance (@{$appliances})
  {
    BDUtil::info(' + ' . $appliance->{'ip_address'});
    foreach $pkg (@{$appliance->{'pkgs'}})
    {
      BDUtil::info('    - ' . $pkg);
    }
  }
  foreach $mapp (@mappList)
  {
    BDUtil::info('Cleanup script for ' . $mapp->{'mapp_name'}
                 . ' (' . $mapp->{'cleanup_script'} . ') will be run on:');
    foreach $appliance (@{$mapp->{'appliances'}})
    {
      BDUtil::info(' - ' . $appliance->{'ip_address'});
    }
  }

  return (0, $appliances, \@mappList, $workUnits);
}

#
#
#
sub uninstallPkgsOnAppliance
{
  my ($appliance, $pkgs) = @_;

  foreach my $mappId (keys %{$appliance->{'mapps'}})
  {
      MAPP::setPkgInstallationStatus($appliance->{'mapps'}->{$mappId},
                                     $appliance->{'appliance_id'}, 'F', '');
  }
  return Pusher::removeRemotePkgs($appliance->{'ip_address'}, $pkgs, 0);
}

1;
