#!/usr/bin/perl -wT
# $Id: db_tool.pl,v 1.26.2.2.2.2.2.4 2005/08/26 01:14:11 ms152511 Exp $
# db_tool.pl - a database tool for scs
# Copyright 2004 Sun Microsystems, Inc., All rights reserved.
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.

use strict;

# global variables 

my $EXITCODE = 0;
my $DBDIR = "/scs/data/db/mgmt";
my $LOGDIR = "/scs/logs";
my $PERLLIBDIR = "/scs/lib/perl5";
my $DBCONNECTIONS = 128;
my $DBSHAREDBUFFERS = $DBCONNECTIONS * 2;
my $DBPORT = 5434;
my $DBLOG = "$LOGDIR/db.log";
my $DBINFOF = "$PERLLIBDIR/DBInfo.pm";
my $DBPIDF = "postmaster.pid";
my $DBSQLDIR = "/scs/lib/sql/";
my $BACKUPFILE = '/tmp/mgmt_db.sql';
my $DBUSER = "mgmt";
my $DBPASSWD = "mgmt";
my $DBNAME  = "mgmt";
my $DBDEBUG = 0;
my $DBENCODING = "UNICODE";
my $DBLOGLEVEL="NOTICE";
my $DBPRINTLEVEL="WARNING";
my $WEBUSER = "tomcat4";
my $WEBGROUP = "tomcat4";
my $DBNAMEDEFAULT = "template1";
my $HBACONF = "$DBDIR/pg_hba.conf";
my $PASSWDF = "$DBDIR/passwd";
my $PGUSER = "postgres";
my $PGSQL = "/usr/bin/psql";
my $PGCTL = "/usr/bin/pg_ctl";
my $PGINIT = "/usr/bin/initdb";
my $PGCREATE = "/usr/bin/createdb";
my $PGDROP = "/usr/bin/dropdb";
my $PGDUMP = "/usr/bin/pg_dump";
my $SU = "/bin/su";
my $ENV = "/usr/bin/env";
my $PERL = "/usr/bin/perl -w";
my $HAVEONECMD = (defined $ARGV[0] && $ARGV[0] ne "");
my %DBSTATUS = ( 'UP'   => 0,
		 'DOWN' => 1,
		 '?'    => 2 );
my $MODE = "";
my %MODES = ( 'INIT'     => 'init',
	      'START'    => 'start',
	      'STOP'     => 'stop',
	      'KILL'     => 'kill',
	      'STATUS'   => 'status',
	      'RESTART'  => 'restart',
	      'DESTROY'  => 'destroy',
              'BACKUP'   => 'backup',
              'RESTORE'  => 'restore',
	      'CLI'      => 'psql',
	      'LOADSQL'  => 'load',
	      'NOSORT'   => 'nosort',
	      'DBVERS'   => 'dbvers',
	      'HELP'     => 'help',
	      'DEBUG'    => 'debug' );

# only allow root to run this script

if ($< > 0) {
    &printError("Only root can run $0");
    exit(1);
}

# secure the env to pass taint checks

$ENV{'PATH'} = "/sbin:/bin:/usr/bin:/usr/sbin";
delete @ENV{'IFS', 'CDPATH', 'ENV', 'BASH_ENV'};

# process the arguments

for (;;) {
    
    $MODE = shift(@ARGV);

    unless (defined($MODE) && $MODE ne "") {
	unless ($HAVEONECMD) {
	    &printUsage();
	    $EXITCODE = 1;
	}
	last;
    }

    if ($MODE eq $MODES{'DEBUG'}) {
	$DBDEBUG = 1;
	next;
    }

    if ($MODE eq $MODES{'DBVERS'}) {
	print "DB Version: " . &dbVersion() . "\n";
	next;
    }

    if ($MODE eq $MODES{'INIT'}) {
	if (&doDbInit() != 0) {
	    $EXITCODE = 1;
	    last;
	}
	next;
    }

    if ($MODE eq $MODES{'START'}) {
	if (&doDbStart() != 0) {
	    $EXITCODE = 1;
	    last;
	}
	next;
    }

    if ($MODE eq $MODES{'STOP'}) {
	if (&doDbStop() != 0) { 
	    $EXITCODE = 1;
	    last;
	}
	next;
    }

    if ($MODE eq $MODES{'KILL'}) {
	if (&doDbKill() != 0){
	    $EXITCODE = 1;
	    last;
	}
	next;
    }

    if ($MODE eq $MODES{'STATUS'}) {
	&doDbStatus();
	next;
    }
    
    if ($MODE eq $MODES{'RESTART'}) {
	if (&doDbStop() != 0 || &doDbStart() != 0) {
	    $EXITCODE = 1;
	    last;
	}
	next;
    }

    if ($MODE eq $MODES{'DESTROY'}) {
	if (&doDbDestroy() != 0) {
	    $EXITCODE = 1;
	    last;
	}
	next;
    }

    if ($MODE eq $MODES{'BACKUP'}) {
        if (&doDbBackup() != 0) {
            $EXITCODE = 1;
            last;
        }
        next;
    }

    if ($MODE eq $MODES{'RESTORE'}) {
        if (&doDbRestore() != 0) {
            $EXITCODE = 1;
            last;
        }
        next;
    }

    if ($MODE eq $MODES{'CLI'}) {
	if (&doPsql() != 0) {
	    $EXITCODE = 1;
	}
	last;
    }

    if ($MODE eq $MODES{'LOADSQL'}) {
	my $SQLFILE = shift @ARGV;
	if ($SQLFILE eq $MODES{'NOSORT'}) {
	    foreach $SQLFILE (@ARGV) {
		$EXITCODE = 1 if (&doLoadSql($SQLFILE) != 0);
	    }
	} else {
	    my @SQLFILES = sort($SQLFILE,@ARGV);
	    foreach $SQLFILE (@SQLFILES) {
		$EXITCODE = 1 if (&doLoadSql($SQLFILE) != 0);
	    }
	}
	last;
    }

    if ($MODE eq $MODES{'HELP'} || 
	$MODE =~ /^-[-]?([hH]|[hH]elp|\?)$/) {
	&printUsage();
	last;
    }

    # default

    &printError("Unknown mode: '$MODE'");
    &printUsage();
    $EXITCODE = 1;
    last;
} 

exit($EXITCODE);

#
# subroutines
#

sub doDbStart
{   
    # remove a stale pid file if necessary

    if (-r "$DBDIR/$DBPIDF" && open(DBPIDF,"$DBDIR/$DBPIDF")) {
	my $pid = -1;
	my $rawPid = <DBPIDF>;
	close(DBPIDF);
	if ($rawPid =~ /([0-9]+)/) {
	    $pid = int($1);
	}
	unlink ("$DBDIR/$DBPIDF") unless ($pid > 0 && kill(0, $pid));
    }

    if (&dbStatus() == $DBSTATUS{'UP'}) {
	&printInfo("The db is already running");
	return 0;
    }

    if (! -d $DBDIR) {
	&printError("The db is not initialized");
	return -1;
    }

    unless (chdir("/")) {
	&printError("Cannot cd to /");
	return -1;
    }

    # get the db user and group

    my $pguid = getpwnam($PGUSER);
    unless (defined($pguid) && $pguid+0 > 0) {
	&printError("Cannot determine db user id");
	return -1;
    }

    my $pggid = getgrnam($PGUSER);
    unless (defined($pggid) && $pggid+0 > 0) {
	&printError("Cannot determine db group id");
	return -1;
    }

    # create the db log file

    if (! -d $LOGDIR) {
	if (&mkdirs($LOGDIR, 0755, $pguid, $pggid)) {
	    &printError("Cannot create $LOGDIR: $!");
	    return -1;
	}
    }

    unless (open(LOGF,">$DBLOG")) {
	&printError("Cannot create db log file '$DBLOG': $!");
	return -1;
    }
    close (LOGF);

    return -1 if (&setAccess($DBLOG, 0644, $pguid, $pggid));

    # pre-7.3 versions of Postgres do not support changing the client logging level
    my $pgVersion = &dbVersion();
    my ($pgSubVersion) = ($pgVersion =~ /^7\.(\d+)\b/);

    my $pgStartCmd = "$PGCTL -o \"-i -h localhost -N $DBCONNECTIONS -B $DBSHAREDBUFFERS -p $DBPORT ";
    if ($pgSubVersion == 4) {
        $pgStartCmd .= "-c client_min_messages=$DBPRINTLEVEL -c log_min_messages=$DBLOGLEVEL ";
    } elsif ($pgSubVersion == 3) {
        $pgStartCmd .= "-c client_min_messages=$DBPRINTLEVEL -c server_min_messages=$DBLOGLEVEL ";
    }
    $pgStartCmd .= "\" -l $DBLOG -D $DBDIR start";
    if (&suRunCmd($PGUSER, $pgStartCmd)) {
	&printError("Could not start the db");
	return -1;
    }
    
    return 0;
}

sub doDbStop
{
    return 0 unless (&dbStatus() == $DBSTATUS{'UP'});

    my $pgStopCmd = "$PGCTL -D $DBDIR stop";
    if (&suRunCmd($PGUSER, $pgStopCmd)) {
	&printError("Could not stop the db");
	return -1;
    }

    return 0;
}

sub doDbKill
{
    my $line = "";
    my $pid = 0;
    my @parts = ();
    my @pids = ();

    &doDbStop();

    unlink("$DBDIR/$DBPIDF");
    
    if (open(PS,"/bin/ps -ef|")) {
	while($line = <PS>) {
	    next unless ($line =~ /postgres/);
	    print "$line";
	    @parts = split(/[ \t\r\n]+/, $line);
	    if ($parts[1] =~ /^([0-9]+)$/) {
		$pid = $1;
	    } else {
		$pid = 0;
	    }
	    next unless ($pid > 1);
	    push(@pids, $pid);
	}
	close(PS);

	foreach $pid (@pids) {
	    next unless (kill(0, $pid));
	    &printInfo("Killing db process with pid '$pid'");
	    kill(1, $pid);
	    sleep(3);
	    kill(9, $pid) if (kill(0, $pid));
	}
    } else {
	&printError("Cannot run ps");
	return -1;
    }
    
    return 0;
}

sub doDbInit
{
    # get the db user and group

    my $pguid = getpwnam($PGUSER);
    unless (defined($pguid) && $pguid+0 > 0) {
	&printError("Cannot determine db user id");
	return -1;
    }

    my $pggid = getgrnam($PGUSER);
    unless (defined($pggid) && $pggid+0 > 0) {
	&printError("Cannot determine db group id");
	return -1;
    }

    my $webgid = getgrnam($WEBUSER);
    unless (defined($webgid) && $webgid+0 > 0) {
	&printError("Cannot determine web group id");
	return -1;
    }

    # create the db directory

    if (! -d $DBDIR) {	
	if (&mkdirs($DBDIR, 0700, $pguid, $pggid)) {
	    &printError("Cannot not create $DBDIR: $!");
	    return -1;
	}
    }

    # initialize the db

    my $pgInitCmd = "$ENV LC_COLLATE=\"C\" $PGINIT -E $DBENCODING -D $DBDIR 2>&1";
    if (&suRunCmd($PGUSER, $pgInitCmd)) {
	&printError("Cannot initialize db");
	return -1;
    }

    my $savedConfig = 0;
    my $dbVers = &dbVersion();

    # PostgreSQL 7.2 seems to required a password, so use a less
    # secure access config file to setup the db, then switch to
    # our regular db config file

    if ($dbVers =~ /^7\.2\b/) {
	# save the default access config file

	unless (rename($HBACONF,"$HBACONF.default")) {
	    &printError("Cannot configure db access parameters");
	    return -1;
	}

	$savedConfig = 1;

	# create a temporary db access control file 
	
	unless (open(HBA,">$HBACONF")) {
	    &printError("Cannot create '$HBACONF': $!");
	    return -1;	
	}
	
	print HBA "local all trust\n";
	
	close(HBA);
    }

    # start the db since we will need to create our db and user

    if (&doDbStart()) {
	&printError("Cannot start the db");
	return -1;
    }

    # wait for the db to start

    sleep(10);

    # create the mgmt db
    
    my $pgCreateCmd = "$PGCREATE -p $DBPORT -U $PGUSER $DBNAME";
    if (&suRunCmd($PGUSER, $pgCreateCmd)) {
	&printError("Cannot create the database");
	&doDbStatus();
	&doDbStop();
	return -1;
    } 

    # create the db user

    my $sql = "CREATE USER $DBUSER WITH PASSWORD \\'$DBPASSWD\\'";
    $pgCreateCmd = "$SU - $PGUSER -c \"echo $sql | ";
    $pgCreateCmd .= "$PGSQL -p $DBPORT -U $PGUSER $DBNAMEDEFAULT\"";

    my $pgCreateRc = system($pgCreateCmd);
    $pgCreateRc /= 256;

    if ($pgCreateRc != 0) {
	&printError("Cannot create db user");
	&doDbStop();
	return -1;
    }

    # determine the db version

    $sql = "'SELECT version()'";
    my $pgVersionCmd = "$SU - $PGUSER -c \"echo $sql | ";
    $pgVersionCmd .= "$PGSQL -p $DBPORT -U $PGUSER $DBNAMEDEFAULT\"";

    unless (open(PGVERSION,"$pgVersionCmd|")) {
	&printError("Cannot determine db version");
	&doDbStop();
	return -1;
    }

    $dbVers = "";

    while (<PGVERSION>) {
	if (/PostgreSQL/) {
	    my @fields = split;
	    $dbVers = $fields[1];
	    $dbVers = "" unless (defined($fields[1]));
	    last;
	}
    }

    close(PGVERSION);
    
    if (&doDbStop()) {
	unlink($HBACONF) if ($savedConfig);
	&printError("Cannot stop the db");
	return -1;
    }

    # remove the temporary access config (if needed)

    unlink($HBACONF) if ($savedConfig);
    

    my $dbVersMajor = 0;
    my $dbVersMinor = 0;

    if ($dbVers =~ /^([0-9]+)\.([0-9]+)/) {
	$dbVersMajor = $1;
	$dbVersMinor = $2;
    } 

    $dbVersMajor = 0 unless (defined($dbVersMajor));
    $dbVersMinor = 0 unless (defined($dbVersMinor));

    if (int($dbVersMajor+0) != 7) {
	&printError("Unsupported db version '$dbVers'");
	return -1;
    }

    my $userAccess = "";
    $userAccess = "all" if (int($dbVersMinor+0) >= 3);

    my $authMethod = "crypt";
    $authMethod = "md5" if (int($dbVersMinor+0) >= 2);

    # save the default access config file
    
    unless ($savedConfig) {
	unless (rename($HBACONF,"$HBACONF.default")) {
	    &printError("Cannot configure db access parameters");
	    return -1;
	}
    }

    # create the real db access control file 

    unless (open(HBA,">$HBACONF")) {
	&printError("Cannot create '$HBACONF': $!");
	return -1;	
    }

    print HBA <<EO_HBA;
local  all $userAccess                                     $authMethod
host   all $userAccess 127.0.0.1   255.255.255.255         $authMethod
host   all $userAccess ::1         ff:ff:ff:ff:ff:ff:ff:ff $authMethod
EO_HBA

    close(HBA);

    return -1 if (&setAccess($HBACONF, 0644, $pguid, $pggid));

    # configure the db passwd 
    
    my $salt  = join('', ('.','/',0..9,'A'..'Z','a'..'z')[rand 64, rand 64]);
    my $pgcrypt = crypt($DBPASSWD,$salt);

    unless (defined($pgcrypt) && $pgcrypt ne "") {
	&printError("Cannot configure db password (crypt failed)");
	return -1;
    }

    unless (open(PASSWD,">$PASSWDF")) {
	&printError("Cannot create password file '$PASSWDF': $!");
	return -1;
    }    
    close (PASSWD);
    
    return -1 if (&setAccess($PASSWDF, 0600, $pguid, $pggid));

    unless (open(PASSWD,">>$PASSWDF")) {
	&printError("Cannot update password file '$PASSWDF': $!");
	return -1;
    }    
    
    print PASSWD "$DBUSER:$pgcrypt\n";

    close (PASSWD);

    # create DBInfo.pm

    if (&mkdirs("$PERLLIBDIR",0755)) {
	&printError("Cannot create db info file '$DBINFOF': $!");
	return -1;
    }

    unless (open(DBINFO,">$DBINFOF")) {
	&printError("Cannot create db info file '$DBINFOF': $!");
	return -1;
    }
    close(DBINFO);

    return -1 if (&setAccess($DBINFOF, 0640, $pguid, $webgid));

    unless (open(DBINFO,">$DBINFOF")) {
	&printError("Cannot create db info file '$DBINFOF': $!");
	return -1;
    }

    print DBINFO <<EO_DBINFO;
\#\!/usr/bin/perl -w
\# DBInfo.pm - SCS Database authentication information
\# Copyright 2003, 2004 Sun Microsystems, Inc., All rights reserved.
\# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms

package DBInfo;

use strict;

\#
\# getDBInfo
\#
\# return the database connection parameters (url, user, pass) for the
\# scs db
\#

sub getDBInfo
{
    return (\'$DBNAME\', \'$DBPORT\', \'$DBUSER\', \'$DBPASSWD\');
}

1;

EO_DBINFO

    close (DBINFO);

    return 0;
}

sub doDbDestroy
{
    my $rc = &doDbKill();
    if ($rc != 0) {
	&printError("Cannot kill the db. Destruction aborted");
	return $rc;
    }

    if (&runCmd("/bin/rm -rf $DBDIR $DBLOG $DBINFOF")) {
	&printError("Cannot destroy db");
	return -1;
    }

    return 0;
}

sub doDbBackup
{
    &printInfo("Backing up database to $BACKUPFILE...");
    if (&suRunCmd($PGUSER,
                  "$ENV PGUSER=$DBUSER PGPASSWORD=$DBPASSWD $PGDUMP -p $DBPORT $DBNAME > $BACKUPFILE")) {
        &printError(" Failed.");
        return -1;
    }
    &printInfo(" Done.");
    return 0;
}

sub doDbRestore
{
    if ( ! -r $BACKUPFILE ) {
        &printError("Cannot find backup file $BACKUPFILE");
        return -1;
    }
    
#     &printInfo("Dropping $DBNAME database...");
#     if (&suRunCmd("$PGDROP -U $PGUSER $DBNAME")) {
#         &printError(" Failed.");
#         return -1;
#     }
#     &printInfo(" Done.");

#     &printInfo("Recreating $DBNAME database...");
#     my $pgCreateCmd = "$PGCREATE -U $PGUSER $DBNAME";
#     if (&suRunCmd($PGUSER, $pgCreateCmd)) {
# 	&printError("Cannot create the database");
# 	&doDbStatus();
# 	return -1;
#     } 

    &printInfo("Destroying existing database...");
    if (&doDbDestroy() != 0) {
        &printError("Could not destroy existing database");
        return -1;
    }

    &printInfo("Recreating database from blank template...");
    if (&doDbInit() != 0) {
        &printError("Could not recreate database");
        return -1;
    }

    &printInfo("Restarting database server...");
    if (&doDbStart() != 0) {
        &printError("Could not restart database");
        return -1;
    }
    
    # wait for database to come up and be ready for connections
    sleep 5;

    &printInfo("Restoring database contents from $BACKUPFILE...");
    if (&doLoadSql($BACKUPFILE) != 0) {
        &printError("Failed to restore database.");
        return -1;
    }

    &printInfo(" Done.");
    return 0;
}

sub doDbStatus
{
    return &suRunCmd($PGUSER, "$PGCTL status -D $DBDIR");
}

sub doPsql    
{
    sanitize_cli_args(\(@ARGV));
    my $psqlCmd = "$ENV PGPASSWORD=$DBPASSWD $PGSQL -p $DBPORT -U $DBUSER $DBNAME @ARGV";
    return &suRunCmd($PGUSER, $psqlCmd);
}

sub sanitize_cli_args (@)
{
    foreach my $arg_ref (@_) {
        die "sanitize_cli_args() requires references to scalars"
            unless ( ref $arg_ref eq 'SCALAR' );
        
        $$arg_ref =~ s/\'/\'\\\'\'/g;
        $$arg_ref = "'" . $$arg_ref . "'";
    }
}

sub doLoadSql    
{
    my $file = $DBSQLDIR;
    my $fileSpec = shift @_;
    return -1 unless (defined($fileSpec) && ! -d $fileSpec);

    if (-r $fileSpec) {
	$file = $fileSpec;
    } else {
	return -1 if ($fileSpec =~ /^\//);
	$file .= "/$fileSpec";
	return -1 unless (-r $file);
    }

    my $psqlCmd = "";
    
    if ($file =~ /\.pl$/) {
	$psqlCmd = "$PERL $file ";
    } else {
	my $psql = $PGSQL;
	$psql .= " -q " unless ($DBDEBUG);
	$psqlCmd = "$ENV PGPASSWORD=$DBPASSWD $psql -p $DBPORT -U $DBUSER ";
	$psqlCmd .= "$DBNAME -f '$file' ";
    }

    $psqlCmd .= ($DBDEBUG ? " 1>&2 " : " > /dev/null 2>&1 ");

    return (&suRunCmd($PGUSER, $psqlCmd) != 0);
}

sub dbStatus
{
    my $status = $DBSTATUS{'?'};
    my $pgStatusCmd = "$PGCTL status -D $DBDIR";

    return $status unless (open(PGSTATUS,"$SU - $PGUSER -c '$pgStatusCmd 2>&1' |"));

    my $line = <PGSTATUS>;    

    close (PGSTATUS);

    if ($line =~ /is running/) {
	$status = $DBSTATUS{'UP'};
    } elsif ($line =~ /not running/) {
	$status = $DBSTATUS{'DOWN'};
    }

    return $status;
}

sub dbVersion
{
    my $pgVersionCmd = "$PGSQL -V";
    return 0 unless (open(PGVERSION,"$pgVersionCmd 2>&1|"));

    my $line = <PGVERSION>;

    close (PGVERSION);

    chomp($line);
    
    my @parts = split(/ /,$line);
    return (defined($parts[2]) && $parts[2] ne "" ? $parts[2] : 0);    
}

sub printInfo
{
    print "INFO   : @_\n";
}

sub printError
{
    print STDERR "ERROR  : @_\n"
}

sub printUsage
{
    my $pgm = $0;
    $pgm =~ s/^.*\///;
    
    # hand format the spacing...

    print <<EOUSAGE;
Usage  : $pgm $MODES{'INIT'}    \tInitializes the db
         $pgm $MODES{'START'}   \tStarts the db (if required)
	 $pgm $MODES{'STOP'}    \tStops the db (if required)
         $pgm $MODES{'RESTART'} \tRestarts the db
	 $pgm $MODES{'KILL'}    \tKills all db related processes
	 $pgm $MODES{'STATUS'}  \tPrints the status of the db
         $pgm $MODES{'DESTROY'} \tStops and destroys the db
         $pgm $MODES{'BACKUP'}  \tBack up the db as an SQL dump to $BACKUPFILE
         $pgm $MODES{'RESTORE'} \tRestore (overwrite) the db from $BACKUPFILE
         $pgm $MODES{'CLI'}     \tAllows CLI access to the db
         $pgm $MODES{'LOADSQL'} \tLoads and SQL file into the db
         $pgm $MODES{'HELP'}    \tPrints this message
EOUSAGE
}

sub mkdirs
{
    my ($curPath, $part, $rc);
    
    my $dir = shift @_;
    return -1 unless (defined($dir) && $dir =~ /^\//);

    my $mode = shift @_;
    $mode = 0755 unless (defined($mode) && $mode+0 >= 0);

    my $gid = -1;
    my $uid = shift @_;
    if (defined($uid) && $uid+0 >= 0) {
	$gid = shift @_;
	unless (defined($gid) && $gid+0 >= 0) {
	    $uid = -1;
	    $gid = -1;
	}
    } else {
	$uid = -1;
    }

    $curPath = "";
    $rc = 0;
    foreach $part (split(/\//,$dir)) {
	$curPath .= "$part/";
	if (! -d $curPath) {
	    unless (mkdir($curPath, $mode)) {
		$rc = -1;
		last;
	    }
	    if ($uid+0 > -1) {
		unless (chown(int($uid+0), int($gid+0), $curPath)) {
		    $rc = -1;
		    last;
		}
	    }
	}
    }

    return $rc;
}

sub setAccess
{
    my $file = shift @_;
    return -1 unless (defined($file) && -r $file);

    my $mode = shift @_;
    return -1 unless (defined($mode) && $mode+0 >= 0);
        
    my $uid = shift @_;
    return -1 unless (defined($uid) && $uid+0 >= 0);

    my $gid = shift @_;
    return -1 unless (defined($gid) && $gid+0 >= 0);

    unless (chmod($mode, $file)) {
	&printError("Cannot set permissions for '$file'");
	return -1;
    }

    unless (chown(int($uid+0), int($gid+0), $file)) {
	&printError("Cannot set owner/group for '$file'");
	return -1;
    }

    return 0;
}

sub suRunCmd
{
    my $user = shift @_;
    return -1 unless (defined($user) && $user ne "");

    if ($user =~ /^([0-9A-Za-z\-\_\.]+)$/) {
	$user = $1;
    } else {
	return -1;
    }

    my $cmd = shift @_;
    return -1 unless (defined($cmd) && $cmd ne "" && $cmd !~ /^\'/);

    if ($cmd =~ /^(\/.+)$/) {
	$cmd = $1;
    } else {
	print "$cmd\n";
	return -1;
    }

    sanitize_cli_args(\$cmd);
    return &runCmd("$SU - $user -c $cmd");
}

sub runCmd
{
    my $cmd = shift @_;
    return -1 unless (defined($cmd) && $cmd ne "");

    if ($cmd =~ /^(\/.+)$/) {
	$cmd = $1;
    } else {
	return -1;
    }

    my $cmdRc = system($cmd);
    $cmdRc /= 256;
    
    return $cmdRc;
}
