#!/usr/bin/perl -w
# $Id: 011_load_builds.pl,v 1.1.2.2.4.6 2005/08/03 21:17:46 dm123365 Exp $
# 011_load_builds.pl - populate tables for mapping architectures to English strings
# Copyright 2004 Sun Microsystems, Inc., All rights reserved.
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms

use strict;
use lib '/scs/lib/perl5'; 
use SCSDB;

# When performing an upgrade, there will be entries already existing
# in the mgmt_build table.  The following cases can arise (build_type => build_class):
#
# 1. type => class - a preexisting correct entry.
# We can overwrite this with the same value safely.
#
# 2. type => type - a host was managed with an unsupported OS that we
#                   now support.
# Overwrite the build_class field with the new string.
#
# 3. <UnknownType> => class - a module listed support for an unsupported type that is now supported.
#
# When we create a new (correct) type=>class mapping, this entry will
# be a duplicate.  The corresponding entries in mgmt_mapp_client_rpm
# must be updated to point to the new entry, and the duplicate entry
# then removed.

my %BUILDTABLE =
    (
        # Solaris
        
        # 32-bit SPARC
        'SunOS|5.9|sparc_32'   =>       'sparc Solaris 9',
        'SunOS|5.9|sparc_32'   =>       'sparc Solaris 9',
        'SunOS|5.10|sparc_32'  =>       'sparc Solaris 10',
        # 64 bit sparc
        'SunOS|5.9|sparc_64'   =>       'sparc-64 Solaris 9',
        'SunOS|5.10|sparc_64'  =>       'sparc-64 Solaris 10',
        # x86
        'SunOS|5.9|i386'       =>       'x86 Solaris 9',
        'SunOS|5.10|i386'      =>       'x86 Solaris 10',
        # x86_64
        'SunOS|5.10|x86_64'    =>       'x86-64 Solaris 10',
        
        # RedHat
        
        # community versions
        'RedHatLinux|7.3|i386'    =>       'x86 Red Hat 7.3',
        'RedHatLinux|8.0|i386'    =>       'x86 Red Hat 8.0',
        'RedHatLinux|9|i386'      =>       'x86 Red Hat 9',
        # Fedora
        'FedoraLinux|1|i386'      =>       'x86 Fedora Core 1',
        'FedoraLinux|2|i386'      =>       'x86 Fedora Core 2',
	'FedoraLinux|3|i386'	  =>	   'x86 Fedora Core 3',
        'FedoraLinux|1|x86_64'    =>       'x86-64 Fedora Core 1',
        'FedoraLinux|2|x86_64'    =>       'x86-64 Fedora Core 2',
        'FedoraLinux|3|x86_64'    =>       'x86-64 Fedora Core 3',
        # Red Hat Enterprise Linux
        'RedHatLinux|2.1WS|i386'  =>       'x86 Red Hat EL WS 2.1',
        'RedHatLinux|2.1AS|i386'  =>       'x86 Red Hat EL AS 2.1',
        'RedHatLinux|2.1ES|i386'  =>       'x86 Red Hat EL ES 2.1',
        'RedHatLinux|3WS|i386'    =>       'x86 Red Hat EL WS 3',
        'RedHatLinux|3AS|i386'    =>       'x86 Red Hat EL AS 3',
        'RedHatLinux|3ES|i386'    =>       'x86 Red Hat EL ES 3',
        'RedHatLinux|3WS|x86_64'  =>       'x86-64 Red Hat EL WS 3',
        'RedHatLinux|3ES|x86_64'  =>       'x86-64 Red Hat EL ES 3',
        'RedHatLinux|3AS|x86_64'  =>       'x86-64 Red Hat EL AS 3',
        'RedHatLinux|4WS|i386'    =>       'x86 Red Hat EL WS 4',
        'RedHatLinux|4AS|i386'    =>       'x86 Red Hat EL AS 4',
        'RedHatLinux|4ES|i386'    =>       'x86 Red Hat EL ES 4',
        'RedHatLinux|4WS|x86_64'  =>       'x86-64 Red Hat EL WS 4',
        'RedHatLinux|4ES|x86_64'  =>       'x86-64 Red Hat EL ES 4',
        'RedHatLinux|4AS|x86_64'  =>       'x86-64 Red Hat EL AS 4',

        # SuSE Professionsal Linux
        
        'SuSELinux|8.2|i386'    =>       'x86 SuSE Linux 8.2',
        'SuSELinux|9.0|i386'    =>       'x86 SuSE Linux 9.0',
        'SuSELinux|9.1|i386'    =>       'x86 SuSE Linux 9.1',
        'SuSELinux|9.2|i386'    =>       'x86 SuSE Linux 9.2',
        'SuSELinux|9.0|x86_64'  =>       'x86-64 SuSE Linux 9.0',
        'SuSELinux|9.1|x86_64'  =>       'x86-64 SuSE Linux 9.1',
        'SuSELinux|9.2|x86_64'  =>       'x86-64 SuSE Linux 9.2',
       
	# SuSE Enterprise Server 
	
	'SLES|9|i386'         =>       'x86 SuSE Linux Enterprise Server 9.0',
	'SLES|9|x86_64'       =>       'x86-64 SuSE Linux Enterprise Server 9.0',
 
        # Java Desktop System
        
        'JDS|1|i386'  =>       'x86 Java Desktop System Release 1',
        'JDS|2|i386'  =>       'x86 Java Desktop System Release 2'
    );

my $Select_Stmt = SCSDB::prepareStmt("SELECT build_id FROM mgmt_build WHERE build_type=?");
my $Insert_Stmt = SCSDB::prepareStmt("INSERT INTO mgmt_build (build_type, build_class) VALUES (?, ?)");
my $Update_Stmt = SCSDB::prepareStmt("UPDATE mgmt_build SET build_class=? WHERE build_type=?");

my $Unknown_Select_Stmt = SCSDB::prepareStmt("SELECT build_id FROM mgmt_build WHERE build_type='<UnknownType>' AND build_class=?");
my $Update_Client_RPM_Stmt = SCSDB::prepareStmt("UPDATE mgmt_mapp_client_rpm SET build_id=? WHERE build_id=?");
my $Delete_Duplicate_Stmt = SCSDB::prepareStmt("DELETE FROM mgmt_build WHERE build_id=?");

die "Could not prepare SQL statements." unless ( defined $Select_Stmt and
                                                 defined $Insert_Stmt and
                                                 defined $Update_Stmt and
                                                 defined $Unknown_Select_Stmt and
                                                 defined $Update_Client_RPM_Stmt and
                                                 defined $Delete_Duplicate_Stmt );

my $rc;
$rc= SCSDB::startTxn();
die "Could not start DB transaction"
    if ( $rc < 0 );

for my $build_type ( sort keys %BUILDTABLE ) {
    my $build_class = $BUILDTABLE{$build_type};
    my $current_id = SCSDB::getResult( $Select_Stmt, $build_type );

    if ( ! $current_id ) {
        # no result found
        $rc = SCSDB::executeStmt( $Insert_Stmt, $build_type, $build_class );
        die "Could not execute INSERT for $build_type"
            if ( $rc < 0 );
        $current_id = SCSDB::getInsertId( 'mgmt_build', 'build_id' );
    } else {
        $rc = SCSDB::executeStmt( $Update_Stmt, $build_class, $build_type );
        die "Could not execute UPDATE for $build_type"
            if ( $rc < 0 );
    }

    # Now check for an old duplicate entry of the form in 3. above
    my $duplicate_id = SCSDB::getResult( $Unknown_Select_Stmt, $build_class );
    if ($duplicate_id) {
        SCSDB::executeStmt( $Update_Client_RPM_Stmt, $current_id, $duplicate_id );
        die "Could not update module build IDs for $build_type"
            if ( $rc < 0 );
        SCSDB::executeStmt( $Delete_Duplicate_Stmt, $duplicate_id );
        die "Could not remove duplicate entry for $build_type"
            if ( $rc < 0 );
    }
}

$rc = SCSDB::commit();
die "Could not commit DB transaction"
    if ( $rc < 0 );
