#!/usr/bin/perl -w
# $Id: SysCmd.pm,v 1.17.4.1.4.1 2005/03/18 21:27:31 bm122834 Exp $
# SysCmd.pm - system independent command access
# Copyright 2004 Sun Microsystems, Inc., All rights reserved.
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms

package SysCmd;

use lib '/scs/lib/perl5';
use strict;
use MIME::Base64;
use IO::Select;
use Cwd;
use Carp qw(cluck);
use BDUtil;

my @usualSuspects =
(
  '/bin',
  '/usr/bin',
  '/sbin',
  '/usr/sbin',
  '/opt/sfw/bin/'
);

my %specialCmds =
(
  'Linux' =>
  {
    'ps'    => 'ps auwx',
    'mkdir' => 'mkdir -p',
    'zcat'  => 'zcat'
  },
  'SunOS' =>
  {
    'ps'    => 'ps -ef',
    'mkdir' => 'mkdir -p',
    'zcat'  => 'gzcat'
  }
);

my $solarisPkgAdminFile = '/scs/etc/pkgAdminFile';

my $osCmd = '/bin/uname';
$osCmd = '/usr/bin/uname' if (! -x $osCmd);
my $os = `$osCmd`;
chomp($os);
if (!exists($specialCmds{$os}))
{
  BDUtil::printError('OS not supported : ' . $os);
}

#
#
#
sub getOS
{
  return $os;
}

#
#
#
sub getCmd
{
  my $cmd = shift;
  my $dontExit = shift || 0;

  my $fCmd;
  foreach my $d (@usualSuspects)
  {
    $fCmd = $d . '/' . $cmd;
    if ( -x $fCmd)
    {
      if (!exists($specialCmds{$os}->{$cmd}))
      {
        return $fCmd . ' ';
      }
      return $d . '/' . $specialCmds{$os}->{$cmd} . ' ';
    }
  }

  if ($dontExit)
  {
    BDUtil::setError("Can't find command : " . $cmd);
    return '';
  }

  BDUtil::printError("Can't find command, exiting : " . $cmd);
}

#
#
#
sub runCmd
{
  my $cmd = shift;
  my $args = shift || '';

  return runFullCmd(getCmd($cmd) . $args);
}

#
#
#
sub runFullCmd
{
  my $cmd = shift;
  return ("", -1) unless (defined($cmd) && $cmd ne "");
  
  open(RUNFULLCMD,"$cmd|") || return ("", -1);

  my $sel = new IO::Select(\*RUNFULLCMD);

  my $outStr = "";
  my @ready = ();

  while (!eof(RUNFULLCMD)) {
      @ready = $sel->can_read(60);
      unless (@ready && scalar(@ready) == 1) {
	  cluck "Select Failed!";
	  last;
      }
      $outStr .= <RUNFULLCMD>;      
  }

  chomp($outStr);

  close (RUNFULLCMD);

  return ($outStr, $?);
}

#
#
#
sub getPingCmd
{
  my $remoteHost = shift;
  my $extraArgs = shift || '';
  my $timeout = shift || 0;

  my $cmd = getCmd('ping') . ' ' . $extraArgs . ' ' . $remoteHost;
  return $cmd if (!$timeout);
  if ($os eq 'Linux')
  {
    $cmd .= ' -w';
  }
  $cmd .= ' ' . $timeout;

  return $cmd;
}

#
#
#
sub unpackGzippedTar
{
  my ($file, $destdir) = @_;

  my $rc = 0;
  if ($os eq 'SunOS')
  {
    system(getCmd('cp') . $file . ' ' . $destdir);
    my $orig_directory = cwd();
    chdir($destdir);
    $rc = system(getCmd('gunzip') . '-S .mapp ' . $file);
    if (!$rc)
    {
      # get file name without extension (assuming .mapp)
      my $file_noext = substr($file, 0, -5);
      $rc = system(getCmd('tar') . 'xf ' . $file_noext);
      unlink($file_noext);
    }
    chdir($orig_directory);
  }
  else
  {
    # Linux tar supports 'z' and '-C' args.
    $rc = system(getCmd('tar') . 'zxf ' . $file . ' -C ' . $destdir . ' 2>/dev/null');
  }

  return $rc;
}

#
#
#
sub installPkg
{
  my $pkg = shift;
  my $adminFile = shift || '';
  my $resourceFile = shift || '';

  my ($errStr, $rc);
  if (($os eq 'Linux') && ($pkg =~ /\.rpm$/))
  {
    ($errStr, $rc) = runCmd('rpm', '-i --nodeps --force ' . $pkg);
    if ($rc > 0)
    {
      $rc *= -1;  # register as error
    }
  }
  elsif (($os eq 'SunOS') && ($pkg !~ /\.rpm$/))
  {
    if (($adminFile eq '') || ( ! -f $adminFile) || ( ! -r $adminFile ))
    {
      checkAdminFile();
      ($errStr, $rc) = runCmd('pkgadd', '-a ' . $solarisPkgAdminFile
                                     . ' -d ' . $pkg . ' all');
    }
    elsif (($resourceFile ne '') && ( -f $resourceFile) && ( -r $resourceFile))
    {
      ($errStr, $rc) = runCmd('pkgadd', '-a ' . $adminFile
                                     . ' -r ' . $resourceFile
                                     . ' -d ' . $pkg . ' all');
    }
    else
    {
      ($errStr, $rc) = runCmd('pkgadd', '-a ' . $adminFile
                                     . ' -d ' . $pkg . ' all');
    }
    if ($rc > 0)
    {
      $rc *= -1;  # register as error
    }
  }
  else
  {
    BDUtil::info($pkg . ' is not compatible for OS (' . $os
                      . '), not installing');
    return 1;
  }

  return $rc;
}

#
#
#
sub uninstallPkg
{
  my ($path, $pkg, $type) = @_;

  my ($errStr, $rc);
  if (($os eq 'Linux') && ($type eq 'rpm'))
  {
    ($errStr, $rc) = runCmd('rpm', '-e --allmatches --nodeps ' . $pkg);
  }
  elsif (($os eq 'SunOS') && ($type eq 'svr4'))
  {
    checkAdminFile();

    (my $pkgNames, $rc) = runFullCmd(
                            getCmd('pkginfo') . '-d ' . $path . '/' . $pkg
                  . ' | ' . getCmd('awk') . "{'print \$2'}");
    return -1 if ($rc);
    foreach my $p (split(/\s+/, $pkgNames))
    {
      ($errStr, $rc) = runFullCmd(getCmd('echo') . 'y | ' . getCmd('pkgrm') . '-a ' . $solarisPkgAdminFile . ' ' . $p);
    }
  }
  else
  {
    BDUtil::info($pkg . ' is not compatible for OS (' . $os
                      . '), not uninstalling');
    return 1;
  }

  return $rc;
}

#
#
#
sub checkAdminFile
{
  return $solarisPkgAdminFile if ( -r $solarisPkgAdminFile);

  open(ADMIN_FILE, '>' . $solarisPkgAdminFile)
    || die('Agh! Cannot write to ' . $solarisPkgAdminFile);

  print ADMIN_FILE<<EOD;
# SCS pkg admin file
mail=
instance=unique
partial=default
runlevel=default
idepend=default
rdepend=default
space=default
setuid=default
conflict=default
action=nocheck
basedir=default
EOD

  close(ADMIN_FILE);

  return $solarisPkgAdminFile;
}

#
#
#
sub checkPkg
{
  my ($pkg, $type) = @_;

  my ($pkgName, $version, $rev, $current, $rc);
  if ($type eq 'rpm')
  {
    ($pkgName, $version, $rev) = ($pkg =~ /^(.+)\-(.+)\-([0-9]+)$/);
    ($current, $rc) = runCmd('rpm', '-q ' . $pkgName . ' 2>/dev/null');
    if ($rc || ($current eq '') || ($current =~ /^package .* is not installed/))
    {
      return (-1, '', '', '', '');
    }

    return (0, $current, $pkgName, $version, $rev);
  }

  ($current, $rc) = runCmd('pkginfo', '-l ' . $pkg . ' 2>/dev/null');
  if ($rc || ($current eq ''))
  {
    return (-1, '', '', '', '');
  }

  ($version, $rev) = ($current =~ /VERSION:\s+([0-9\.]+),REV=([0-9\.]+)/);
  if (!defined($version) || !$version)
  {
    return (-1, '', '', '', '');
  }
  $rev ||= 0;

  return (0, $pkg . '-' . $version . '-' . $rev, $pkg, $version, $rev);
}

#
#
#
sub getPkgInfo
{
  my $pkg = shift;

  if (($os eq 'Linux') && exists($pkg->{'rpm'}))
  {
    return ('rpm', $pkg->{'rpm'});
  }

  if (($os eq 'SunOS') && exists($pkg->{'svr4'}))
  {
    return ('svr4', $pkg->{'svr4'});
  }

  return (0, 0);
}

#
#
#
sub getCrontabPath
{
  if ($os eq 'Linux')
  {
    return ('/etc/cron.d', 'mgmtSchedule');
  }
  return ('/var/spool/cron/crontabs', '/root');
}

#
#
#
sub getCronEntry
{
  my $prog = shift;

  if ($os eq 'Linux')
  {
    return 'root ' . $prog . ' >&/dev/null';
  }
  return $prog . ' 2>&1 >/dev/null';
}

#
#
#
sub commitSchedulerCrontab
{
  my $delimiterStr = shift;
  my $crontabStr = shift;

  my $tmpCrontab = '/tmp/newCrontab.' . $$;

  if (!open (CRONTAB, '>' . $tmpCrontab))
  {
    BDUtil::setError("Can't open tmp crontab $tmpCrontab file for writing!");
    return -1;
  }

  if ($os eq 'Linux')
  {
    print CRONTAB $crontabStr . "\n";
    close CRONTAB;
    SysCmd::runCmd('mv', $tmpCrontab . ' /etc/cron.d/mgmtScheduler');
    # So that crond picks up the change...
    my $now = time();
    utime ($now, $now, '/etc/cron.d');
    return 0;
  }

  # Solaris

  if (!open (OLDCRONTAB, '/var/spool/cron/crontabs/root'))
  {
    close CRONTAB;
    BDUtil::setError('Failed to open crontab /var/spool/cron/crontabs/root !');
    return -1;
  }

  my $line;
  my $inSched = 0;
  while ($line = <OLDCRONTAB>)
  {
    if ($line =~ /$delimiterStr/)
    {
      $inSched = !$inSched;
      next;
    }
    next if ($inSched);
    print CRONTAB $line;
  }
  close OLDCRONTAB;

  print CRONTAB "\n" . '# start ' . $delimiterStr . "\n";
  print CRONTAB $crontabStr . "\n";
  print CRONTAB '# end' . $delimiterStr . "\n";
  close CRONTAB;
  SysCmd::runCmd('crontab', $tmpCrontab);
  unlink($tmpCrontab);

  return 0;
}

#
#
#
sub getRunlevel
{
  my ($outStr, $rc, $runLevel);
  if ($os eq 'Linux')
  {
    ($outStr, $rc) = runCmd('runlevel');
    ($runLevel) = ($outStr =~ /^\S\s(.)$/);
  }
  else
  {
    ($outStr, $rc) = runCmd('who', '-r');
    ($runLevel) = ($outStr =~ /run-level ([0-9])/);
  }
  return $runLevel;
}

#
#
#
sub sendEmail
{
  my ($fromAddr, $toAddr, $subject, $body, $bodyFile, $header) = @_;

  $bodyFile ||= 0;
  return -1 if (!$body && !$bodyFile);

  my $tmpMailFile = '/tmp/mail.' . $$;
  return -1 if (!open(TMP_MAIL, ">$tmpMailFile"));

  my $cmd = getCmd('mail');

  my $rc;
  if ($bodyFile)
  {
    ($body, $rc) = runCmd('cat', $bodyFile);
    return -1 if ($rc);
  }
  $subject = encode_base64($subject);
  chomp $subject;
  # Format the Subject and Body to encode for Double Byte Chars
  $header = "To:". $toAddr . "\nSubject:=?UTF-8?B?". $subject . "?=\nContent-type: text/plain; charset=UTF-8; format=flowed\nContent-transfer-encoding: BASE64\n";
 $body = encode_base64($body);
# BDUtil::info("HEADER=>" . $header);
# BDUtil::info("BODY=>" . $body);
  print TMP_MAIL $header;
  print TMP_MAIL "\n";
  print TMP_MAIL $body;
  close (TMP_MAIL);

  
  $cmd = "cat $tmpMailFile|/usr/sbin/sendmail -F \"$fromAddr\" $toAddr";
# BDUtil::info("Mail Cmd=> $cmd");
  $rc = system($cmd);

  unlink($tmpMailFile);

  return $rc;
}

1;
