# $Id: SCSUtil.pm,v 1.9 2004/12/22 01:08:01 ms152511 Exp $
# SCSUtil - miscellaneous utilities for Control Station
# Copyright 2004 Sun Microsystems, Inc.  All rights reserved
# Use is subject to license terms.

package SCSUtil;

use strict;
use IO::Socket;
use lib '/scs/lib/perl5';
use ErrorUtil;

#
# Determine which OS we are running on.
#
sub getOS () {
    my $vender = '';
    
    # get the correct path to uname.
    my $unameCmd = '/bin/uname';
    $unameCmd = '/usr/bin/uname' if ( !-x $unameCmd );
    
    # execute the uname command.
    my $uname = `$unameCmd`;
    chomp($uname);
    
    #
    if ( $uname eq 'SunOS' ) { $vender = 'solaris'; }
    elsif ( $uname eq 'Linux' ) {
        if ( -f '/etc/redhat-release' ) { $vender = 'redhat'; }
        elsif ( -f '/etc/SuSE-release' ) { $vender = 'suse'; }
        
        else {
            ErrorUtil::setError( 'OS not supported : ' . $vender );
            return;
        }
    }
    
    return $vender;
}

#
# isIPAddress
#
# Decide whether a string represents an IP address.
#
sub isIPAddress ($) {
    return 0 unless (defined $_[0]);
    return ($_[0] =~ /^
            (?: (?: \d |     #   0-  9
                 [01]?\d\d | #  10-199
                 2[0-4]\d  | # 200-249
                 25[0-5]     # 250-255
                 )
             \.
             ){3}            # first three octets
            (?: \d |
             [01]?\d\d |
             2[0-4]\d  |
             25[0-5]
             )               # final octet
            $/x);
}

#
# isHostname
#
# Decide whether a string follows the syntax of a hostname.
# Note that this function also accepts IP addresses.
#
sub isHostname ($)
{
    return 0 unless defined($_[0]);
    return ($_[0] =~ /^
            [[:alnum:]]    # First character is alpha-numeric. (RFC 1123)
            [[:alnum:].-]* # Remaining characters are drawn from the
                           # alphabet (A-Z), digits (0-9), minus sign (-),
                           # and period (.) (RCF 952)
            [[:alnum:]]    # The last character must not be a minus sign
                           # or period.  (RFC 952)
            $/x);
}

#
# getIPAddress
#
# Gets the IP address for a host, resolving if necessary.
# Returns -1 if the argument does not represent a
# resolvable hostname or IP address.
#
sub getIPAddress ($) {
    my ($host) = @_;
    return -1 unless (defined $host);

    my $ipaddr = -1;

    my $tmpip = inet_aton($host);
    if (defined $tmpip) {
        $ipaddr = inet_ntoa($tmpip);
        $ipaddr = -1 unless (defined $ipaddr);
    }

    return $ipaddr;
}

#
# getHostname
#
# Gets the canonical hostname for an IP address via reverse DNS lookup.
# Returns -1 if the argument is not a valid IP address or the reverse
# lookup fails.
#
sub getHostname ($) {
    my ($ipaddr) = @_;
    return -1 unless (isIPAddress($ipaddr));

    my $packed_addr = gethostbyname($ipaddr);
    return -1 if (not defined $packed_addr);

    my $hostname = (gethostbyaddr($packed_addr, AF_INET))[0];
    return -1 if (not defined $hostname);

    return $hostname;
}
    

#
# checkEmailAddress
#
# Checks a email address for the basic sytax.
# Provided by: http://www.castellum.net
#
sub checkEmailAddress ($) {
    return (defined $_[0] and
            $_[0] =~ /^[A-Z0-9][\-\.A-Z0-9]*\@\[?[\-\.A-Z0-9]+\.([A-Z]{2,3}|[0-9]{1,3})\]?$/i);
}

1;
