#!/usr/bin/perl -w
# $Id: SCSDB.pm,v 1.30 2004/11/22 20:54:38 ms152511 Exp $
# SCSDB.pm - SCS db access routines
# Copyright 2004 Sun Microsystems, Inc., All rights reserved.
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms

package SCSDB;

use lib '/scs/lib/perl5';
use strict;
use DBI;
use Carp qw(cluck);

#
# Subroutines
#
# Open / close connection:
# ------------------------
#  openConnection  : No need to call this, done automatically.
#  closeConnection : Call on program exit, or it will be done automatically.
#  forkProcess     : Call this to make sure both parent & child can use DB
#                    instead of fork(), returns pid.
#
# Command functions: (for insert, update, delete)
# ------------------
#  runCommand      : returns 0 upon success, -1 otherwise.
#  getInsertId     : returns last ID given to auto_increment column for insert
#  getRowsAffected : returns number of rows affected by last command
#
# Query functions taking a SQL string argument, return value:
# -----------------------------------------------------------
# (call this...       if you want this...)
#
#  getResult       :  Scalar (single record, single value)
#  getResultList   :  Reference to list of scalars (records have only one item)
#  getResultHash   :  Hash reference (single record)
#  getResultHashes :  Reference to list of hash references
#  hasResults      :  Boolean on whether or not results exist for query
#
# Diagnostic functions:
# ---------------------
#  errorCode         : set error message, return error code
#  getDBerrorString  : Get message set by DB engine for a failed query
#  getErrorMessage   : Get message set by errorCode() function in an API
#  getLastSQL        : Get last SQL command processed by DB engine
#

my $MKTABLE_SELECTSTMT = 0;
my $DROPTABLES_TABLESELECTSTMT = 0;
my $DROPTABLES_SEQSELECTSTMT = 0;
my $DROPTABLES_INDEXSELECTSTMT = 0;
my $DBDEBUG = 0;
my $DBH = 0;
my $DBQUERY = 0;
my $DBROW = 0;
my $DBLASTSQL = '';
my $DBTXN = 0;
my @DBVERSINFO = ();

#
# Error / Return codes
#

my %errorCodes = ( 'OKAY'   =>  0,    # Everything's cool
		   'EINVAL' => -1,    # Invalid condition
		   'EDBERR' => -2,    # DB Error
		   'EDUP'   => -3,    # Duplicate
		   'EPARM'  => -4 );  # Missing parameters

my $errorMessage = '';

#
# enableDBDebugging
#
# turns on database debugging
#

sub enableDBDebugging
{
    $DBDEBUG = 1;
}

#
# disableDBDebugging
#
# turns on database debugging
#

sub disableDBDebugging
{
    $DBDEBUG = 0;
}

#
# printSQLDebug
#
# private subroutine used to print out sql statements
#

sub printSQLDebug
{
    return unless ($DBDEBUG);    
    print STDERR ">>>>>>>>>>\nSQL: @_\n<<<<<<<<<<\n";
}

#
# printDBDebug
#
# private subroutines used to print out debugging messages
#

sub printDBDebug
{
    return unless ($DBDEBUG);
    cluck "DEBUG: @_";
}

#
# openConnection
#
# Opens a db connection
#

sub openConnection
{
    if ($DBH) {
	printDBDebug("Connection already open");
	return -1;
    }
    
    my ($dbUrl, $dbPort, $dbUser, $dbPass) = do {
        require DBInfo;
        DBInfo::getDBInfo();
    };

    my $dbOpts = shift @_;
    $dbOpts->{'PrintError'} = 1 
	unless (defined($dbOpts) && ref($dbOpts) eq "HASH");

    $DBH = DBI->connect("DBI:Pg:dbname=$dbUrl;port=$dbPort", 
			$dbUser, 
			$dbPass,
			$dbOpts);

    unless ($DBH) {
	printDBDebug("Connection failed");
	exit (1);
    }

    return 0;
}

#
# closeConnection
#
# closes a db connection
#

sub closeConnection
{
    # finish the statements

    if (defined($MKTABLE_SELECTSTMT) && 
	int($MKTABLE_SELECTSTMT+0) != 0) {
        finishStmt($MKTABLE_SELECTSTMT);
        $MKTABLE_SELECTSTMT = 0;
    }

    if (defined($DROPTABLES_TABLESELECTSTMT) && 
	int($DROPTABLES_TABLESELECTSTMT+0) != 0) {
        finishStmt($DROPTABLES_TABLESELECTSTMT);
        $DROPTABLES_TABLESELECTSTMT = 0;
    }

    if (defined($DROPTABLES_SEQSELECTSTMT) && 
	int($DROPTABLES_SEQSELECTSTMT+0) != 0) {
        finishStmt($DROPTABLES_SEQSELECTSTMT);
        $DROPTABLES_SEQSELECTSTMT = 0;
    }

    # rollback any existing transactions

    SCSDB::rollBack() if ($DBTXN);

    # finish any open queries

    if ($DBQUERY) {
	$DBQUERY->finish();
	$DBQUERY = 0;
    }

    # disconnect from the db

    if ($DBH) {
	$DBH->disconnect();
	$DBH = 0;
    }

    return 0;
}

#
# forkProcess
#
# ???
#

sub forkProcess
{
    closeConnection();
    my $pid = fork();
    openConnection();
    return $pid;
}

#
# runCommand
#
# Run a SQL command that does not return any rows, like INSERT
#

sub runCommand
{
    my $sql = shift @_;
    return -1 unless (defined($sql) && $sql ne "");

    # save this command as the last sql command executed

    $DBLASTSQL = $sql;
    
    printSQLDebug($sql);

    openConnection() unless ($DBH);

    # execute the sql 
    
    unless ($DBH->do($sql)) {
	printDBDebug("DBI->do failed");
	return -1;
    }

    return 0;
}

#
# startTxn
#
# starts a db transaction
#

sub startTxn
{    
    if ($DBTXN) {
	printDBDebug("Nested transaction");
	return -1;
    }

    openConnection() unless ($DBH);
    
    $DBH->{AutoCommit} = 0;
    $DBH->{RaiseError} = 1;

    $DBTXN = 1;

    return 0;
}

#
# rollBack
#
# rolls back a db transaction
#

sub rollBack
{    
    return -1 unless ($DBH);

    unless ($DBTXN) {
	printDBDebug("Rollback called outside of transaction");
	return -1;
    }

    $DBH->rollback;

    $DBH->{AutoCommit} = 1;
    $DBH->{RaiseError} = 0;

    $DBTXN = 0;

    return 0;
}

#
# commit
#
# commit a db transaction
#

sub commit 
{
    return -1 unless ($DBH);

    unless ($DBTXN) {
	printDBDebug("Commit called outside of transaction");
	return -1;
    }

    $DBH->commit();
    
    $DBH->{AutoCommit} = 1;
    $DBH->{RaiseError} = 0;
    
    $DBTXN = 0;

    return 0;
}

#
# prepareStmt
#
# prepares a sql statement
#

sub prepareStmt
{
    my $sql = shift @_;
    return 0 unless (defined($sql) && $sql ne "");

    openConnection() unless ($DBH);

    my $DBLASTSQL = $sql;

    printSQLDebug($sql);

    my $stmt = $DBH->prepare($sql);
    unless(defined($stmt) && $stmt) {
	printDBDebug("Cannot prepare statement");
	$stmt = 0;
    }
    
    return $stmt;
}

#
# finishStmt
#
# calls finish on a prepared statement
#

sub finishStmt
{
    my $stmt = shift @_;
    $stmt->finish() if (defined($stmt));
}

#
# mkTable
#
# creates a table if it doesn't exist
#

sub mkTable
{
    my $table = shift @_;
    return -1 unless (defined($table) && isValidFieldName($table));

    my $sql = "";
    my $haveTable = "";

    unless (defined($MKTABLE_SELECTSTMT) && 
	    int($MKTABLE_SELECTSTMT+0) != 0) {
	$sql = "SELECT tablename FROM pg_tables WHERE tablename = ?";
	$MKTABLE_SELECTSTMT = prepareStmt($sql);
	return -1 unless ($MKTABLE_SELECTSTMT);
    }
    
    return -1 if (executeStmt($MKTABLE_SELECTSTMT, $table) != 0);

    my $DBROW = $MKTABLE_SELECTSTMT->fetchrow_arrayref();
    
    $haveTable = $DBROW->[0] 
	if (defined($DBROW) && ref($DBROW) eq "ARRAY");
    
    return 0 if ($haveTable eq $table);

    $sql = "CREATE TABLE $table (";
    my $field = "";
    my $field_sep = " ";
    foreach $field (@_) {
	$sql .= "$field_sep$field";
	$field_sep = ", ";
    }
    $sql .= ")";
    
    return runCommand($sql);
}

#
# getInsertId
#
# Return last ID given to auto_increment column.  Uses currval to make
# sure the ID corresponds to the last operation from the current
# session.
#

sub getInsertId
{
    my $table = shift @_;
    return -1 unless (defined($table) && isValidFieldName($table));

    my $field = shift @_;
    return -1 unless (defined($field) && isValidFieldName($field));

    my $seq = shift @_;
    $seq = "" unless (defined($seq) && isValidFieldName($seq));

    my $insertId = 0;

    @DBVERSINFO = getDBVersion()
	if (scalar(@DBVERSINFO) <= 0);

    # !!! HACK !!!
    # For version 7.3 or newer first try the auto generated seq
    # naming scheme [table]_[field]_seq. If that fails and a
    # seq name is given try that.

    my $sql_auto = "SELECT currval('$table" . "_" . "$field" . "_seq')";

    if ($DBVERSINFO[1] > 7 || ($DBVERSINFO[1] == 7 && $DBVERSINFO[2] >= 3)) {
	$insertId = getResult($sql_auto);
	return $insertId if ($insertId+0 > 0);
    }

    my $sql = ($seq ne "" ? "SELECT currval('$seq')" : $sql_auto);

    return getResult($sql);
}

#
# getRowsAffected
#
# returns the number of rows affect by the last execution of a prepared statement
#

sub getRowsAffected
{
    return -1 unless (scalar(@_) == 1);

    my $stmt = shift @_;
    return -1 unless (defined($stmt) && int($stmt+0) != 0);

    return $stmt->rows;
}

#
# executeStmt
#
# executes a prepared statement
#

sub executeStmt
{
    return -1 unless (scalar(@_) > 1);

    my $stmt = shift @_;
    return -1 unless (defined($stmt) && int($stmt+0) != 0);

    if ($DBDEBUG) {
	printSQLDebug($stmt->{Statement},'(',join(', ', @_), ')');
    }

    unless ($stmt->execute(@_)) {
	printDBDebug("Execute statement failed");
	return -1;
    }

    return 0;
}

#
# executeQuery
#
# prepares and executes a specified sql statement
#

sub executeQuery
{
    my $sql = shift @_;
    return 0 unless (defined($sql) && $sql ne "");

    # save the sql 
    
    $DBLASTSQL = $sql;

    printSQLDebug($sql);

    openConnection() unless ($DBH);

    $DBQUERY->finish() if ($DBQUERY);

    $DBQUERY = $DBH->prepare($sql);
    return 0 unless ($DBQUERY);

    unless ($DBQUERY->execute()) {
	printDBDebug("Query execution failed");
	$DBQUERY->finish();
	$DBQUERY = 0;
    }
    
    return $DBQUERY;
}

#
# hasResults
#
# Check to see if a query yields any results
#

sub hasResults
{
    if (scalar(@_) > 1) {
	my $stmt = shift @_;
	return 0 unless (defined($stmt) && int($stmt+0) != 0);
	return 0 if (executeStmt($stmt, @_) != 0);

	return ($stmt->fetchrow_arrayref() ? 1 : 0);
    }

    my $sql = shift @_;
    return 0 unless (defined($sql) && $sql ne "");

    $DBQUERY = executeQuery($sql);
    return 0 unless ($DBQUERY);
    
    $DBROW = $DBQUERY->fetchrow_arrayref();
    return ($DBROW ? 1 : 0);
}

#
# getResultHashes
#
# Return a reference to a list of hash references containing
# the results of the given SQL query.
#

sub getResultHashes
{
    my @results = ();
	
    if (scalar(@_) > 1) {
	my $stmt = shift @_;
	return 0 unless (defined($stmt) && int($stmt+0) != 0);	
	return 0 if (executeStmt($stmt, @_) != 0);
	
	my $row = $stmt->fetchrow_hashref();
	while (defined($row) && ref($row) eq "HASH") {
	    push (@results, $row);
	    $row = $stmt->fetchrow_hashref();
	} 
	
	return \@results;
    }

    my $sql = shift @_;
    return 0 unless (defined($sql) && $sql ne "");

    $DBQUERY = executeQuery($sql);    
    return 0 unless ($DBQUERY);

    $DBROW = $DBQUERY->fetchrow_hashref();
    while ($DBROW) {
	push (@results, $DBROW);
	$DBROW = $DBQUERY->fetchrow_hashref();
    }

    return \@results;
}

#
# getResultHash
#
# Return a hash reference for a single record
#

sub getResultHash
{
    if (scalar(@_) > 1) {
	my $stmt = shift @_;
	return 0 unless (defined($stmt) && int($stmt+0) != 0);
	return 0 if (executeStmt($stmt, @_) != 0);

	return $stmt->fetchrow_hashref();
    }
    
    my $sql = shift @_;
    return 0 unless (defined($sql) && $sql ne "");

    $DBQUERY = executeQuery($sql);
    return 0 unless ($DBQUERY);

    return $DBQUERY->fetchrow_hashref();
}

#
# getResultList
#
# Return a reference to a list of scalars.
# The given SQL must only request one item per row.
#

sub getResultList
{
    my @results = ();

    if (scalar(@_) > 1) {

	my $stmt = shift @_;
	return 0 unless (defined($stmt) && int($stmt+0) != 0);
	return 0 if (executeStmt($stmt, @_) != 0);

	my $row = $stmt->fetchrow_arrayref();
	while (defined($row) && ref($row) eq "ARRAY" && 
	       defined($row->[0])) {
	    push(@results, $row->[0]);
	    $row = $stmt->fetchrow_arrayref();
	}

	return \@results;
    }

    my $sql = shift @_;
    return 0 unless (defined($sql) && $sql ne "");

    $DBQUERY = executeQuery($sql);
    return 0 unless ($DBQUERY);

    $DBROW = $DBQUERY->fetchrow_arrayref();
    while ($DBROW) {
	push (@results, $DBROW->[0]);
	$DBROW = $DBQUERY->fetchrow_arrayref();
    }

    return \@results;
}

#
# getResult
#
# Return a scalar value from a query
# Hence, the query must return a single row with a single field
# Returns 0 if no row returned.
#

sub getResult
{
    if (scalar(@_) > 1) {
	my $stmt = shift @_;
	return 0 unless (defined($stmt) && int($stmt+0) != 0);
	return 0 if (executeStmt($stmt, @_) != 0);
	
	my $row = $stmt->fetchrow_arrayref();
	return (defined($row) && ref($row) eq "ARRAY" && 
		defined($row->[0]) ? $row->[0] : 0);
    }

    my $sql = shift @_;
    return 0 unless (defined($sql) && $sql ne "");
    
    $DBQUERY = executeQuery($sql);
    return 0 unless ($DBQUERY);

    $DBROW = $DBQUERY->fetchrow_arrayref();    
    return ($DBROW ? $DBROW->[0] : 0);
}

#
# errorCode
#
# ???
#

sub errorCode
{
    my $code = shift @_;
    
    unless (defined($code)) {
	printDBDebug("Error code undefined");
	return -1;
    }

    my $msg = shift @_ || '';
    $errorMessage = $msg;

    my $ec = $errorCodes{$code};
    return (defined($ec) && $ec+0 <= 0 ? $ec : -1);
}

#
# getErrorMessage
#
# ???
#

sub getErrorMessage
{
    return $errorMessage;
}

#
# getDBerrorString
#
# returns the db's error string
#

sub getDBerrorString
{
    return $DBH->errstr;
}

#
# getLastSQL
#
# returns the last sql statement executed
#

sub getLastSQL
{
    return $DBLASTSQL;
}

#
# isValidFieldName
#
# checks to see if a given field name is a valid sql field name
#

sub isValidFieldName
{
    my $field = shift @_;
    return (defined($field) && $field =~ /^[a-zA-Z0-9\_\-]+$/);
}

#
# encodeValue
#
# encodes a given value 
#

sub encodeValue
{
    my $value = "";
    my $cnt = 0;

    $cnt = scalar(@_);

    if ($cnt > 1) {
	$value = shift @_;
    } elsif ($cnt == 1) {
	$value = join(' ', @_);
    } 

    $value =~ s/\'/\&\#039\;/g;
    return $value;
}

#
# decodeValue
#
# decodes a given value
#

sub decodeValue
{    
    my $value = "";
    my $cnt = 0;

    $cnt = scalar(@_);

    if ($cnt > 1) {
	$value = shift @_;
    } elsif ($cnt == 1) {
	$value = join(' ', @_);
    } 

    $value =~ s/\&\#039\;/\'/g;
    return $value;
}

#
# idList
#
# returns a list of ints (ids) from a given list
#

sub idList
{
    my @ids = ();
    my $intId = 0;
    my $id = 0;

    foreach $id (@_) {
	next unless (defined($id));
	$intId = int($id+0);
	push (@ids, $intId) if ($intId > 0);
    }

    return @ids;
}

#
# insertHash
#
# inserts a hash into the db
#

sub insertHash
{
    my $table = shift @_;
    return -1 unless (defined($table) && isValidFieldName($table));

    my $field = shift @_;
    return -1 unless (defined($field) && isValidFieldName($field));

    my $insertHash = shift @_;
    return -1 unless (defined($insertHash) && 
		      ref($insertHash) eq "HASH");

    my $seq = shift @_;
    $seq = "" unless (defined($seq) && isValidFieldName($seq));

    my $colName = "";
    my $value = "";
    my $cols = "";
    my $values = "";

    # NOTE: Resist the temptation to add automatic value quoting. 
    #       Sometimes numbers are passed in. These should not be
    #       quoted. If you need values to be quoted, make sure you
    #       pass in a value that contains the necessary quoting: 
    #       myCol => "'foo bar'"
    #       You can optionally call the encodeValue() method to
    #       encode all single quotes

    foreach $colName (sort(keys(%{$insertHash}))) {
	next unless (defined($colName) && 
		     isValidFieldName($colName));
	$cols .= "$colName, ";	

	$value = $insertHash->{$colName};
	$value = "" unless (defined($value));

	if ($value =~ /^\'.*\'$/s) {
	    $value =~ s/^\'//s;
	    $value =~ s/\'$//s;
            $values .= "'" . SCSDB::encodeValue($value) . "'" . ", ";
	} else {
            $values .= 
                ($value =~ /^[\-]?[0-9]+$/ ? int($value) : 
                 "'" . SCSDB::encodeValue($value) . "'") . ", ";
        }
    }

    $cols =~ s/, $//;
    $values =~ s/, $//;

    my $sql = "INSERT INTO $table ($cols) VALUES($values)";

    return -1 if (SCSDB::runCommand($sql));

    return SCSDB::getInsertId($table, $field, $seq);
}

#
# updateHash
#
# ???
#

sub updateHash
{
    my $table = shift @_;
    return -1 unless (defined($table) && isValidFieldName($table));

    my $fields = shift @_;
    my $matchFields = [ ];
    if (ref($fields) eq 'ARRAY')
    {
      $matchFields = $fields;
    }
    else
    {
      $matchFields = [ $fields ];
    }
    my $field;
    foreach $field (@{$matchFields})
    {
      return -1 unless (defined($field) && isValidFieldName($field));
    }

    my $updateHash = shift @_;
    return -1 unless (defined($updateHash) &&
                      ref($updateHash) eq "HASH");

    my $colName = "";
    my $value = "";
    my $cols = "";
    my $values = "";
    my $sep = '';

    my $sql = 'select ' . $matchFields->[0] . ' from ' . $table . ' where ';
    foreach $field (@{$matchFields})
    {
      $sql .= $sep . $field . " = '" . $updateHash->{$field} . "'";
      $sep = ' and ';
    }
    $sql .= ' for update';

    if (SCSDB::hasResults($sql))
    {
      $sql = 'UPDATE ' . $table . ' SET';
      foreach $colName (sort(keys(%{$updateHash}))) {
        next unless (defined($colName) &&
                     isValidFieldName($colName));
        $sql .= ' ' . $colName . " = " . $updateHash->{$colName} . ",";
        $value = $updateHash->{$colName};
      }
      $sql =~ s/,$//;
      $sep = ' WHERE ';
      foreach $field (@{$matchFields})
      {
        $sql .= $sep . $field . " = '" . $updateHash->{$field} . "'";
        $sep = ' and ';
      }

      return SCSDB::runCommand($sql);
    }
    return 0;
}

#
# replace
#
# simulates mysql's REPLACE INTO. Not recommended.
#

sub replace    
{
    my $sqlSelect = shift @_;
    return -1 unless (defined($sqlSelect) && $sqlSelect ne "");

    my $sqlUpdate = shift @_;
    return -1 unless (defined($sqlUpdate));

    my $sqlInsert = shift @_;
    return -1 unless (defined($sqlInsert) && $sqlInsert ne "");

    startTxn();

    my $sql = $sqlInsert;

    eval {
	if (hasResults($sqlSelect)) {
	  $sql = $sqlUpdate;
        }
        if (runCommand($sql)) {
            rollBack();
	} else {
	    commit();
	}
    };

    if ($@) {
	rollBack();
	return -1;
    }

    return 0;
}

#
# dropTable
#
# drops the specified table and corresponding index and sequence names
#

sub dropTable
{
    my $table = shift @_;
    return -1 unless (defined($table) && isValidFieldName($table));

    my @seqs = ();
    my @indexes = ();
    my $parm = "";

    foreach $parm (@_) {
	next unless (defined($parm) && isValidFieldName($parm));
	if ($parm =~ /_seq$/) {
	    push (@seqs, $parm);
	    next;
	}
	if ($parm =~ /_(key|index)$/) {
	    push(@indexes, $parm);
	    next;
	}
    }
    
    my $sql = "DROP TABLE $table";
    runCommand($sql);

    foreach $parm (@seqs) {
	$sql = "DROP SEQUENCE $parm";
	runCommand($sql);
    }

    foreach $parm (@indexes) {
	$sql = "DROP INDEX $parm";
	runCommand($sql);
    }

    return 0;
}

#
# dropTables
#
# drops tables (sequences and indexes) matching a specified pattern
#

sub dropTables
{
    my $match = shift @_;
    return -1 unless (defined($match) && $match ne "");


    my $seqMatch = shift @_;
    $seqMatch = $match unless (defined($seqMatch) && $seqMatch ne "");

    my $indexMatch = shift @_;
    $indexMatch = $match unless (defined($indexMatch) && $indexMatch ne ""); 

    my $sql = "";

    unless (defined($DROPTABLES_TABLESELECTSTMT) && 
	    int($DROPTABLES_TABLESELECTSTMT+0) != 0) {
        $sql = 
	    'SELECT tablename FROM pg_tables WHERE tablename like ?';
        $DROPTABLES_TABLESELECTSTMT = prepareStmt($sql);
        return -1 unless ($DROPTABLES_TABLESELECTSTMT);
    }

    unless (defined($DROPTABLES_SEQSELECTSTMT) &&
	    int($DROPTABLES_SEQSELECTSTMT+0) != 0) {
	$sql = 'SELECT relname FROM pg_class WHERE relname like ? ';
	$sql .= "AND relkind = 'S'";
	$DROPTABLES_SEQSELECTSTMT = prepareStmt($sql);
	return -1 unless ($DROPTABLES_SEQSELECTSTMT);
    }

    unless (defined($DROPTABLES_INDEXSELECTSTMT) &&
	    int($DROPTABLES_INDEXSELECTSTMT+0) != 0) {
	$sql = 'SELECT relname FROM pg_class WHERE relname like ? ';
	$sql .= "AND relkind = 'i'";
	$DROPTABLES_INDEXSELECTSTMT = prepareStmt($sql);
	return -1 unless ($DROPTABLES_INDEXSELECTSTMT);
    }

    my $row = "";

    my $results = getResultList($DROPTABLES_TABLESELECTSTMT, $match);
    if (defined($results) && ref($results) eq "ARRAY") {
	print join(', ',@{$results}) . "\n";
	foreach $row (@{$results}) {
	    dropTable($row);
	}
    } else {
	printDBDebug("No matching tables");
    }

    $results = getResultList($DROPTABLES_SEQSELECTSTMT, $seqMatch);
    if (defined($results) && ref($results) eq "ARRAY") {
	print join(', ',@{$results}) . "\n";
	foreach $row (@{$results}) {	    
	    runCommand("DROP SEQUENCE $row");
	}
    } else {
	printDBDebug("No matching sequences");
    }

    $results = getResultList($DROPTABLES_INDEXSELECTSTMT , $indexMatch);
    if (defined($results) && ref($results) eq "ARRAY") {
	print join(', ',@{$results}) . "\n";
	foreach $row (@{$results}) {	    
	    runCommand("DROP INDEX $row");
	}
    } else {
	printDBDebug("No matching index");
    }
    
    return 0;
}

#
# getDBVersion
#
# private function that returns the db version
#

sub getDBVersion
{
    my $vers = SCSDB::getResult("SELECT version()");
    my @parts = split(/[ \t]/, $vers);
    $vers = $parts[1];
    @parts = split(/\./, $vers);
    return ($vers, @parts);
}

1;
