# Copyright 2004 Sun Microsystems, Inc., All rights reserved.
# SUN PROPRIETARY/CONFIDENTIAL.  Use is subject to license terms.

package Notification;

use lib '/scs/lib/perl5';
use strict;
use SCSDB;

#
# Subroutines
#
# post_notification(%)
#  Post a notification and notify any registered listeners for the same type.
#  Returns a negative result code on DB failure, or if successful returns
#  the number of listeners notified in response to this notification.
#
# add_listener(%)
#  Register a listener for a given type.  Returns the listener_id assigned
#  to the listener on success, or a negative number if an error occurs.
#
# remove_listener(%)
#  Unregister a specific listener by id.  This may result in a
#  notification beind sent to one or more posters if those posters are
#  waiting on this listener to receive an AFTER_ALL notification.
#  Returns a negative number if the specified listener was not found,
#  or an error occurs.
#
# acknowledge_notification(%)
#  Acknowledge a specific notification as having been recieved by a
#  specific listener.  May cause the original poster to receive a
#  return notification as a result.  Returns an error if no such
#  notification was psoted to the specified listener, or if it has
#  already been acknowledged.
#

my $Debug_Notification = 0;

# valid callback_type arguments allowed for post_notification()
our %CALLBACK_TYPES = ( 'NONE'       => 'NONE',
			'AFTER_EACH' => 'AFTER_EACH',
			'AFTER_ALL'  => 'AFTER_ALL');

# storage for prepared statements

my $Post_Notification_Insert_Notification_Stmt;
my $Post_Notification_Select_Listener_Stmt;
my $Post_Notification_Insert_Notified_Listeners_Stmt;

my $Add_Listener_Insert_Stmt;

my $Remove_Listener_Select_Unacknowledged_Notifications_For_Update_Stmt;
my $Remove_Listener_Select_Listener_Stmt;
my $Remove_Listener_Select_Posts_Needing_Return_Notification_Stmt;
my $Remove_Listener_Delete_Listener_Stmt;

my $Send_Poster_Notification_Select_Notification_Stmt;

my $Acknowledge_Notification_Select_Post_Notifications_For_Update_Stmt;
my $Acknowledge_Notification_Select_Acknowledged_Stmt;
my $Acknowledge_Notification_Update_Acknowledged_Stmt;
my $Acknowledge_Notification_Select_Poster_Callback_Type_Stmt;
my $Acknowledge_Notification_Select_Unacknowledged_Listeners_Stmt;

#
# post_notification()
#
# Expects a hash with the following arguments:
#
#   notification_type => The notification will be sent to all listeners
#                        matching this notification_type.
#   description       => Human-readable description stored in the DB.
#   data              => String passed to matching listeners in the notification.
#   callback_type     => One of the entries in Notification::CALLBACK_TYPES.
#   callback_cmd      => The shell command to be executed when some or
#                        all listeners have acknowledged this notification,
#                        as determined by callback_type.
#   async             => Boolean true if listener callbacks (if any) should be
#                        executed in the background.  Otherwise post_notification()
#                        blocks until all listener callbacks complete execution.
#                        If this argument isn't provided, blocks by default.
#
# Returns a negative result code from Notification::ERROR_CODES if an
# error occurs.  Otherwise returns the number of listeners
# successfully notified (may be 0).
#
# Operates as an atomic transaction.  This prevents additional
# listeners from being added between the time we look for listeners
# and the time we post the notifications.
#
sub post_notification (%)
{
    my (%args) = @_;

    # add notification to posted_notifications table
    if (not defined $Post_Notification_Insert_Notification_Stmt) {
	my $sql = 'INSERT INTO mgmt_posted_notifications ';
	$sql .= '(notification_type, description, data, post_time, callback_cmd, callback_type) ';
	$sql .= "VALUES (?,?,?,now(),?,?)";

	$Post_Notification_Insert_Notification_Stmt = SCSDB::prepareStmt($sql);
	if ($Post_Notification_Insert_Notification_Stmt == 0) {
	    undef $Post_Notification_Insert_Notification_Stmt;
	    return SCSDB::errorCode('EDBERR', 'Could not prepare SQL statement to insert notification');
	}
    }

    &SCSDB::startTxn();
    my $rc = SCSDB::executeStmt($Post_Notification_Insert_Notification_Stmt,
			       $args{'notification_type'},
			       $args{'description'},
			       $args{'data'},
			       (not defined $args{'callback_cmd'} or $args{'callback_cmd'} eq "") ?
			       undef : $args{'callback_cmd'},
			       $args{'callback_type'});
    if ($rc < 0) {
	&SCSDB::rollBack();
	return SCSDB::errorCode('EDBERR', "Could not execute SQL statement to insert notification of type '" .
				$args{'notification_type'} . "'");
    }

    my $post_id = SCSDB::getInsertId('mgmt_posted_notifications', 'post_id');

    # look for matching listeners
    if (not defined $Post_Notification_Select_Listener_Stmt) {
	my $sql = 'SELECT listener_id, name, callback_cmd FROM mgmt_notification_listeners ';
	$sql .= 'WHERE notification_type = ?';

	$Post_Notification_Select_Listener_Stmt = SCSDB::prepareStmt($sql);
	if ($Post_Notification_Select_Listener_Stmt == 0) {
	    undef $Post_Notification_Select_Listener_Stmt;
	    &SCSDB::rollBack();
	    return SCSDB::errorCode('EDBERR', 'Could not prepare SQL statement to select matching listeners');
	}
    }
    
    my $matching_listener_records_list_ref = SCSDB::getResultHashes($Post_Notification_Select_Listener_Stmt,
								    $args{'notification_type'});
    # reference to list of references to hashes
    if (not $matching_listener_records_list_ref) {
	&SCSDB::rollback();
	return SCSDB::errorCode('EDBERR', "Could not execute SQL statement to select listeners of type '" .
				$args{'notification_type'} . "'");
    } elsif (scalar @$matching_listener_records_list_ref == 0) {
	# no results, hence no matching listeners to notify
	&SCSDB::commit();
	return 0;
    }
    my @matching_listener_record_refs = @$matching_listener_records_list_ref; 
    # list of references to hashes

    # prepare to add record of notifications
    if (not defined $Post_Notification_Insert_Notified_Listeners_Stmt) {
	my $sql = 'INSERT INTO mgmt_notified_listeners (post_id, listener_id, acknowledged, ack_time) ';
	$sql .= 'VALUES (?,?, FALSE, NULL)';

	$Post_Notification_Insert_Notified_Listeners_Stmt = SCSDB::prepareStmt($sql);
	if ($Post_Notification_Insert_Notified_Listeners_Stmt == 0) {
	    undef $Post_Notification_Insert_Notified_Listeners_Stmt;
	    &SCSDB::rollBack();
	    return SCSDB::errorCode('EDBERR', 'Could not prepare SQL statement to insert notification for listener');
	}
    }

    # iterate over listener record hash references
    foreach my $listener_record_ref (@matching_listener_record_refs) {
	my $listener_id = $listener_record_ref->{'listener_id'};
	
	# add individual notifications to notified_listeners table
	$rc = SCSDB::executeStmt($Post_Notification_Insert_Notified_Listeners_Stmt,
				 $post_id, $listener_id);
	if ($rc < 0) {
	    &SCSDB::rollBack();
	    return SCSDB::errorCode('EDBERR', "Could not execute SQL statement to insert notification for listener with id '$listener_id'");
	}
    }

    &SCSDB::commit();

    # send notifications via the listeners' callbacks
    foreach my $listener_record_ref (@matching_listener_record_refs) {

	_invoke ( $args{'async'},
		  $listener_record_ref->{'callback_cmd'},    # command
		  $args{'notification_type'},                # arguments
		  $listener_record_ref->{'name'},
		  $listener_record_ref->{'listener_id'},
		  $args{'data'},
		  $post_id );

    }

    # return number of listeners successfully notified
    return scalar(@matching_listener_record_refs);
}

#
# add_listener()
#
# Expects a hash with the following arguments:
#
#   notification_type => The listener will only be notified about notification
#                        matching this notification_type.
#   name              => Will be passed back to the listener along with notifications.
#   description       => Human-readable description stored in the DB.  Optional.
#   callback_cmd      => The shell command to be executed when a matching
#                        notification is posted.
#
# Returns a negative number from Notification::ERROR_CODES if an error occured, otherwise
# returns the listener_id assigned to this listener.
#
sub add_listener (%)
{
    my (%args) = @_;

    if (not defined $Add_Listener_Insert_Stmt) {
	my $sql = 'INSERT INTO mgmt_notification_listeners ';
	$sql .= '(name, description, notification_type, callback_cmd) ';
	$sql .= 'VALUES (?,?,?,?)';

	$Add_Listener_Insert_Stmt = SCSDB::prepareStmt($sql);
	if ($Add_Listener_Insert_Stmt == 0) {
	    undef $Add_Listener_Insert_Stmt;
	    return SCSDB::errorCode('EDBERR', 'Could not prepare SQL statement to insert listener');
	}
    }

    my $rc = SCSDB::executeStmt($Add_Listener_Insert_Stmt,
				$args{'name'},
				$args{'description'},
				$args{'notification_type'},
				$args{'callback_cmd'});
    return SCSDB::errorCode('EDBERR', "Could not execute SQL statement to insert listener for type '" .
			    $args{'notification_type'} . "'") if ($rc < 0);
    
    return SCSDB::getInsertId('mgmt_notification_listeners','listener_id');
}

#
# remove_listener()
#
# Expects a hash with the following arguments:
#
#   listener_id => The id of the listener to remove, as returned from add_listener().
#   async       => remove_listener() may result in the invocation of poster callbacks.
#                  If async evaluates to a true boolean value, these callbacks are
#                  invoked in the background.  If async is boolean false or unspecified,
#                  remove_listener() blocks until all poster callbacks complete.
#
# For the purposes of AFTER_ALL return notifications, removing a
# listener is basically a "silent acknowledgement" of all
# notifications sent to this listeners that have not yet been
# acknowledged.  Thus we require locking on all such notifications,
# similar to acknowledge_notification().  Otherwise if a post is
# waiting on two listener notifications (N1, N2) to receive
# acknowledgements, and the listener associated with N1 is removed,
# but between the SELECT that decides if N1 is the last unacknowledged
# notification and the following DELETE for the listener (and related
# listener notifications), N2 is acknowledged, no AFTER_ALL
# notification would be generated at all.  No notification would be
# sent when N1 is removed, because at the time of the SELECT N2 was
# still present and unacknowledged, and similarly no return
# notification would be sent when N2 is acknowledged, because N1 was
# not yet DELETEd.  The row-locking prevents this from occuring.
#
sub remove_listener (%)
{
    my (%args) = @_;

    # Make sure this listener_id exists.
    if (not defined $Remove_Listener_Select_Listener_Stmt) {
	my $sql = 'SELECT COUNT(*) FROM mgmt_notification_listeners ';
	$sql .= 'WHERE listener_id = ?';

	$Remove_Listener_Select_Listener_Stmt = SCSDB::prepareStmt($sql);
	if ($Remove_Listener_Select_Listener_Stmt == 0) {
	    undef $Remove_Listener_Select_Listener_Stmt;
	    return SCSDB::errorCode('EDBERR', 'Could not prepare SQL statement to check for listener');
	}
    }

    my $rc = &SCSDB::startTxn();
    return SCSDB::errorCode('EDBERR', "Could not initiate database transaction to delete listener '"
			    . $args{'listener_id'} . "'") if ( $rc < 0 );

    my $listeners_found_ref = SCSDB::getResultHash( $Remove_Listener_Select_Listener_Stmt,
						    $args{'listener_id'} );
    if ( ref($listeners_found_ref) ne 'HASH' ) {
	&SCSDB::rollBack();
	return SCSDB::errorCode('EDBERR', "Could not execute statement to check for listener with id '" .
				$args{'listener_id'} . "'");
    } elsif ( $listeners_found_ref->{'count'} != 1 ) {
	&SCSDB::rollBack();
	return SCSDB::errorCode('EINVAL', "No listener with id '" . $args{'listener_id'} . "' found");
    }

    # Setup row locking on unacknowledged notifications for this
    # listener.  This will cause any acknowledgements for posts
    # waiting on this listener to block while attempting to aquire row
    # locks on every acknowledgement.
    if (not defined $Remove_Listener_Select_Unacknowledged_Notifications_For_Update_Stmt) {
	my $sql = 'SELECT post_id FROM mgmt_notified_listeners ';
	$sql   .= 'WHERE listener_id = ? AND acknowledged = FALSE FOR UPDATE';
	
	$Remove_Listener_Select_Unacknowledged_Notifications_For_Update_Stmt = SCSDB::prepareStmt($sql);
	if ($Remove_Listener_Select_Unacknowledged_Notifications_For_Update_Stmt == 0) {
	    undef $Remove_Listener_Select_Unacknowledged_Notifications_For_Update_Stmt;
	    &SCSDB::rollBack();
	    return SCSDB::errorCode('EDBERR', 'Could not prepare SQL statement to select unacknowledged notifications for update');
	}
    }

    my $waiting_posts_ref = SCSDB::getResultList($Remove_Listener_Select_Unacknowledged_Notifications_For_Update_Stmt,
						 $args{'listener_id'});
    if ( ref($waiting_posts_ref) ne 'ARRAY' ) {
	&SCSDB::rollBack();
	return SCSDB::errorCode('EDBERR', 'Could not execute statement to select unacknowledged notifications for update');
    }

    # Find all posted notifications awaiting an acknowledgement from this listener
    # but are *not* awaiting acknowledgement from any other listeners
    # and originally requested notification after all listeners acknowledge.

    if (not defined $Remove_Listener_Select_Posts_Needing_Return_Notification_Stmt) {
	my $sql = 'SELECT post_id FROM ';
	$sql .=     'mgmt_posted_notifications AS p ';
	$sql .=   "WHERE callback_type = '" . $CALLBACK_TYPES{'AFTER_ALL'} . "' AND post_id IN ";
	$sql .=     '(SELECT post_id FROM ';
	$sql .=         'mgmt_posted_notifications JOIN mgmt_notified_listeners USING (post_id) ';
	$sql .=       'WHERE listener_id = ? ';
	$sql .=         'AND acknowledged = FALSE) ';
	$sql .=     'AND NOT EXISTS ';
	$sql .=     '(SELECT listener_id FROM mgmt_notified_listeners AS n ';
	$sql .=       'WHERE listener_id <> ? ';
	$sql .=          'AND acknowledged = FALSE ';
	$sql .=          'AND n.post_id = p.post_id)';

	$Remove_Listener_Select_Posts_Needing_Return_Notification_Stmt = SCSDB::prepareStmt($sql);
	if ($Remove_Listener_Select_Posts_Needing_Return_Notification_Stmt == 0) {
	    undef $Remove_Listener_Select_Posts_Needing_Return_Notification_Stmt;
	    &SCSDB::rollBack();
	    return SCSDB::errorCode('EDBERR', 'Could not prepare SQL statement to select posts needing return notification');
	}
    }

    # These posters need to be notifed that all (remaining) acknowledgements
    # have been received.
    my $affected_notifications_ref = SCSDB::getResultList($Remove_Listener_Select_Posts_Needing_Return_Notification_Stmt,
							  $args{'listener_id'},
							  $args{'listener_id'});

    # *** FIXME: getResult* functions are inconsistent with runCommand
    # for return values (return 0 instead of -1).
    if ( ref($affected_notifications_ref) ne 'ARRAY' ) {
	&SCSDB::rollBack();
	return SCSDB::errorCode('EDBERR', 'Could not execute SQL statement to select posts needing return notification');
    }
    
    # This SQL command should be executed in the same
    # transaction as the prior SELECT, or else another notification
    # that matches this listener's preferences may be posted, but
    # never acknowledged, nor will the poster be notified even after
    # this listener vanished.

    # actually delete the listener entry
    if (not defined $Remove_Listener_Delete_Listener_Stmt) {
	my $sql = 'DELETE FROM mgmt_notification_listeners ';
	$sql .= 'WHERE listener_id = ?';

	$Remove_Listener_Delete_Listener_Stmt = SCSDB::prepareStmt($sql);
	if ($Remove_Listener_Delete_Listener_Stmt == 0) {
	    undef $Remove_Listener_Delete_Listener_Stmt;
	    &SCSDB::rollBack();
	    return SCSDB::errorCode('EDBERR', 'Could not prepare SQL statement to delete listener');
	}
    }

    $rc = SCSDB::executeStmt($Remove_Listener_Delete_Listener_Stmt,
			     $args{'listener_id'});
    if ($rc < 0) {
	&SCSDB::rollBack();
	return SCSDB::errorCode('EDBERR', "Could not execute SQL statement to delete listener with id '" .
				$args{'listener_id'} . "'");
    }

    # finish the transaction so that the notified posters can do their business.
    
    &SCSDB::commit();

    if ($affected_notifications_ref) {
	# found some posters that need AFTER_ALL notifications now
	foreach my $post_id (@$affected_notifications_ref) {
	    _send_poster_notification($args{'async'},$post_id);
	}
    }

    return SCSDB::errorCode('OKAY');
}

#
# acknowledge_notification()
#
# Expects a hash with the following arguments:
#
#   listener_id => The listener acknowledging the post, as passed to the listener's
#                  callback.
#   post_id     => The post being acknowledged, as passed to the listener's callback.
#   async       => remove_listener() may result in the invocation of poster callbacks.
#                  If async evaluates to a true boolean value, these callbacks are
#                  invoked in the background.  If async is boolean false or unspecified,
#                  remove_listener() blocks until all poster callbacks complete.
#
# After marking the notification as having been acknowledged, checks
# if there are any remaining notifications for the same post that have
# yet to be acknwledged.  If none remain, we conclude that this
# notification was the last one remaining, and send the AFTER_ALL
# notification to the poster if originally requested.  Does row-level
# locking on the post_id via SELECT FOR UPDATE, which prevents other
# notifications from being acknowledged between us marking this
# notification as acknowledged and checking for other unacknowledged
# notifications.  If we didn't do this we could get multiple
# notifications thinking that they were the last notification, and
# thus send duplicate AFTER_ALL return notifications.
#
sub acknowledge_notification (%)
{
    my (%args) = @_;

    # setup atomicity
    if (not defined $Acknowledge_Notification_Select_Post_Notifications_For_Update_Stmt) {
	my $sql = 'SELECT listener_id FROM mgmt_notified_listeners ';
	$sql   .= 'WHERE post_id = ? FOR UPDATE';

	$Acknowledge_Notification_Select_Post_Notifications_For_Update_Stmt = SCSDB::prepareStmt($sql);
	if ($Acknowledge_Notification_Select_Post_Notifications_For_Update_Stmt == 0) {
	    undef $Acknowledge_Notification_Select_Post_Notifications_For_Update_Stmt;
	    return SCSDB::errorCode('EDBERR', 'Could not prepare SQL statement to select notifications for update');
	}
    }

    my $rc = SCSDB::startTxn();
    if ($rc < 0) {
	&SCSDB::rollBack();
	return SCSDB::errorCode('EDBERR', 'Could not start database transaction');
    }

    my $listener_ids_ref = SCSDB::getResultList($Acknowledge_Notification_Select_Post_Notifications_For_Update_Stmt,
						$args{'post_id'});
    if ( ref($listener_ids_ref) ne 'ARRAY' ) {
	&SCSDB::rollBack();
	return SCSDB::errorCode('EDBERR', "Could not select notifications associated with post '" .
				$args{'post_id'} . "' for update");
    }

    # check current status of this listener notification
    if (not defined $Acknowledge_Notification_Select_Acknowledged_Stmt) {
	my $sql = 'SELECT acknowledged FROM mgmt_notified_listeners ';
	$sql   .= 'WHERE listener_id = ?';
	$sql   .= ' AND post_id = ?';

	$Acknowledge_Notification_Select_Acknowledged_Stmt = SCSDB::prepareStmt($sql);
	if ($Acknowledge_Notification_Select_Acknowledged_Stmt == 0) {
	    undef $Acknowledge_Notification_Select_Acknowledged_Stmt;
	    &SCSDB::rollBack();
	    return SCSDB::errorCode('EDBERR', 'Could not prepare SQL statement to check whether notification is already acknowledged');
	}
    }

    # This is messy, but since getResult returns 0 on a DB failure, we need to do this
    # to distinguish an error from a result of FALSE.
    my $notified_listener_ref = SCSDB::getResultHash($Acknowledge_Notification_Select_Acknowledged_Stmt,
						     $args{'listener_id'},
						     $args{'post_id'});
    if ( not defined $notified_listener_ref ) {
	&SCSDB::rollBack();
	return SCSDB::errorCode('EINVAL', "No notification for listener id '" . $args{'listener_id'} .
				"' and post id '" . $args{'post_id'} . "' found");
    }
    if ( ref($notified_listener_ref) ne 'HASH' ) {
	&SCSDB::rollBack();
	return SCSDB::errorCode('EDBERR', "Could not execute SQL statement to find notification " .
				"for listener id '" . $args{'listener_id'} .
				"' and post id '" . $args{'post_id'} . "'");
    }
    my $acknowledged = $notified_listener_ref->{'acknowledged'};
    _debug("acknowledged = $acknowledged");
    if ($acknowledged) {
	&SCSDB::rollBack();
	return SCSDB::errorCode('EINVAL', "Listener with id '" . $args{'listener_id'} .
				"' has already acknowledged post id '" . $args{'post_id'} . "'");
    }

    # set to acknowledged
    if (not defined $Acknowledge_Notification_Update_Acknowledged_Stmt) {
	my $sql = "UPDATE mgmt_notified_listeners SET acknowledged = TRUE, ack_time = now() ";
	$sql .= 'WHERE listener_id = ?';
	$sql .= ' AND post_id = ?';

	$Acknowledge_Notification_Update_Acknowledged_Stmt = SCSDB::prepareStmt($sql);
	if ($Acknowledge_Notification_Update_Acknowledged_Stmt == 0) {
	    undef $Acknowledge_Notification_Update_Acknowledged_Stmt;
	    &SCSDB::rollBack();
	    return SCSDB::errorCode('EDBERR', 'Could not prepare SQL statement to mark notification as acknowledged');
	}
    }

    $rc = SCSDB::executeStmt($Acknowledge_Notification_Update_Acknowledged_Stmt,
			     $args{'listener_id'},
			     $args{'post_id'});
    if ($rc < 0) {
	&SCSDB::rollBack();
	return SCSDB::errorCode('EDBERR', "Could not execute SQL statement to mark notification for post id '" . 
				$args{'post_id'} . "' as acknowledged by listener '" . $args{'listener_id'} . "'");
    }

    # check if notification requires a return notification
    if (not defined $Acknowledge_Notification_Select_Poster_Callback_Type_Stmt) {
	my $sql = 'SELECT callback_type FROM mgmt_posted_notifications ';
	$sql .= 'WHERE post_id = ?';

	$Acknowledge_Notification_Select_Poster_Callback_Type_Stmt = SCSDB::prepareStmt($sql);
	if ($Acknowledge_Notification_Select_Poster_Callback_Type_Stmt == 0) {
	    undef $Acknowledge_Notification_Select_Poster_Callback_Type_Stmt;
	    &SCSDB::rollBack();
	    return SCSDB::errorCode('EDBERR', 'Could not prepare SQL statement to select poster callback type');
	}
    }

    my $callback_type_ref = SCSDB::getResultHash($Acknowledge_Notification_Select_Poster_Callback_Type_Stmt,
						 $args{'post_id'});
    if ( ref($callback_type_ref) ne 'HASH' ) {
	&SCSDB::rollBack();
	return SCSDB::errorCode('EDBERR', "Could not execute SQL statement to select callback type for poster '" .
				$args{'post_id'} . "'");
    }
    my $callback_type = $callback_type_ref->{'callback_type'};

    # end transaction before sending acknowledgements
    &SCSDB::commit();

    if ($callback_type eq $CALLBACK_TYPES{'AFTER_EACH'}) {

	# send per-acknowledgement notification to poster if requested
	_send_poster_notification($args{'async'},$args{'post_id'});

    } elsif ($callback_type eq $CALLBACK_TYPES{'AFTER_ALL'}) {

	# send an AFTER_ALL notification if there are no pending acknowledgements
	if (not defined $Acknowledge_Notification_Select_Unacknowledged_Listeners_Stmt) {
	    my $sql = 'SELECT listener_id FROM mgmt_notified_listeners ';
	    $sql .= 'WHERE post_id = ?';
	    $sql .= ' AND acknowledged = FALSE';

	    $Acknowledge_Notification_Select_Unacknowledged_Listeners_Stmt = SCSDB::prepareStmt($sql);
	    if ($Acknowledge_Notification_Select_Unacknowledged_Listeners_Stmt == 0) {
		undef $Acknowledge_Notification_Select_Unacknowledged_Listeners_Stmt;
		&SCSDB::rollBack();
		return SCSDB::errorCode('EDBERR', 'Could not prepare SQL statement to select unacknowledged notifications');
	    }
	}
	
	my $listeners_with_pending_acks_ref = SCSDB::getResultList($Acknowledge_Notification_Select_Unacknowledged_Listeners_Stmt,
								   $args{'post_id'});
	if ( ref($listeners_with_pending_acks_ref) ne 'ARRAY' ) {
	    &SCSDB::rollBack();
	    return SCSDB::errorCode('EDBERR', "Could not execute SQL statement to select unacknowledged notifications for post with ID '"
				    . $args{'post_id'} . "'");
	}

	if (scalar(@$listeners_with_pending_acks_ref) == 0) {
	    _send_poster_notification($args{'async'},$args{'post_id'});
	}

    }

    return SCSDB::errorCode('OKAY');
}

#
# _send_poster_notification()
#
# Internal routine to unify sending a return notification to a poster.
#
sub _send_poster_notification ($$)
{
    my ($async,$post_id) = @_;

    if (not defined $Send_Poster_Notification_Select_Notification_Stmt) {
	my $sql = 'SELECT callback_cmd, notification_type, callback_type, data ';
	$sql .= 'FROM mgmt_posted_notifications WHERE post_id = ?';

	$Send_Poster_Notification_Select_Notification_Stmt = SCSDB::prepareStmt($sql);
	if ($Send_Poster_Notification_Select_Notification_Stmt == 0) {
	    undef $Send_Poster_Notification_Select_Notification_Stmt;
	    return SCSDB::errorCode('EDBERR', 'Could not prepare SQL statement to select post');
	}
    }

    my $query_result_ref = SCSDB::getResultHash( $Send_Poster_Notification_Select_Notification_Stmt, $post_id );
    return SCSDB::errorCode('EDBERR', "Could not execute SQL statement to select post with id '$post_id'") if ( ref($query_result_ref) ne 'HASH' );

    _invoke ( $async,
	      $query_result_ref->{'callback_cmd'},
	      $query_result_ref->{'notification_type'},
	      $query_result_ref->{'callback_type'},
	      $query_result_ref->{'data'} );

    return SCSDB::errorCode('OKAY');
}

#
# _invoke()
#
# Takes two or more arguments:
#
#   async : If true, the command is forked off and executed in the background.
#           otherwise invoke waits for the process to complete before returning.
#   cmd   : The command to invoke.
#
# All remaining arguments are passed to the command being invoked.
#
sub _invoke ($$@)
{
    my ($async, $cmd, @cmdargs) = @_;

    if (defined $async && $async) {
	my $child_pid = fork();
	if (defined $child_pid and $child_pid == 0) {
	    exec($cmd, @cmdargs);
	    exit();
	}
    } else {
	system($cmd, @cmdargs);
    }
}

sub _debug (@)
{
    if ($Debug_Notification) { warn "NOTIFY.DEBUG: @_"; }
}

1;
