#!/usr/bin/perl -w
# $Id: MAPP.pm,v 1.21.2.1 2005/01/22 21:53:22 ms152511 Exp $
# MAPP.pm - Module loading functions
# Copyright 2003, 2004 Sun Microsystems, Inc., All rights reserved.
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms

package MAPP;

use strict;
use lib '/scs/lib/perl5';
use SCSDB;
use Builds;
use BDUtil;

#
# Subroutines
#
# getMAPP
# getManagedAppliancesByBuild
# getApplianceBuilds
# getAllMAPPIds
# getAllMAPPs
# getMAPPbyName
# getMAPPSchedules
# getBuildPackages
# getPackageID
# getServerPackages
# getInstalledPackages
# getInstalledAppliances
# getNumMAPPs
# getApplianceSchedules
# getEligibleAppliances
# getEligibleApplianceList
# getDeniedVersion
#
# addMAPP
# addClientPackage
# addInstalledApplication
# setPkgInstallationStatus
# addServerPackage
# addApplianceToSchedules
# registerTask
# registerApplianceIcon
# setMAPPStatus
#
# removeMAPP
# removeInstalledPackages
# removeApplianceFromSchedules
#

my $STMT_getMAPP = 0;
my $STMT_setPkgInstallationStatus;

#
# getMAPP
#
# returns the MAPP entry for the MAPP with the specified id
#

sub getMAPP
{
    my $mappId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid module id')
	unless (defined($mappId) && $mappId+0 > 0);

    unless ($STMT_getMAPP) {
	my $sql = 'SELECT * FROM mgmt_mapp WHERE mapp_id = ?';
	$STMT_getMAPP = SCSDB::prepareStmt($sql);
	return SCSDB::errorCode('EDBERR', "Cannot prepare '$sql'")
	    unless ($STMT_getMAPP);
    }

    return (SCSDB::errorCode('OKAY'), 
	    SCSDB::getResultHash($STMT_getMAPP,int($mappId+0)));
}

#
# getMAPPIdByName
#
#

sub getMAPPIdByName
{
    my $name = shift @_;
    return -1 unless (defined($name) && $name ne "");
    
    my $sql = "";

    if ($name+0 > 0 && $name =~ /^[0-9]+$/) {
	$sql = 'SELECT mapp_name FROM mgmt_mapp WHERE mapp_id = ';
	$sql .= int($name+0);
	
	my $mappName = SCSDB::getResult($sql);
	return -1 unless (defined($mappName) && $mappName ne "");
	return $name+0;
    } 

    $sql = "SELECT mapp_id FROM mgmt_mapp WHERE mapp_name = '$name'";

    my $mappId = SCSDB::getResult($sql);
    return -1 unless (defined($mappId) && $mappId+0 > 0);
    
    return int($mappId+0);
}

#
# getMAPPSchedules
#
# returns the schedules for a specified mapp name
#

sub getMAPPSchedules
{
    my $mappName = shift @_;
    return SCSDB::errorCode('EPARM', 'Mapp name is null')
	unless (defined($mappName) && $mappName ne "");

    my $sql = "SELECT schedule_id, name, type, verbose_settings ";
    $sql .= "FROM mgmt_schedule WHERE mapp_name = '$mappName'";

    return SCSDB::getResultHashes($sql);
}

#
# removeMAPP
#
# deletes the MAPP with the specified MAPP id
#
# WARNING: This needs a TXN!
#

sub removeMAPP
{
    my $mappId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid module id')
	unless (defined($mappId) && $mappId+0 > 0);

    my @tables = ('mgmt_schedulable_task',
		  'mgmt_appliance_icon');
    my $table = "";
    my $sql = "";
    my $err = "";

    foreach $table (@tables) {
	$sql = "DELETE FROM $table WHERE mapp_id = ";
	$sql .= int($mappId+0);
	if (SCSDB::runCommand($sql)) {
	    $err = "Cannot delete '$mappId' from $table";
	    last;
	}
    }
    
    return SCSDB::errorCode('EDBERR', $err) if ($err ne "");

    $sql = 'SELECT mapp_client_rpm_id FROM mgmt_mapp_client_rpm ';
    $sql .= 'WHERE mapp_id = ' . int($mappId+0);

    my $clientRPMs = SCSDB::getResultList($sql);

    if (defined($clientRPMs) && ref($clientRPMs) eq "ARRAY") {
	my @rpmIds = SCSDB::idList(@{$clientRPMs});
	if (scalar(@rpmIds) > 0) {
	    $sql = 'DELETE FROM mgmt_installed_rpm WHERE ';
	    $sql .= 'mapp_client_rpm_id IN (';
	    $sql .= join(', ', @rpmIds);
	    $sql .= ')';
	
	    if (SCSDB::runCommand($sql)) {
		$err = "Cannot delete '$mappId' from mgmt_installed_rpm";
		return SCSDB::errorCode('EDBERR',$err);
	    }
	}

	$sql = 'DELETE FROM mgmt_mapp_client_rpm WHERE mapp_id = ';
	$sql .= int($mappId+0);

	if (SCSDB::runCommand($sql)) {
	    $err = 
		"Cannot delete '$mappId' from mgmt_mapp_client_rpm";
	    return SCSDB::errorCode('EDBERR',$err);
	}
    }
    
    @tables = ('mgmt_mapp_server_rpm',
	       'mgmt_backup_path',
	       'mgmt_mapp');
    $err = "";

    foreach $table (@tables) {
	$sql = "DELETE FROM $table WHERE mapp_id = ";
	$sql .= int($mappId+0);
	if (SCSDB::runCommand($sql)) {
	    $err = "Cannot delete '$mappId' from $table";
	    last;
	}
    }

    return SCSDB::errorCode('EDBERR', $err) if ($err ne "");

    return SCSDB::errorCode('OKAY');
}

#
# getManagedAppliancesByBuild
#
# returns the host ids that match a given list of builds
#

sub getManagedAppliancesByBuild
{
    my $buildList = shift @_;
    return (SCSDB::errorCode('OKAY'), 0)
	unless (defined($buildList) && ref($buildList) eq "ARRAY");
    
    my @buildIds = SCSDB::idList(@{$buildList});

    my $sql;
    $sql  = 'SELECT appliance_id, '
          . 'COALESCE(host_name,ip_address) AS ip_address, '
          . 'build_id ';
    $sql .= 'FROM mgmt_appliance ';
    $sql .= "WHERE status IN ('M') "
          . 'AND build_id IN ('
          . join(', ', @buildIds) . ')';
    
    return (SCSDB::errorCode('OKAY'), SCSDB::getResultHashes($sql));
}

#
# getApplianceBuilds
#
# returns the build ids for a specified list of host ids
#

sub getApplianceBuilds
{
    my $idList = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid id list')
	unless (defined($idList) && ref($idList) eq "ARRAY");

    my @ids = SCSDB::idList(@{$idList});

    my $sql = 'SELECT appliance_id, host_name AS ip_address, ';
    $sql .= 'build_id FROM mgmt_appliance WHERE appliance_id IN (';
    $sql .= join(', ', @ids);
    $sql .= ')';

    return SCSDB::getResultHashes($sql);
}

#
# getMAPPbyName
#
# returns mapp_id, status and version for the mapp with the specified
# name
#

sub getMAPPbyName
{
    my $mappName = shift @_;
    return (SCSDB::errorCode('EPARM'), undef)
        unless (defined($mappName) && $mappName ne '');

    my $sql = 'SELECT mapp_id, status, version FROM mgmt_mapp ';
    $sql .= "WHERE mapp_name = '$mappName'";

    return SCSDB::getResultHash($sql);
}

#
# getAllMAPPIds
#
# returns ids of all available MAPPS
#

sub getAllMAPPIds
{
    my $sql = 'SELECT mapp_id FROM mgmt_mapp ORDER BY mapp_id';
    return SCSDB::getResultList($sql);
}

#
# getAllMAPPs
#
# returns information about all available MAPPS
#

sub getAllMAPPs
{
    my $sql = 'SELECT * FROM mgmt_mapp ORDER BY mapp_id';
    return SCSDB::getResultHashes($sql);
}

#
# addMAPP
#
# adds a MAPP 
#

sub addMAPP
{
    my $name = shift @_;
    return SCSDB::errorCode('EPARM', 'MAPP name is null')
	unless (defined($name) && $name ne "");

    my $vendor = shift @_;
    return SCSDB::errorCode('EPARM', 'Vendor is null')
	unless (defined($vendor) && $vendor ne "");

    my $version = shift @_;
    return SCSDB::errorCode('EPARM', 'Version is null')
	unless (defined($version) && $version ne "");

    my $updateBool = shift @_;    
    $updateBool = 'Y' unless (defined($updateBool));

    return SCSDB::errorCode('EPARM', 
			   "Invalid update flag '$updateBool'")
	unless ($updateBool =~ /^[YN]$/);

    my $initScript = shift @_;
    $initScript = "" unless (defined($initScript));

    my $cleanupScript = shift @_;
    $cleanupScript = "" unless (defined($cleanupScript));

    my $nameTag = shift @_;
    $nameTag = $name unless (defined($nameTag));

    my $vendorTag = shift @_;
    $vendorTag = $vendor unless (defined($vendorTag));

    my $versionTag = shift @_;
    $versionTag = $version unless (defined($versionTag));

    my %mappHash = ( 'mapp_name'      => "'$name'",
		     'version'        => "'$version'",
		     'vendor'         => "'$vendor'",
		     'name_tag'       => "'$nameTag'",
		     'version_tag'    => "'$versionTag'",
		     'vendor_tag'     => "'$vendorTag'",
		     'update_bool'    => "'$updateBool'",
		     'init_script'    => "'$initScript'",
		     'cleanup_script' => "'$cleanupScript'",
		     'status'         => "'N'" );

    my $mappId = 
	SCSDB::insertHash('mgmt_mapp', 'mapp_id', \%mappHash);
    return SCSDB::errorCode('EDBERR', 
			   "Cannot add MAPP '$name-$version'")
	unless (defined($mappId) && $mappId+0 > 0);
    
    return $mappId;
}

#
# setMAPPStatus
#
# sets the status of the specified mapp to the specified status
#

sub setMAPPStatus
{
    my $mappId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid mapp id')
	unless (defined($mappId) && $mappId+0 > 0);

    my $status = shift @_;
    return SCSDB::errorCode('EPARM', 'Status is null')
	unless (defined($status) && $status ne "");

    my %validMAPPStatus = ( 'N'  => 1, 
			    'U'  => 1, 
			    'D'  => 1, 
			    'AS' => 1, 
			    'AF' => 1, 
			    'DF' => 1 );

    my $mappStatus = $validMAPPStatus{$status};    

    return SCSDB::errorCode('EPARM', "Invalid status '$status'")
	unless (defined($mappStatus) && $mappStatus+0 > 0);

    my $sql = 'SELECT mapp_name FROM mgmt_mapp WHERE mapp_id = ';
    $sql .= int($mappId+0);

    return SCSDB::errorCode('EINVAL', "Invalid mapp id '$mappId'")
	unless (SCSDB::hasResults($sql));

    $sql = "UPDATE mgmt_mapp SET status = '$status' WHERE ";
    $sql .= 'mapp_id = ' . int($mappId+0);

    return SCSDB::errorCode('EDBERR', 
			   "Cannot set status of mapp '$mappId'")
	if (SCSDB::runCommand($sql));

    return SCSDB::errorCode('OKAY');
}

#
# addClientPackage
#
# ???
#

sub addClientPackage
{
    my $buildId = 0;
    my @buildIds = ();
    my %buildToPkgHash = ();
    my $sql = "";
    my $err = "";
    my %pkgRecord = ();
    my $rpmId = 0;
    my %clientRpmHash = ();

    my $buildIdsList = shift @_;
    unless (defined($buildIdsList) && 
	    ref($buildIdsList) eq "ARRAY") {
	return (SCSDB::errorCode('EPARM', 'Invalid build id list'), 
		\%buildToPkgHash);
    }

    @buildIds = SCSDB::idList(@{$buildIdsList});
    unless (scalar(@buildIds) > 0) {
	return (SCSDB::errorCode('EPARM', 'Invalid build id list'),
		\%buildToPkgHash);
    }
	
    my $mappId = shift @_;
    unless (defined($mappId) && $mappId+0 > 0) {
	return (SCSDB::errorCode('EPARM', 'Invalid mapp id'),
		\%buildToPkgHash);
    }

    my $rpmName = shift @_;
    unless (defined($rpmName) && $rpmName ne "") {
	return (SCSDB::errorCode('EPARM', 'RPM name is null'),
                \%buildToPkgHash);
    }

    my $path = shift @_;
    unless (defined($path) && $path =~ /^\//) {
	return (SCSDB::errorCode('EPARM', 'Invalid path'),
                \%buildToPkgHash);
    }

    foreach $buildId (@buildIds) {

	$sql = 'SELECT mapp_client_rpm_id FROM mgmt_mapp_client_rpm';
	$sql .= ' WHERE mapp_id = ' . int($mappId+0);
	$sql .= " AND rpm_name = '$rpmName' AND path_info = '$path'";
	$sql .= ' AND build_id = ' . int($buildId+0);

	next if (SCSDB::hasResults($sql));

	%clientRpmHash = ( 'mapp_id'   => int($mappId+0),
			   'build_id'  => int($buildId+0),
			   'rpm_name'  => "'$rpmName'",
			   'path_info' => "'$path'",
			   'status'    => "'N'" );

	$rpmId = 
	    SCSDB::insertHash('mgmt_mapp_client_rpm',
			     'mapp_client_rpm_id',
			     \%clientRpmHash,
			     'mgmt_mapp_cli_mapp_client_r_seq');
	
	unless ($rpmId+0 > 0) {
	    $err = "Cannot add rpm to mapp_id '$mappId'";
	    last;
	}
	
	$buildToPkgHash{int($buildId+0)} = 
	{ 'mapp_client_rpm_id' => int($rpmId+0),
	  'rpm_name'  => $rpmName,
	  'path_info' => $path,
	  'mapp_id'   => int($mappId+0) };
    }

    return (SCSDB::errorCode('EDBERR', $err), \%buildToPkgHash)
	if ($err ne "");

    return (SCSDB::errorCode('OKAY'), \%buildToPkgHash);
}

#
# getBuildPackages
#
# gets the client pkgs for a given mapp and build
#

sub getBuildPackages
{
    my $mappId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid mapp id')
	unless (defined($mappId) && $mappId+0 > 0);

    my $buildId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid build id')
	unless (defined($buildId) && $buildId+0 > 0);

    my $sql = 'SELECT * FROM mgmt_mapp_client_rpm WHERE mapp_id = ';
    $sql .= int($mappId+0) . ' AND build_id = ' . int($buildId+0);
    $sql .= ' ORDER BY mapp_client_rpm_id';

    return (SCSDB::errorCode('OKAY'), SCSDB::getResultHashes($sql));
}

#
# getPackageID
#
# returns the package id for ???
#

sub getPackageID
{
    my $mappId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid mapp id')
	unless (defined($mappId) && $mappId+0 > 0);
    
    my $buildType = shift @_;
    return SCSDB::errorCode('EPARM', 'Build type is null')
	unless (defined($buildType) && $buildType ne "");
    
    my $buildClass = shift @_;
    $buildClass = "" unless (defined($buildClass));

    my $rpmName = shift @_;
    return SCSDB::errorCode('EPARM', 'RPM Name is null')
	unless (defined($rpmName) && $rpmName ne "");

    my $rpmPath = shift @_;
    return SCSDB::errorCode('EPARM', 'RPM Path is null')
	unless (defined($rpmPath) && $rpmPath ne "");

    my $sql = 'SELECT a.mapp_client_rpm_id, a.status FROM ';
    $sql .= 'mgmt_mapp_client_rpm AS a, mgmt_build AS b WHERE ';
    $sql .= 'a.build_id = b.build_id AND a.mapp_id = ';
    $sql .= int($mappId+0);
    $sql .= " AND b.build_type = '$buildType' AND a.rpm_name = ";
    $sql .= "'$rpmName' AND a.path_info = '$rpmPath'";

    my $res = SCSDB::getResultHash($sql);
    return SCSDB::errorCode('EINVAL', 'No such package')
	unless (defined($res) && ref($res) eq "HASH");

    return (SCSDB::errorCode('OKAY'),
	    $res->{'mapp_client_rpm_id'},
	    $res->{'status'});
}

#
# addServerPackage
#
# adds the specified pkg to the specified mapp id
#

sub addServerPackage
{
    my $mappId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid mapp id')
	unless (defined($mappId) && $mappId+0 > 0);

    my $rpmName = shift @_;
    return SCSDB::errorCode('EPARM', 'Pkg name is null')
	unless (defined($rpmName) && $rpmName ne "");

    # Get rid of the <arch>.rpm suffix
    my $pkgName = $rpmName;
    my $pkgType = 'svr4';

    if ($rpmName =~ /\.rpm$/) {
	($pkgName) = ($rpmName =~ /^(.*)\.[^\.]+\.rpm$/);
	$pkgType = 'rpm';
    }

    my %serverRpmHash = ( 'mapp_id'  => int($mappId+0),
			  'rpm_name' => "'$pkgName'",
			  'pkg_type' => "'$pkgType'" );

    my $mgmtPkgId = 
	SCSDB::insertHash('mgmt_mapp_server_rpm',
			  'mapp_server_rpm_id',
			  \%serverRpmHash,
			  'mgmt_mapp_ser_mapp_server_r_seq');

    return SCSDB::errorCode('EDBERR', 'Cannot add server pkg')
	unless (defined($mgmtPkgId) && $mgmtPkgId+0 > 0);

    return $mgmtPkgId;
}

#
# addBackupPath
#
# adds a backup path for the specified mapp
#

sub addBackupPath
{
    my $mappId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid mapp id')
	unless (defined($mappId) && $mappId+0 > 0);
    
    my $backupPath = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid backup path')
	unless (defined($backupPath) && $backupPath =~ /^\//);

    my %backupHash = ( 'mapp_id'     => int($mappId+0),
		       'backup_path' => "'$backupPath'" );

    return SCSDB::insertHash('mgmt_backup_path', 
			     'backup_path_id', 
			     \%backupHash,
			     'mgmt_backup_p_backup_path_i_seq');
}

#
# getServerPackages
#
# returns the server pkgs for the specified mapp
#

sub getServerPackages
{
    my $mappId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid mapp id')
	unless (defined($mappId) && $mappId+0 > 0);

    my $sql = 'SELECT * FROM mgmt_mapp_server_rpm WHERE mapp_id = ';
    $sql .= int($mappId+0) .  ' ORDER BY mapp_server_rpm_id';

    return (SCSDB::errorCode('OKAY'), SCSDB::getResultHashes($sql));
}

#
# getPackagesByBuild
#
# returns the pkg list by build ???
#

sub getPackagesByBuild
{
    my $buildId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid build id')
	unless (defined($buildId) && $buildId+0 > 0);

    my $mappList = shift @_ || 0;

    my $sql = 'SELECT * FROM mgmt_mapp_client_rpm WHERE build_id = ';
    $sql .= int($buildId+0) . " ";

    if (defined($mappList) && ref($mappList) eq "ARRAY") {
	my @mappIds = SCSDB::idList(@{$mappList});
	if (scalar(@mappIds) > 0) {
	    $sql .= 'AND mapp_id in (';
	    $sql .= join(', ', @mappIds);
	    $sql .= ') ';
	}
    }

    $sql .= 'ORDER BY mapp_client_rpm_id';
    
    return (SCSDB::errorCode('OKAY'), SCSDB::getResultHashes($sql));
}

##
# getInstalledPackages
##
sub getInstalledPackages
{
    my $applianceList = shift @_ || 0;
    my $mappId = shift @_ || 0;
    my $moduleList = shift @_ || 0; 
   
    my $sql = "SELECT p.rpm_name, p.path_info, p.mapp_id, ";
    $sql .= "p.mapp_client_rpm_id, a.appliance_id, a.host_name ";
    $sql .= "AS ip_address FROM mgmt_mapp_client_rpm p, ";
    $sql .= "mgmt_installed_rpm i, mgmt_appliance a WHERE ";
    $sql .= "i.mapp_client_rpm_id = p.mapp_client_rpm_id AND ";
    $sql .= "i.appliance_id = a.appliance_id AND i.status = 'P' ";
    
    if (defined($applianceList) && ref($applianceList) eq "ARRAY") {
	my @hostIds = SCSDB::idList(@{$applianceList});	
	if (scalar(@hostIds) > 0) {
	    $sql .= 'AND i.appliance_id IN (';
	    $sql .= join(', ', @hostIds);
	    $sql .= ') ';
	}
    }

    $sql .= ' AND p.mapp_id = ' . int($mappId+0) . ' '
	if ($mappId);

    if (defined($moduleList) && ref($moduleList) eq "ARRAY") {
	my @mappIds = SCSDB::idList(@{$moduleList});
	if (scalar(@mappIds) > 0) {
	    $sql .= 'AND p.mapp_id IN (';
	    $sql .= join(', ', @mappIds);
	    $sql .= ') ';
	}
    }

    $sql .= 'ORDER BY a.appliance_id';
    
    return (SCSDB::errorCode('OKAY'), SCSDB::getResultHashes($sql));
}

#
# addInstalledApplication
#
# ???
#

sub addInstalledApplication
{
    my $pkgId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid pkg id')
	unless (defined($pkgId) && $pkgId+0 > 0);

    my $hostId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid host id')
	unless (defined($hostId) && $hostId+0 > 0);

    my $status = shift @_;
    return SCSDB::errorCode('EPARM', 'Status is null')
	unless (defined($status) && $status ne "");

    my $reason = shift @_;
    $reason = '' unless (defined($reason));

    my %installedRpmHash = ( 'mapp_client_rpm_id' => $pkgId,
			     'appliance_id'       => $hostId,
			     'status'             => "'$status'",
			     'reason'             => "'$reason'" );

    my $installedAppId = 
	SCSDB::insertHash('mgmt_installed_rpm',
			  'installed_rpm_id',
			  \%installedRpmHash,
			  'mgmt_installe_installed_rpm_seq');

    return SCSDB::errorCode('EDBERR', 'Cannot add installed_rpm')
	unless (defined($installedAppId) && $installedAppId+0 > 0);
    
    return $installedAppId;
}

#
# setPkgInstallationStatus
#
# Updates the mgmt_installed_rpm table to reflect (un)installed
# status or failures.
#

sub setPkgInstallationStatus ($$$$)
{
    my $pkgId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid pkg id')
       unless (defined($pkgId) && $pkgId+0 > 0);

    my $hostId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid host id')
       unless (defined($hostId) && $hostId+0 > 0);

    my $status = shift @_;
    return SCSDB::errorCode('EPARM', 'Status is null')
       unless (defined($status) && $status ne "");

    my $reason = shift @_;
    $reason = '' unless (defined($reason));

    unless ($STMT_setPkgInstallationStatus) {
        my $sql = 'UPDATE mgmt_installed_rpm SET status = ?, reason = ? ';
        $sql .= 'WHERE mapp_client_rpm_id = ? AND appliance_id = ?';
        $STMT_setPkgInstallationStatus = SCSDB::prepareStmt($sql);
        return SCSDB::errorCode('EDBERR', "Cannot prepare '$sql'")
            unless ($STMT_setPkgInstallationStatus);
    }

    my $rc = SCSDB::executeStmt($STMT_setPkgInstallationStatus,
                                $status, $reason,
                                $pkgId, $hostId);

    return SCSDB::errorCode('EDBERR', 'Cannot change status in mgmt_installed_rpm table')
       if ( $rc < 0 );

    my $rowsAffected = SCSDB::getRowsAffected($STMT_setPkgInstallationStatus);
    return SCSDB::errorCode('EDBERR', "No entry found in mgmt_installed_rpm for package ID=$pkgId on appliance=$hostId")
        if ( $rowsAffected == 0 );
    return SCSDB::errorCode('EDBERR', "Unexpected entries found in mgmt_installed_rpm for package ID=$pkgId on appliance=$hostId")
        if ( $rowsAffected != 1);

    return 0;
}
 
#
# getInstalledAppliances
#
# ??? 
#

sub getInstalledAppliances
{
    my $mappId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid module id')
	unless (defined($mappId) && $mappId+0 > 0);

    my $sql = 'SELECT DISTINCT i.appliance_id FROM ';
    $sql .= 'mgmt_installed_rpm AS i, mgmt_mapp_client_rpm AS r ';
    $sql .= 'WHERE r.mapp_id = ' . int($mappId+0);
    $sql .= " AND i.status = 'P' AND ";
    $sql .= 'i.mapp_client_rpm_id = r.mapp_client_rpm_id';

    return (SCSDB::errorCode('OKAY'), SCSDB::getResultList($sql));
}

#
# getNumMAPPs
#
# ???
#

sub getNumMAPPs
{
    my $mappCnt = 0;
    my $initScriptCnt = 0;
    my @mappList = ();
    my $mappId = 0;

    my $mappListRef = shift @_;
    @mappList = SCSDB::idList(@{$mappListRef})
	if (defined($mappListRef) && ref($mappListRef) eq "ARRAY");

    my $sql = 'SELECT mapp_id, init_script FROM mgmt_mapp ';

    if (scalar(@mappList) > 0) {
	$sql .= 'WHERE mapp_id in (';
	$sql .= join(', ', @mappList);
	$sql .= ')';
    }
    
    my $res = SCSDB::getResultHashes($sql);
    if (defined($res) && ref($res) eq "ARRAY") {
	my $mapp = "";
	my $initScript = "";
	foreach $mapp (@{$res}) {
	    next unless (defined($mapp) && ref($mapp) eq "HASH");
	    $mappId = $mapp->{'mapp_id'};
	    next unless (defined($mappId) && $mappId+0 > 0);
	    $mappCnt++;
	    $initScript = $mapp->{'init_script'};
	    $initScriptCnt++ 
		if (defined($initScript) && $initScript ne "");
	}
    }
   
    return ($mappCnt, $initScriptCnt);
}

#
# getApplianceSchedules
#
# ???
#

sub getApplianceSchedules
{
    my $mappName = shift @_;
    return SCSDB::errorCode('EPARM', 'MAPP name is null')
	unless (defined($mappName) && $mappName ne "");
    
    my $filterNA = shift @_;
    $filterNA = 1 unless (defined($filterNA));

    my $sql = 'SELECT schedule_id FROM mgmt_schedule WHERE ';
    $sql .= "use_appliances = 'Y' AND mapp_name = '$mappName' ";

    $sql .= "AND new_appliances = 'Y'"
	if ($filterNA);

    return SCSDB::getResultList($sql);
}

#
# addApplianceToSchedules
#
# ???
#
# WARNING: NEEDS A TXN
#

sub addApplianceToSchedules
{
    my $hostId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid host id')
	unless (defined($hostId) && $hostId+0 > 0);
    
    my $scheduleList = shift @_;
    if (defined($scheduleList) && ref($scheduleList) eq "ARRAY") {
	my $scheduleId = 0;
	my $sql = "";
	my $err = "";
	my %schedHash = ();
	my $schedInsertId = 0;

	foreach $scheduleId (@{$scheduleList}) {
	    next unless (defined($scheduleId) && $scheduleId+0 > 0);

	    %schedHash = ( 'appliance_id' => int($hostId+0),
			   'schedule_id'  => int($scheduleId+0) );

	    $schedInsertId = 
	      SCSDB::insertHash('mgmt_schedule_appliance',
				'schedule_appliance_id',
				\%schedHash,
				'mgmt_schedule_schedule_appl_seq');

	    unless (defined($schedInsertId) && 
		    $schedInsertId+0 > 0) {
		$err = "Cannot add schedule for '$hostId'";
		last;
	    }
	}

	return SCSDB::errorCode('EDBERR', $err)
	    if ($err ne "");
    }
    
    return SCSDB::errorCode('OKAY');
}

#
# removeApplianceFromSchedules
#
# ???
#

sub removeApplianceFromSchedules
{
    my $hostId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid host id')
	unless (defined($hostId) && $hostId+0 > 0);
    
    my $scheduleList = shift @_;
    return SCSDB::errorCode('OKAY')
	unless (defined($scheduleList) && 
		ref($scheduleList) eq "ARRAY");

    my @scheduleIds = SCSDB::idList(@{$scheduleList});
    return SCSDB::errorCode('OKAY')
	unless (scalar(@scheduleIds) > 0);

    my $sql = 'DELETE FROM mgmt_schedule_appliance WHERE ';
    $sql .= 'appliance_id = ' . int($hostId+0);
    $sql .= ' AND schedule_id IN (';
    $sql .= join(', ', @scheduleIds);
    $sql .= ')';

    return SCSDB::runCommand($sql);
}

#
# registerTask
#
# ???
#

sub registerTask
{
    my $mappId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid MAPP id')
	unless (defined($mappId) && $mappId+0 > 0);

    my $task = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid task')
	unless (defined($task) && ref($task) eq "HASH");

    my %taskHash = ( 'mapp_id'        => int($mappId+0),
		     'name'           => 
		     "'" . $task->{'name'} . "'",
		     'target_url'     => 
		     "'" . $task->{'target_url'} . "'",
		     'target_tab'     => 
		     "'" . $task->{'target_tab'} . "'",
		     'is_schedulable' => 
		     "'" . $task->{'schedulable'} . "'" );

    return SCSDB::insertHash('mgmt_schedulable_task',
			     'schedulable_task_id',
			     \%taskHash,
			     'mgmt_schedula_schedulable_t_seq');
}

#
# registerApplianceIcon
#
# ???
#

sub registerApplianceIcon
{
    my $mappId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid MAPP id')
	unless (defined($mappId) && $mappId+0 > 0);

    my $icon = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid icon')
	unless (defined($icon) && ref($icon) eq "HASH");

    my %iconHash = ( 'mapp_id'     => int($mappId+0),
		     'icon'        => 
		     "'" . $icon->{'icon'} . "'",
		     'description' => 
		     "'" . $icon->{'description'} . "'",
		     'target_url'  => 
		     "'" . $icon->{'target_url'} . "'" );

    return SCSDB::insertHash('mgmt_appliance_icon',
			     'appliance_icon_id',
			     \%iconHash,
			     'mgmt_applianc_appliance_ico_seq');
}

#
# getEligibleAppliances
#
# ???
#

sub getEligibleAppliances
{
    my $mappId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid MAPP id')
	unless (defined($mappId) && $mappId+0 > 0);

    my @hostIds = ();
    my $hostList = shift @_;
    @hostIds = SCSDB::idList(@{$hostList})
	if (defined($hostList) && ref($hostList) eq "ARRAY");

    my $sql = 'SELECT DISTINCT a.appliance_id, a.host_name, ';
    $sql .= 'b.build_class, i.status, i.reason FROM mgmt_appliance ';
    $sql .= 'AS a LEFT JOIN mgmt_build AS b ON a.build_id = ';
    $sql .= 'b.build_id LEFT JOIN mgmt_installed_rpm AS i ON ';
    $sql .= 'i.appliance_id = a.appliance_id LEFT JOIN ';
    $sql .= 'mgmt_mapp_client_rpm r ON i.mapp_client_rpm_id = ';
    $sql .= 'r.mapp_client_rpm_id WHERE mapp_id = ' . int($mappId+0);

    if (scalar(@hostIds) > 0) {
	$sql .= 'AND a.appliance_id  IN (';
	$sql .= join(', ', @hostIds);
	$sql .= ') ';
    }

    $sql .= 'ORDER BY a.appliance_id';

    return SCSDB::getResultHashes($sql);
}

#
# getEligibleApplianceList
#
# ???
#

sub getEligibleApplianceList
{
    my $mappId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid MAPP id')
	unless (defined($mappId) && $mappId+0 > 0);

    my @hostIds = ();
    my $hostList = shift @_;
    @hostIds = SCSDB::idList(@{$hostList})
	if (defined($hostList) && ref($hostList) eq "ARRAY");

    my $sql = 'SELECT DISTINCT a.appliance_id FROM mgmt_appliance ';
    $sql .= 'AS a LEFT JOIN mgmt_build AS b ON a.build_id = ';
    $sql .= 'b.build_id LEFT JOIN mgmt_installed_rpm AS i ON ';
    $sql .= 'i.appliance_id = a.appliance_id LEFT JOIN ';
    $sql .= 'mgmt_mapp_client_rpm r ON i.mapp_client_rpm_id = ';
    $sql .= 'r.mapp_client_rpm_id WHERE mapp_id = ' . int($mappId+0);

    if (scalar(@hostIds) > 0) {
	$sql .= 'AND a.appliance_id  IN (';
	$sql .= join(', ', @hostIds);
	$sql .= ') ';
    }

    $sql .= 'ORDER BY a.appliance_id';

    return SCSDB::getResultList($sql);
}

#
# getDeniedVersion
#
# returns the max version of the mapp with the specified name from
# the deny mapp table
#

sub getDeniedVersion
{
    my $moduleName = shift @_;
    return SCSDB::errorCode('EPARM', 'Module name is null')
	unless (defined($moduleName) && $moduleName ne "");
    
    my $sql = 'SELECT max_version FROM mgmt_deny_mapp WHERE ';
    $sql .= "mapp_name = '$moduleName'";

    return SCSDB::getResult($sql);
}

1;
