#!/usr/bin/perl -w
# $Id: Group.pm,v 1.4 2004/10/27 16:36:54 ms152511 Exp $
# Group.pm - Manipulate database for groups (views) of hosts
#
# Copyright 2004 Sun Microsystems, Inc., All rights reserved.
# SUN PROPRIETARY/CONFIDENTIAL.

package Group;

use strict;
use lib '/scs/lib/perl5';
use SCSDB;
use Appliance;

#
# Subroutines
#
# getGroupIDByName
# getGroupDetails
# getGroupHosts
#
# addGroup
# removeGroup
# modifyGroupDetails
#
# addHostsToGroup
# removeHostsFromGroup
#

#
# Constants
#

my $GroupTable       = 'mgmt_view';
my $GroupIDField     = 'view_id';
my $GroupNameField   = 'view_name';
my $GroupDescField   = 'view_description';
my $HostMappingTable = 'mgmt_view_to_appliance';
my $HostIDField      = 'appliance_id';

#
# getGroupIDByName
#
# Given a group name, returns the corresponding group ID.
# Returns the group ID if found, or EINVAL if no group is found.
#
{
    # scope prepared statement locally
    my $query_stmt;

    sub getGroupIDByName ($)
    {
        my $group_name = shift @_;
        return ( SCSDB::errorCode( 'EPARM',
                                   { 'tag' => 'error_groupNameMissing' } ) )
            unless (defined $group_name && $group_name ne '');
        
        if (not defined $query_stmt) {
            my $sql = "SELECT $GroupIDField FROM $GroupTable " .
                "WHERE $GroupNameField = ?";
            $query_stmt = SCSDB::prepareStmt($sql);
            if ($query_stmt == 0) {
                undef $query_stmt;
                return SCSDB::errorCode( 'EDBERR',
                                         { 'tag' => 'error_prepareSQLStmt' } );
            }
        }
        
        my $group_id = SCSDB::getResult( $query_stmt,
                                         $group_name );
        
        if ( $group_id == 0 ) {
            return SCSDB::errorCode( 'EINVAL',
                                     { 'tag'        => 'error_groupNameNotFound' ,
                                       'group_name' => $group_name } );
        } else {
            return $group_id;
        }
    }                 
}

#
# getGroupDetails
#
# Given a group ID, returns a hash reference containing the name and
# description fields of the specified group.
#
{
    my $query_stmt;
    
    sub getGroupDetails ($)
    {
        my $group_id = shift @_;
        return ( SCSDB::errorCode( 'EPARM',
                                   { 'tag' => 'error_groupIDMissing' } ) )
            unless (defined $group_id && $group_id ne '');
        return ( SCSDB::errorCode( 'EINVAL',
                                   { 'tag'      => 'error_invalidGroupID', 
                                     'group_id' => $group_id } ) )
            unless ($group_id+0 > 0);
        
        if (not defined $query_stmt) {
            my $sql = "SELECT $GroupIDField, $GroupNameField, $GroupDescField FROM $GroupTable " .
                "WHERE $GroupIDField = ?";
            $query_stmt = SCSDB::prepareStmt($sql);
            if ($query_stmt == 0) {
                undef $query_stmt;
                return SCSDB::errorCode( 'EDBERR',
                                         { 'tag' => 'error_prepareSQLStmt' } );
            }
        }
        
        my $group_detail_hash_ref = SCSDB::getResultHash( $query_stmt,
                                                          $group_id );
        
        if ( not defined $group_detail_hash_ref ) {
            return SCSDB::errorCode( 'EINVAL',
                                     { 'tag'      => 'error_groupIDNotFound' , 
                                       'group_id' => $group_id } );
        } elsif ( $group_detail_hash_ref == 0 ) {
            return SCSDB::errorCode( 'EDBERR',
                                     { 'tag'      => 'error_executeSQLStmt' } );
        } else {
            return $group_detail_hash_ref;
        }
    }
}

#
# getGroupHosts
#
# Given a group ID, returns a reference to a list of the hosts
# contained within the specified group.  If the group contains no
# hosts, returns a reference to an empty list.  If the group does not
# exist, returns a negative integer.
#
{
    my $query_stmt;
    
    sub getGroupHosts ($)
    {
        my $group_id = shift @_;
        return ( SCSDB::errorCode( 'EPARM',
                                   { 'tag' => 'error_groupIDMissing' } ) )
            unless (defined $group_id && $group_id ne '');
        return ( SCSDB::errorCode('EINVAL',
                                   { 'tag'      => 'error_invalidGroupID',
                                     'group_id' => $group_id } ) )
            unless ($group_id+0 > 0);

        # check that group exists
        my $group_detail_hash_ref = getGroupDetails($group_id);
        if ( ref($group_detail_hash_ref) ne 'HASH' ) {
            return $group_detail_hash_ref;
        }
        
        if (not defined $query_stmt) {
            my $sql = "SELECT $HostIDField FROM $HostMappingTable " .
                "WHERE $GroupIDField = ?";
            $query_stmt = SCSDB::prepareStmt($sql);
            if ($query_stmt == 0) {
                undef $query_stmt;
                return SCSDB::errorCode( 'EDBERR',
                                         { 'tag' => 'error_prepareSQLStmt' } );
            }
        }
        
        my $host_id_list_ref = SCSDB::getResultList( $query_stmt,
                                                     $group_id );
        
        if ( $host_id_list_ref == 0 ) {
            return SCSDB::errorCode( 'EDBERR',
                                     { 'tag' => 'error_executeSQLStmt' } );
        } else {
            return $host_id_list_ref;
        }
    }
}    

#
# addGroup
#
# Creates a group with the given name and description.  Returns the
# group ID (a positive integer) of the newly created group.  If an
# error occurs or a group with a conflicting name exists, returns a
# negative integer.
#
{
    my $insert_stmt;
    sub addGroup ($$)
    {
        my $group_name = shift @_;
        return ( SCSDB::errorCode( 'EINVAL',
                                   { 'tag' => 'error_groupNameMissing' } ) )
            unless (defined $group_name && $group_name ne '');
        # default to empty (not null) group description
        my $group_desc = shift @_ || '';
        
        # check if a group by this name already exists
        my $existing_group_id = getGroupIDByName($group_name);
        return ( SCSDB::errorCode( 'EDUP',
                                   { 'tag'        => 'error_groupAlreadyExists', 
                                     'group_name' => $group_name } ) )
            if ($existing_group_id > 0);

        if (not defined $insert_stmt) {
            my $sql = "INSERT INTO $GroupTable ($GroupNameField, $GroupDescField) " .
                "VALUES (?, ?)";
            $insert_stmt = SCSDB::prepareStmt($sql);
            if ($insert_stmt == 0) {
                undef $insert_stmt;
                return SCSDB::errorCode( 'EDBERR',
                                         { 'tag' => 'error_prepareSQLStmt' } );
            }
        }

        my $rc = SCSDB::executeStmt( $insert_stmt, $group_name, $group_desc );
        return SCSDB::errorCode( 'EDBERR',
                                 { 'tag' => 'error_executeSQLStmt' } )
            if ($rc < 0);

        my $group_id = SCSDB::getInsertId( $GroupTable, $GroupIDField );
        return SCSDB::errorCode( 'EDBERR',
                                 { 'tag' => 'error_getInsertID' } )
            if ($rc < 0);

        return $group_id;
    }
}

#
# removeGroup
#
# Delete a group with the given group ID.
#
{
    my $delete_stmt;

    sub removeGroup ($)
    {
        my $group_id = shift @_;
        return ( SCSDB::errorCode( 'EPARM',
                                   { 'tag' => 'error_groupIDMissing' } ) )
            unless (defined $group_id && $group_id ne '');
        return ( SCSDB::errorCode( 'EINVAL',
                                   { 'tag'      => 'error_invalidGroupID', 
                                     'group_id' => $group_id } ) )
            unless ($group_id+0 > 0);
        
        if (not defined $delete_stmt) {
            my $sql = "DELETE FROM $GroupTable " .
                "WHERE $GroupIDField = ?";
            $delete_stmt = SCSDB::prepareStmt($sql);
            if ($delete_stmt == 0) {
                undef $delete_stmt;
                return SCSDB::errorCode( 'EDBERR',
                                         { 'tag' => 'error_prepareSQLStmt' } );
            }
        }
        
        my $rc = SCSDB::executeStmt( $delete_stmt, $group_id );
        return SCSDB::errorCode( 'EDBERR',
                                 { 'tag' => 'error_executeSQLStmt' } )
            if ($rc < 0);

        my $rows_deleted = SCSDB::getRowsAffected( $delete_stmt );
        return SCSDB::errorCode( 'EDBERR',
                                 { 'tag' => 'error_groupDeleteCount' } )
            if ($rows_deleted < 0 || $rows_deleted > 1);

        return SCSDB::errorCode( 'EINVAL',
                                 { 'tag'      => 'error_groupIDNotFound' , 
                                   'group_id' => $group_id } )
            if ($rows_deleted != 1);

        return SCSDB::errorCode( 'OKAY', '' );
    }
}

#
# modifyGroupDetails
#
# Change a groups' name and description, provided a group ID.  Returns
# a negative integer if the group is not found, or if there is already
# another group existing that already has the desired name.
#
{
    my $select_for_update_stmt;
    my $update_stmt;
    
    sub modifyGroupDetails ($$$)
    {
        my $rc;
        
        my $group_id = shift @_;
        return ( SCSDB::errorCode( 'EPARM',
                                   { 'tag' => 'error_groupIDMissing' } ) )
            unless (defined $group_id && $group_id ne '');
        return ( SCSDB::errorCode( 'EINVAL',
                                   { 'tag'      => 'error_invalidGroupID', 
                                     'group_id' => $group_id } ) )
            unless ($group_id+0 > 0);
        
        my $new_group_name = shift @_;
        return ( SCSDB::errorCode( 'EINVAL',
                                   { 'tag' => 'error_groupNameMissing' } ) )
            unless (defined $new_group_name && $new_group_name ne '');
        # default to empty (not null) group description
        my $new_group_desc = shift @_ || '';

        # setup transaction
        $rc = SCSDB::startTxn();
        return SCSDB::errorCode( 'EDBERR',
                                 { 'tag' => 'error_startTxn' } )
            if ($rc < 0);

        # verify that this group exists
        if (not defined $select_for_update_stmt) {
            my $sql = "SELECT $GroupIDField FROM $GroupTable " .
                "WHERE $GroupIDField = ? FOR UPDATE";
            $select_for_update_stmt = SCSDB::prepareStmt($sql);
            if ($select_for_update_stmt == 0) {
                undef $select_for_update_stmt;
                SCSDB::rollBack();
                return SCSDB::errorCode( 'EDBERR',
                                         { 'tag' => 'error_prepareSQLStmt' } );
            }
        }
        
        my $select_result = SCSDB::getResultList( $select_for_update_stmt,
                                                  $group_id );
        # should return reference to array with one entry
        if ( $select_result == 0 ||
             ref($select_result) ne 'ARRAY' ||
             scalar(@$select_result) < 0 || scalar(@$select_result) > 1 ) {
            SCSDB::rollBack();
            return SCSDB::errorCode( 'EDBERR',
                                     { 'tag' => 'error_executeSQLStmt' } );
        } elsif ( scalar @$select_result != 1 ) {
            SCSDB::rollBack();
            return SCSDB::errorCode( 'EINVAL',
                                     { 'tag'       => 'error_groupIDNotFound' , 
                                       'group_id'  => $group_id } );
        } elsif ( $$select_result[0] ne $group_id ) {
            SCSDB::rollBack();
            return SCSDB::errorCode( 'EDBERR',
                                     { 'tag' => 'error_badGroupIDReturned' } );
        }
        
        # check if another group by this name already exists
        my $existing_group_id = getGroupIDByName($new_group_name);
        if ($existing_group_id > 0 && $existing_group_id != $group_id) {
            SCSDB::rollBack();
            return ( SCSDB::errorCode( 'EDUP',
                                       { 'tag'        => 'error_groupAlreadyExists',
                                         'group_name' => $new_group_name } ) );
        }
        
        if (not defined $update_stmt) {
            my $sql = "UPDATE $GroupTable " .
                "SET $GroupNameField = ?, $GroupDescField = ? " .
                "WHERE $GroupIDField = ?";
            $update_stmt = SCSDB::prepareStmt($sql);
            if ($update_stmt == 0) {
                undef $update_stmt;
                SCSDB::rollBack();
                return SCSDB::errorCode( 'EDBERR',
                                         { 'tag' => 'error_prepareSQLStmt' } );
            }
        }

        $rc = SCSDB::executeStmt( $update_stmt,
                                  $new_group_name, $new_group_desc,
                                  $group_id );
        if ($rc < 0) {
            SCSDB::rollBack();
            return SCSDB::errorCode( 'EDBERR',
                                     { 'tag' => 'error_executeSQLStmt' } );
        }

        my $rows_updated = SCSDB::getRowsAffected( $update_stmt );
        if ($rows_updated != 1) {
            SCSDB::rollBack();
            return SCSDB::errorCode( 'EDBERR',
                                     { 'tag' => 'error_groupUpdateCount' } );
        }
        
        SCSDB::commit();
        return SCSDB::errorCode( 'OKAY', undef );
    }
}

#
# addHostsToGroup
#
# Given a group ID and a reference to a list of host IDs, adds the
# hosts to the group.  If an error occurs, the group does not exist,
# or any of the specified hosts do not exist or are already a member
# of the given group, the group remains unchanged and an error is
# returned to the caller.
#
{
    my $insert_stmt;
    
    sub addHostsToGroup ($$)
    {
        # validate arguments
        my $group_id = shift @_;
        return ( SCSDB::errorCode( 'EPARM',
                                   { 'tag' => 'error_groupIDMissing' } ) )
            unless (defined $group_id && $group_id ne '');
        return ( SCSDB::errorCode( 'EINVAL',
                                   { 'tag'      => 'error_invalidGroupID', 
                                     'group_id' => $group_id } ) )
            unless ($group_id+0 > 0);
        
        my $host_ids_ref = shift @_;
        return ( SCSDB::errorCode( 'EPARM',
                                   { 'tag' => 'error_addHostIDListMissing' } ) )
            unless (defined $host_ids_ref);
        return ( SCSDB::errorCode( 'EINVAL',
                                   { 'tag' => 'error_invalidHostIDList' } ) )
            unless (ref($host_ids_ref) eq 'ARRAY');
        return ( SCSDB::errorCode( 'EPARM',
                                   { 'tag' => 'error_noHostIDsSpecified' } ) )
            unless (scalar @{$host_ids_ref} > 0);

        # verify that group exists and pass along error if it does not
        my $group_details_hash_ref = getGroupDetails($group_id);
        return $group_details_hash_ref
            if ($group_details_hash_ref < 0);

        # prepare SQL statement if necessary
        if (not defined $insert_stmt) {
            my $sql = "INSERT INTO $HostMappingTable ($GroupIDField, $HostIDField) " .
                "VALUES (?, ?)";
            $insert_stmt = SCSDB::prepareStmt($sql);
            if ($insert_stmt == 0) {
                undef $insert_stmt;
                return SCSDB::errorCode( 'EDBERR',
                                         { 'tag' => 'error_prepareSQLStmt' } );
            }
        }

        # setup transaction
        my $rc = SCSDB::startTxn();
        return SCSDB::errorCode( 'EDBERR',
                                 { 'tag' => 'error_startTxn' } )
            if ($rc < 0);
        
        # loop over all host IDs to add
        foreach my $host_id (@$host_ids_ref) {

            # try to add the host to the group
            $rc = SCSDB::executeStmt( $insert_stmt,
                                      $group_id, $host_id );

            if ($rc < 0) {
                # something went wrong, let's try to figure out
                # exactly what
                
                # clean up transaction
                SCSDB::rollBack();

                # verify host ID is valid and exists
                my $ipaddr = Appliance::getApplianceIPaddress($host_id);
                unless ($ipaddr) {
                    # FIXME: couldn't get information for this host;
                    # for now assume this means the host doesn't
                    # exist.
                    return SCSDB::errorCode( 'EINVAL',
                                             { 'tag'     => 'error_hostIDNotFound',
                                               'host_id' => $host_id } );
                }

                # check whether host is already a member of this group
                my $group_member_hosts_ref = getGroupHosts($group_id);
                return $group_member_hosts_ref
                    if ($group_member_hosts_ref < 0);

                return SCSDB::errorCode( 'EINVAL',
                                         { 'tag'      => 'error_hostAlreadyInGroup',
                                           'host_id'  => $host_id,
                                           'group_id' => $group_id } )
                    if (grep( $host_id, @$group_member_hosts_ref ));
                
                # neither of those conditions, so likely a DB problem
                return SCSDB::errorCode( 'EDBERR',
                                         { 'tag' => 'error_executeSQLStmt' } );
            }
        }
        
        # clean up transaction
        $rc = SCSDB::commit();
        return SCSDB::errorCode( 'EDBERR',
                                 { 'tag' => 'error_commit' } )
            if ($rc < 0);
        
        return SCSDB::errorCode( 'OKAY', undef );
    }
}

#
# removeHostsFromGroup
#
# Given a group ID and a reference to a list of host IDs, removes the
# hosts from the group.  If an error occurs, the group does not exist,
# or any of the specified hosts do not exist or are not members of the
# given group, the group remains unchanged and an error is returned to
# the caller.
#
{
    my $delete_stmt;
    
    sub removeHostsFromGroup ($$)
    {
        # validate arguments
        my $group_id = shift @_;
        return ( SCSDB::errorCode( 'EPARM',
                                   { 'tag' => 'error_groupIDMissing' } ) )
            unless (defined $group_id && $group_id ne '');
        return ( SCSDB::errorCode( 'EINVAL',
                                   { 'tag'      => 'error_invalidGroupID',
                                     'group_id' => $group_id } ) )
            unless ($group_id+0 > 0);
        
        my $host_ids_ref = shift @_;
        return ( SCSDB::errorCode( 'EPARM',
                                   { 'tag' => 'error_hostIDListMissing' } ) )
            unless (defined $host_ids_ref);
        return ( SCSDB::errorCode( 'EINVAL',
                                   { 'tag' => 'error_invalidHostIDList' } ) )
            unless (ref($host_ids_ref) eq 'ARRAY');
        return ( SCSDB::errorCode( 'EPARM',
                                   { 'tag' => 'error_noHostIDsSpecified' } ) )
            unless (scalar @{$host_ids_ref} > 0);

        # verify that group exists and pass along error if it does not
        my $group_details_hash_ref = getGroupDetails($group_id);
        return $group_details_hash_ref
            if ($group_details_hash_ref < 0);

        # prepare SQL statement if necessary
        if (not defined $delete_stmt) {
            my $sql = "DELETE FROM $HostMappingTable " .
                "WHERE $GroupIDField = ? AND " .
                    "  $HostIDField  = ?";
            $delete_stmt = SCSDB::prepareStmt($sql);
            if ($delete_stmt == 0) {
                undef $delete_stmt;
                return SCSDB::errorCode( 'EDBERR',
                                         { 'tag' => 'error_prepareSQLStmt' } );
            }
        }

        # setup transaction
        my $rc = SCSDB::startTxn();
        return SCSDB::errorCode( 'EDBERR',
                                 { 'tag' => 'error_startTxn' } )
            if ($rc < 0);
        
        # loop over all host IDs to add
        foreach my $host_id (@$host_ids_ref) {

            # try to remove the host from the group
            $rc = SCSDB::executeStmt( $delete_stmt,
                                      $group_id, $host_id );
            if ($rc < 0) {
                # clean up transaction
                SCSDB::rollBack();
                return SCSDB::errorCode( 'EDBERR',
                                         { 'tag' => 'error_executeSQLStmt' } );
            }

            # see if we acomplished anything
            my $rows_deleted = SCSDB::getRowsAffected( $delete_stmt );
            if ($rows_deleted < 0 || $rows_deleted > 1) {
                SCSDB::rollBack();
                return SCSDB::errorCode( 'EDBERR',
                                         { 'tag' => 'error_groupMemberDeleteCount' } );
            }

            if ($rows_deleted != 1) {
                SCSDB::rollBack();
                return SCSDB::errorCode( 'EINVAL',
                                         { 'tag'     => 'error_hostIDNotFound' ,
                                           'host_id' => $host_id } ); 
            }
            
        }
        
        # clean up transaction
        $rc = SCSDB::commit();
        return SCSDB::errorCode( 'EDBERR',
                                 { 'tag' => 'error_commit' } )
            if ($rc < 0);

        return SCSDB::errorCode( 'OKAY', undef );
    }
}

1;
