#!/usr/bin/perl -wT
# $Id: CLIids.pm,v 1.3 2004/10/20 20:15:38 ms152511 Exp $
# MAPP.pm - Module loading functions
# Copyright 2003, 2004 Sun Microsystems, Inc., All rights reserved.
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms

package CLIids;

use strict;
use Carp qw(cluck);
use lib '/scs/lib/perl5';
use BDUtil;
use SCSDB;
use Appliance;
use CLIips;

my $infoMsgs = 0;
sub setInfoMsgs
{
  CLIips::setInfoMsgs();
  $infoMsgs = 1;
}

#
#
#
sub printUsage
{
  print STDERR<<EOD;

Hosts may be specified singly by:

 - host ID
 - IP address, i.e. 10.6.1.1
 - hostname

or muliply by:

 - a range of IP addresses

   Examples:
     * 10.6.1.1-10.6.1.3 : 3 addresses, could also be specified as:
     * 10.6.1.1-3 : same as above
     * 10.6.1.100-200 : 101 addresses, between 10.6.1.100 and 10.6.1.200
     * 10.6.1.1-2.255 : 510 addresses, between 10.6.1.1 and 10.6.2.255
     * 0.0.0.0-255.255.255.255 : Every single possible IP address

 - a range of host IDs, e.g. 4-15
 - group name or ID, prepended with a _g, e.g. _g"Group 1", _g4

Host IDs will then be looked up for all of the above parameters and a
distinct list of relevant hosts will then be processed.

If an argument corresponds to no valid host IDs, e.g. a group does not exist,
no hosts match a given IP range, then a warning will be generated.

EOD
}

#
#
#
sub setExitOnError
{
  CLIips::setExitOnError();
}

#
# getHostIds
#
# Takes a list reference to the command line arguments to be
# processed.  Returns a reference to an array of host IDs that were
# valid when this subroutine was invoked.
#
# If any element of the argument list is invalid, e.g. a nonexistant
# group, then the behavior depends on whether the setExitOnError()
# routine has been called.  If so, then the entire script aborts.
# Otherwise a warning is printed and processing proceeds with the
# next argument.
#
sub getHostIds ($)
{
    my $args_ref = shift;
    my %hostIds  = ();
    
    if ( !$args_ref                or
         ref($args_ref) ne 'ARRAY' or 
         scalar(@$args_ref) == 0 )
    {
        CLIips::processError('No hosts specifiers provided');
        return [];
    }

    foreach my $arg (@$args_ref) {

        my $rc;
        my @arg_host_ids = ();
        my ($ipaddrs_ref,$group_spec,$host_id);

        if ( $rc = SCSUtil::isIPAddress($arg) ) {

            # IP addresses
            if ( $rc < 0 ) {
                CLIips::processError($arg . ' is an invalid IP address');
            } elsif ( $rc > 0 ) {
                @arg_host_ids = _processIPs( [$arg] );
                if ( scalar @arg_host_ids > 0 ) {
                    foreach my $host_id (@arg_host_ids) {
                        $hostIds{$host_id} = 1;
                    }
                } else {
                    CLIips::processError('IP address ' . $arg .
                                         ' does not correspond to a host');
                }
            }

        } elsif ( ( ($rc, $ipaddrs_ref) = CLIips::isIPrange($arg) ) &&
                  $rc ) {

            # IP ranges
            if ( $rc < 0 ) {
                CLIips::processError($arg . ' is an invalid IP range');
            } elsif ( $rc > 0 ) {
                # proper IP range
                @arg_host_ids = _processIPs($ipaddrs_ref);
                if ( scalar @arg_host_ids > 0 ) {
                    foreach my $host_id (@arg_host_ids) {
                        $hostIds{$host_id} = 1;
                    }
                } else {
                    CLIips::processError($arg . ' does not correspond to any hosts');
                }
            }

        } elsif ( ($group_spec) = $arg =~ /^_g(.+)$/ ) {
            
            # group specifier (ID/name)

            # try by name lookup first
            my $group_id = getGroupIDByName($group_spec);
            
            if ( $group_id > 0 ) {
                # successful name lookup
                @arg_host_ids = _processGroupID($group_id);
                if ( scalar @arg_host_ids > 0 ) {
                    foreach my $host_id (@arg_host_ids) {
                        $hostIds{$host_id} = 1;
                    }
                } else {
                    CLIips::processError('Group ' . $group_spec . ' contains no hosts');
                }
            } elsif ( ($group_spec) =~ /^\d+$/ && getGroupDetails( $group_spec ) > 0 ) {
                # found a group by direct ID match
                @arg_host_ids = _processGroupID($group_spec);
                if ( scalar @arg_host_ids > 0 ) {
                    foreach my $host_id (@arg_host_ids) {
                        $hostIds{$host_id} = 1;
                    }
                } else {
                    CLIips::processError('Group with ID ' . $group_spec . ' contains no hosts');
                }
            } else {
                CLIips::processError('No group with name/ID ' . $group_spec);
            }

        } elsif ( ($host_id) = $arg =~ /^(\d+)$/ ) {
            
            # simple host ID

            my $host_details = Appliance::getApplianceDetail($host_id);
            if ( ref($host_details)      eq 'ARRAY' &&
                 ref($host_details->[0]) eq 'HASH' ) {
                $hostIds{$host_id} = 1;
            } else {
                CLIips::processError('No host with ID ' . $host_id);
            }

        } elsif ( my ($host_id_range_start, $host_id_range_end) =
                  $arg =~ /^(\d+)-(\d+)$/ ) {

            # range of host IDs

            @arg_host_ids = _processHostIdRange( $host_id_range_start,
                                                 $host_id_range_end );
            if ( scalar @arg_host_ids > 0 ) {
                foreach my $host_id (@arg_host_ids) {
                    $hostIds{$host_id} = 1;
                }
            } else {
                CLIips::processError('Host range ' . $arg . ' includes no valid hosts');
            }

        } else {

            # presumably this is a resolvable hostname

            @arg_host_ids = _processHostname($arg);
            if ( scalar @arg_host_ids > 0 ) {
                foreach my $host_id (@arg_host_ids) {
                    $hostIds{$host_id} = 1;
                }
            } else {
                CLIips::processError('No host named ' . $arg . ' found');
            }

        } # end if-elsif ladder
        
    } # end $arg foreach

    # make sure we return a list reference
    return [ sort keys %hostIds ];
}

#
# type-specific query routines
#
# Each of these routines queries the database for hosts that match the
# corresponding criterion.  Each routine returns a list of matching
# hosts, but may return none at all (an empty list) if there are no
# matching hosts in the database.  
#

#
# _processIPs
#
# Takes a reference to a list of IPs.  Returns a list of host_ids
# corresponding to provided IPs.  Not all provided IPs may generate a
# host ID.
#
sub _processIPs ($)
{
    my ($ipaddrs_ref) = @_;
    cluck "_processIPs expects a reference to an array of IP addresses"
        if ( not defined $ipaddrs_ref or
             ref $ipaddrs_ref ne 'ARRAY' );

    my @host_ids;

    foreach my $ipaddr (@$ipaddrs_ref) {
        my $host_id = Appliance::getApplianceId($ipaddr);
        push( @host_ids, $host_id )
            if ( $host_id > 0 );
    }

    return @host_ids;
}   

sub _processGroupID ($)
{
    my ($group_id) = @_;
    my $group_hosts_ref = Group::getGroupHosts($group_id);
    
    if (ref $group_hosts_ref eq 'ARRAY') {
        return @$group_hosts_ref;
    } else {
        return ();
    }
}

sub _processHostIdRange ($$)
{
    my ($startId, $endId) = @_;
    my @host_ids = ();
    for my $host_id ($startId..$endId) {
        my $host_details = Appliance::getApplianceDetail($host_id);
        if ( ref($host_details) eq 'ARRAY' &&
             ref($host_details->[0]) eq 'HASH' ) {
            push( @host_ids, $host_id );
        }
    }
    return @host_ids;
}

sub _processHostname ($)
{
    my ($hostname) = @_;
    my $host_id = Appliance::getApplianceIdByHostname($hostname);
    return ($host_id)
        if ($host_id > 0);

    # no exact hostname match, so let's try a DNS-resolved IP match
    my $ipaddr = SCSUtil::getIPAddress($hostname);
    $host_id = Appliance::getApplianceId($ipaddr);
    return ($host_id)
        if ($host_id > 0);

    return ();
}                      

1;

