#!/usr/bin/perl -w
# $Id: Appliance.pm,v 1.11 2004/10/05 20:03:20 ms152511 Exp $
# Appliance.pm - Host access api
# Copyright 2003, 2004 Sun Microsystems, Inc., All rights reserved.
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms

package Appliance;

use lib '/scs/lib/perl5';
use strict;
use SCSDB;
use Builds;

#
# Subroutines
#
# getAppliances
# getApplianceId
# getApplianceDetail
# getApplianceIPaddress
# getApplianceIPaddressList
# getApplianceEmail
# getApplianceEmailById
# getAppliancesByIpAddress
# getAppliancesByBuildAndStatus
# getAppliancesByStatus
#
# getModuleAppliances
#
# addAppliance
# addScannedAppliance
#
# removeApplianceRecords
# removeAppliance
#
# setScannedAppliance
# setApprovedAppliance
# setManagedAppliance
# setManagedFailed
# resetUnmanagedStatus
# setHostname
# setMacAddress
# setDescription
# setEmail
# setApplianceStatus
# setPasswd
# updateIpAddress
#

#
# getApplianceId
#
# returns the host id by ip addr
#

sub getApplianceId
{
    my $ipaddr = shift @_;
    return SCSDB::errorCode('EINVAL', 
			   'IP Address is null')
	unless (defined $ipaddr && $ipaddr ne "");

    my $sql = "SELECT appliance_id FROM mgmt_appliance WHERE ";
    $sql .= "status != 'D' AND ip_address = '$ipaddr'";

    my $id = SCSDB::getResult($sql);
    return SCSDB::errorCode('EINVAL', "No host with ip '$ipaddr'")
	unless ($id);

    return $id;
}

#
# getApplianceIdByHostname
#
# returns the host id by hostname
#

sub getApplianceIdByHostname
{
    my $hname = shift @_;
    return SCSDB::errorCode('EINVAL', 'Hname is null')
	unless (defined($hname));

    $hname =~ s/[ \t\r\n]//g;

    return SCSDB::errorCode('EINVAL', 'Hname is null')
	unless (defined($hname) && $hname ne "");    

    my $sql = "SELECT appliance_id FROM mgmt_appliance WHERE ";
    $sql .= "status != 'D' AND host_name = '$hname'";

    my $id = SCSDB::getResult($sql);
    return SCSDB::errorCode('EINVAL', "No host with hostname '$hname'")
	unless ($id);

    return $id;
}

#
# getApplianceIPAddress
#
# returns the ip address for a given host id
#

sub getApplianceIPaddress
{
    my $id = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid host id') 
	unless (defined($id) && $id+0 > 0);

    my $sql = "SELECT ip_address FROM mgmt_appliance WHERE ";
    $sql .= 'appliance_id = ' . int($id+0);

    my $ip = SCSDB::getResult($sql);

    return (SCSDB::errorCode('EINVAL', "No host with id '$id'"), '')
	unless ($ip);

    return (SCSDB::errorCode('OKAY'), $ip);
}


# getApplianceIPaddressList
#
# returns the host id, ip_address and host_name for a set of host ids
#

sub getApplianceIPaddressList
{
    my $sql = 'SELECT appliance_id, ip_address, host_name FROM ';
    $sql .= "mgmt_appliance ";

    my @ids = SCSDB::idList(@_);

    if (scalar @ids == 0) {
        return (SCSDB::errorCode('OKAY'), []);
    }

    $sql .= 'WHERE appliance_id in (';
    $sql .= join(', ', @ids);
    $sql .= ') ';

    $sql .= "ORDER BY appliance_id";

    return (SCSDB::errorCode('OKAY'), SCSDB::getResultHashes($sql));
}

#
# getApplianceEmail
#
# returns the email contact for a specified ip
#

sub getApplianceEmail
{
    my $ipaddr = shift @_;
    return (SCSDB::errorCode('EPARM', 'IP address is null'), '')
	unless (defined($ipaddr) && $ipaddr ne "");

    my $sql = "SELECT appliance_id, email_addr FROM mgmt_appliance ";
    $sql .= "WHERE ip_address = '$ipaddr'";

    my $res = SCSDB::getResultHash($sql);
    unless (defined($res) && 
	    ref($res) eq "HASH" &&
	    $res->{'appliance_id'}+0 > 0) {
	return (SCSDB::errorCode('EINVAL',
				"No host with ip '$ipaddr'"), '');
    }

    return (SCSDB::errorCode('OKAY'), $res->{'email_addr'});
}

#
# getApplianceEmailById
#
# returns the email contact for a specified host id
#

sub getApplianceEmailById
{
    my $id = shift @_;
    return (SCSDB::errorCode('EPARM', 'Invalid host id'), '')
	unless (defined($id) && $id+0 > 0);

    my $sql = "SELECT ip_address, email_addr FROM mgmt_appliance ";
    $sql .= 'WHERE appliance_id = ' . int($id+0);

    my $res = SCSDB::getResultHash($sql);
    unless (defined($res) && 
	    ref($res) eq "HASH" &&
	    $res->{'ip_address'} ne "") {
	return (SCSDB::errorCode('EINVAL', 
				"Invalid host id '$id'"), '');
    }
    
    return (SCSDB::errorCode('OKAY'), $res->{'email_addr'});
}

#
# getAppliancesByIpAddress
#
# returns appliance_id, ip_address and hostname for specified ips
#

sub getAppliancesByIpAddress
{
    my $sql = "SELECT appliance_id, ip_address, host_name FROM ";
    $sql .= "mgmt_appliance ";

    my @ips = ();
    my $ip = shift @_;

    push (@ips, "'$ip'") if (defined($ip) && $ip ne "");
	
    foreach $ip (@_) {
	push (@ips, "'$ip'") if (defined($ip) && $ip ne "");
    }

    if (scalar(@ips) > 0) {
	$sql .= "WHERE ip_addres in (";
	$sql .= join(', ',@ips);
	$sql .= ") ";
    }

    $sql .= "ORDER BY appliance_id";

    return (SCSDB::errorCode('OKAY'), SCSDB::getResultHashes($sql));
}

#
# addAppliance
#
# adds a host 
#

sub addAppliance ($$$$$$)
{
  my $ipaddr = shift @_;
  return SCSDB::errorCode('EPARM', 
			 'IP address is null')
      unless (defined($ipaddr) && $ipaddr ne "");

  my $id = getApplianceId($ipaddr);
  return SCSDB::errorCode('EINVAL', 'Host exists')
      if (defined($id) && $id+0 > 0);

  my $hname = shift @_;
  my $username = shift @_;
  return SCSDB::errorCode('EPARM', 
			 'Username is null')
      unless (defined($username));

  my $password = shift @_;
  return SCSDB::errorCode('EPARM', 
			 'Password is null')
      unless (defined($password));

  my $email = shift @_;
  $email = "" unless (defined($email));

  my $desc = shift @_;
  $desc = "" unless (defined($desc));

  my %hostHash = ( 'ip_address'  => "'$ipaddr'",
		   'host_name'   => "'$hname'",
#		   'mac_address' => "'ff:ff:ff:ff:ff:ff'",
		   'build_id'    => 1,
#		   'status'      => "'U'",
		   'reason'      => "",
		   'email_addr'  => "'$email'",
		   'username'    => "'$username'",
		   'password'    => "'$password'",
		   'description' => "'$desc'" );

  return SCSDB::insertHash('mgmt_appliance', 'appliance_id', 
			  \%hostHash);
}

#
# addScannedAppliance
#
# Adds a host that was automatically scanned, and hence
# no detailed login or build information is yet available.
#
sub addScannedAppliance ($$$)
{
  my ( $ipaddr, $hostname, $fingerprint ) = @_;
  return SCSDB::errorCode('EPARM', 
			 'IP address is null')
      unless (defined($ipaddr) && $ipaddr ne "");

  my $id = getApplianceId($ipaddr);
  return SCSDB::errorCode('EINVAL', 'Host exists')
      if (defined($id) && $id+0 > 0);

  return SCSDB::errorCode('EPARM',
                          'Bad fingerprint')
      if ( not _isValidFingerprint($fingerprint) );

  my %hostHash = ( 'ip_address'  => $ipaddr,
                   'host_name'   => $hostname,
		   'status'      => 'S',
                   'fingerprint' => $fingerprint );

  return SCSDB::insertHash('mgmt_appliance', 'appliance_id', 
			  \%hostHash);
}
#
# removeApplianceRecords
#
# removes host records for the specified host id
#
# WARNING: THIS NEEDS A TXN!
#

sub removeApplianceRecords
{
    my $id = shift @_;
    return SCSDB::errorCode('EPARM', "Invalid host id")
	unless (defined($id) && $id+0 > 0);

    my @tables = ();
    my $table = "";
    my $err = "";
    my $sql = "";

    # if the second arg is all then add mgmt_appliance as the first
    # table to delete this host from

    my $all = shift @_;
    push(@tables, 'mgmt_appliance')
	if (defined($all) && $all eq 'all');

    push (@tables, 
	  'mgmt_installed_rpm',
	  'mgmt_view_to_appliance',
	  'mgmt_schedule_appliance' );

    foreach $table (@tables) {
	$sql = "DELETE FROM $table WHERE appliance_id = ";
	$sql .= int($id+0);
	if (SCSDB::runCommand($sql)) {
	    $err = "Cannot delete host '$id' from $table";
	    last;
	}
    }

    return SCSDB::errorCode('EDBERR', $err) if ($err ne "");
    
    return SCSDB::errorCode('OKAY');
}

#
# removeAppliance
#
# removes the specified system
#

sub removeAppliance
{
    my $id = shift @_;
    return SCSDB::errorCode('EPARM', "Invalid host id")
	unless (defined($id) && $id+0 > 0);
    
    return removeApplianceRecords($id, 'all');
}

#
# getAppliances
#
# returns hosts with a specified status (or all if no status is
# specified)
#

sub getAppliances
{
    my $status = shift @_ || 0;

    my $sql = "SELECT appliance_id FROM mgmt_appliance ";
    $sql .= "WHERE status = '$status'" 
	if ($status);

    return (SCSDB::errorCode('OKAY'), SCSDB::getResultList($sql));
}

#
# getApplianceDetail
#
# returns details for the specified host ids
#

sub getApplianceDetail
{
    my $sql = "SELECT a.*, b.build_class, b.build_type FROM ";
    $sql .= "mgmt_appliance AS a, mgmt_build AS b WHERE ";
    $sql .= "a.build_id = b.build_id ";

    my @ids = SCSDB::idList(@_);

    if (scalar(@ids) > 0) {
	$sql .= "AND appliance_id IN (";
	$sql .= join(', ', @ids);
	$sql .= ") ";
    }

    $sql .= "ORDER BY a.appliance_id";

    return (SCSDB::errorCode('OKAY'), SCSDB::getResultHashes($sql));
}

#
# setScannedAppliance
#
# Updates a host with a provided SSH public host key fingerprint
# and sets it to scanned ('S') status.
#
sub setScannedAppliance
{
    my ($id, $fingerprint) = @_;

    return SCSDB::errorCode( 'EPARM', 'Invalid host id' )
        unless ( defined($id) && ( $id+0 > 0 ) );

    return SCSDB::errorCode( 'EPARM', 'Bad fingerprint format' )
        unless ( _isValidFingerprint($fingerprint) );

    my $sql = "UPDATE mgmt_appliance SET status = 'S', ";
    $sql   .= "fingerprint = '$fingerprint' WHERE appliance_id = '$id'";

    return SCSDB::errorCode('EDBERR', 'Cannot change status')
        if (SCSDB::runCommand($sql));

    return SCSDB::errorCode('OKAY');
}

#
# setApprovedAppliance
#
# sets a host's state to approved
#

sub setApprovedAppliance ($$)
{
    my ($id, $fingerprint) = @_;

    return SCSDB::errorCode( 'EPARM', 'Invalid host id' )
        unless ( defined($id) && ( $id+0 > 0 ) );

    return SCSDB::errorCode( 'EPARM', 'Bad fingerprint format' )
        unless ( _isValidFingerprint($fingerprint) );

    my $sql = "UPDATE mgmt_appliance SET status = 'A', ";
    $sql   .= "fingerprint = '$fingerprint' WHERE appliance_id = '$id'";

    return SCSDB::errorCode('EDBERR', 'Cannot change status')
        if (SCSDB::runCommand($sql));

    return SCSDB::errorCode('OKAY');
}

#
# setManagedAppliance
#
# sets a host's state to managed
#
sub setManagedAppliance
{
    my ($id, $build_type, $build_class, $build_id, $sql);

    $id = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid host id')
	unless (defined($id) && $id+0 > 0);

    $build_type = shift @_;
    $build_type = "<UnknownType>" 
	unless (defined($build_type) && $build_type ne "");

    $build_id = Builds::getBuildIdByType($build_type);

    unless ($build_id) {
	$build_class = $build_type;
	$build_class = "Sun $build_type"
	    if ($build_type =~ /^[1-9]/);
	$build_id = Builds::addBuild($build_type, $build_class);
    }

    return SCSDB::errorCode('EDBERR', 'Cannot determine build id')
	unless (defined($build_id) && $build_id+0 > 0);

    $sql = "UPDATE mgmt_appliance SET build_id = '$build_id'";
    $sql .= ", status = 'M' WHERE appliance_id = '$id'";

    return SCSDB::errorCode('EDBERR', "Cannot set build id")
	if (SCSDB::runCommand($sql));

    return SCSDB::errorCode('OKAY');
}

#
# setManagedFailed
#
# sets a hosts' status to management failed
#

sub setManagedFailed
{
    my $id = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid host id')
	unless (defined($id) && $id+0 > 0);

    my $reason = shift @_;
    $reason = "Unknown" unless (defined($reason) && $reason ne "");

    my $sql = "UPDATE mgmt_appliance SET status = 'F', reason = ";
    $sql .= "'$reason', password = '', username = '' WHERE ";
    $sql .= "appliance_id = '$id'";

    return SCSDB::errorCode('EDBERR', "Set status F failed for '$id'")
	if (SCSDB::runCommand($sql));

    return removeApplianceRecords($id);
}

#
# resetUnmanagedStatus
#
# returns a system to unmanaged status
#

sub resetUnmanagedStatus
{
    my $id = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid host id')
	unless (defined($id) && $id+0 > 0);

    my $username = shift @_;
    $username = "" unless (defined($username));

    my $passwd = shift @_;
    $passwd = "" unless (defined($passwd));

    my $reason = shift @_;
    $reason = "" unless (defined($reason));

    
    my $sql = "UPDATE mgmt_appliance SET username = '$username', ";
    $sql .= "password = '$passwd', status = 'U', reason = ";
    $sql = "'$reason' WHERE appliance_id = '$id'";

    return SCSDB::errorCode('EDBERR', 
			   "Cannot set host id '$id' unmanaged")
	if (SCSDB::runCommand($sql));

    return SCSDB::errorCode('OKAY');
}

#
# setHostname
#
# set the hostname for the specified host id
#

sub setHostname
{
    my $id = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid host id')
	unless (defined($id) && $id+0 > 0);

    my $newHname = shift @_;
    return SCSDB::errorCode('EPARM', 'Hostname is null')
	unless (defined($newHname) && $newHname ne "");

    my $sql = "UPDATE mgmt_appliance SET host_name = '$newHname' ";
    $sql .= "WHERE appliance_id '$id'";

    return SCSDB::errorCode('EDBERR', 
			   "Cannot set hostname for host id '$id'")
	if (SCSDB::runCommand($sql));
  
    return SCSDB::errorCode('OKAY');
}

#
# setMacAddress
#
# sets the MAC address for the specified host id
#

sub setMacAddress
{
    my $id = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid host id')
	unless (defined($id) && $id+0 > 0);

    my $maddr = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid mac address')
	unless (defined($maddr) && 
		$maddr =~ /^([0-9a-fA-F]?[0-9a-fA-F]\:?){5}[0-9a-fA-F]?[0-9a-fA-F]$/);

    my $sql = "UPDATE mgmt_appliance SET mac_address = '$maddr' ";
    $sql .= "WHERE appliance_id = '$id'";

    return SCSDB::errorCode('EDBERR', 
			   "Cannot update mac addr for '$id'")
	if (SCSDB::runCommand($sql));

    return SCSDB::errorCode('OKAY');
}

#
# setDescription
#
# sets the Description for a given host id
#

sub setDescription
{
    my $id = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid host id')
	unless (defined($id) && $id+0 > 0);

    my $desc = shift @_;
    $desc = "" unless (defined($desc));

    my $sql = "UPDATE mgmt_appliance SET description = '$desc' ";
    $sql .= "WHERE appliance_id = '$id'";

    return SCSDB::('EDBERR', "Cannot update desc for '$id'")
	if (SCSDB::runCommand($sql));

    return SCSDB::errorCode('OKAY');
}

#
# setEmail
#
# sets the email contact for a given host
#

sub setEmail
{
    my $id = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid host id')
	unless (defined($id) && $id+0 > 0);

    my $email = shift @_;
    $email = "" unless (defined($email));

    my $sql = "UPDATE mgmt_appliance SET email_addr = '$email' ";
    $sql .= "WHERE appliance_id = '$id'";

    return SCSDB::('EDBERR', "Cannot update email for '$id'")
	if (SCSDB::runCommand($sql));

    return SCSDB::errorCode('OKAY');
}

##
# getAppliancesByBuildAndStatus
##
#
# If Build ID has a type, then just use that Build ID
# else if type is null, then find all build id's with this class
# So you either have one single build id or a list, and return the list
# of appliances for all of them.
#
##
sub getAppliancesByBuildAndStatus
{
  my ($buildId, $status) = @_;

  my $build;
  return SCSDB::errorCode('EDBERR', 'build ID = ' . $buildId . ' does not exist')
    if (!($build = SCSDB::getResultHash('select build_type'
                                          . ', build_class'
                                      . ' from mgmt_build'
                                     . ' where build_id = ' . $buildId)));
  my $buildType  = $build->{'build_type'};
  my $buildClass = $build->{'build_class'};

  # If null type then get all appliances with this class
  if ($buildType eq '')
  {
    return (SCSDB::errorCode('OKAY'),
            SCSDB::getResultHashes('select a.appliance_id'
                                     . ', a.ip_address'
                                 . ' from mgmt_appliance as a'
                                     . ', mgmt_build as b'
                                . ' where a.build_id = b.build_id'
                                  . ' and b.build_class = "' . $buildClass . '"'
                                  . ' and a.status = "' . $status . '"'));
  }

  # Just return appliances with this specific build ID
  return (SCSDB::errorCode('OKAY'),
          SCSDB::getResultHashes('select appliance_id'
                                   . ', ip_address'
                               . ' from mgmt_appliance'
                              . ' where build_id = ' . $buildId
                                . ' and status = "' . $status . '"'));
}

##
# getAppliancesByStatus
##
sub getAppliancesByStatus
{
  my $status = shift || '';

  return (SCSDB::errorCode('OKAY'),
          SCSDB::getResultHashes('select appliance_id'
                                   . ', ip_address'
                                   . ', build_id'
                               . ' from mgmt_appliance'
                              . ' where status = "' .  $status . '"'));
}

##
# getApplianceStatus
##
sub getApplianceStatus
{
    my $id = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid host id')
	unless (defined($id) && $id+0 > 0);

    my $sql = "SELECT status FROM mgmt_appliance WHERE appliance_id";
    $sql .= " = '$id'";

    return SCSDB::getResult($sql);
}

#
# setApplianceStatus
#
# sets the status code for a given host id
#

sub setApplianceStatus
{
    my $id = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid host id')
	unless (defined($id) && $id+0 > 0);

    my $status = shift @_;
    return SCSDB::errorCode('EPARM', 'Status is null')
	unless (defined($status) && $status ne "");

    my $sql = "UPDATE mgmt_appliance SET status = '$status' WHERE ";
    $sql .= "appliance_id = '$id'";

    return SCSDB::errorCode('EDBERR', "Cannot set status for id '$id'")
	if (SCSDB::runCommand($sql));

  return SCSDB::errorCode('OKAY');
}

#
# setPasswd
#
# sets the password for a given host id
#

sub setPasswd
{
    my $id = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid host id')
	unless (defined($id) && $id+0 > 0);
    
    my $passwd = shift @_;
    $passwd = "" unless (defined($passwd));
    
    
    my $sql = "UPDATE mgmt_appliance SET username = '', password = ";
    $sql .= "'$passwd' WHERE appliance_id = '$id'";

    return SCSDB::errorCode('EDBERR', 
			   "Cannot update user and pass for '$id'")
	if (SCSDB::runCommand($sql));

    return SCSDB::errorCode('OKAY');    
}

#
# updateIpAddress
#
# updates the IP addr for a given host id
#

sub updateIpAddress
{
    my $id = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid host id')
	unless (defined($id) && $id+0 > 0);

    my $ipaddr = shift @_;
    return SCSDB::errorCode('EPARM', 'IP address is null')
	unless (defined($ipaddr) && $ipaddr ne "");

    my $sql = "UPDATE mgmt_appliance SET ip_address = '$ipaddr' ";
    $sql .= "WHERE appliance_id = '$id'";

    return SCSDB::runCommand($sql);
}

#
# getModuleAppliances
#
# ???
#

sub getModuleAppliances
{
  my $moduleName = shift @_;
  return SCSDB::errorCode('EPARM', 'Module name is null')
      unless (defined($moduleName) && $moduleName ne "");

  my $sql = "SELECT DISTINCT i.appliance_id FROM mgmt_installed_rpm";
  $sql .= " i LEFT JOIN mgmt_mapp_client_rpm r ON ";
  $sql .= "i.mapp_client_rpm_id = r.mapp_client_rpm_id LEFT JOIN ";
  $sql .= "mgmt_mapp m ON r.mapp_id = m.mapp_id WHERE m.mapp_name ";
  $sql .= "= '$moduleName' AND i.status = 'P'";

  return SCSDB::getResultList($sql);
}

#
# Utility Routines
#

sub _isValidFingerprint ($)
{
    return $_[0] =~ /^(?:[0-9a-f][0-9a-f]:){15}[0-9a-f][0-9a-f]$/;
}

1;
