#!/usr/bin/perl -wT
# $Id: init_tool.pl,v 1.4 2004/04/27 22:41:04 sv122840 Exp $
# init_tool.pl - an init script tool for scs
# Copyright 2004 Sun Microsystems, Inc., All rights reserved.
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.

use lib "/scs/lib/perl5";
use strict;

use InitUtil;

# global variables

my $EXITCODE = 0;
my $INITBASEDIR = '/scs/etc/init.d/';
my $CHKCONFIG = '/sbin/chkconfig';
my $CP = "";
my %MODES = ( 'INSTALL' => 'add',
	      'REMOVE'  => 'del',
	      'ENABLE'  => 'enable',
	      'DISABLE' => 'disable',
	      'EXISTS'  => 'type',
	      'HELP'    => 'help',
	      'START'   => 'start',
	      'STOP'    => 'stop',
	      'RESTART' => 'restart' );
my $MODE = shift(@ARGV);

# only allow root to run this script

if ($< > 0) {
    &printError("Only root can run $0");
    exit(1);
}

# secure the env to pass taint checks

$ENV{'PATH'} = "/sbin:/bin:/usr/bin:/usr/sbin";
delete @ENV{'IFS', 'CDPATH', 'ENV', 'BASH_ENV'};

CASE_MODE : {

    unless (defined($MODE) && $MODE ne "") {
	&printUsage();
	$EXITCODE = 1;
	last CASE_MODE;
    }

    if ($MODE eq $MODES{'INSTALL'}) {
	$EXITCODE = 1 if (&doInitInstall($ARGV[0]));
	last CASE_MODE;
    }

    if ($MODE eq $MODES{'REMOVE'}) {
	$EXITCODE = 1 if (&doInitUninstall($ARGV[0]));
	last CASE_MODE;
    }

    if ($MODE eq $MODES{'ENABLE'} || $MODE eq $MODES{'DISABLE'}) {
	$EXITCODE = 1 if (&doCheckConfig($MODE, $ARGV[0]));
	last CASE_MODE;
    }

    if ($MODE eq $MODES{'START'} || $MODE eq $MODES{'STOP'}) {	
	$EXITCODE = 1 if (InitUtil::doInit($MODE, @ARGV));
	last CASE_MODE;
    }

    if ($MODE eq $MODES{'RESTART'}) {
	$EXITCODE = 1 if (InitUtil::restart(@ARGV));
	last CASE_MODE;
    }

    if ($MODE eq $MODES{'EXISTS'}) {
	my ($rc, $initScript) = InitUtil::getInitScript(@ARGV);
	if ($rc || ! -f $initScript) {
	    $EXITCODE = 1;
	} else {
	    print "$initScript\n";
	}
	last CASE_MODE;
    }

    if ($MODE eq $MODES{'HELP'} || 
	$MODE =~ /^-[-]?([hH]|[hH]elp|\?)$/) {
	&printUsage();
	last CASE_MODE;
    }

    # default
    
    &printError("Unknown mode: '$MODE'");
    &printUsage();
    $EXITCODE = 1;
    last;
}

exit($EXITCODE);

#
# subroutines
#

#
# doInitInstall
#
# installs the specified init script in the system's init script dir
#

sub doInitInstall
{
    my $scriptName = shift @_;    
    return -1 unless (defined($scriptName) && $scriptName ne "");
    
    my $oldAbsPath = ($scriptName =~ /^\// ? "" : $INITBASEDIR);
    $oldAbsPath .= $scriptName;

    if ($oldAbsPath =~ /^(\/.*)$/) {
	$oldAbsPath = $1;
    } else {
	return -1;
    }

    unless (defined($oldAbsPath) && &isReadableFile($oldAbsPath)) {
	&printError("Invalid script '$scriptName'");
	return -1;
    }
    
    my $initDir = InitUtil::getInitDir();
    unless (defined($initDir) && -d $initDir) {
	&printError("Cannot determine system init directory");
	return -1;
    }
    
    my $newAbsPath = $oldAbsPath;
    $newAbsPath =~ s/^.*\///;
    if ($newAbsPath eq "") {
	&printError("Cannot create system copy of '$scriptName'");
	return -1;
    } 

    $newAbsPath = "$initDir/$newAbsPath";

    return 0 if (&isReadableFile($newAbsPath));

    return 0 if (symlink($oldAbsPath, $newAbsPath));

    &printError("Cannot install '$scriptName'");

    return -1;
}

#
# doInitUninstall
#
# removes the specified init script from the system's init script dir
#

sub doInitUninstall
{
    my $scriptName = shift @_;
    return -1 unless (defined($scriptName) && $scriptName ne "");
	
    my ($rc, $initScript) = InitUtil::getInitScript($scriptName);
    if ($rc || ! -f $initScript) {
	&printError("Cannot find init script for '$scriptName'");
	return -1;
    }

    if ($initScript =~ /^(\/.*)$/) {
	$initScript = $1;
    } else {
	return -1;
    }

    unlink($initScript);    
    return 0;
}

#
# doCheckConfig
#
# enables/disables an init script using chkconfig
#

sub doCheckConfig
{
    my $mode = shift @_;    
    $mode = $MODES{'ENABLE'} unless (defined($mode));

    my $scriptName = shift @_;
    return -1 
	unless (defined($scriptName) && $scriptName ne "");

    $scriptName =~ s/^.*\///;
    if ($scriptName eq "") {
	&printError("Invalid script");
	return -1;
    }

    if ($scriptName =~ /^([a-zA-Z0-9\_\-\.]+)$/) {
	$scriptName = $1;
    } else {
	return -1;
    }

    my $chkconfigCmd = "";
    my $ckconfigRc = 0;

    if ($mode eq $MODES{'ENABLE'}) {

	# add the init script

	$chkconfigCmd = "$CHKCONFIG --add $scriptName";	
	$ckconfigRc = system($chkconfigCmd);
	$ckconfigRc /= 256;
	if ($ckconfigRc) {
	    &printError("Cannot enable '$scriptName'");
	    return -1;
	}

	# turn the init script on 
	
	$chkconfigCmd = "$CHKCONFIG $scriptName on";	
	$ckconfigRc = system($chkconfigCmd);
	$ckconfigRc /= 256;	
	if ($ckconfigRc) {
	    &printError("Cannot enable '$scriptName'");
	    return -1;
	}

	return 0;
    }

    if ($mode eq $MODES{'DISABLE'}) {

	# turn off the init script

	$chkconfigCmd = "$CHKCONFIG $scriptName off";	
	$ckconfigRc = system($chkconfigCmd);
	$ckconfigRc /= 256;	
	if ($ckconfigRc) {
	    &printError("Cannot disable '$scriptName'");
	    return -1;
	}

	# delete the init script

	$chkconfigCmd = "$CHKCONFIG --del $scriptName";	
	$ckconfigRc = system($chkconfigCmd);
	$ckconfigRc /= 256;
	if ($ckconfigRc) {
	    &printError("Cannot enable '$scriptName'");
	    return -1;
	}

	return 0;
    }

    &printError("Invalid mode '$MODE'");
    return -1;
}

sub printUsage
{
    my $pgm = $0;
    $pgm =~ s/^.*\///;

    print <<EOUSAGE;
Usage  : $pgm $MODES{'INSTALL'}     [script] \tInstall the init script
         $pgm $MODES{'REMOVE'}     [script] \tRemoves the init script
         $pgm $MODES{'ENABLE'}  [script] \tEnables the init script
         $pgm $MODES{'DISABLE'} [script] \tDisables the init script
         $pgm $MODES{'START'}   [script] \tStarts the init script
         $pgm $MODES{'STOP'}    [script] \tStops the init script
	 $pgm $MODES{'EXISTS'}    [script] \tReturns full path to the
                                      \tinit script
         $pgm $MODES{'RESTART'} [script] [delay]
                                  \tRestarts the init script
EOUSAGE
}

sub printError
{
    print STDERR "ERROR: @_\n"
}

sub isReadableFile
{
    my $file = shift @_;
    return (defined($file) && -f $file && -r $file);
}

sub cpFile
{
    my $oldPath = shift @_;
    return -1 
	unless (defined($oldPath) && &isReadableFile($oldPath));

    if ($oldPath =~ /^(\/.+)$/) {
	$oldPath = $1;
    } else {
	return -1;
    }

    my $newPath = shift @_;
    return -1 unless (defined($newPath));

    if ($newPath =~ /^(\/.+)$/) {
	$newPath = $1;
    } else {
	return -1;
    }
        
    my $cpCmd = "$CP '$oldPath' '$newPath'";
    my $cpRc = system($cpCmd);
    $cpRc /= 256;

    return $cpRc;
}
