#!/usr/bin/perl -w
# $Id: InitUtil.pm,v 1.5 2004/07/15 17:38:25 jb122832 Exp $
# InitUtil.pm - init script utilities
# Copyright 2004 Sun Microsystems, Inc., All rights reserved.
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.

package InitUtil;

use strict;
use lib '/scs/lib/perl5';
use Carp;

my @INITDIRS = ( '/etc/init.d', '/etc/rc.d/init.d' );
my %INITMODES = ( 'START'       => 'start',
		  'STOP'        => 'stop',
		  'HARDSTOP'    => 'hardstop',
		  'RESTART'     => 'restart' );

my $INITLCKDIR = "/var/lock/subsys";
my $INITRUNDIR = "/var/run";

#
# getInitDir
#
# determines the path of init script dir for the system
#

sub getInitDir
{
    my $i = 0;
    my $dir = "";

    for ($i = 0; $i < scalar(@INITDIRS); $i++) {
	if (-d $INITDIRS[$i]) {
	    $dir = $INITDIRS[$i];
	    last;
	}
    }

    return $dir;
}

#
# getInitScript
#
# returns the full path to the specified init script
#

sub getInitScript
{
    my $scriptName = shift @_;
    return (-1, "")
	unless (defined($scriptName) && $scriptName ne "");
	
    $scriptName =~ s/^.*\///;
    return (-1, "") if ($scriptName eq "");
    
    my $initDir = &getInitDir();
    return (-1, "") unless (defined($initDir) && -d $initDir);

    my $absPath = "$initDir/$scriptName";
    return (0, $absPath) if (-f $absPath && -x $absPath);

    return (-1, "");
}

#
# doInit
#
# starts or stops the specified init script and waits for a
# specified number of seconds after performing the action
#
# args: $mode        -> start/stop/hardstop
#       $scriptName  -> name of the script
#       $delay       -> wait time after performing the action
#

sub doInit
{
    my $hardStop = 0;
    
    my $mode = shift @_;
    $mode = $INITMODES{'START'} unless (defined($mode));

    CASE_INITMODE : {

	if ($mode eq $INITMODES{'HARDSTOP'}) {
	    $hardStop = 1;
	    $mode = $INITMODES{'STOP'};
	}

	last CASE_INITMODE if ($mode eq $INITMODES{'START'} ||
			       $mode eq $INITMODES{'STOP'});
	
	# default 

	cluck("Invalid Mode, $mode, assuming $INITMODES{'START'}");
	$mode = $INITMODES{'START'};
    }

    my $scriptName = shift @_;
    return -1 unless (defined($scriptName) && $scriptName ne "");

    my $delay = shift @_;
    $delay = 0 unless (defined($delay) && int($delay+0) > 0);

    my ($rc, $initScript) = &getInitScript($scriptName);
    if ($rc || ! -x $initScript) {
	carp("Cannot find init script for '$scriptName'");
	return -1;
    }

    if ($initScript =~ /^(\/.*)$/) {
	$initScript = $1;
    } else {
	return -1;
    }

    my $initRc = -256;

    if ($mode eq $INITMODES{'STOP'}) {
	$initRc = system("$initScript stop");
	$initRc /= 256;

	my $stopped = ($initRc ? 0 : 1);

	my $lckF = "$INITLCKDIR/$scriptName";
	my $pidF = "$INITRUNDIR/$scriptName.pid";
	
	# handle a left over pid file

	if (-r $pidF) {
	   	    
	    # if the process is stopped remove the pidf, otherwise
	    # read the pid file, figure out the pid and remove the
	    # pid file if the pid doesn't exist. If the pid does 
	    # exist, then kill -9 the pid if hard stop is requested.
	    # NOTE: If the proc has died, but the pid has been 
	    # recycled, we might kill the wrong process

	    unless ($stopped) {
		my $pid = 0;
		
		if (open(PIDF, $pidF)) {
		    $pid = <PIDF>;
		    close(PIDF);
		    if ($pid =~ /^([0-9]+)$/) {
			$pid = $1;
		    } else {
			$pid = 0;
		    }
		}

		my $intPid = int($pid+0);
		
		if ($intPid > 1 && kill(0, $intPid)) {
		    if ($hardStop) {
			kill(9, $intPid);
			$stopped = 1;
		    } else {
			carp ("'$scriptName' still running");
		    }
		} else {
		    $stopped = 1;
		}
	    }
	    
	    unlink($pidF) if ($stopped);
	}

	if (-r $lckF) {
	    if ($stopped) {
		unlink($lckF);
	    } else {
		carp("Not removing lock file '$lckF'");
	    }
	}
    } else {
	$initRc = system("$initScript start");
	$initRc /= 256;
	carp ("'$scriptName' may not have started") if ($initRc);
    }

    # only wait if stop/start was successful
    
    sleep(int($delay+0)) if ($initRc == 0 && $delay);

    return $initRc;
}

#
# start
#
# starts the specified init script
#

sub start
{
    return doInit($INITMODES{'START'},@_);
}

#
# stop
#
# stops the specified init script
#

sub stop
{
    return doInit($INITMODES{'STOP'},@_);
}

#
# hardstop
#
# performs a hard stop (kill -9 if needed) of the specified init
# script
#

sub hardstop
{
    return doInit($INITMODES{'HARDSTOP'},@_);
}

#
# restart
#
# restarts the specified init script by calling stop and start.
#

sub restart
{
    my $scriptName = shift @_;
    return -1 unless (defined($scriptName) && $scriptName ne "");
    
    my $delay = shift @_;
    $delay = 10 unless (defined($delay) && int($delay+0) > 0);

    my $stopMode = shift @_;
    if (defined($stopMode) && $stopMode) {
	hardstop($scriptName, $delay);
    } else {
	stop($scriptName, $delay);
    }
    
    return start($scriptName, $delay);
}

1;
