# $Id: Flock.pm,v 1.4 2004/07/22 21:00:56 jb122832 Exp $

# Copyright (c) 2003 Sun Microsystems, Inc. All rights reserved
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.

#
# File locking functions
# written by Duncan Laurie <duncan@cobaltnet.com> (Cobalt::Util)
# hacked up again by Jeff Bilicki <jeff.bilicki@sun.com>
# Viva la SPECIAL SAUCE!

package Flock;

use strict;
use POSIX;
use Fcntl qw(F_WRLCK F_RDLCK F_UNLCK);
use FileHandle;

use vars qw(@ISA @EXPORT @EXPORT_OK);
require Exporter;
@ISA = qw(Exporter);
@EXPORT = qw(FILE_lock_r FILE_lock_w FILE_lock_rw FILE_unlock FILE_unlock_rw);
@EXPORT_OK = qw();

use vars qw($DEBUG $lock_dir);
$DEBUG=0;
$lock_dir = '/etc/locks';

# inputs:  filein = name of file to lock (& full path)
# return:  two filehandles, the first for the input and the second for output
#
# for writing, while opening the $file for reading
# 
sub FILE_lock_rw
{
    my $filein = shift;
    local *FHIN = shift;
    local *FHOUT = shift;
    my $LOCK_EX = 2;
    my $Lockdir = "/etc/locks";

    # extract filename from fully-qualified filename
    my @fary = split("/", $filein);
    my $ret = pop(@fary);
    my $fileout = "$Lockdir/".pop(@fary)."_$ret";

    # we want to have exclusive locking on the input file
    print STDERR "FILE_lock_rw:  opening $filein... " if ($DEBUG);
    unless(open(FHIN, "< $filein")) {
		print STDERR "error!\n" if ($DEBUG);
		return -1;
    }
    print STDERR "done\n" if ($DEBUG);

    print STDERR "FILE_lock_rw:  locking $filein... " if ($DEBUG);
    unless(flock(FHIN,$LOCK_EX)) {
		print STDERR "error!\n" if ($DEBUG);
		return -1;
    }
    print STDERR "done\n" if ($DEBUG);

    print STDERR "FILE_lock_rw:  opening $fileout... " if ($DEBUG);
    unless(open(FHOUT, "> $fileout")) {
		print STDERR "error!\n" if ($DEBUG);
		return -1;
    }
    print STDERR "done\n" if ($DEBUG);

    print STDERR "FILE_lock_rw:  locking $fileout... " if ($DEBUG);
    unless(flock(FHOUT,$LOCK_EX)) {
		print STDERR "error!\n" if ($DEBUG);
		return -1;
    }
    print STDERR "done\n" if ($DEBUG);

    return 1;
}

# inputs:  file = name of file to lock (& full path)
# return:  filehandle
# opens the $file for reading
sub FILE_lock_w
{
    my $file = shift;
    local *FH = shift;
    my $LOCK_EX = 2;

    # open the file
    print STDERR "FILE_lock_w:  opening $file... " if ($DEBUG);
    unless(open(FH, "> $file")) {
		print STDERR "error!\n" if ($DEBUG);
		return -1;
    }
    print STDERR "done\n" if ($DEBUG);

    # lock the file
    print STDERR "FILE_lock_w:  locking $file... " if ($DEBUG);
    unless(flock(FH,$LOCK_EX)) {
		print STDERR "error!\n" if ($DEBUG);
		return -1;
    }
    print STDERR "done\n" if ($DEBUG);

    return 1;
}

# inputs:  file = name of file to lock (& full path)
# return:  filehandle
# opens the $file for reading
sub FILE_lock_r
{
    my $file = shift;
    local *FH = shift;
    my $LOCK_EX = 2;

    # open the file
    print STDERR "FILE_lock_r:  opening $file... " if ($DEBUG);
    unless(open(*FH,"< $file")) {
		print STDERR "error!\n" if ($DEBUG);
		return -1;
    }
    print STDERR "done\n" if ($DEBUG);

    # lock the file
    print STDERR "FILE_lock_r:  locking $file... " if ($DEBUG);
    unless(flock(FH,$LOCK_EX)) {
		print STDERR "error!\n" if ($DEBUG);
		return -1;
    }
    print STDERR "done\n" if ($DEBUG);

    return 1;
}

# inputs:  array of filehandles to unlock and close
# return:  1 if success
sub FILE_unlock
{
    my (@fhs) = @_;
    my ($fh);
    my $LOCK_UN = 8;

    foreach $fh (@fhs) {
		print STDERR "FILE_unlock:  unlocking filehandle >$fh<... " if ($DEBUG);
		flock($fh, $LOCK_UN);
		close($fh);
		print STDERR "done\n" if ($DEBUG);
    }

    return 1;
}

#
# Cobalt::Util::FILE_unlock_rw
#
# inputs:  $filein = full path to file
#          $fhin = input filehandle returned by FILE_lock_rw
#          $fhout = output filehandle returned by FILE_lock_rw
# return:  1 if succes
#
sub FILE_unlock_rw
{
    my ($filein,$fhin,$fhout) = @_;
    my $LOCK_UN = 8;
    my $Lockdir = "/etc/locks";

    # extract filename from fully-qualified filename
    my @fary = split("/", $filein);
    my $ret = pop(@fary);
    my $fileout = "$Lockdir/".pop(@fary)."_$ret";

    print STDERR "FILE_unlock_rw:  unlocking $fileout... " if ($DEBUG);
    flock($fhout, $LOCK_UN);
    close($fhout);
    print STDERR "done\n" if ($DEBUG);

    print STDERR "FILE_unlock_rw:  moving $fileout to $filein... " if ($DEBUG);
    qx[ mv -f $fileout $filein ];
    print STDERR "done\n" if ($DEBUG);

    print STDERR "FILE_unlock_rw:  unlocking $filein... " if ($DEBUG);
    flock($fhin, $LOCK_UN);
    close($fhin);
    print STDERR "done\n" if ($DEBUG);

    return 1;
}

#
# Cobalt::Util::lock_services
#
# inputs: 1 to lock, 0 to unlock, 2 to check lock
#
# return:
#
sub lock_services
{
  my ($on) = @_;
  my $Servicelockfile="/etc/locks/LCK..services";

  if ( $on eq 1 ) {
    open( LOCKER, ">$Servicelockfile" );
    close( LOCKER );
  } else {
    if ( $on eq 0 ) {
      unlink( "$Servicelockfile" );
    } else {
      return ( -e $Servicelockfile );
    }
  }
}


# writes the entries in the array referenced by $arrayref to $filehandle
# in a format easily understood and returned by Cobalt::Util::read_status
# $filehandle can be an open file handle or the full path to a file
# 
sub update_status
{
	my $fh = shift;
	my $status = shift;

	# for the call to fcntl, still not sure what it could be used for
        my $buff;
	my $close_file = 0;

	# check if $fh is an open filehandle or a filename
	if (not ref $fh) {
		$fh = new FileHandle(">$fh");
		if (not defined $fh) {
			return undef;
		}
		$close_file = 1;
	}

        fcntl $fh, F_WRLCK, $buff;

        seek $fh, 0, 0;

	for my $entry (@{ $status }) {
		print $fh "$entry\n";
	}

        fcntl $fh, F_UNLCK, $buff;

	$fh->close() if $close_file;
}

# essentially slurps in an entire file and returns an array with one line per
# entry, but with the added plus of doing file locking on the file handle
# $filehandle can be an open file handle or the full path to a file to read
#
sub read_status
{
	my $fh = shift;
	my $buff;
	my $close_file = 0;

	if (not ref $fh) {
		$fh = new FileHandle($fh);
		if (not defined $fh) {
			return undef;
		}
		$close_file = 1;
	}

        fcntl $fh, F_RDLCK, $buff;

        my @info = <$fh>;

        fcntl $fh, F_UNLCK, $buff;

	$fh->close() if $close_file;

	chomp @info;
	
	return @info;
}

1;
