#!/usr/bin/perl -w

# Copyright (c) 2003 Sun Microsystems, Inc. All rights reserved
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.

#
# $Id: verifyStates.pl,v 1.1 2004/03/31 22:08:57 sv122840 Exp $
#

use strict;
use lib '/scs/lib/perl5';
use MAPP;
use Appliance;
use BDi18n;
use BDUtil;
use Progress;
use PropertyUtil;

my $sbin_dir = PropertyUtil::getProperty('dir.sbin');
my $ps_cmd = PropertyUtil::getProperty('cmd.ps');
my $grep_cmd = PropertyUtil::getProperty('cmd.grep');

my $lookForProcess = $ps_cmd . ' | ' . $grep_cmd . ' -v grep | ' . $grep_cmd;
my $importProcess  = 'devmgr.pl';
my $mappProcess    = 'moduleMgr.pl';
my $waitIterations = 6;
my $waitSeconds    = 5;
my $scheduleTool   = $sbin_dir . '/schedule.pl';

Progress::initProgress(\@ARGV);
Progress::setProcessName('Verify States');
BDi18n::setDomain('base-mgmt-db');
BDUtil::openLog('verifyStates');

#
# If an import or mappLoader process is running, then it may be in the middle
# of something.  The only way this could really happen is if the DB died, AM
# restarted it, and it was in the middle of something like an authentication
# so it doesn't care that the DB was down for a bit (auto-reconnect)
#
for (my $i=0 ; $i < $waitIterations; $i++)
{
  if (!system($lookForProcess . ' ' . $importProcess))
  {
    BDUtil::info('There is a ' . $importProcess . ' running, waiting '
                      . $waitSeconds . ' seconds.');
    sleep($waitSeconds);
  }

  if (!system($lookForProcess . ' ' . $mappProcess))
  {
    BDUtil::info('There is a ' . $mappProcess . ' running, waiting '
                      . $waitSeconds . ' seconds.');
    sleep($waitSeconds);
  }
}

if (!system($lookForProcess . ' ' . $importProcess) ||
    !system($lookForProcess . ' ' . $mappProcess))
{
  # Just figget it then
  BDUtil::info('Process still running, aborting.');
  BDUtil::closeLog();
  exit 0;
}

#
#
#
sub dbQuit
{
  BDUtil::dbError();
  BDUtil::closeLog();
  exit -1;
}

#
# Tasks that are set to status R (Running) must have had their
# process die prematurely, set to Failed.
#
my $numDefunctTasks = BDDB::getResult('select count(*)'
                                     . ' from mgmt_task'
                                    . ' where status = "R"');
if ($numDefunctTasks)
{
  my $msg = BDi18n::getMsg('taskDied');
  BDUtil::info("Setting $numDefunctTasks defunct tasks to status 'F'");
  dbQuit() if (BDDB::runCommand('update mgmt_task'
                                . ' set status = "F"'
                                   . ', message = "' . $msg . '"'
                              . ' where status = "R"'));
}
else
{
  BDUtil::info('No defunct tasks');
}

my ($rc, $appliances, $mapps);

#
# If these states are present they must be deleted:
#  D : Was in the middle of a remove, having completed the pkg uninstalls
#  U : Died in the early stages of import
#  M : Died after agent was installed but not pkgs were installed or
#      registered, best to make them re-import it.
#
BDUtil::info('Verifying appliance record integrity');
foreach my $status (qw(D U M))
{
  ($rc, $appliances) = Appliance::getAppliancesByStatus($status);
  dbQuit() if ($rc);
  if ($appliances)
  {
    foreach my $appliance (@{$appliances})
    {
      BDUtil::info('Appliance id = ' . $appliance->{'appliance_id'} . ' (' .
       $appliance->{'ip_address'} . ') has state "' . $status . '", purging.');
      dbQuit() if (Appliance::removeAppliance($appliance->{'appliance_id'}));
    }
  }
  else
  {
    BDUtil::info('No appliances left in "' . $status . '" state');
  }
}

#
# MAPPs that are in state N must have not completely installed,
# state D not completely deleted.
#
($rc, $mapps) = MAPP::getMAPPs();
dbQuit() if ($rc);
if ($mapps)
{
  BDUtil::info('Verifying MAPP integrity');
  foreach my $mapp (@{$mapps})
  {
    if (($mapp->{'status'} eq 'D') || ($mapp->{'status'} eq 'N'))
    {
      # Something bad happened during install or uninstall, remove
      BDUtil::info('MAPP id ' . $mapp->{'mapp_id'} . ' has status '
                              . $mapp->{'status'} . ', purging');
      dbQuit() if (MAPP::removeMAPP($mapp->{'mapp_id'}));
    }
    elsif ($mapp->{'status'} eq 'AF')
    {
      # Something bad happened during pkg installs after server side was ok
      # Set it to good and then let the user finish the job manually
      BDUtil::info('MAPP id ' . $mapp->{'mapp_id'}
                    . ' has status AF, setting to AS');
      dbQuit() if (MAPP::setMAPPStatus($mapp->{'mapp_id'}, 'AS'));
    }
  }
}
else
{
  BDUtil::info('No MAPPs installed');
}

#
# Schedules that are of type once that are set to run in the past
# must have not been able to run for some reason, i.e. the box being
# down, so remove these
#
BDUtil::info('Verifying schedule integrity');
my $sql = 'select schedule_id'
             . ', name'
             . ', verbose_settings'
             . ", unix_timestamp(now()) -  unix_timestamp(concat(replace(once_date, ' ', '-'), ' ', replace(once_time, ' ', ':'))) as time_diff"
         . ' from mgmt_schedule'
        . ' where type = "once"'
       . ' having time_diff > 0';

my $msg;
my $schedules = BDDB::getResultHashes($sql);
if ($schedules)
{
  foreach my $sched (@{$schedules})
  {
    system ($scheduleTool . ' delete ' . $sched->{'schedule_id'});
    $msg = BDi18n::getMsg('gotRidOfSchedule',
                          { 'name' => $sched->{'name'},
                            'settings' => $sched->{'verbose_settings'} });
    BDUtil::info($msg);
    Progress::warningEvent($msg);
  }
}
else
{
  $msg = BDi18n::getMsg('noDefunctSchedules');
  BDUtil::info($msg);
  Progress::infoEvent($msg);
}

BDUtil::closeLog();
