#!/usr/bin/perl -w

# Copyright (c) 2003 Sun Microsystems, Inc. All rights reserved
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.


#
# Schedule tool
# Author: Phil Ploquin (phil.ploquin@sun.com)
# Copyright 2001, 2002 Sun Microsystems, Inc., All rights reserved.
# $Id: schedule.pl,v 1.23 2004/11/15 19:31:18 ms152511 Exp $
#

use strict;
use lib '/scs/lib/perl5';
use SCSDB;
use Schedule;
use BDi18n;
use SysCmd;

my $runScheduleProg = '/scs/sbin/runSchedule.pl';

BDi18n::setDomain('base-mgmt-event');

printUsage() if (!defined($ARGV[0]) || ($ARGV[0] eq '-h') ||
                 ($ARGV[0] eq 'help') || ($ARGV[0] eq '--help'));
my $mode = shift @ARGV;

my %modeFunc =
(
  'add'    => \&addMode,
  'list'   => \&listMode,
  'save'   => \&saveMode,
  'clear'  => \&clearMode,
  'delete' => \&deleteMode,
  'update' => \&updateMode
);

printUsage('Unknown mode : ' . $mode)
  if (!exists($modeFunc{$mode}));
&{$modeFunc{$mode}}(\@ARGV);

exit 0;

#
#-----------------------------------------------------------
#

sub printUsage
{
  my $errorMsg = shift;

  if (defined($errorMsg))
  {
    print STDERR "\nError : " . $errorMsg . "\n";
    print STDERR "\n '$0' with no arguments for full usage.\n\n";
    exit 0;
  }

  my $p = '-N"Foo" -p/bin/foo -g"-x -y8" -a1,2,8,12,7 -n';

  print <<EOS;

Usage: $0 <mode> <arguments>

Modes:
======
help   : Print this help.
add    : Add a schedule item.
list   : List the existing schedules and their ids.
delete : Delete a schedule.
update : Change a schedule (same as delete followed by add.)

The add mode:
=============

  Usage: $0 add <type [args]> <arguments>

  Arguments:
  ----------
  -N<schedule name>     Name for the scheduled item.
  -p<program path>      Full path of program to be executed.
  -g<program args>      Constant arguments to pass to program.
  -a<host list>         Comma seperated list of host ids.
  -u                    Use hosts. Implied when -a is specified.
  -n                    Append newly imported hosts to list.
  -A                    Don't allow choosing of new hosts in higher UI.
  -R                    Read-only schedule (can't delete from higher level UI.)
  -E                    Notification email address.
  -S                    Notify when tasks start.
  -F                    Notify when tasks finish.
  -C                    Type is constant (can't change from higher level UI.)
  -H                    Hide spawned tasks from all tasks viewer.
  -M                    Control Module code name (so that if that module is
                        removed, the schedule can be automatically removed as
                        well.)

  Types:
  ------
   * once -d<yyyymmdd> -t<hhmm>

     Run once at the specified date / time.  Default date / time is now.

   * hourly -f<frequency> -s<start hour> -l<last hour> -m<minute[, .., minute]>

     Run every n hours at the specified minute(s).  Default frequency is 1,
     for once per hour.  To set for a range of hours use the first and last
     hour parameters.

   * once_a_day -t<hhmm>

     Run once every day at the specified time.

   * daily -f<frequency> -h<hour[,hour,..,hour]> -m<minute>

     Run every n days at the specified time(s).  Default frequency is 1
     for every day, set to 2 to run every two days, etc.

   * weekly -w<day[,day,..,day]> -t<hhmm>

     Run every week on the specified day(s).  Available days are:
     sun, mon, tue, wed, thu, fri, and sat.  The task will run at
     the specified time.

   * monthly -d<day num[,day num,..,day num]> -t<hhmm>

     Run every month on the specified day(s) of the month.  The task
     will run at the specified time.

  Examples:
  ---------

   Run right now:
    $0 add once $p

   Run at 3:14 PM:
    $0 add once -t1514 $p

   Run June 21st, 2004 at 8:15 AM:
    $0 add once -d20040621 -t0815 $p

   Every hour, at the beginning of the hour:
    $0 add hourly -m0 $p

   Every hour, at 15 and 45 past the hour:
    $0 add hourly -m15,45 $p

   Every 2 hours, from 4 PM to 8 AM, at 15 minutes past the hour:
    $0 add hourly -f2 -m15 -s16 -l08 $p

   Every day at 3 AM and 7 AM, 15 minutes past the hour:
    $0 add daily -h3,7 -m15 $p

   Every three days at midnight:
    $0 add daily -f3 -h0 -m0 $p
 
   Every monday, wednesday, and friday at 11:30 PM:
    $0 add weekly -wmon,wed,fri -t2330 $p

   Every 1st and 15th of the month at 2 AM:
    $0 add monthly -d1,15 -t0200 $p


The update mode:
================

  Usage: $0 update <schedule id> <add parameters>

  After specifiying the schedule id we are updating, specify the same
  parameters as for the add mode.  This is the same as running this program
  in delete mode and then add mode.

The delete mode:
================

  Usage: $0 delete <schedule id>

  Get the list of schedule id's with the list mode.

The list mode:
==============

  Usage: $0 list

  Gives a listing of all currently scheduled items.

The save mode:
==============

  Usage: $0 save

  Not necessary to call this unless the at queue and the crontab file
  have somehow gotten out of sync with the kept list of scheduled items,
  This is not dangerous for regularly scheduled items, but it will lead
  to two of each once scheduled items being in the queue if called
  frivelously.

The clear mode:
===============

  Usage: $0 clear

  Clear out the system schedules but leave them in the cache.

Notes:
======

 Do not put spaces between flags and arguments.

EOS

  exit 0;
}

#-----------------------------------------------------------------

#
#
#
sub listMode
{
  my $argv = shift;

  my $schedules = Schedule::getSchedules();
  if (!$schedules)
  {
    print "No schedules currently in the system.\n";
    return;
  }

  print " Id  Type          Name                  Settings\n";
  print " --  ----          ----                  --------\n";
  foreach my $sched (@{$schedules})
  {
    printf "%3d  %-12s  %-20s  %s\n",
           $sched->{'schedule_id'},
           $sched->{'type'},
           $sched->{'name'},
           $sched->{'verbose_settings'};
  }
}

#
#
#
sub deleteMode
{
  my $argv = shift @_;

  deleteScheduleEntry($argv->[0]);
}

#
#
#
sub updateMode
{
  my $argv = shift;
  my $id   = shift @$argv;

  printUsage('Must specify a schedule type')
    if (!defined($argv->[0]));
  my $type = shift @{$argv};

  my $rules     = buildArgRules($type);
  my $paramHash = parseArgs($type, $argv, $rules);

  # if the args parsed correctly, go ahead and delete the existing entry

  deleteScheduleEntry($id);
  addScheduleEntry($type, $paramHash, $id);
}

#
#
#
sub saveMode
{
  my $argv = shift;

  setAt();
  saveCrontab();
}

#
#
#
sub clearMode
{
  SysCmd::commitSchedulerCrontab('SCS_SCHEDULER', '');
  my $schedList = Schedule::getSchedules(1);
  return if (!$schedList);
  foreach my $sched (@{$schedList})
  {
    SysCmd::runCmd('atrm', $sched->{'at_ident'});
    Schedule::setAtIdent($sched->{'schedule_id'}, 0);
  }
}

#
#
#
sub addMode
{
  my $argv = shift;
  my $useThisId = shift || 0;

  printUsage('Must specify a schedule type')
    if (!defined($argv->[0]));
  my $type = shift @{$argv};

  my $rules     = buildArgRules($type);
  my $paramHash = parseArgs($type, $argv, $rules);
  addScheduleEntry($type, $paramHash, $useThisId);
}

#
#
#
sub getHour
{
  my $hour = shift;

  return (12, 'AM') if ($hour == 0);
  return (12, 'PM') if ($hour == 12);
  return ($hour - 12, 'PM') if ($hour > 12);
  return ($hour, 'AM');
}

#
#
#
sub onceVerb
{
  my $p = shift;

  return '' if (!$p->{'once_time'});

  my ($th, $ta) = getHour($p->{'once_time'}->[0]);
  my $str;

  if (!$p->{'once_date'})
  {
    $str = BDi18n::getMsg('once_format_time',
        { 'rtime' => ($th * 1) . ':' . $p->{'once_time'}->[1] . ' ' . $ta } );
  }
  else
  {
    my ($lastDigit) = ($p->{'once_date'}->[2] =~ /([0-9])$/);
    $lastDigit = 5  # ugh
      if (($p->{'once_date'}->[2] > 10) && ($p->{'once_date'}->[2] < 14));
    my $df = BDi18n::getMsg('month_' . ($p->{'once_date'}->[1] * 1))
           . ' ' . ($p->{'once_date'}->[2] * 1)
                      . BDi18n::getMsg('end_' . $lastDigit)
           . ' ' . $p->{'once_date'}->[0];
    $str = BDi18n::getMsg('once_format_date_and_time', { 'rdate' => $df,
          'rtime' => ($th * 1) . ':' . $p->{'once_time'}->[1] . ' ' . $ta } );
  }

  return $str;
}

#
#
#
sub hourlyVerb
{
  my $p = shift;

  my $str;
  if ($p->{'hourly_frequency'} > 1)
  {
    $str = BDi18n::getMsg('hourly_format_every_n',
                  { 'frequency' => $p->{'hourly_frequency'} } );
  }
  else
  {
    $str = BDi18n::getMsg('hourly_format_every_1');
  }

  if (($p->{'hourly_start_hour'} != 0) ||
      ($p->{'hourly_last_hour'} != 23))
  {
    my ($fh, $fa) = getHour($p->{'hourly_start_hour'});
    my ($lh, $la) = getHour($p->{'hourly_last_hour'});
    $str .= BDi18n::getMsg('hourly_format_between',
              { 'firstHour' => ($fh * 1) . ' ' . $fa,
                'lastHour'  => ($lh * 1) . ' ' . $la } );
  }

  # Want minutes to show up as 1, 12 instead of 01, 12
  for (my $i=0; $i < scalar(@{$p->{'hourly_minutes'}}); $i++)
  {
    $p->{'hourly_minutes'}->[$i] = $p->{'hourly_minutes'}->[$i] * 1;
  }
  # But want them sorted like 01, 12
  $p->{'hourly_minutes'} = [ sort { sprintf("%02d", $a) <=> sprintf("%02d", $b) } @{$p->{'hourly_minutes'}} ];

  if ((scalar(@{$p->{'hourly_minutes'}}) > 1) ||
      ($p->{'hourly_minutes'}->[0] > 0))
  {
    if (scalar(@{$p->{'hourly_minutes'}}) == 1)
    {
      $str .= BDi18n::getMsg('hourly_format_minutes_n',
                   { 'minutes' => $p->{'hourly_minutes'}->[0] } );
    }
    else
    {
      my $lastX = pop @{$p->{'hourly_minutes'}};
      $str .= BDi18n::getMsg('hourly_format_minutes_n',
            { 'minutes' => join (', ', @{$p->{'hourly_minutes'}} )
                          . ' and ' . $lastX } );
      push (@{$p->{'hourly_minutes'}}, $lastX);
    }
  }
  else
  {
    $str .= BDi18n::getMsg('hourly_format_minutes_0');
  }

  return $str;
}

#
#
#
sub once_a_dayVerb
{
  my $p = shift;

  my ($th, $ta) = getHour($p->{'once_time'}->[0]);
  my $str = BDi18n::getMsg('every_day_at',
       { 'rtime' => ($th * 1) . ':' . $p->{'once_time'}->[1] . ' ' . $ta } );

  return $str;
}

#
#
#
sub dailyVerb
{
  my $p = shift;

  my $str;
  if ($p->{'daily_frequency'} > 1)
  {
    $str = BDi18n::getMsg('daily_format_every_n_days',
                    { 'frequency' => $p->{'daily_frequency'} } );
  }
  else
  {
    $str = BDi18n::getMsg('daily_format_every_day');
  }

  $p->{'daily_hours'} = [ sort { sprintf("%02d", $a) <=> sprintf("%02d", $b) } @{$p->{'daily_hours'}} ];

  my $m = '';
  my ($hh, $ha);
  my $numHours = scalar(@{$p->{'daily_hours'}});
  for (my $i=0; $i < $numHours; $i++)
  {
    if (($i == $numHours - 1) && ($numHours > 1))
    {
      $m .= ' and ';
    }
    elsif ($i)
    {
      $m .= ', ';
    }
    ($hh, $ha) = getHour($p->{'daily_hours'}->[$i]);
    $m .= sprintf "%d:%02d %s", $hh, $p->{'daily_minute'}, $ha;
  }

  $str .= BDi18n::getMsg('daily_format_times', { 'times' => $m });

  return $str;
}

#
#
#
sub weeklyVerb
{
  my $p = shift;

  my $str = '';
  my $numDays = scalar(@{$p->{'weekly_days'}});
  for (my $i=0; $i < $numDays; $i++)
  {
    if (($i == $numDays - 1) && ($numDays > 1))
    {
      $str .= ' and ';
    }
    elsif ($i)
    {
      $str .= ', ';
    }
    $str .= BDi18n::getMsg('day_' . $p->{'weekly_days'}->[$i]) . 's';
  }

  my ($hh, $ha) = getHour($p->{'weekly_time'}->[0]);
  $str .= BDi18n::getMsg('weekly_format_time',
    { 'rtime' => sprintf "%d:%02d %s", $hh, $p->{'weekly_time'}->[1], $ha } );

  return $str;
}

#
#
#
sub monthlyVerb
{
  my $p = shift;

  my $d = '';
  my $lastDigit;

  $p->{'monthly_days'} = [ sort { sprintf("%02d", $a) <=> sprintf("%02d", $b) } @{$p->{'monthly_days'}} ];

  my $numDays = scalar(@{$p->{'monthly_days'}});
  my $day;
  for (my $i=0; $i < $numDays; $i++)
  {
    $day = $p->{'monthly_days'}->[$i];
    if (($i == $numDays - 1) && ($numDays > 1))
    {
      $d .= ' and ';
    }
    elsif ($i)
    {
      $d .= ', ';
    }
    $d .= $day;
    ($lastDigit) = ($day =~ /([0-9])$/);
    $d .= BDi18n::getMsg('end_' . $lastDigit);
  }

  my $str = BDi18n::getMsg('monthly_format_days', { 'day' => $d } );

  my ($hh, $ha) = getHour($p->{'monthly_time'}->[0]);
  $str .= BDi18n::getMsg('monthly_format_time',
    { 'rtime' => sprintf "%d:%02d %s", $hh, $p->{'monthly_time'}->[1], $ha } );

  return $str;
}

#
#
#
sub buildArgRules
{
  my $type = shift @_;

  my %globalRules = 
  (
  # flag     structure   elements  default  field name   required
  # ----     --- 0 ---   --- 1 --  -- 2 --  --- 3 ----   --- 4 --
    'N' => [ 'scalar',   'string',  '',     'name',                 1 ],
    'p' => [ 'scalar',   'string',  '',     'program_path',         1 ],
    'g' => [ 'scalar',   'string',  '',     'program_args',         0 ],
    'a' => [ 'list',     'num',     '',     'appliances',           0 ],
    'u' => [ 'scalar',   'boolean', 'N',    'use_appliances',       0 ],
    'n' => [ 'scalar',   'boolean', 'N',    'new_appliances',       0 ],
    'A' => [ 'scalar',   'boolean', 'N',    'dont_allow_new',       0 ],
    'R' => [ 'scalar',   'boolean', 'N',    'read_only',            0 ],
    'E' => [ 'scalar',   'string',  '',     'notification_email',   0 ],
    'S' => [ 'scalar',   'boolean', 'N',    'email_when_starting',  0 ],
    'F' => [ 'scalar',   'boolean', 'N',    'email_when_done',      0 ],
    'C' => [ 'scalar',   'boolean', 'N',    'constant_type',        0 ],
    'X' => [ 'scalar',   'string',  '',     'special_fields_lib',   0 ],
    'M' => [ 'scalar',   'string',  '',     'mapp_name',            0 ],
    'H' => [ 'scalar',   'boolean', 'N',    'hide_tasks',           0 ]
  );

  my %typeRules =
  (
    'once' =>
    {
      'd' => [ 'scalar',   'date',    '',     'once_date',         0 ],
      't' => [ 'scalar',   'time',    '',     'once_time',         0 ]
    },
    'hourly' =>
    {
      's' => [ 'scalar',   'hour',    '0',    'hourly_start_hour', 0 ],
      'l' => [ 'scalar',   'hour',    '23',   'hourly_last_hour',  0 ],
      'm' => [ 'list',     'minute',  '',     'hourly_minutes',    1 ],
      'f' => [ 'scalar',   'num',     '1',    'hourly_frequency',  0 ]
    },
    'once_a_day' =>
    {
      't' => [ 'scalar',   'time',    '',     'once_time',         0 ]
    },
    'daily' =>
    {
      'h' => [ 'list',     'hour',    '',     'daily_hours',       1 ],
      'm' => [ 'scalar',   'minute',  '',     'daily_minute',      1 ],
      'f' => [ 'scalar',   'num',     '1',    'daily_frequency',   0 ]
    },
    'weekly' =>
    {
      'w' => [ 'list',     'dayname', '',     'weekly_days',       1 ],
      't' => [ 'scalar',   'time',    '',     'weekly_time',       1 ]
    },
    'monthly' =>
    {
      'd' => [ 'list',     'daynum',  '',     'monthly_days',      1 ],
      't' => [ 'scalar',   'time',    '',     'monthly_time',      1 ]
    }
  );

  printUsage('Unknown type : ' . $type)
    if (!exists($typeRules{$type}));

  foreach my $rule (keys %{$typeRules{$type}})
  {
    $globalRules{$rule} = $typeRules{$type}->{$rule};
  }

  return \%globalRules;
}

#
#
#
sub parseArgs
{
  my ($type, $args, $rules) = @_;

  my ($arg, $flag, $content, @elements, $ele, $ftype, $year,
      $month, $day, $hour, $min, %unique);

  my %dayMap =
  (
    'sun' => 0,
    'mon' => 1,
    'tue' => 2,
    'wed' => 3,
    'thu' => 4,
    'fri' => 5,
    'sat' => 6
  );

  my %flagHash = ();
  foreach $arg (@{$args})
  {
    ($flag, $content) = ($arg =~ /^\-([a-zA-Z])(.*)$/);
    printUsage('Bad argument form : ' . $arg)
      if (!defined($flag));
    $content = '' if (!defined($content));
    printUsage('Unknown flag in : ' . $arg)
      if (!exists($rules->{$flag}));
    $flagHash{$flag} = $content;
  }

  printUsage('Must specify time if specifying date')
   if (($type eq 'once') && exists($flagHash{'d'}) && !exists($flagHash{'t'}));

  my %paramHash = ();
  foreach $flag (keys %{$rules})
  {
    $ftype = $rules->{$flag}->[1];

    printUsage('Flag ' . $flag . ' is required (' . $rules->{$flag}->[3] . ')')
        if ($rules->{$flag}->[4] &&
            (!exists($flagHash{$flag}) || ($flagHash{$flag} eq '')));
    $flagHash{$flag} = 'Y' if (($ftype eq 'boolean') &&
                               exists($flagHash{$flag}));
    $flagHash{$flag} = $rules->{$flag}->[2]
      if (!exists($flagHash{$flag}));

    $content = $flagHash{$flag};

    if ($rules->{$flag}->[0] eq 'list')
    {
      @elements = split(/,/, $content);
    }
    elsif ($content ne '')
    {
      @elements = ($content);
    }
    else
    {
      @elements = ();
    }

    %unique = ();
    my @results = ();
    foreach $ele (@elements)
    {
      next if (exists($unique{$ele}));
      $unique{$ele} = 1;

      if ($ftype eq 'string')
      {
        push (@results, $ele);
      }
      elsif ($ftype eq 'boolean')
      {
        push (@results, $ele);
      }
      elsif ($ftype eq 'num')
      {
        printUsage($ele . ' is not a number in ' . $arg)
          if ($ele !~ /^[0-9]+$/);
        push (@results, $ele);
      }
      elsif ($ftype eq 'date')
      {
        ($year, $month, $day) = ($ele =~ /^([0-9][0-9][0-9][0-9])([0-9][0-9])([0-9][0-9])$/);
        printUsage($ele . ' is an invalid date, must be in the form yyyymmdd')
          if (!defined($year) || !defined($month) || !defined($day)
              || ($month < 1) || ($month > 12) || ($day < 1) || ($day > 31));
        push (@results, [$year, $month, $day]);
      }
      elsif ($ftype eq 'time')
      {
        ($hour, $min) = ($ele =~ /^([0-9][0-9])([0-9][0-9])$/);
        printUsage($ele . ' is an invalid time, must be in the form hhmm')
          if (!defined($hour) || !defined($min) || ($hour < 0) || ($hour > 23)
              || ($min < 0) || ($min > 59));
        push (@results, [$hour, $min]);
      }
      elsif ($ftype eq 'hour')
      {
        printUsage($ele . ' is an invalid hour, must be between 0 and 23')
          if (($ele !~ /^[0-9]+$/) || ($ele < 0) || ($ele > 23));
        push (@results, $ele);
      }
      elsif ($ftype eq 'minute')
      {
        printUsage($ele . ' is an invalid minute, must be between 0 and 59')
          if (($ele !~ /^[0-9]+$/) || ($ele < 0) || ($ele > 59));
        push (@results, $ele);
      }
      elsif ($ftype eq 'dayname')
      {
        printUsage('Unknown day : ' . $ele)
          if (!exists($dayMap{$ele}));
        push (@results, $dayMap{$ele});
      }
      elsif ($ftype eq 'daynum')
      {
        printUsage($ele . ' is an invalid day num, must be between 1 and 31')
          if (($ele !~ /^[0-9]+$/) || ($ele < 1) || ($ele > 31));
        push (@results, $ele);
      }
      else
      {
        print "Internal error : field type = $ftype\n";
        exit 0;
      }
    }

    if ($rules->{$flag}->[0] eq 'list')
    {
      $paramHash{$rules->{$flag}->[3]} = defined($results[0])?\@results:'';
    }
    elsif (defined($results[0]))
    {
      $paramHash{$rules->{$flag}->[3]} = $results[0];
    }
    else
    {
      $paramHash{$rules->{$flag}->[3]} = '';
    }
  }

  return \%paramHash;
}

#
#
#
sub addScheduleEntry
{
  my $type      = shift @_;
  my $paramHash = shift @_;
  my $useThisId = shift @_;

  my %typeVerboseFunc =
  (
    'once'       => \&onceVerb,
    'hourly'     => \&hourlyVerb,
    'once_a_day' => \&once_a_dayVerb,
    'daily'      => \&dailyVerb,
    'weekly'     => \&weeklyVerb,
    'monthly'    => \&monthlyVerb
  );
  $paramHash->{'verbose_settings'} = &{$typeVerboseFunc{$type}}($paramHash);

  $paramHash->{'schedule_id'} = $useThisId
    if ($useThisId);

  my $scheduleId = Schedule::addSchedule($type, $paramHash);
  if ($scheduleId <= 0)
  {
    print STDERR 'ERROR : ' . SCSDB::getErrorMessage() . "\n";
    exit 0;
  }

  $paramHash->{'schedule_id'} = $scheduleId;

  if ($type eq 'once')
  {
    setAt($paramHash);
  }
  else
  {
    saveCrontab();
  }
}

#
#
#
sub deleteScheduleEntry
{
  my $id = shift @_;

  my $schedule;

  if (!defined($id) || ($id !~ /^[0-9]+$/)
        || !($schedule = Schedule::getSchedule($id)))
  {
    print STDERR "Can't find schedule, available schedules:\n\n";
    listMode();
    exit 0;
  }

  if (Schedule::deleteSchedule($schedule->{'schedule_id'}))
  {
    print STDERR 'ERROR : ' . SCSDB::getErrorMessage() . "\n";
    exit -1;
  }

  if ($schedule->{'type'} eq 'once')
  {
    SysCmd::runCmd('atrm', $schedule->{'at_ident'});
  }
  else
  {
    saveCrontab();
  }
}

#
#
#
sub getRunScheduleCmd
{
  my $sched = shift;

  return $runScheduleProg . ' ' . $sched->{'schedule_id'} . ' >/dev/null 2>/dev/null';
}

#
#
#
sub setAt
{
  my $schedule = shift || 0;

  my %monthMap =
  (
    '01' => 'Jan',
    '02' => 'Feb',
    '03' => 'Mar',
    '04' => 'Apr',
    '05' => 'May',
    '06' => 'Jun',
    '07' => 'Jul',
    '08' => 'Aug',
    '09' => 'Sep',
    '10' => 'Oct',
    '11' => 'Nov',
    '12' => 'Dec'
  );

  my $schedList;
  if ($schedule)
  {
    $schedList = [$schedule];
  }
  else
  {
    $schedList = Schedule::getSchedules(1);
  }

  return if (!$schedList);

  my ($cmd, $res, $atIdent);
  foreach my $sched (@{$schedList})
  {
    next if ($sched->{'at_ident'});
    $cmd = SysCmd::getCmd('echo') . '"' . getRunScheduleCmd($sched) . '" | '
         . SysCmd::getCmd('at') . ' ';
    if (!$sched->{'once_time'})
    {
      $cmd .= 'now';
    }
    else
    {
      $cmd .= join(':', @{$sched->{'once_time'}});
      if ($sched->{'once_date'})
      {
        if (SysCmd::getOS() eq 'Linux')
        {
          $cmd .= ' ' . join('-', @{$sched->{'once_date'}});
        }
        else
        {
          $cmd .= ' ' . $monthMap{$sched->{'once_date'}[1]}
                . $sched->{'once_date'}[2]
                . ',' . $sched->{'once_date'}[0];
        }
      }
    }
    $cmd .= ' 2>&1 1>/dev/null';
    $res = `$cmd`;
    ($atIdent) = ($res =~ /^job\s+(\S+)/);
    if (defined($atIdent) && $atIdent)
    {
      Schedule::setAtIdent($sched->{'schedule_id'}, $atIdent);
    }
    else
    {
      print STDERR "Error adding task to at\n";
    }
  }
}

#
#
#
sub saveCrontab
{
  my $schedList = Schedule::getSchedules(2);

  my %typeFunc =
  (
    'hourly'     => \&hourlyCrontab,
    'once_a_day' => \&once_a_dayCrontab,
    'daily'      => \&dailyCrontab,
    'weekly'     => \&weeklyCrontab,
    'monthly'    => \&monthlyCrontab
  );

  my $crontabStr =<<EOS;
#
# Control Station Schedule crontab
#  ** Do NOT modify by hand **
#
#WANT_LOG=1
EOS
  if ($schedList)
  {
    foreach my $sched (@{$schedList})
    {
      $crontabStr .= "\n#\n# " . $sched->{'name'} . ' : '
                        . $sched->{'verbose_settings'} . "\n#\n";
      $crontabStr .= sprintf "%-20s ", &{$typeFunc{$sched->{'type'}}}($sched);
      $crontabStr .= SysCmd::getCronEntry($runScheduleProg . ' '
                                            . $sched->{'schedule_id'}) . "\n";
    }
  }

  SysCmd::commitSchedulerCrontab('SCS_SCHEDULER', $crontabStr);
}

#
#
#
sub hourlyCrontab
{
  my $s = shift;

  my $str = join(',', @{$s->{'hourly_minutes'}}) . ' ';
  $str .= $s->{'hourly_start_hour'} . '-' . $s->{'hourly_last_hour'};
  $str .= '/' . $s->{'hourly_frequency'}
    if ($s->{'hourly_frequency'} > 1);
  $str .= ' * * *';

  return $str;
}

#
#
#
sub once_a_dayCrontab
{
  my $s = shift;

  my $str = $s->{'once_time'}->[1] . ' ' . $s->{'once_time'}->[0] . ' * * *';

  return $str;
}

#
#
#
sub dailyCrontab
{
  my $s = shift;

  my $str = $s->{'daily_minute'} . ' '
          . join(',', @{$s->{'daily_hours'}}) . ' *';
  $str .= '/' . $s->{'daily_frequency'} if ($s->{'daily_frequency'} > 1);
  $str .= ' * *';

  return $str;
}

#
#
#
sub weeklyCrontab
{
  my $s = shift;

  my $str = $s->{'weekly_time'}->[1] . ' ' . $s->{'weekly_time'}->[0] . ' * * '
          . join(',', @{$s->{'weekly_days'}});

  return $str;
}

#
#
#
sub monthlyCrontab
{
  my $s = shift;

  my $str = $s->{'monthly_time'}->[1] . ' ' . $s->{'monthly_time'}->[0]
          . ' ' . join(',', @{$s->{'monthly_days'}}) . ' * * ';

  return $str;
}
