#!/usr/bin/perl -w
# $Id: Task.pm,v 1.15 2004/10/20 22:01:33 ms152511 Exp $
# Task.pm - lib for manipulating the task table
# Copyright 2004 Sun Microsystems, Inc., All rights reserved.
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms

package Task;

use lib "/scs/lib/perl5";
use strict;
use SCSDB;
use POSIX qw(setsid);
use POSIX ":sys_wait_h";

#
# Subroutines
# -----------
#  addTask
#  setTaskStatus
#  setTaskMessage
#  setTaskItems
#  setTaskFailed
#  setTaskCompleted
#  setTaskWarning
#  spawnTask
#  advanceProgress
#  updateProgress
#  getTask
#  getAllTasks
#

my $CONCURRENT_TASKS = 5;
my $TASK_POLL_INTERVAL = 10;

#
# setMaxConcurrentTasks - sets the maximum number of concurrent tasks
#
# !!!!! DO NOT USE THIS UNLESS YOU KNOW WHAT YOU ARE DOING !!!!!
#

sub setMaxConcurrentTasks
{
    my $maxTasks = shift @_;    
    return unless (defined($maxTasks));

    my $intMaxTasks = int($maxTasks+0);
    $CONCURRENT_TASKS = $intMaxTasks if ($intMaxTasks > 0);
}

#
# setTaskPollInterval - sets the task polling interval for parallel tasks
#

sub setTaskPollInterval
{
    my $interval = shift @_;
    return unless (defined($interval));

    my $intInterval = int($interval+0);
    $TASK_POLL_INTERVAL = $intInterval if ($intInterval > 0);
}

#
# spawnTasks
#
# In: cmd, args (scalar or ref), parallel items
#
# Out: Hash reference (key => parallelization parameter, value => rc)
#

sub spawnTasks
{
    my %rcHash = ();
    my $allocateTask = 1;

    my $taskCmd = shift @_;

    if (defined($taskCmd) && $taskCmd eq "NO_TASK") {
	$allocateTask = 0;
	$taskCmd = shift @_;
    }

    return \%rcHash unless (defined($taskCmd) && -x $taskCmd);

    # add -t and -1 to the arg list (provided that NO_TASK hasn't been
    # specified

    my @taskArgs = ();
    push (@taskArgs, '-t', '-1') if ($allocateTask);

    my $taskArgsRef = shift @_;
    return \%rcHash unless (defined($taskArgsRef));

    my %argHash = ();
    my $taskItem = "";
    my $value = "";

    #
    # build the arg list for the individual instances; we understand
    # a few different calling conventions:
    #
    # 1. spawnTasks($cmd,\@(...),item1,item2,...)
    #
    #    In this style we call exec as: exec($cmd,@(...),item1)
    #    and so on. If \@(...) is omitted (the second arg is a scalar,
    #    and not an array/hash reference), then it is treated as the
    #    first item to work on.
    #
    # 2. spawnTasks($cmd,\%( 'item1' => [item1Arg1,item1Arg2,...],
    #                        'item2' => [item2Arg1,item2Arg2,...],
    #                        ... ))
    #  
    #    In this style we call exec as: exec($cmd,@{$argHash{$item1}})
    #    and so on.
    #

    if (ref($taskArgsRef) eq "ARRAY") {
	foreach $taskItem (@{$taskArgsRef}) {
	    if (!defined($taskItem) || $taskItem eq "") {
		push (@taskArgs,'""');
		next;
	    }
	    push (@taskArgs, $taskItem);
	}

	foreach $taskItem (@_) {
	    $argHash{$taskItem} = [@taskArgs, $taskItem];
	}
    } elsif (ref($taskArgsRef) eq "HASH") {
	foreach $taskItem (keys(%{$taskArgsRef})) {
	    $value = $taskArgsRef->{$taskItem};
	    next unless (defined($value));
	    if (ref($value) eq "ARRAY") {
		$argHash{$taskItem} = $value;
	    } else {
		$argHash{$taskItem} = [$value];
	    }
	}
    } else {	
	foreach $taskItem (@_) {
	    $argHash{$taskItem} = [@taskArgs, $taskItem];
	}
    }

    return \%rcHash unless(scalar(keys(%argHash)) > 0);

    # make sure that the max number of concurrent tasks and the
    # wait interval are defined

    my $maxTasks = 
	(defined($CONCURRENT_TASKS) ? int($CONCURRENT_TASKS+0) : 0);
    $maxTasks = 5 unless ($maxTasks > 0);

    my $interval =
	(defined($TASK_POLL_INTERVAL) ? int($TASK_POLL_INTERVAL+0) : 0);
    $interval = 10 unless ($interval > 0);
    
    my $taskCnt = 0;
    my @pids = ();
    my %pidToTask = ();
    my $slot = 0;
    my $nextSlot = 0;
    my $pid = 0;
    my $rc = 0;
    my $ec = 0;
    my $rcHashKey = "";

    # close the db connection so that we don't run into problems after
    # fork()'ing

    SCSDB::closeConnection();

    # now run each of the commands 

    foreach $taskItem (sort(keys(%argHash))) {
	
	next unless (defined($taskItem));

	$value = $argHash{$taskItem};
	next unless (defined($value) && ref($value) eq "ARRAY");

	$nextSlot = -1;

	# find an execution slot for this instance

	while ($nextSlot < 0) {

	    # wait for at least one task to finish 

	    for ($slot = 0; $slot < $maxTasks; $slot++) {
		$pid = $pids[$slot];
		next unless (defined($pid) && $pid != 0);
		    
		$rc = waitpid($pid, &WNOHANG);
		
		# if wait returns 0, then pid is still running
		#    need to examine the next slot
		# if wait returns $pid, then exit code is in $?
		#    we can use this slot                   
		# if wait returns -1, then pid doesn't exist
		#                     (should never happen)
		    
		next if ($rc == 0);
		
		$ec = ($rc == $pid ? $?/256 : -1);
		    
		$rcHashKey = $pidToTask{$pid};
		$rcHash{$rcHashKey} = $ec
		    if (defined($rcHashKey) && $rcHashKey ne "");
		
		$pids[$slot] = 0;
		$taskCnt--;
	    }
	
	    # set nextSlot to the first available slot

	    for ($slot = 0; $slot < $maxTasks; $slot++) {
		$pid = $pids[$slot];
		next if (defined($pid) && $pid != 0);
		$nextSlot = $slot;
		last;
	    }
	    
	    # wait upto interval seconds if we don't have a free slot

	    sleep($interval) unless ($nextSlot >= 0);
	}

	# fork and exec

	$pid = fork();

	if ($pid > 0) {
	    # parent
	    $pids[$nextSlot] = $pid;
	    $pidToTask{$pid} = $taskItem;
	    $taskCnt++;
	    next;
	}

	# child

	$| = 1;
	exec($taskCmd, @{$value});
	exit(127);
    }

    # wait for all processes to finish

    while ($taskCnt > 0) {
	for ($slot = 0; $slot < $maxTasks; $slot++) {
	    $pid = $pids[$slot];
	    next unless (defined($pid) && $pid != 0);
		    
	    $rc = waitpid($pid, &WNOHANG);
		    
	    next if ($rc == 0);
		
	    $ec = ($rc == $pid ? $?/256 : -1);
	    
	    $rcHashKey = $pidToTask{$pid};
	    $rcHash{$rcHashKey} = $ec
		if (defined($rcHashKey) && $rcHashKey ne "");
	    
	    $pids[$slot] = 0;
	    $taskCnt--;	    
	}
	sleep($interval);
    }
    
    # reopen the db connection

    SCSDB::openConnection();

    # return a reference to the result hash

    return \%rcHash;
}

#
# spawnTask
#
# In : Task name, Num items (optional), message (optional)
#
# Out : New Task ID (positive integer) or error code (negative)
#

sub spawnTask
{
    my $parent = 1;
    my $taskId = addTask(@_);
    return (-1, $parent) 
	unless (defined($taskId) && $taskId+0 > 0);

    my $pid = fork();

    &SCSDB::closeConnection();

    if ($pid > 0) {
	waitpid($pid,0);
	return ($taskId, $parent);
    } 
    
    # child

    $parent = 0;
    $pid = fork();
    
    if ($pid > 0) {
	exit(0);
    }
    
    # grandchild

    setsid();
    chdir('/');
    umask(0);

    &Progress::setTaskId($taskId);

    &SCSDB::openConnection();

    return ($taskId, $parent);
}

#
# addTask
#
# In : Task name, Num items (optional), message (optional)
#
# Out : New Task ID (positive integer) or error code (negative)
#

sub addTask
{
    my $name = shift @_;
    $name = "" unless (defined($name));
    
    my $totalItems = shift @_ || 0;

    return addTaskWithVisibility($name, $totalItems, 'N', @_);
}

#
# addHiddenTask
#
# adds a hidden task
#

sub addHiddenTask
{
    my $name = shift @_;
    $name = "" unless (defined($name));
    
    my $totalItems = shift @_ || 0;

    return addTaskWithVisibility($name, $totalItems, 'Y', @_);
}

#
# addTaskWithVisibility
#
# adds a task with the spcified visibility 
#

sub addTaskWithVisibility
{
    my $name = shift @_;
    $name = SCSDB::encodeValue($name);
    
    my $totalItems = shift @_ || 0;

    my $hidden = shift @_;
    $hidden = 'N' unless (defined($hidden) && $hidden eq "Y");

    my $msg = SCSDB::encodeValue(@_);

    my %taskHash = ( 'start'        => 'now()',
		     'stop'         => "'epoch'",  # time_t = 0 
		     'status'       => "'R'",
		     'message'      => "'$msg'",
		     'msg_vars'     => "",
		     'name'         => "'$name'",
		     'ultimate_url' => "",
		     'hidden'       => "'$hidden'",
		     'items_done'   => 0,
		     'total_items'  => int($totalItems+0),
		     'schedule_id'  => 0 );

    return SCSDB::insertHash('mgmt_task', 'task_id', \%taskHash);
}

#
# setTaskSchedule
#
# sets the schedule id for a task
#

sub setTaskSchedule
{
    my $taskId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid task id')
	unless (defined($taskId) && $taskId+0 > 0);

    my $schedId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid schedule id')
	unless (defined($schedId) && $schedId+0 > 0);

    my $sql = 'UPDATE mgmt_task SET schedule_id = ';
    $sql .= int($schedId+0) . ' WHERE task_id = ' . int($taskId+0);
    
    return SCSDB::errorCode('EDBERR', 
			   "Cannot set schedule for task '$taskId'")
	if (SCSDB::runCommand($sql));

    return SCSDB::errorCode('OKAY');
}

#
# setTaskStatus
#
# In : Status, Task ID, Message (optional)
#
# Out : Return code
#

sub setTaskStatus
{
    my $status = shift @_;
    return SCSDB::errorCode('EPARM', 'Status is null')
	unless (defined($status) && $status =~ /^[CWRFU]$/);
    
    my $taskId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid task id')
	unless (defined($taskId) && $taskId+0 > 0);
    
    my $msg = SCSDB::encodeValue(@_);
    
    my $sql = "UPDATE mgmt_task SET status = '$status', message = ";
    $sql .= "'$msg', stop = now() ";
    $sql .= "WHERE task_id = " . int($taskId+0);
    
    return SCSDB::errorCode('EDBERR', 
			   "Cannot set status for task '$taskId'")
	if (SCSDB::runCommand($sql));
    
    return SCSDB::errorCode('OKAY');
}

#
# setTaskMessage
#
# Set the message
#

sub setTaskMessage
{
    my $taskId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid task id')
	unless (defined($taskId) && $taskId+0 > 0);
    
    my $msg = SCSDB::encodeValue(@_);
    
    my $sql = "UPDATE mgmt_task SET message = '$msg' ";
    $sql .= "WHERE task_id = " . int($taskId+0);
    
    return SCSDB::errorCode('EDBERR', 
			   "Cannot update message for task '$taskId'")
	if (SCSDB::runCommand($sql));
    
    return SCSDB::errorCode('OKAY');
}

#
# setTaskItems
#
# Set the number of work units
#

sub setTaskItems
{
    my $taskId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid task id')
	unless (defined($taskId) && $taskId+0 > 0);
    
    my $count = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid count')
	unless (defined($count) && $count+0 >= 0);
    
    my $sql = 'UPDATE mgmt_task SET total_items = ';
    $sql .= int($count+0) . ' WHERE task_id = ' . int($taskId+0);
    
    return SCSDB::errorCode('EDBERR', 
			   "Cannot update work units for '$taskId'")
	if (SCSDB::runCommand($sql));
    
    return SCSDB::errorCode('OKAY');
}

#
# setTaskFailed
#
# In : Task ID, Message (optional)
#
# Out : Return code
#

sub setTaskFailed
{
    return setTaskStatus('F', @_);
}

#
# setTaskStatusAndAdvanceProgress
#
# sets the task status and advances the progress
#

sub setTaskStatusAndAdvanceProgress
{
    my $status = shift @_;
    return SCSDB::errorCode('EPARM', 'Status is null')
	unless (defined($status) && $status =~ /^[CWRFU]$/);
    
    my $taskId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid task id')
	unless (defined($taskId) && $taskId+0 > 0);

    my ($ec, $task) = getTask($taskId);
    return SCSDB::errorCode('EINVAL', 'Invalid task id')
	unless (defined($task) && ref($task) eq "HASH");
    
    if ($task->{'items_done'} != $task->{'total_items'}) {
	my $total = $task->{'total_items'};
	my $complete = $task->{'items_done'};	
	advanceProgress($taskId, $total - $complete);
    }
    
    return setTaskStatus($status, $taskId, @_);
}

#
# setTaskCompleted
#
# In : Task ID, Message (optional)
#
# Out : Return code
#

sub setTaskCompleted
{
    return setTaskStatusAndAdvanceProgress('C', @_);
}

#
# setTaskWarning
#
# In : Task ID, Message (optional)
#
# Out : Return code
#

sub setTaskWarning
{
    return setTaskStatusAndAdvanceProgress('W', @_);
}

#
# advanceProgress
#
# advances the progress of the task by the specified increment
#

sub advanceProgress
{
    my $taskId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid task id')
	unless (defined($taskId) && $taskId+0 > 0);

    my $numUnits = shift @_;
    $numUnits = 1 unless (defined($numUnits) && $numUnits+0 > 0);

    my $msg = SCSDB::encodeValue(@_);
    
    my $sql = 'UPDATE mgmt_task SET items_done = items_done + ';
    $sql .= int($numUnits+0);    
    $sql .= ", message = '$msg'" 
	if ($msg ne "");
    $sql .= " WHERE task_id = " . int($taskId+0);
    
    return SCSDB::errorCode('EDBERR', 
			   "Cannot update task '$taskId' progress")
	if (SCSDB::runCommand($sql));

    return SCSDB::errorCode('OKAY');
}

#
# updateProgress
#
# In : Task ID, Message (optional)
#
# Out : Return code
#
# Increment our progress
#
sub updateProgress
{
    my $taskId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid task id')
	unless (defined($taskId) && $taskId+0 > 0);

    return advanceProgress($taskId, 1, @_);
}

#
# updateUltimateUrl
#
# sets the ultimate url for the task
#

sub updateUltimateURL
{
    my $taskId = shift @_;
    return SCSDB::errorCode('EPARM', 'Invalid task id')
	unless (defined($taskId) && $taskId+0 > 0);
    
    my $newUrl = shift @_;
    return SCSDB::errorCode('EPARM', 'Ultimate url is null')
	unless (defined($newUrl));

    $newUrl = SCSDB::encodeValue($newUrl);
    $newUrl =~ s/ [\t\r\n]/\&\#032\;/g;
    return SCSDB::errorCode('EPARM', 'Ultimate url is null')
	if ($newUrl eq "");

    my $sql = "UPDATE mgmt_task SET ultimate_url = '$newUrl' ";
    $sql .= 'WHERE task_id = ' . int($taskId+0);

    return SCSDB::errorCode('EDBERR',
			   "Cannot set task '$taskId' ultimate url")
	if (SCSDB::runCommand($sql));

    return SCSDB::errorCode('OKAY');
}

#
# getTask
#
# In : Task ID
#
# Out : Return code, Reference to hash
#

sub getTask
{
    my $taskId = shift @_;
    return (SCSDB::errorCode('EPARM', 'Invalid task id'), 0)
	unless (defined($taskId) && $taskId+0 > 0);

    my $sql = "SELECT * FROM mgmt_task ";    
    $sql .= 'WHERE task_id = ' . int($taskId+0);

    my $task = SCSDB::getResultHash($sql);
    
    return (SCSDB::errorCode('EINVAL', 
			    "Invalid task id '$taskId'"), 0)
	unless (defined($task) && ref($task) eq "HASH");

    return (SCSDB::errorCode('OKAY'), $task);
}

#
# getTaskStatus
#
# In : Task ID
#
# Out : Task status
#

sub getTaskStatus
{
    my $taskId = shift @_;
    return 'U' unless (defined($taskId) && $taskId+0 > 0);
    
    my ($ec, $task) = Task::getTask($taskId);
    return 'U' unless (defined($task) && ref($task) eq "HASH");
    
    my $status = $task->{'status'};
    
    return ($status =~ /^[CWRF]$/ ? $status : 'U');
}

#
# getAllTasks
#
# In : Nothing
#
# Out : Return code, reference to list of hashes
#

sub getAllTasks
{
    my $sql = "SELECT * FROM mgmt_task ORDER BY start";

    return (SCSDB::errorCode('OKAY'), SCSDB::getResultHashes($sql));
}

1;
