#!/usr/bin/perl -w
# $Id: Progress.pm,v 1.8 2004/07/20 00:13:34 sv122840 Exp $
# Progress.pm - lib to facilitate progress updates for UI
# Copyright 2004 Sun Microsystems, Inc., All rights reserved.
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms

package Progress;

use lib '/scs/lib/perl5';
use strict;
use Task;
use Event;
use BDUtil;

#
# Subroutines
# -----------
# initProgress
# setProcessName
# currentlyDone
# currentWarning
# setWorkItems
# advanceProgress
# updateProgress
# progressComplete
# progressWarned
# progressFailed
# putEvent
# infoEvent
# warningEvent
# errorEvent
# setMessage
# updateUltimateURL

my $taskId = 0;
my $whoIam = '';

#
# initProgress - initialize the progess bar with the task id
# 
# In:  argument list supplied to the command
# Out: task id of the command
#
# NOTES:
#
# The task id can be passed in one of two ways:
#
# 1. If the first arg is -t and the second arg exists, the
#    second arg is assumed to be the task id
# 2. If -t isn't given but the environment variable _TASK_ID_
#    is defined, then its value is assumed to the be the task id
#
# The argument following -t is always prefered to _TASK_ID_.
# 
# If the task id is -1, then a new task is allocated with the
# task name "Background Task". 
#

sub initProgress
{
    my $haveOptTask = 0;
    my $argList = shift @_;
    $whoIam = shift @_ || '';
    $taskId = 0;

    if (defined($argList) && ref($argList) eq "ARRAY") {
	$haveOptTask = 1 
	    if (defined($argList->[0]) && $argList->[0] eq "-t" &&
		defined($argList->[1]));
    }
    
    if ($haveOptTask) {
	shift @{$argList};   # Waste the -t
	$taskId = shift @{$argList};
    } else {
	$taskId = $ENV{"_TASK_ID_"} if (defined($ENV{"_TASK_ID_"}));
    }
    
    $taskId = 0 unless (defined($taskId));
    $taskId = Task::addTask("Background Task")
	if ($taskId eq "-1");

    if ($taskId =~ /^[0-9]+$/ && int($taskId+0) > 0) {
	&BDUtil::info('Using progress bar, task ID = ' . $taskId);
	if ($whoIam eq '') {
	    my $task = Task::getTask($taskId);
	    if (defined($task) && ref($task) eq "HASH") {
		$whoIam = $task->{'name'};
	    }
	}
    } 

    $ENV{"_TASK_ID_"} = $taskId;
    return $taskId;
}

#
# getTaskId - returns the current task id
#

sub getTaskId
{
  return $taskId;
}

#
# setTaskId - sets the current task id
#
# !!!!! DO NOT USE THIS UNLESS YOU KNOW WHAT YOU ARE DOING !!!!!
#

sub setTaskId
{
    my $tid = shift @_;
    $taskId = $tid if (defined($tid) && $tid+0 > 0);
}

#
# setWorkItems - sets the number of work items for this task
#

sub setWorkItems
{
  return 0 if (!$taskId);
  my $n = shift;
  BDUtil::info('Setting progress units to ' . $n);
  return Task::setTaskItems($taskId, $n);
}

#
# advanceProgress - Advance progress n times
#

sub advanceProgress
{
  return 0 if (!$taskId);
  my $n = shift;
  $n = 1 if (!defined($n));
  return if (!$n);

  BDUtil::info('Advancing progress ' . $n . ' units');
  return Task::advanceProgress($taskId, $n);
}

#
# Time to upgrade the progress
#
sub updateProgress
{
  return 0 if (!$taskId);
  BDUtil::info('Updating progress');
  return Task::updateProgress($taskId, @_);
}

#
# Set the task status to C
#
sub progressComplete
{
  return 0 if (!$taskId);
  infoEvent(@_);
  return Task::setTaskCompleted($taskId, @_);
}

#
# Set the task status to W
#
sub progressWarned
{
  return 0 if (!$taskId);
  warningEvent(@_);
  return Task::setTaskWarning($taskId, @_);
}


#
# Set the task status to F
#
sub progressFailed
{
  return 0 if (!$taskId);
  errorEvent(@_);
  return Task::setTaskFailed($taskId, @_);
}

#
# putEvent
#
# In : Process name, Event Type, Message
#
# Out : Return code
#
sub putEvent
{
  return 0 if (!$taskId);
  my ($processName, $eventType, $message) = @_;

  return Event::addEvent($taskId, $eventType, '127.0.0.1', $processName, 0, $message);
}

#
# infoEvent
#
sub infoEvent
{
  return 0 if (!$taskId);
  return Event::addEvent($taskId, 'info', '127.0.0.1', $whoIam, 0, @_);
}

#
# warningEvent
#
sub warningEvent
{
  return 0 if (!$taskId);
  return Event::addEvent($taskId, 'warning', '127.0.0.1', $whoIam, 0, @_);
}

#
# errorEvent
#
sub errorEvent
{
  return 0 if (!$taskId);
  return Event::addEvent($taskId, 'error', '127.0.0.1', $whoIam, 0, @_);
}

#
# setMessage
#
sub setMessage
{
  return 0 if (!$taskId);
  return Task::setTaskMessage($taskId, @_);
}

#
# setProcessName
#
sub setProcessName
{
  $whoIam = shift;
}

#
# currentlyDone() : number of items already done
#
sub currentlyDone
{
  return 0 if (!$taskId);
  my $task = Task::getTask($taskId);
  return $task->{'items_done'};
}

#
# currentTaskStatus : returns the task final state - 0 (green)
#                                                  - 1 (yellow)
#                                                  - 2 (red)  
#
sub currentTaskStatus 
{
  return 0 if (!$taskId);

  my $events = Event::getEvents($taskId);
  my $warnEvents = Event::getEvents($taskId, 'warning');
  my $errorEvents = Event::getEvents($taskId, 'error');

  if ( scalar(@{$events}) <= 0 ) {
    $events = Event::getEvents($taskId, 'error');
  }
 
  if ( ((scalar(@{$warnEvents})) == (scalar(@{$events}))) || ((scalar(@{$errorEvents})) == (scalar(@{$events}))) ) {
    return 2;
  }
  elsif ( (scalar(@{$warnEvents})) || (scalar(@{$errorEvents})) ) {
    return 1;
  }
  else {
    return 0;
  }
}

#
#
#
sub updateUltimateURL
{
  my $newURL = shift;

  return 0 if (!$taskId);
  return Task::updateUltimateURL($taskId, $newURL);
}

1;

