package amUpdate;

# Copyright (c) 2003 Sun Microsystems, Inc. All rights reserved
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.

use strict;
use lib '/scs/lib/perl5';
use SysCmd;

# AMupdate.pm - active monitoring subroutines
#
# $Id: amUpdate.pm,v 1.18 2004/12/08 20:27:14 rs131644 Exp $
# This file is modified from HealthCmd.pm and healthData.pm in version 1.

### Modules
use lib "/scs/lib/perl5";
use AM;
use SCSDB;
use BDConfig;
use BDi18n;
use Alerts;
use Cwd;     # used in getcwd
use lib qw( /scs/lib/perl5 );
use Fcntl qw(:DEFAULT :flock);
use Fcntl qw(O_CREAT O_RDONLY O_WRONLY O_APPEND);
use PropertyUtil;
use SysCmd;

### Global Vars
my $Debug 	= 0;	# turn on or off debugging data
my $etc_dir = PropertyUtil::getProperty('dir.etc');
my $logs_dir = PropertyUtil::getProperty('dir.logs');

my $confdir  = $etc_dir . '/am/';	# location of config data
my $logfile  = $logs_dir . '/AM.log';	# location of active monitor logfile
my $statedir = '/var/tmp/am/';		# location of state data

# event keys: ip, service, data
my %amTables = ();        # table of ( ip=>{service=>{data1=>"x",data2=>"x"}} )

my $mail_ignore=0;       # ids that have sent am admin notification
my $mail_ignorefile     = '/var/tmp/am_mail_ignore';   # admin mail
my $am_mailfile        = "/var/tmp/amNofifyEmail";    # temp mail file
my $am_emailaddr;      # email address of am notification
my $mail_tosend = 0;    # boolean for sending mail

my $rc;                 # hold return codes

umask 0000;             # play REALLY nice..

my $ip_address = get_ipaddress();


sub logger
# logger - log message to specified logfile
# Arguments: message to log
# Returns:   nothing
# Callers:   all
{
	my $msg	= shift;
	my $now	= localtime;

	if (! sysopen(LOG, $logfile, O_CREAT|O_WRONLY|O_APPEND, 0644)) {
		print STDERR "failed to open logfile $logfile: $!\n";
		return;
	}
	print LOG "$now: $msg\n\n";
	close(LOG);

}

sub debug
# debug - output specified message to STDERR
# Argements: message
# Returns:   null
# Callers:   depends...
{
	my $msg = shift;

	( ! $Debug ) ? return : 0;

	print STDERR "debug: $msg\n";

	return;
}

sub error
# error - error handler
# Arguments: error message
# Returns:   nothing
# Callers:   all
{
	my $msg	= shift;
	my $now	= localtime;

	# output to logfile and die
	logger("$now: $msg");
	die "$now am: $msg\n";
}

sub get_ipaddress
{
	my ($ifconfig, $address, $rc);
	logger('Getting ip address...');

	foreach my $intf ('eth0', 'hme0', 'bge0')
	{
          ($ifconfig, $rc) = SysCmd::runFullCmd(SysCmd::getCmd('ifconfig') . $intf . ' 2>/dev/null | ' . SysCmd::getCmd('grep') . 'inet 2>/dev/null');
          last if (!$rc && ($ifconfig ne ''));
	}
	return 0 if (($ifconfig eq '') || $rc);

	($address) = ($ifconfig =~ /inet addr:(\S+)/);
	if (!defined($address) || !$address)
        {
	  ($address) = ($ifconfig =~ /inet (\S+)/);
	  return 0 if (!defined($address) || !$address);
        }

	logger("Success, address: $address");

	return $address;
}

sub parse_cmdline
# parse_cmdline - set and check command-line variables
# Arguments: hash reference of options
# Returns:   null
# Callers:   main()
{
	my $optref 	= shift;
	my $eventchannel = 0;

	logger('Parsing command-line options...');

	# Usage?
	if ($optref->{h}) {
		usage();
		exit 0;
	}

	# Debugging?
	if ($optref->{d}) { $Debug = 1 }

	# alternate state dir?
	if ($optref->{'s'}) { $statedir = $optref->{'s'} }

	# alternate conf dir?
	if ($optref->{c}) { $confdir = $confdir->{c} }
}

sub get_config
# get_config - return all the configuration files in specified directory
# Arguments: none
# Returns:   array of config files
# Callers:   main()
{
	my $cur	= getcwd;

	logger("Getting config files in $confdir");

	opendir(DIR, $confdir) or error("Failed to open $confdir: $!\n");
	my @configs = grep { /.+\.conf$/ } readdir(DIR);
	close(DIR);

	for (@configs) { $_ = $confdir . $_ }

	return @configs;
}

sub runConfProg
# runConfProg - execute program in a conf file
# Arguments: config data (version,program,detail....)
# Returns:   success
# Callers:   Schedule::Cron
{
	my $hashref = shift;
	my $args = 0;
	my $return;

	my $program = $hashref->{program};

	# build optional args
	my $yellow = $hashref->{yellowalarm};
	my $red = $hashref->{redalarm};
	( $yellow || $red ) ? $args = "$yellow $red" : 0;

	$program .= " $args" if (defined $args && 
				$args ne "" &&
				$args ne "0");
	$program .= " > /dev/null";
	$return = system($program);
	
	# set return code
	$return = $return/256;

	# state > 3 is error in program, set alarm
	($return > 3) ? $return = 3 : 0;

	return $return;
}


sub updateStateInfo
# state - holds state of monitoring programs, dumps to outfile and global amTable
# Arguments: config data (version,program,detail....)
# Returns:   success or failure
# Callers:   runConfProg()
{
	my ($ji, $state, $hashref) = @_;
	my ($version, $program, $detail, $vendor, $interval, $name, $description, $state0msg,
            $state1msg, $state2msg, $state3msg, $url) = 
					( $hashref->{version}, 
					   $hashref->{program}, 
					   $hashref->{detail}, 
					   $hashref->{vendor}, 
					   $hashref->{interval}, 
					   $hashref->{name}, 
					   $hashref->{description}, 
					   $hashref->{state0msg}, 
				   	   $hashref->{state1msg}, 
		                           $hashref->{state2msg}, 
   					   $hashref->{state3msg},
   					   $hashref->{url});
	my $sendevents=0;
	my $datachanged=0;
	my $statemsg = "";
	my $dbfile = $statedir . "am" . sprintf("%03d", $ji) . ".data";
	
	# Create hash data
	my %data = (
			jobid		=> $ji,
			version 	=> $version,
			program 	=> $program,
			detail  	=> $detail,
			vendor		=> $vendor,
			interval	=> $interval,
			name		=> $name,
			description	=> $description,
			state		=> $state,
			state0msg	=> $state0msg,
			state1msg	=> $state1msg,
			state2msg	=> $state2msg,
			state3msg	=> $state3msg,
			url 		=> $url
	);

	( ! -e $statedir ) ? mkdir $statedir, 0700 : 0;

	# Lock db then do the work
	if (!sysopen(DBFILE, $dbfile, O_RDONLY) ) {
		logger("failed to open $dbfile, new entry");
		$sendevents = 1;
		$datachanged = 1;
	} else {
		my ($k1, $v1);
		while(<DBFILE>) {
			chomp;
			($k1, $v1) = split;
			if ($data{$k1} ne $v1 ) {
				$datachanged = 1;
				if ($k1 eq 'state') {
				 	$sendevents = 1;
					last;
				}
			} else {
				if (($k1 eq 'state') && ($datachanged ==1)) {
					#no state change, $datachanged is already set
					#no need to continue checking
					last;
				}
			}
		}
		close(DBFILE);
	}

	if ($datachanged) {
		my ($k2, $v2);
		sysopen(NEW, "$dbfile.new", O_WRONLY|O_CREAT, 0644) or
			die "failed to open $dbfile.new for writing\n";
		while (($k2, $v2) = each %data) {
			# optional url entry could be NULL
			if (($k2 eq "url") && (!defined($v2))) {
				next;
			}
			$v2 =~ s/ /\%20/g;
			print NEW "$k2 $v2 \n";
		}
		close(NEW);
		rename("$dbfile.new", $dbfile);
		logger("Update $dbfile");
	}

	if (!$sendevents) {
		return $sendevents;
	}

	if ($state == 0) {
		$statemsg = $state0msg;
	} elsif ($state == 1) {
		$statemsg = $state1msg;
	} elsif ($state == 2) {
		$statemsg = $state2msg;
	} elsif ($state == 3) {
		$statemsg = $state3msg;
	}


        # hold the info in a common format
        logger("adding event for $name");
        $amTables{$name} = {
                "date"          => time(),
                "state"         => $state,
                "service"       => $name,
                "vendor"        => $vendor,
                "detail"        => $detail,
                "message"       => $statemsg,
                "status_url"    => $url
        };

	return $sendevents;
}


sub process_am_event
# process_am_event - handle the creation and adding of events for device
# arguments: ip address of device to handle from the event table
# returns:   success or failure
# callers:   main
{
	my $state = 0;

	foreach my $service (keys %amTables) {
		logger("processing $service for $ip_address");

		my $url = "http://$ip_address" . 
			$amTables{$service}{'detail'};

		if ((AM::addAMService(
			$amTables{$service}{'service'},0)) < 0 )
		{
			my $error = SCSDB::getErrorMessage();
			logger("failed to insert service: $error");
		}

		if ((AM::addAMState( $service,
			$amTables{$service}{'state'},
			$amTables{$service}{'vendor'},
			$amTables{$service}{'message'}, $url,
			$amTables{$service}{'status_url'})) < 0 ) 
		{
			my $error = SCSDB::getErrorMessage();
			logger("failed to write event for $ip_address: $error");
		}
	}

	return 1;
}


sub updateMailIgnoreFile
# updateMailIgnoreFile - if mail ignore file exist and no warning status, remove 
# the ignore file
# argumets: none
# returns:  none, may remove $mail_ignorefile
{
	if (! -e $mail_ignorefile) {
		# ignore file doesn't exist, no need to remove
		return;
	}

	# check the worst state of control station
	my ($ok, $amRef) = AM::getWorstState();
	if ( $ok < 0 ) {
		next;   # how should we fail?
	}

	if ($amRef->{'state'} < 2) {
		# back to normal now, remove the mail ignore file
		SysCmd::runCmd('rm', $mail_ignorefile);
	}
}


sub sendEmails
# sendEmails - find out devices that has problem but not in ignore files, and send
# sys admins about these devices
# argumets: none, set global variables $i18n, $am_emailaddr, $mail_tosend
# returns:  none
# callers:  main()
{
	my $ret;

	# if there is no appliance need to be sent email about, the sub routine would exit here.
	if (!($ret = getBadDeviceDetails())) {
		return 0;
	}

	$am_emailaddr = BDConfig::getConfig("mgmtAmEmail");
	logger("Active monitoring email: $am_emailaddr");

	#am notify email header
	if ($am_emailaddr) {

		logger("Creating active monitor mail file to $am_emailaddr");
		unlink($am_mailfile);
		if (! sysopen(AM_MAIL, $am_mailfile, O_WRONLY|O_CREAT,0666)) { 
			logger("Failed to open temp $am_mailfile: $!\n");
			return 0;
		}

		# output header
		if (am_outputHeader() < 0) {
			logger("Failed to output header information to $am_mailfile\n");
			return 0;
		}
	}

	addHostDetail();
	

	# send am notify email if there is bad devices to notify 
	# (bad device and there is no am mail ignore file, mail_tosend would be set)
	if ($am_emailaddr) {
		close(AM_MAIL);

		# send it off
		if ($mail_tosend) {
			if (am_mailit() < 0) {
				logger("Failed to mail notification email $am_mailfile\n");
				return 0;
			}
			unlink($am_mailfile);
		}
	}

}

sub getBadDeviceDetails
# getBadDeviceDetails- 
# arguments: none
# returns:   null
# callers:   main_sendEmails
{
	# grab status
	my ($rtc, $ref) = AM::getWorstState();

	# quit if no warning status is found
	if ( $ref->{'state'} < 2) {
		logger("Not sending email, no problems found");
		Alerts::removeAlert("/sdui/activeMonitor.do?method=viewDBTable");
		return 0;
	} else {
		Alerts::addAlert("Active Monitor Critical Event", "/sdui/activeMonitor.do?method=viewDBTable", "/sdui/images/MerlotRedAlert.gif");
	}

	if (-e $mail_ignorefile)  {
		# no need to send email again
		return 0;
	}

	return 1;

}

sub addHostDetail
# addHostDetail - get details from the active monitor tables and add to the email
#                 to both am notification email and appliance notification email
#		  and send out appliance notification email
# arguments: host_name
# returns:   null
# callers:   main_sendEmails
{
	my ($service, $message, $amRef, $line);

	my $hostname=`/bin/hostname`;
	($rc, $amRef) = AM::getAllStates();

	# risk of blank notifications?
	if ( ($rc < 0) || (scalar(@{$amRef}) <= 0) ) {
		return;
	}

	#if there is a am email address and not among am email ignore list

	if ($amRef) {
		foreach my $amr (@{$amRef}) {
			next if ($amr->{'state'} < 2);
			$service = $amr->{'service_name'};
			$message = $amr->{'message'};
			$line = BDi18n::getMsg
			('[[base-mgmt-am.notifyLine]]',
				{ 
				'hostname' =>, $hostname, 
				'service' => $service,
				'message'  =>  $message
				}
			);

			if ($am_emailaddr) {
				if (! -e $mail_ignorefile) {
					print AM_MAIL $line;
					$mail_tosend = 1;
					SysCmd::runCmd('touch', $mail_ignorefile);
				}
			}

		}
	}

}


sub am_mailit
# am_mailit - interface to i18nmail
# arguments: mail message
# returns:   success or failure
# callers:   main_sendEmails
{
	logger("Sending active monitor email to $am_emailaddr");
	my $subject = BDi18n::getMsg('[[base-mgmt-am.notifySubject]]');

	SysCmd::sendEmail('Sun Control Station', $am_emailaddr, $subject, 0, $am_mailfile);
	# system("/bin/mail -s \"$subject\" $am_emailaddr -- -f \"Sun Control Station\"< $am_mailfile");

	return 1;
}

sub am_outputHeader
# am_outputHeader - create the intiro to the mail
# arguments: none
# returns:   success or failure
# callers:   main_sendEmails
{
	logger("Outputting email header...");
	my $header = BDi18n::getMsg('[[base-mgmt-am.notifyHeader]]');

	print AM_MAIL "$header\n\n";

	return 1;
}

1;
