#!/usr/bin/perl -wT
# $Id: pkg_install.pl,v 1.4.2.5.2.1.2.1 2005/02/28 23:13:19 steveod Exp $
# pkg_install.pl - install a pkg 
# Copyright 2004 Sun Microsystems, Inc., All rights reserved.
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.

use strict;
use File::Spec;

# secure the env to pass taint checks

$ENV{'PATH'} = "/sbin:/bin:/usr/bin:/usr/sbin";
delete @ENV{'IFS', 'CDPATH', 'ENV', 'BASH_ENV'};

# globals

my $EXITCODE = 0;
my $RPM = "/bin/rpm";
my $SVR4_PKGADD = "/usr/sbin/pkgadd";

# package types

my %PKGTYPES = ( 'RPM'         => 'rpm',
		 'SVR4'        => 'svr4',
		 'UNSUPPORTED' => 'unsupported' );

# package states 

my %PKGSTATES = ( 'NOTINSTALLED' => '0',
		  'INSTALLED'    => '1',
		  'INVALID'      => '2' );

# determine the packaging format for this system

my $PKGTYPE = getPkgType();
my $PKG = "";
my @PKGS = ();

# verbose mode

my $VERBOSE = 0;
if (defined($ARGV[0]) && $ARGV[0] eq "-v") {
    $VERBOSE = 1;
    shift @ARGV;
}

# make sure at least one arg is given

if (scalar(@ARGV) < 1) {
    printUsage();
    exit($EXITCODE);
}

CASE_PKGTYPE : {
    if (!defined($PKGTYPE)) {
	printError("Supported packaging system not found");
	$EXITCODE = 1;
	last CASE_PKGTYPE;
    }
    
    if ($PKGTYPE eq $PKGTYPES{'RPM'}) {
	if (defined($ARGV[0]) && $ARGV[0] eq "check") {
	    shift @ARGV;
	    $EXITCODE = checkRPM(@ARGV);
	} elsif (defined($ARGV[0]) && $ARGV[0] eq "uninstall") {
	    shift @ARGV;
	    $EXITCODE = uninstallRPM(@ARGV);
	} else {
	    $EXITCODE = installRPM(@ARGV);
	}
	last CASE_PKGTYPE;
    }
    
    if ($PKGTYPE eq $PKGTYPES{'SVR4'}) {
	if (defined($ARGV[0]) && $ARGV[0] eq "uninstall") {
	    shift @ARGV;
	    $EXITCODE = uninstallSVR4(@ARGV);
	} else {
	    $EXITCODE = installSVR4(@ARGV);
	}
	last CASE_PKGTYPE;
    }

    # default
    
    printError("Supported packaging system not found");
    $EXITCODE = 1;
}


exit($EXITCODE < 0 ? 1 : $EXITCODE);

#
# subroutines
#

# getPkgType - determines the packaging format of this system

sub getPkgType
{
    if (-x $RPM) {
	return $PKGTYPES{'RPM'};
    } 

    if (-x $SVR4_PKGADD) {
	return $PKGTYPES{'SVR4'};
    } 

    return $PKGTYPES{'UNSUPPORTED'};
}

# installSVR4 - installs SVR4 packages

sub installSVR4
{
    my $SVR4_PATCHADD = "/usr/sbin/patchadd";
    my $SOLARIS_TAR = "/usr/bin/tar";
    my $SOLARIS_UNZIP = "/usr/bin/unzip";
    my $vol = "";
    my $dir = "";
    my $file = "";
    my $abspath = "";
    my $pkg = "";
    my $pkgname = "";
    my $pkgext = "";
    my @pkgs = ();
    my $test = 0;
    my $batch = 0;
    my $pkgaddCmd = "";
    my $pkgaddRc = 0;
    my $tarRc = 0;
    my $unzipRc = 0;
    my $rc = 0;
    my $tarCmd = "";
    my $unzipCmd = "";
    my $scriptfile = "";
    my $responsefile = "";
    my $adminfile = "";

    # see if any options were specified

    CASE_INSTALL_SVR4_MODES : {
	return -1 if (scalar(@_) < 1 || !defined($_[0]) || $_[0] eq "");

	if ($_[0] eq "test") {
	    $test = 1;
	    shift @_;
	    redo CASE_INSTALL_SVR4_MODES;
	}
	
	if ($_[0] eq "batch") {
	    $batch = 1;
	    shift @_;
	    redo CASE_INSTALL_SVR4_MODES;
	}
    }

    # create the list of pkgs to pass the taint check
    foreach $pkg (@_) {	
	if (defined($pkg) && -r $pkg && 
	    $pkg =~ /^([a-zA-Z0-9\-\_\/\.]+)$/) {
	    $pkg = $1;
	} else {
	    next;
	}

	($vol, $dir, $file) = File::Spec->splitpath($pkg);	

	if (defined($file) &&
	    $file =~ /^[a-zA-Z0-9]?[a-zA-Z0-9\_\-\.]+[a-zA-Z]$/) {
	    $abspath = File::Spec->rel2abs($pkg);	    
	    push (@pkgs, $abspath) 
		if (defined($abspath) && -r $abspath);
	}
    }
    
    return -1 if (scalar(@pkgs) < 1);

    # install the svr4 pkgs one at a time
    foreach $pkg (@pkgs) {
    $scriptfile = "$pkg.script";
    $responsefile = "$pkg.response";
    $adminfile = "$pkg.admin";

    if (-e $scriptfile) {
       $pkgaddCmd = "$scriptfile $pkg";
    }
	elsif ($pkg =~ /\.tar$/) {
	    ($vol, $dir, $file) = File::Spec->splitpath($pkg);
	    $pkgname = substr($file, 0, length($file) - 4);

	    $tarCmd = "(cd /tmp; $SOLARIS_TAR -xf $pkg ";
	    $tarCmd .= "> /dev/null " if ($VERBOSE == 0);
	    $tarCmd .= " 2>&1 )";

	    $tarRc = system($tarCmd);
	    $tarRc /= 256;
	    return $tarRc if ($tarRc > 0);
	    $pkgaddCmd = "$SVR4_PKGADD ";
        if (-e $responsefile) {
            $pkgaddCmd .= "-r $responsefile ";
        } else {
            $pkgaddCmd .= "-n ";
        }
        if (-e $adminfile) {
            $pkgaddCmd .= "-a $adminfile ";
        }
	    $pkgaddCmd .= "-d /tmp $pkgname";
	} elsif ($pkg =~ /\.zip$/) {
	    ($vol, $dir, $file) = File::Spec->splitpath($pkg);
	    $pkgname = substr($file, 0, length($file) - 4);

	    $unzipCmd = "$SOLARIS_UNZIP -d /tmp -oqq $pkg ";
	    $unzipCmd .= "> /dev/null " if ($VERBOSE == 0);
	    $unzipCmd .= " 2>&1";

	    $unzipRc = system($unzipCmd);
	    $unzipRc /= 256;
	    return $unzipRc if ($unzipRc > 0);

	    $pkgaddCmd = "$SVR4_PATCHADD /tmp/$pkgname";
	} else {
        if (-e $scriptfile) {
            $pkgaddCmd = "$scriptfile $pkg";
        } else {
	        $pkgaddCmd = "$SVR4_PKGADD ";
            if (-e $responsefile) {
                $pkgaddCmd .= "-r $responsefile ";
            } else {
                $pkgaddCmd .= "-n ";
            }
            if (-e $adminfile) {
                $pkgaddCmd .= "-a $adminfile ";
            }
	        $pkgaddCmd .= "-d $pkg all";
        }
    }

	$pkgaddCmd .= " > /dev/null " if ($VERBOSE == 0);
	$pkgaddCmd .= " 2>&1";

	$pkgaddRc = system($pkgaddCmd);
	$pkgaddRc /= 256;
	
	$rc = 1 if ($pkgaddRc > 0);
    }
    
    return $rc;
}


# installRPM - installs RPMs

sub installRPM
{
    my $vol = "";
    my $dir = "";
    my $file = "";
    my $abspath = "";
    my $pkg = "";
    my @pkgs = ();
    my $test = 0;
    my $batch = 0;
    my $upgrade = 0;
    my $scriptfile = "";
    my $rpmCmd = "$RPM ";
    my $rpmRc = 0;
    my $rc = 0;

    # see if any options were specified

    CASE_INSTALL_RPM_MODES : {
	return -1 if (scalar(@_) < 1 || !defined($_[0]) || $_[0] eq "");

	if ($_[0] eq "test") {
	    $test = 1;
	    shift @_;
	    redo CASE_INSTALL_RPM_MODES;
	}
	
	if ($_[0] eq "batch") {
	    $batch = 1;
	    shift @_;
	    redo CASE_INSTALL_RPM_MODES;
	}

	if ($_[0] eq "upgrade") {
	    $upgrade = 1;
	    shift @_;
	    redo CASE_INSTALL_RPM_MODES;
	}
    }

    # create the list of pkgs to pass the taint check
    foreach $pkg (@_) {	
	if (defined($pkg) && -r $pkg && 
	    $pkg =~ /^([a-zA-Z0-9\-\_\/\.]+)$/) {
	    $pkg = $1;
	} else {
	    next;
	}

	($vol, $dir, $file) = File::Spec->splitpath($pkg);	

	if (defined($file) &&
	    $file =~ /^[a-zA-Z0-9]?[a-zA-Z0-9\_\-\.]+[a-zA-Z]$/) {
	    $abspath = File::Spec->rel2abs($pkg);	    
	    push (@pkgs, $abspath) 
		if (defined($abspath) && -r $abspath);
	}
    }
    
    return -1 if (scalar(@pkgs) < 1);

    # if a batch install is requested, install all the rpms in a
    # single pass 

    if ($batch == 1) {
	$rpmCmd = "$RPM " . ($test == 0 ? "" : "--test ");
	$rpmCmd .= "--quiet " if ($VERBOSE == 0);
	$rpmCmd .= ($upgrade == 0 ? "-i" : "-U") . " @pkgs ";
	$rpmCmd .= "> /dev/null " if ($VERBOSE == 0);
	$rpmCmd .= " 2>&1";

	$rpmRc = system($rpmCmd);
	$rpmRc /= 256;

        return $rpmRc;
    } 
	
    # install the rpms one at a time

    foreach $pkg (@pkgs) {
    $scriptfile = "$pkg.script";
    if (-e $scriptfile) {
        $rpmCmd = "$scriptfile $pkg";
    } else {
	    $rpmCmd = "$RPM " . ($test == 0 ? "" : "--test ");
	    $rpmCmd .= "--quiet " if ($VERBOSE == 0);
	    $rpmCmd .= ($upgrade == 0 ? "-i" : "-U") . " $pkg ";
	    $rpmCmd .= "> /dev/null " if ($VERBOSE == 0);
	    $rpmCmd .= " 2>&1";
    }
	
	$rpmRc = system($rpmCmd);
	$rpmRc /= 256;
	
	$rc = 1 if ($rpmRc > 0);
    }
    
    return $rc;
}

# uninstallSVR4 - uninstalls SVR4 packages and Solaris patches

sub uninstallSVR4
{
    my $vol = "";
    my $dir = "";
    my $file = "";
    my $abspath = "";
    my $pkg = "";
    my @pkgs = ();
    my $test = 0;
    my $batch = 0;
    my $pkgrmCmd = "";
    my $pkgrmRc = 0;
    my $rc = 0;
    my $adminfile = "";

    # see if any options were specified

    CASE_INSTALL_RPM_MODES : {
	return -1 if (scalar(@_) < 1 || !defined($_[0]) || $_[0] eq "");

	if ($_[0] eq "test") {
	    $test = 1;
	    shift @_;
	    redo CASE_INSTALL_RPM_MODES;
	}
	
	if ($_[0] eq "batch") {
	    $batch = 1;
	    shift @_;
	    redo CASE_INSTALL_RPM_MODES;
	}
    }

    # create the list of pkgs to pass the taint check

    foreach $pkg (@_) {	
	if (defined($pkg) && $pkg =~ /^([a-zA-Z0-9\-\_\/\.]+)$/) {
	    $pkg = $1;
	    push (@pkgs, $pkg) 
	} else {
	    next;
	}
    }
    
    # uninstall the rpms one at a time

    foreach $pkg (@pkgs) {
	if ($pkg =~ /^[0-9\-]+$/) {
	    $pkgrmCmd = "/usr/sbin/patchrm $pkg"; 
	} else {
	    $pkgrmCmd = "/usr/sbin/pkgrm "; 
        $adminfile = "/tmp/$pkg.admin";
        if (-e $adminfile) {
            $pkgrmCmd .= "-a $adminfile ";
        }
	    $pkgrmCmd .= "-n $pkg"; 
	}
    $pkgrmCmd .= "> /dev/null " if ($VERBOSE == 0);
	$pkgrmCmd .= " 2>&1";

	$pkgrmRc = system($pkgrmCmd);
	$pkgrmRc /= 256;
	
	$rc = 1 if ($pkgrmRc > 0);
    }
    
    return $rc;
}

# uninstallRPM - uninstalls RPMs

sub uninstallRPM
{
    my $vol = "";
    my $dir = "";
    my $file = "";
    my $abspath = "";
    my $pkg = "";
    my @pkgs = ();
    my $test = 0;
    my $batch = 0;
    my $rpmCmd = "$RPM ";
    my $rpmRc = 0;
    my $rc = 0;

    # see if any options were specified

    CASE_INSTALL_RPM_MODES : {
	return -1 if (scalar(@_) < 1 || !defined($_[0]) || $_[0] eq "");

	if ($_[0] eq "test") {
	    $test = 1;
	    shift @_;
	    redo CASE_INSTALL_RPM_MODES;
	}
	
	if ($_[0] eq "batch") {
	    $batch = 1;
	    shift @_;
	    redo CASE_INSTALL_RPM_MODES;
	}
    }

    # create the list of pkgs to pass the taint check

    foreach $pkg (@_) {	
	if (defined($pkg) && $pkg =~ /^([a-zA-Z0-9\-\_\/\.]+)$/) {
	    $pkg = $1;
	    push (@pkgs, $pkg) 
	} else {
	    next;
	}
    }
    
    # if a batch uninstall is requested, uninstall all the rpms in a
    # single pass 

    if ($batch == 1) {
	$rpmCmd = "$RPM " . ($test == 0 ? "" : "--test ");
	$rpmCmd .= "--quiet " if ($VERBOSE == 0);
	$rpmCmd .= "-e @pkgs ";
	$rpmCmd .= "> /dev/null " if ($VERBOSE == 0);
	$rpmCmd .= " 2>&1";

	$rpmRc = system($rpmCmd);
	$rpmRc /= 256;

    return $rpmRc;
    } 
	
    # uninstall the rpms one at a time

    foreach $pkg (@pkgs) {
	$rpmCmd = "$RPM " . ($test == 0 ? "" : "--test ");
	$rpmCmd .= "--quiet " if ($VERBOSE == 0);
	$rpmCmd .= "-e $pkg ";
	$rpmCmd .= "> /dev/null " if ($VERBOSE == 0);
	$rpmCmd .= " 2>&1";

	$rpmRc = system($rpmCmd);
	$rpmRc /= 256;
	
	$rc = 1 if ($rpmRc > 0);
    }
    
    return $rc;
}


# checkRPM - checks if the specified RPMs are installed

sub checkRPM
{
    my $pkg = "";
    my $rpmRc = 0;
    my $state = "";
    
    return -1 if (scalar(@_) < 0);

    foreach $pkg (@_) {
	if (defined($pkg) && 
	    $pkg =~ /^([a-zA-Z0-9]?[a-zA-Z0-9\_\-\.]+)$/) {
	    $pkg = $1;
	    $rpmRc = system("$RPM -q --quiet $pkg > /dev/null 2>&1");
	    $rpmRc /= 256;
	    if ($rpmRc > 0) {
		$state = $PKGSTATES{'NOTINSTALLED'};
	    } else {
		$state = $PKGSTATES{'INSTALLED'};
	    }
	} else {
	    $state = $PKGSTATES{'INVALID'};
	}

	print "$pkg $state\n";
    }

    return 0;
}

# printError - prints an error message to STDERR

sub printError
{
    print STDERR "ERROR: @_\n";
}

# printUsage - prints the usage statement

sub printUsage
{
    print STDERR <<EOUSAGE;
usage: pkg_install.pl [batch] [test] [pkgs]
       pkg_install.pl check [pkgs]
       pkg_install.pl uninstall [batch] [test] [pkgs]
EOUSAGE
}
