#!/usr/bin/perl -w

# Copyright (c) 2003 Sun Microsystems, Inc. All rights reserved
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.


#
# Package manager for managed appliances
# Author: Phil Ploquin (phil.ploquin@sun.com)
# $Id: pkgMgr.pl,v 1.8 2004/09/21 19:17:15 ms152511 Exp $
#

use strict;

#
# Usage
#
sub printUsage ($;$)
{
  my $needToRelease = shift @_;
  my $complaint     = shift @_;

  print STDERR 'Error : ' . $complaint . "\n" if (defined($complaint));

  print STDERR<<EOD;

Usage: $0 <locking> [<mode> <pkg 1> [pkg 2] ... [pkg n]]

The first field given may be used to set and unset locks:
'lock' obtains a lock at startup and keeps it there after the program exits.
'unlock' releases the lock upon program exit.
'full' obtains a lock at startup and releases upon exit.
'none' doesn't bother with locks.

  mode         description
  ----         -----------
  check        Check whether or not the specified package(s) are already
               installed.  Each row returned is the reference count for
               each package.
  add          Install the specified package(s) if necessary, increment
               reference count(s).
  del          Uninstall the specified package(s) if the reference count(s)
               are at 1, and decrease the reference counts regardless.
  purge        Uninstall all installed packages and reset the reference
               counts to 0.

For all modes except check, each row returned is a code and the package name:
code = 0 : Successfully installed / uninstalled package.
code > 0 : Increased / decreased reference counts (no install or uninstall.)
code < 0 : Error, third field here is reason.

Packages can be RPMs, SVR4 package data streams, and more in the future.

EOD

  releaseLock() if ($needToRelease);
  exit 1;
}

#
# Constants
#
my $filePath             = '/usr/mgmt/etc';
my $referenceFile        = 'pkg_count';
my $typeFile             = 'pkg_type';
my $lockFile             = 'pkg_lock';
my $lockTimeout          = 30;  # seconds till the current lock is regarded
                                # as bogus and released.
my $pkgPath              = '/var/csadmin';
my $mkdirPath            = '/bin/mkdir';
$mkdirPath = '/usr/bin/mkdir' if (! -x $mkdirPath);
my $mkdirCmd             = $mkdirPath . ' -p';
my $headCmd              = '/usr/bin/head';
$headCmd = '/bin/head' if (! -x $headCmd);

#
# Pkg installation, removal constants, functions for different types of pkgs
#

my $defaultType = 'rpm';

my %pkgUtil =
(
  'rpm' =>
  {
    'init_good'     => 0,
    'ident_pkg_fn'  => \&isRPM,
    'pkg_name_fn'   => \&rpmPkgName,
    'install_fn'    => \&installRPM,
    'uninstall_fn'  => \&uninstallRPM,
    'check_fn'      => \&checkRPM,
    'install_cmd'   => '/bin/rpm -i --force',
    'uninstall_cmd' => '/bin/rpm -e --allmatches --nodeps',
    'check_cmd'     => '/bin/rpm -q'
  },
  'svr4' =>
  {
    'init_good'     => 0,
    'ident_pkg_fn'  => \&isSvr4Pkg,
    'pkg_name_fn'   => \&svr4PkgName,
    'install_fn'    => \&installSvr4Pkg,
    'uninstall_fn'  => \&uninstallSvr4Pkg,
    'check_fn'      => \&checkSvr4Pkg,
    'install_cmd'   => '/usr/mgmt/libexec/yesit.sh /usr/sbin/pkgadd -d',
    'uninstall_cmd' => '/usr/mgmt/libexec/yesit.sh /usr/sbin/pkgrm',
    'check_cmd'     => '/usr/bin/pkginfo'
  }
);

#
# Main
#
printUsage(0) if (!defined($ARGV[0]));
my $locking = shift @ARGV;

printUsage( 0, 'Unknown locking option: ' . $locking )
  if (($locking ne 'lock') && ($locking ne 'unlock')
   && ($locking ne 'full') && ($locking ne 'none'));

system ($mkdirCmd . ' ' . $filePath) if (! -d $filePath);

obtainLock()
  if (($locking eq 'lock') || ($locking eq 'full'));

my $mode = shift @ARGV || 0;
if ($mode)
{
  my ($pkgCounts, $pkgTypes) = loadPkgCounts();

  if ($mode eq 'check')
  {
    printUsage( ($locking eq 'unlock') || ($locking eq 'full'),
                'Need to specify at least one package.' )
        if (!defined($ARGV[0]));
    checkPkgs($pkgCounts, $pkgTypes, \@ARGV);
  }
  elsif ($mode eq 'add')
  {
    printUsage( ($locking eq 'unlock') || ($locking eq 'full'),
                'Need to specify at least one package.' )
        if (!defined($ARGV[0]));
    addPkgs($pkgCounts, $pkgTypes, \@ARGV);
  }
  elsif ($mode eq 'del')
  {
    printUsage( ($locking eq 'unlock') || ($locking eq 'full'),
                'Need to specify at least one package.' )
        if (!defined($ARGV[0]));
    delPkgs($pkgCounts, $pkgTypes, \@ARGV);
  }
  elsif ($mode eq 'purge')
  {
    purgePkgs($pkgCounts, $pkgTypes);
  }
  else
  {
    printUsage( ($locking eq 'unlock') || ($locking eq 'full'),
                'Unknown mode : ' . $mode );
  }

  savePkgCounts($pkgCounts, $pkgTypes);
}

releaseLock()
  if (($locking eq 'unlock') || ($locking eq 'full'));

exit 0;

#------------------------------------------------------------------
#                       utility
#------------------------------------------------------------------
sub printError
{
  my $errorMsg = shift;

  print 'ERROR  ' . $errorMsg . "\n";
  releaseLock();
  exit 2;
}

sub runCmd
{
  my ($cmd, $pkg, $printCode) = @_;
  $printCode = 1 if (! defined($printCode));

  $cmd .= ' >/dev/null 2>/dev/null';
  my $rc = system($cmd);
  $rc *= -1 if ($rc > 0);
  return $rc if (!$printCode);

  print $rc . ' ' . $pkg . "\n";

  return 0;
}

sub getPkgName
{
  my ($type, $pkg) = @_;

  return $pkg if ($type eq '');
  return &{$pkgUtil{$type}->{'pkg_name_fn'}}($pkg) || $pkg;
}

#------------------------------------------------------------------
#------------------------------------------------------------------

#
#
#
sub getPkgType
{
  my ($pkgName, $pkgTypes) = @_;

  return $pkgTypes->{$pkgName}
    if (exists($pkgTypes->{$pkgName}));

  my @t;
  my $theType = '';
  foreach my $type (keys %pkgUtil)
  {
    if (&{$pkgUtil{$type}->{'ident_pkg_fn'}}($pkgName))
    {
      if (!$pkgUtil{$type}->{'init_good'})
      {
        foreach my $t (qw(install_cmd uninstall_cmd check_cmd))
        {
          @t = split(/ /, $pkgUtil{$type}->{$t});
          if (! -x $t[0])
          {
            printError("Can't find $t[0] on this system for $pkgName");
          }
        }
        $pkgUtil{$type}->{'init_good'} = 1;
      }
      $theType = $type;
      last;
    }
  }

  $theType = $defaultType if ($theType eq '');

  # Save the type
  $pkgTypes->{getPkgName($theType, $pkgName)} = $theType;

  return $theType;
}

#
#
#
sub obtainLock
{
  if ( -f $filePath . '/' . $lockFile)
  {
    my $fileAge = time() - (stat($filePath . '/' . $lockFile))[9];
    if ($fileAge >= $lockTimeout)
    {
      # Something bad must have happened to the last lock setter,
      # chances are they're never coming back
      releaseLock();
    }
    else
    {
      # Wait till the lock is freed or take it over after time expires
      while ($fileAge < $lockTimeout)
      {
        last if ( ! -f $filePath . '/' . $lockFile);
        sleep(1);
        ++$fileAge;
      }
      if ( -f $filePath . '/' . $lockFile)
      {
        # Too bad
        releaseLock();
      }
    }
  }

  open (LOCK_FILE, '>' . $filePath . '/' . $lockFile)
    || printError('Could not create lock file');
  close (LOCK_FILE);
}

#
#
#
sub releaseLock
{
  if ( -f $filePath . '/' . $lockFile)
  {
    unlink($filePath . '/' . $lockFile);
  }
}

#
#
#
sub loadPkgCounts
{
  my %pkgCounts = ();
  my %pkgTypes = ();

  return (\%pkgCounts, \%pkgTypes)
    if (!open (PKG_COUNT_FILE, $filePath . '/' . $referenceFile));

  my (@t, $line);
  while ($line = <PKG_COUNT_FILE>)
  {
    @t = split (/\s+/, $line);
    printError('Reference count file is corrupt')
      if (!defined($t[0]) || !defined($t[1]) || defined($t[2]) || ($t[1] < 0));
    $pkgCounts{$t[0]} = $t[1];
  }
  close (PKG_COUNT_FILE);

  return (\%pkgCounts, \%pkgTypes)
    if (!open (PKG_TYPE_FILE, $filePath . '/' . $typeFile));

  while ($line = <PKG_TYPE_FILE>)
  {
    @t = split (/\s+/, $line);
    printError('Pkg type file is corrupt')
      if (!defined($t[0]) || !defined($t[1]) || defined($t[2]));
    $pkgTypes{$t[0]} = $t[1];
  }
  close (PKG_TYPE_FILE);

  return (\%pkgCounts, \%pkgTypes);
}

#
#
#
sub savePkgCounts
{
  my ($pkgCounts, $pkgTypes) = @_;

  return 0 if (!$pkgCounts);

  # Make sure the counts are all true
  my ($p, $type, $pkg);
  foreach $p (sort keys %{$pkgCounts})
  {
    $type = getPkgType($p, $pkgTypes);
    $pkg = getPkgName($type, $p);
    verifyCount($pkgCounts, $type, $pkg);
  }

  printError('Unable to open ' . $filePath . '/' . $referenceFile . ' for writing')
    if (!open(PKG_COUNT_FILE, '>' . $filePath . '/' . $referenceFile));

  foreach $p (sort keys %{$pkgCounts})
  {
    if ($pkgCounts->{$p} > 0)
    {
      printf PKG_COUNT_FILE "%-60s %2d\n", $p, $pkgCounts->{$p};
    }
    elsif ( -f $filePath . '/pkg_contents/' . $p)
    {
      unlink($filePath . '/pkg_contents/' . $p);
    }
  }

  close (PKG_COUNT_FILE);

  printError('Unable to open ' . $filePath . '/' . $typeFile . ' for writing')
    if (!open(PKG_TYPE_FILE, '>' . $filePath . '/' . $typeFile));

  foreach $p (sort keys %{$pkgTypes})
  {
    printf PKG_TYPE_FILE "%-60s %s\n", $p, $pkgTypes->{$p}
      if (exists($pkgCounts->{$p}) && ($pkgCounts->{$p} > 0));
  }

  close (PKG_TYPE_FILE);
}

#
#
#
sub verifyCount
{
  my ($pkgCounts, $type, $pkg) = @_;

  return 0 if ($type eq '');
  my ($cmd, $fpkg) = &{$pkgUtil{$type}->{'check_fn'}}($pkg);
  my $installed = !(runCmd($cmd, $fpkg, 0));
  if ($installed)
  {
    if (!exists($pkgCounts->{$pkg}) || ($pkgCounts->{$pkg} == 0))
    {
      $pkgCounts->{$pkg} = 1;
    }
  }
  else
  {
    $pkgCounts->{$pkg} = 0;
  }

  return 0;
}

#------------------------------------------------------------------
#                          modes
#------------------------------------------------------------------

#
# checkPkgs : print the new pkgs that are not yet installed
#             (have reference counts of zero.)  If the package
#             is actually installed while its count is 0,
#             set the count to 1.
#
sub checkPkgs
{
  my ($pkgCounts, $pkgTypes, $pkgs) = @_;

  my ($tpkg, $pkgName, $type);
  foreach $tpkg (@{$pkgs})
  {
    $type = getPkgType($tpkg, $pkgTypes);
    $pkgName = getPkgName($type, $tpkg);
    verifyCount($pkgCounts, $type, $pkgName);
    print $pkgCounts->{$pkgName} . ' ' . $pkgName . "\n";
  }

  return 0;
}

#
# add
#
sub addPkgs
{
  my ($pkgCounts, $pkgTypes, $newPkgs) = @_;

  my ($type, $pkg, $pkgName, $cmd, $fpkg);
  foreach $pkg (@{$newPkgs})
  {
    $type = getPkgType($pkg, $pkgTypes);
    $pkgName = getPkgName($type, $pkg);
    verifyCount($pkgCounts, $type, $pkgName);

    if ($pkgCounts->{$pkgName} == 0)
    {
      if ( ! -r $pkgPath . '/' . $pkg)
      {
        printError("Can't find $pkgPath/$pkg to install");
      }

      # First, try to uninstall all other versions, nevermind errors
      # cuz some old mapp rpms that never get uninstalled can interfere
      # with new ones being installed.  grrrr.

      ($cmd, $fpkg) = &{$pkgUtil{$type}->{'uninstall_fn'}}($pkgName);
      runCmd($cmd, $fpkg, 0);

      # Now install it
      ($cmd, $fpkg) = &{$pkgUtil{$type}->{'install_fn'}}($pkg);
      if (!runCmd($cmd, $fpkg))
      {
        $pkgCounts->{$pkgName} = 1;
      }
    }
    else
    {
      ++$pkgCounts->{$pkgName};
      print '1 ' . $pkg . "\n";
    }
  }

  return 0;
}

#
# del
#
sub delPkgs
{
  my ($pkgCounts, $pkgTypes, $pkgs) = @_;

  my ($tpkg, $type, $pkg, $cmd, $fpkg);
  foreach $tpkg (@{$pkgs})
  {
    $type = getPkgType($tpkg, $pkgTypes);
    $pkg = getPkgName($type, $tpkg);
    verifyCount($pkgCounts, $type, $pkg);
    if ($pkgCounts->{$pkg} == 1)
    {
      ($cmd, $fpkg) = &{$pkgUtil{$type}->{'uninstall_fn'}}($pkg);
      runCmd($cmd, $fpkg);
      $pkgCounts->{$pkg} = 0;
    }
    elsif ($pkgCounts->{$pkg} > 1)
    {
      --$pkgCounts->{$pkg};
      print '1 ' . $pkg . "\n";
    }
    else
    {
      print '-1 ' . $pkg . " Not installed\n";
    }
  }

  return 0;
}

#
# purge
#
sub purgePkgs
{
  my ($pkgCounts, $pkgTypes) = @_;

  my ($tpkg, $type, $pkg, $cmd, $fpkg);
  my @pkgs = keys %{$pkgCounts};
  foreach $tpkg (@pkgs)
  {
    $type = getPkgType($tpkg, $pkgTypes);
    $pkg = getPkgName($type, $tpkg);
    verifyCount($pkgCounts, $type, $pkg);

    ($cmd, $fpkg) = &{$pkgUtil{$type}->{'uninstall_fn'}}($pkg);
    runCmd($cmd, $fpkg);
    $pkgCounts->{$pkg} = 0;
  }

  return 0;
}

#---------------------------------------------------------

#------------------ RPM pkg fns -------------------

#
#
#
sub initForRPMs
{
  my @rpmrcPaths =
  (
    '/usr/lib/rpm/rpmrc',
    '/usr/lib/rpmrc'
  );
  my $rpmRc = '';

  my $tRc;
  foreach $tRc (@rpmrcPaths)
  {
    if ( -r $tRc )
    {
      $rpmRc = $tRc;
      last;
    }
  }
  printError("Can't find a rpmrc anywhere in " . join(' , ', @rpmrcPaths))
    if ($rpmRc eq '');
  foreach my $t (qw(install_cmd uninstall_cmd check_cmd))
  {
    $pkgUtil{'rpm'}->{$t} .= ' --rcfile ' . $rpmRc;
  }
}

#
#
#
sub isRPM
{
  my $pkgName = shift;

  return 0 if ($pkgName !~ /\.rpm$/);
  initForRPMs() if (!$pkgUtil{'rpm'}->{'init_good'});
  return 1;
}

#
#
#
sub rpmPkgName
{
  my $pkg = shift;

  # Get rid of the <arch>.rpm suffix
  my ($pkgName) = ($pkg =~ /^(.*)\.[^\.]+\.rpm$/);
  return $pkgName;
}

#
#
#
sub installRPM
{
  my $pkg = shift;

  return ($pkgUtil{'rpm'}->{'install_cmd'} . ' ' . $pkgPath . '/' . $pkg, $pkg);
}

#
#
#
sub uninstallRPM
{
  my $pkg = shift;

  my ($noVersionName) = ($pkg =~ /^(.*)\-[0-9]+\.[0-9]+\-[0-9]+$/);
  if (defined($noVersionName))
  {
    return ($pkgUtil{'rpm'}->{'uninstall_cmd'} . ' ' . $noVersionName, $noVersionName);
  }
  printError('Badly named RPM : ' . $pkg . ', expecting version at end');
}

#
#
#
sub checkRPM
{
  my $pkg = shift;

  return ($pkgUtil{'rpm'}->{'check_cmd'} . ' ' . $pkg, $pkg);
}

#-------------- SVR4 Pkg fns ------------

#
#
#
sub isSvr4Pkg
{
  my $pkg = shift;

  return 0 if ( ! -f $pkgPath . '/' . $pkg); # need the pkg for the test
  my $cmd = $headCmd . ' -1 ' . $pkgPath . '/' . $pkg;
  my $firstLine = `$cmd`;
  return 1 if ($firstLine =~ /PaCkAgE DaTaStReAm/);

  return 0;
}

#
#
#
sub svr4PkgName
{
  my $pkg = shift;

  # nothing special
  return $pkg;
}

#
#
#
sub installSvr4Pkg
{
  my $pkg = shift;

  system ($mkdirCmd . ' ' . $filePath . '/pkg_contents')
    if (! -d $filePath . '/pkg_contents');
  open (CONTENT_FILE, '>' . $filePath . '/pkg_contents/' . $pkg)
    || printError('Could not create content file!');
  my $cmd = $pkgUtil{'svr4'}->{'check_cmd'} . ' -d ' . $pkgPath . '/' . $pkg;
  my @lines = split (/\n/, `$cmd`);
  my @t;
  foreach my $line (@lines)
  {
    @t = split(/\s+/, $line);
    print CONTENT_FILE $t[1] . "\n";
  }
  close (CONTENT_FILE);

  return ($pkgUtil{'svr4'}->{'install_cmd'} . ' ' . $pkgPath . '/' . $pkg . ' all', $pkg);
}

#
#
#
sub uninstallSvr4Pkg
{
  my $pkg = shift;

  return ($pkgUtil{'svr4'}->{'uninstall_cmd'} . getPkgContents($pkg), $pkg);
}

#
#
#
sub checkSvr4Pkg
{
  my $pkg = shift;

  return ($pkgUtil{'svr4'}->{'check_cmd'} . getPkgContents($pkg), $pkg);
}

#
#
#
sub getPkgContents
{
  my $pkg = shift;

  return $pkg if (!open (CONTENT_FILE, $filePath . '/pkg_contents/' . $pkg));

  my $line;
  my $pkgStr = '';
  while ($line = <CONTENT_FILE>)
  {
    chomp($line);
    $pkgStr .= ' ' . $line;
  }
  close (CONTENT_FILE);

  if (($pkgStr eq '') || ($pkgStr =~ /^\s+$/))
  {
    # So that pkginfo doesn't get invoked with no args...
    unlink($filePath . '/pkg_contents/' . $pkg);
    return $pkg;
  }

  return $pkgStr;
}
